'''
 ====================================================================
 Copyright (c) 2003-2005 Barry A Scott.  All rights reserved.

 This software is licensed as described in the file LICENSE.txt,
 which you should have received as part of this distribution.

 ====================================================================

    wb_list_panel.py

'''
import wx
import wb_exceptions
import wb_ids
import wb_shell_commands
import wb_dialogs
import os
import sys
import cPickle

class ListSortData:
    def __init__( self, order=1, field=0 ):
        self.order = order
        self.field = field

    def setField( self, field ):
        if self.field == field:
            # toggle order
            self.order *= -1
        else:
            # new field forward
            self.field = field
            self.order = 1

    def getField( self ):
        return self.field

    def getOrder( self ):
        return self.order


class ListItemState:
    def __init__( self ):
        self.modified = False
        self.versioned = False
        self.new_versioned = False
        self.unversioned = False
        self.need_checkin = False
        self.conflict = False
        self.file_exists = False
        self.is_project_parent = False

    def printState( self, title='' ):
        print '-----------',title
        for item in self.__dict__.items():
            print 'ListState: %s -> %r' % item

class WbListPanel(wx.Panel):
    ''' WbListPanel '''

    def __init__( self, app, frame, parent ):
        wx.Panel.__init__(self, parent, -1)

        self.app = app
        self.frame = frame
        try_wrapper = wb_exceptions.TryWrapperFactory( self.app.log )

        self.filter_field = 'Name'
        self.filter_text = ''

        self.header_panel = HeaderPanel( self, app, self.filter_field )
        self.header_panel.setFilterChangedHandler( self.OnFilterChanged )

        self.v_sizer = wx.BoxSizer( wx.VERTICAL )

        self.id_list = wx.NewId()
        self.list_ctrl = wx.ListCtrl( self, self.id_list,
                        wx.DefaultPosition,
                        wx.DefaultSize,
                        wx.LC_REPORT|wx.NO_BORDER )


        self.v_sizer.Add( self.header_panel, 0, wx.EXPAND|wx.ALL, 0 )
        self.v_sizer.Add( self.list_ctrl, 1, wx.EXPAND|wx.ALL, 0 )

        if wx.Platform == '__WXMAC__':
            acc_init =[
                (wx.ACCEL_ALT, ord('A'), wb_ids.id_SP_Add),
                (wx.ACCEL_ALT, ord('D'), wb_ids.id_SP_DiffWorkBase),
                (wx.ACCEL_ALT, ord('E'), wb_ids.id_File_Edit),
                (wx.ACCEL_ALT, ord('L'), wb_ids.id_SP_History),
                (wx.ACCEL_ALT, ord('I'), wb_ids.id_SP_Info),
                (wx.ACCEL_ALT, ord('P'), wb_ids.id_SP_Properties),
                (wx.ACCEL_ALT, ord('R'), wb_ids.id_SP_Revert),
                (wx.ACCEL_ALT, ord('U'), wb_ids.id_SP_Update),
                (wx.ACCEL_ALT, 8, wb_ids.id_SP_Delete),
                (wx.ACCEL_NORMAL, wx.WXK_RETURN, wb_ids.id_Return_Hotkey),
                (wx.ACCEL_ALT, ord('O'), wb_ids.id_Shell_Open),
                ]
        else:
            acc_init =[
                (wx.ACCEL_CTRL, ord('A'), wb_ids.id_SP_Add),
                (wx.ACCEL_CTRL, ord('D'), wb_ids.id_SP_DiffWorkBase),
                (wx.ACCEL_CTRL, ord('E'), wb_ids.id_File_Edit),
                (wx.ACCEL_CTRL, ord('H'), wb_ids.id_SP_History),
                (wx.ACCEL_CTRL, ord('I'), wb_ids.id_SP_Info),
                (wx.ACCEL_CTRL, ord('P'), wb_ids.id_SP_Properties),
                (wx.ACCEL_CTRL, ord('R'), wb_ids.id_SP_Revert),
                (wx.ACCEL_CTRL, ord('U'), wb_ids.id_SP_Update),
                (wx.ACCEL_NORMAL, wx.WXK_DELETE, wb_ids.id_SP_Delete),
                (wx.ACCEL_NORMAL, wx.WXK_RETURN, wb_ids.id_Return_Hotkey),
                ]
            if wx.Platform == '__WXWIN__':
                acc_init.append( (wx.ACCEL_CTRL, ord('O'), wb_ids.id_Shell_Open) )

        acc_tab = wx.AcceleratorTable( acc_init )
        self.list_ctrl.SetAcceleratorTable( acc_tab )

        wx.EVT_SIZE( self, try_wrapper( self.OnSize ) )

        wx.EVT_LIST_COL_CLICK( self.list_ctrl, self.id_list, self.app.eventWrapper( self.OnColClick ))

        wx.EVT_LIST_ITEM_ACTIVATED( self.list_ctrl, self.id_list, self.app.eventWrapper( self.OnItemActivated ) )

        wx.EVT_LIST_ITEM_RIGHT_CLICK( self.list_ctrl, self.id_list, self.app.eventWrapper( self.OnRightClick ) )
        if wx.Platform in ['__WXMSW__','__WXMAC__']:
            wx.EVT_LIST_BEGIN_DRAG( self.list_ctrl, self.id_list, self.app.eventWrapper( self.OnDragBegin ) )
        #wx.EVT_CHAR( self.list_ctrl, self.OnChar )

        wx.EVT_LIST_ITEM_SELECTED( self.list_ctrl, self.id_list, self.OnItemSelected )
        wx.EVT_LIST_ITEM_DESELECTED( self.list_ctrl, self.id_list, self.OnItemDeselected )

        wx.EVT_SET_FOCUS( self.list_ctrl, self.OnSetFocus )
        wx.EVT_KILL_FOCUS( self.list_ctrl, self.OnKillFocus )  

        self.SetAutoLayout( True )
        self.SetSizer( self.v_sizer )
        self.v_sizer.Fit( self )
        self.Layout()

        view_prefs = self.app.prefs.getView()
        self.sort_data = ListSortData( view_prefs.sort_order, view_prefs.sort_field )

        self.list_handler = None

        self.last_char_timestamp = 0
        self.inc_search_string = ''

    def OnSetFocus( self, event ):
        self.frame.setEventHandler( self )

    def OnKillFocus( self, event ):
        #self.frame.clearEventHandler()
        pass

    def OnItemSelected( self, event ):
        self.frame.clearUpdateUiState()
        self.frame.setEventHandler( self )

    def OnItemDeselected( self, event ):
        self.frame.clearUpdateUiState()
        self.frame.setEventHandler( self )

    def OnDragBegin( self, event ):
        #print 'WbListPanel.OnDragBegin'

        all_filenames = [self.list_handler.getFilename( row ) for row in self.getSelectedRows()]

        # create our own data format and use it in a
        # custom data object
        svn_data = wx.CustomDataObject(wx.CustomDataFormat("WorkBench.svn_wc_path"))
        svn_data.SetData( cPickle.dumps( all_filenames ) )


        # Now make a data object for the text and also a composite
        # data object holding both of the others.
        text_data = wx.TextDataObject( 'the text of the list ctrl file name' )

        data = wx.DataObjectComposite()
        data.Add( svn_data )
        data.Add( text_data )

        # And finally, create the drop source and begin the drag
        # and drop opperation
        src = wx.DropSource( self )
        src.SetData( data )
        #print 'Begining DragDrop'
        src.DoDragDrop( wx.Drag_AllowMove )
        #print 'DragDrop completed: %d' % result

    def savePreferences( self ):
        view_prefs = self.app.prefs.getView()
        view_prefs.sort_order = self.sort_data.getOrder()
        view_prefs.sort_field = self.sort_data.getField()

    def getProjectInfo( self ):
        if self.list_handler:
            return self.list_handler.getProjectInfo()
        return None

    def clearHandler( self ):
        self.list_handler = None
        # empty the list
        self.list_ctrl.DeleteAllItems()

    def updateHandler( self ):
        if self.list_handler:
            self.list_handler.setupColumns()
            self.list_handler.updateStatus()
        self.drawList()

    def setHandler( self, list_handler ):
        self.list_handler = list_handler
        self.list_handler.setupColumns()
        self.list_handler.updateStatus()

        self.filter_field = ''
        self.header_panel.clearFilterText()

        self.drawList()

    def drawList( self ):
        # empty the list
        self.list_ctrl.DeleteAllItems()

        if self.list_handler:
            self.list_handler.initList( self.sort_data, self.filter_field, self.filter_text )

    def getHandler( self ):
        return self.list_handler

    def updateHeader( self, url_name, path_name ):
        self.header_panel.updateHeader( url_name, path_name )

    def OnFilterChanged( self, field, text ):
        self.filter_field = field
        self.filter_text = text
        self.drawList()

    def OnSize( self, event ):
        w, h = self.GetClientSizeTuple()
        self.v_sizer.SetDimension( 0, 0, w, h )

    def OnItemActivated(self, event):
        if not self.list_handler:
            return

        for row in self.getSelectedRows():
            filename = self.list_handler.getFilename( row )
            if self.list_handler.mayOpen( row ):
                self.app.selectTreeNode( filename )

            elif not os.path.isdir( filename ):
                wb_shell_commands.EditFile( self.app, self.list_handler.getProjectInfo(), filename )

    def isTreeHandler( self ):
        return False

    def isListHandler( self ):
        return True

    def getUpdateUiState( self ):
        if self.list_handler is None:
            return None
        return self.list_handler.getState( self.getSelectedRows() )

    def OnRightClick( self, event ):
        if not self.list_handler:
            return

        self.frame.getUpdateUiState()
        if True:
            menu = self.list_handler.getContextMenu()
            self.list_ctrl.PopupMenu( menu, event.GetPoint() )
            menu.Destroy()

    def OnColClick( self, event ):
        self.sort_data.setField( self.list_handler.getColumnId( event.m_col ) )
        if not self.list_handler:
            return

        self.list_handler.sortList( self.sort_data )

    def OnChar( self, event ):
        timestamp = event.GetTimestamp()
        delta = timestamp - self.last_char_timestamp

        if delta > 2000:
            self.inc_search_string = ''

        search_again = False
        if wx.WXK_SPACE <= event.GetKeyCode() < wx.WXK_DELETE and not event.HasModifiers():
            self.inc_search_string += chr(event.GetKeyCode())
        elif event.GetKeyCode() == wx.WXK_BACK:
            self.inc_search_string = self.inc_search_string[0:-1]
        elif event.GetKeyCode() == ord('g')&31:    # Ctrl-G
            search_again = True
        else:
            event.Skip()
            self.inc_search_string = ''

        self.last_char_timestamp = timestamp

        self.incrementalSearch( self.inc_search_string, search_again )


    def incrementalSearch( self, text, search_again ):
        #
        #    find all filenames that contain the text
        #    notice all selected items
        #    at end select one item which is:
        #        if one selected at start the first that contains string starting at the selected one
        #        else the first that contains the string from the start of the list
        #
        ###print 'incrementalSearch: "%s" %r' % (text, search_again)

        if text == '':
            self.frame.setSearch( '' )
            return

        selected_items = []
        matching_items = []
        for item_index in range( self.list_ctrl.GetItemCount() ):
            if self.list_ctrl.GetItemState( item_index, wx.LIST_STATE_SELECTED ):
                ###print 'incrementalSearch %d selected' % item_index
                selected_items.append( item_index )

            row = self.list_ctrl.GetItemData( item_index )
            filename = self.list_handler.getFilename( row )
            if text in os.path.basename( filename ):
                ###print 'incrementalSearch %d %s in %s' % (item_index, text, filename)
                matching_items.append( item_index )

        ###print 'matching_items %r' % matching_items
        ###print 'selected_items %r' % selected_items
        if len(matching_items) == 0:
            # nothing to do - bell maybe?
            self.frame.setSearch( 'Not found - "%s"' % self.inc_search_string )
            return

        new_select = None
        if len(selected_items) == 1:
            # use first item starting at the selected item
            for item_index in matching_items:
                if search_again:
                    if item_index > selected_items[0]:
                        new_select = item_index
                        break
                else:
                    if item_index >= selected_items[0]:
                        new_select = item_index
                        break
        if new_select is None:
            new_select = matching_items[0]

        for item_index in selected_items:
            if item_index != new_select:
                ###print 'incrementalSearch deselect %d, ' % item_index,
                self.list_ctrl.SetItemState( item_index, 0, wx.LIST_STATE_SELECTED|wx.LIST_STATE_FOCUSED )
                ###print self.list_ctrl.GetItemState( item_index, wx.LIST_STATE_SELECTED|wx.LIST_STATE_FOCUSED )

        self.list_ctrl.SetItemState( new_select,
            wx.LIST_STATE_SELECTED|wx.LIST_STATE_FOCUSED,
            wx.LIST_STATE_SELECTED|wx.LIST_STATE_FOCUSED )
        self.list_ctrl.EnsureVisible( new_select )
        ###print 'incrementalSearch new_select=%d' % new_select
        self.frame.setSearch( 'Found "%s"' % self.inc_search_string )

    # command handlers
    def OnFileEdit( self ):
        if not self.list_handler:
            return

        for row in self.getSelectedRows():
            filename = self.list_handler.getFilename( row )

            wb_shell_commands.EditFile( self.app, self.list_handler.getProjectInfo(), filename )

    def OnShellOpen( self ):
        if not self.list_handler:
            return

        for row in self.getSelectedRows():
            filename = self.list_handler.getFilename( row )

            wb_shell_commands.ShellOpen( self.app, self.list_handler.getProjectInfo(), filename )

    def OnSpAdd( self ):
        return self.Sp_Dispatch( 'Cmd_File_Add' )

    def OnSpAnnotate( self ):
        return self.Sp_Dispatch( 'Cmd_File_Annotate' )

    def OnSpCheckin( self ):
        return self.Sp_Dispatch( 'Cmd_File_Checkin' )

    def OnSpCleanup( self ):
        return self.Sp_Dispatch( 'Cmd_File_Cleanup' )

    def OnSpDelete( self ):
        return self.Sp_Dispatch( 'Cmd_File_Delete' )

    def OnSpDiffWorkBase( self ):
        return self.Sp_Dispatch( 'Cmd_File_DiffWorkBase' )

    def OnSpDiffWorkHead( self ):
        return self.Sp_Dispatch( 'Cmd_File_DiffWorkHead' )

    def OnSpDiffMineNew( self ):
        return self.Sp_Dispatch( 'Cmd_File_DiffMineNew' )

    def OnSpDiffOldMine( self ):
        return self.Sp_Dispatch( 'Cmd_File_DiffOldMine' )

    def OnSpDiffOldNew( self ):
        return self.Sp_Dispatch( 'Cmd_File_DiffOldNew' )

    def OnSpHistory( self ):
        return self.Sp_Dispatch( 'Cmd_File_History' )

    def OnSpInfo( self ):
        return self.Sp_Dispatch( 'Cmd_File_Info' )

    def OnSpProperties( self ):
        return self.Sp_Dispatch( 'Cmd_File_Properties' )

    def OnSpRename( self ):
        return self.Sp_Dispatch( 'Cmd_File_Rename' )

    def OnSpRevert( self ):
        return self.Sp_Dispatch( 'Cmd_File_Revert' )

    def OnSpResolved( self ):
        return self.Sp_Dispatch( 'Cmd_File_Resolved' )

    def OnSpUpdate( self ):
        return self.Sp_Dispatch( 'Cmd_File_Update' )

    def Sp_Dispatch( self, sp_func_name ):
        self.app.trace.info( 'WbListPanel.Sp_Dispatch( %s ) event' % sp_func_name )
        if not self.list_handler:
            return

        sp_func = getattr( self.list_handler, sp_func_name )

        self.app.trace.info( 'WbListPanel.Sp_Dispatch( %s ) calling' % sp_func_name )
        return sp_func( self.getSelectedRows() )

    def getSelectedRows( self ):
        all_rows = []
        item_index = -1
        while True:
            item_index = self.list_ctrl.GetNextItem( item_index, wx.LIST_NEXT_ALL, wx.LIST_STATE_SELECTED )
            if item_index < 0:
                break
            all_rows.append( self.list_ctrl.GetItemData( item_index ) )

        all_rows.sort()

        return all_rows
        
class HeaderPanel(wx.Panel):
    ''' HeaderPanel '''

    def __init__( self, parent, app, filter_field ):
        wx.Panel.__init__(self, parent, -1)

        self.app = app


        self.background_colour = wx.SystemSettings.GetColour( wx.SYS_COLOUR_3DFACE )

        self.v_sizer = wx.BoxSizer( wx.VERTICAL )
        self.h_sizer1 = wx.BoxSizer( wx.HORIZONTAL )
        self.h_sizer2 = wx.BoxSizer( wx.HORIZONTAL )

        self.url_text_ctrl = wx.TextCtrl( self, 0, '', style=wx.TE_READONLY )
        self.path_text_ctrl = wx.TextCtrl( self, 0, '', style=wx.TE_READONLY )

        self.filter_changed_handler = None

        self.filter_field_choices = ['Name','Author']
        self.filter_choice_ctrl = wx.Choice( self, wx.NewId(), choices=self.filter_field_choices )
        self.filter_choice_ctrl.SetSelection( self.filter_field_choices.index( filter_field ) )
        if wx.Platform == '__WXMAC__':
            self.filter_text_ctrl = wx.TextCtrl( self, wx.NewId(), '', size=(-1,-1) )
        else:
            self.filter_text_ctrl = wx.TextCtrl( self, wx.NewId(), '', size=(-1,10) )
        self.filter_clear_button = wx.Button( self, wx.NewId(), 'X', style=wx.BU_EXACTFIT, size=(30, -1) )

        # share the space 50/50
        border = 3
        self.h_sizer1.Add( self.url_text_ctrl, 50, wx.EXPAND|wx.ALL, border )
        self.h_sizer1.Add( self.path_text_ctrl, 50, wx.EXPAND|wx.TOP|wx.BOTTOM|wx.RIGHT, border )

        border = 1
        self.h_sizer2.Add( self.filter_choice_ctrl, 0, wx.EXPAND|wx.LEFT|wx.RIGHT, border )
        if wx.Platform == '__WXMAC__':
            self.h_sizer2.Add( self.filter_text_ctrl, 1, wx.EXPAND|wx.ALL, border )
        else:
            self.h_sizer2.Add( self.filter_text_ctrl, 1, wx.EXPAND|wx.ALL, border+2 )
        self.h_sizer2.Add( self.filter_clear_button, 0, wx.EXPAND|wx.LEFT|wx.RIGHT, border )

        border = 3
        self.v_sizer.Add( self.h_sizer1, 0, wx.EXPAND|wx.ALL, 0 )
        self.v_sizer.Add( self.h_sizer2, 1, wx.EXPAND|wx.ALL, 0 )

        wx.EVT_BUTTON( self, self.filter_clear_button.GetId(), self.OnClearFilterText )
        wx.EVT_TEXT( self, self.filter_text_ctrl.GetId(), self.OnFilterTextChanged )
        wx.EVT_CHOICE( self, self.filter_choice_ctrl.GetId(), self.OnFilterTypeChanged )

        self.SetAutoLayout( True )
        self.SetSizer( self.v_sizer )
        self.v_sizer.Fit( self )
        self.Layout()

    def setFilterChangedHandler( self, handler ):
        self.filter_changed_handler = handler

    def __callFilterChangedHandler( self ):
        self.filter_changed_handler(
            self.filter_field_choices[ self.filter_choice_ctrl.GetSelection() ],
            self.filter_text_ctrl.GetValue() )

    def updateHeader(self, url_name, path_name ):
        if url_name is None:
            url_name = ''
        if path_name is None:
            path_name = ''

        self.url_text_ctrl.SetValue( url_name )
        self.path_text_ctrl.SetValue( path_name )

        self.SetBackgroundColour( self.background_colour )
        self.Refresh()

    def clearFilterText( self ):
        self.filter_text_ctrl.Clear()

    def OnClearFilterText( self, event=None ):
        self.filter_text_ctrl.Clear()
        self.__callFilterChangedHandler()

    def OnFilterTypeChanged( self, event ):
        self.filter_text_ctrl.Clear()
        self.__callFilterChangedHandler()

    def OnFilterTextChanged( self, event ):
        self.__callFilterChangedHandler()

class ListHandler:
    def __init__( self, list_panel ):
        self.list_panel = list_panel

    def initList( self, sort_data ):
        raise wb_exceptions.InternalError( 'initList not implemented' )

    def sortList( self, sort_data ):
        raise wb_exceptions.InternalError( 'sortList not implemented' )
