/*
 *  Copyright (C) 2008 Nicolas Vion <nico@picapo.net>
 *
 *   This file is part of swac-explore.
 *
 *   Swac-explore is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   Swac-explore is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with swac-explore; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <iostream>
#include <gtkmm/aboutdialog.h>
#include <sigc++/compatibility.h>
#include <gtkmm/stock.h>
#include <giomm.h>

#include "explorer.hh"
#include "macros.inc"


Explorer::Explorer(SqlDatabase *_db, Gst *_gst, Glib::RefPtr<Gnome::Conf::Client> _gconf) {
	db = _db;
	gst = _gst;
	gconf = _gconf;

	SELECT_FIELD_INIT;

	combobox2->signal_changed().connect(SigC::slot(*this, &Explorer::on_combobox2_changed), false);
	entry3->signal_key_press_event().connect(SigC::slot(*this, &Explorer::search_key_press), false);
	entry3->signal_focus_out_event().connect(SigC::slot(*this, &Explorer::search_focus_out), false);
	treeview1->signal_key_press_event().connect(SigC::slot(*this, &Explorer::view_key_press), false);
	treeview2->signal_key_press_event().connect(SigC::slot(*this, &Explorer::view_key_press), false);
	toolbutton1->signal_clicked().connect(SigC::slot(*this, &Explorer::play), false);

	toolbutton5->signal_clicked().connect(SigC::slot(*this, &Explorer::on_about), false);
	toolbutton2->signal_clicked().connect(SigC::slot(*this, &Explorer::on_quitter1_activate), false);
	combobox1->signal_changed().connect(SigC::slot(*this, &Explorer::on_combobox1_changed), false);
	treeview2->signal_cursor_changed().connect(SigC::slot(*this, &Explorer::on_treeview2_cursor_changed), false);
	treeview1->signal_cursor_changed().connect(SigC::slot(*this, &Explorer::on_treeview1_cursor_changed), false);

	show_lang(gconf->get_string(GCONF_LANG));

	if (combobox1->get_active_row_number() == -1)
		combobox1->set_active(0);	
}

void Explorer::show_lang(Glib::ustring selected) { 
	store_lang->clear();

	SqlQuery query(db);
	query.prepare("SELECT DISTINCT SWAC_LANG, languages.title FROM sounds INNER JOIN languages ON SWAC_LANG=languages.iso_639_3 ORDER BY SWAC_LANG;");

	while (query.step() == SQLITE_ROW) {
		Gtk::TreeRow row = *(store_lang->append());
		row[listext.value] = query.columnText(0);
		row[listext.caption] = query.columnText(1);
		if (row[listext.value] == selected)
			combobox1->set_active(row);
	}	
}

void Explorer::show_sound_info(Glib::ustring idx) { 
	SqlQuery query(db);
	query.prepare("SELECT " SOUND_INFO_FIELDS " FROM sounds INNER JOIN packages ON sounds.packages_idx=packages.idx WHERE sounds.idx=?;");	
	query.bind(idx, 1);
	while (query.step() == SQLITE_ROW)
		SOUND_INFO_SET(query);	
	query.clearBindings();
}

void Explorer::on_treeview1_cursor_changed() {
	if (Gtk::TreeModel::iterator iter = treeview1->get_selection()->get_selected()) {
  		Gtk::TreeModel::Row row = *iter;
		show_sound_info(row[listview.idx]);
	}
}

int Explorer::show_sounds(Glib::ustring filter) { 
	store_sounds->clear();

	SqlQuery query(db);
	query.prepare(
		"SELECT " SOUNDS_LIST_FIELDS " FROM sounds "
		"INNER JOIN packages ON sounds.packages_idx=packages.idx "
		"WHERE " + filter + " "
		"ORDER BY SWAC_LANG, SWAC_TEXT;"
	);

	int results = 0;
	while (query.step() == SQLITE_ROW) {
		results++;
		SOUNDS_LIST_SET(query);
	}
	return results;
}

void Explorer::show_index_alphaidx(Glib::ustring lang) { 
	store_index->clear();

	SqlQuery query(db);
	query.prepare("SELECT DISTINCT str FROM alphaidx WHERE lang=? ORDER BY str;");
	query.bind(lang, 1);
	while (query.step() == SQLITE_ROW) {
		Gtk::TreeRow row = *(store_index->append());
		row[listbox.caption] = query.columnText(0);
	}
}

void Explorer::show_index_else(Glib::ustring filter, Glib::ustring field) { 
	store_index->clear();

	SqlQuery query(db);
	query.prepare("SELECT `" + field + "` FROM sounds WHERE " + filter + " GROUP BY `" + field + "` ORDER BY `" + field + "`");
	while (query.step() == SQLITE_ROW) {
		Gtk::TreeRow row = *(store_index->append());
		row[listbox.caption] = query.columnText(0);
	}
}

void Explorer::select_index_alphaidx(Glib::ustring lang, Glib::ustring str) {
	store_sounds->clear();

	std::string sql(
		"SELECT " SOUNDS_LIST_FIELDS " "
		"FROM alphaidx "
		"INNER JOIN sounds ON sounds.idx=alphaidx.sounds_idx "
		"INNER JOIN packages ON packages.idx=alphaidx.packages_idx "
		"WHERE str=? AND lang=? "
		"ORDER BY SWAC_LANG,SWAC_TEXT;"
	);

	SqlQuery query(db);
	query.prepare(sql);
	query.bind(str, 1);
	query.bind(lang, 2);
	while (query.step() == SQLITE_ROW)
		SOUNDS_LIST_SET(query);
}

void Explorer::on_treeview2_cursor_changed() {
	if (Gtk::TreeModel::iterator iter = treeview2->get_selection()->get_selected()) {
  		Gtk::TreeModel::Row row = *iter;
		if (user_index_field == "SWAC_ALPHAIDX")
			select_index_alphaidx(user_lang, row[listbox.caption]);
		else
			show_sounds("`" + user_index_field + "`=" + db->str(row[listbox.caption]) + " AND `SWAC_LANG`=" + db->str(user_lang));

		treeview1->set_cursor(Gtk::TreePath(1));
	}
}

void Explorer::show_index() {
	if (user_index_field == "SWAC_ALPHAIDX")
		show_index_alphaidx(user_lang.c_str());
	else
		show_index_else("`SWAC_LANG`=" + db->str(user_lang), user_index_field);
}

void Explorer::on_combobox1_changed() {
	if (Gtk::TreeModel::iterator iter = combobox1->get_active()) {
		Gtk::TreeModel::Row row = *iter;

		user_lang = row[listext.value];
		gconf->set(GCONF_LANG, row[listext.value]);

		show_index();
	}
}

void Explorer::on_combobox2_changed() {
	if (Gtk::TreeModel::iterator iter = combobox2->get_active()) {
		Gtk::TreeModel::Row row = *iter;

		user_index_field = row[listext.value];
		show_index();
	}
}

void Explorer::search(Glib::ustring str) {
	if (show_sounds("`SWAC_LANG`=" + db->str(user_lang) + " AND `SWAC_TEXT` LIKE " + db->str("%" + str + "%")) != 0) 
		treeview1->grab_focus();
	else
		tool_search_icon->set(Gtk::StockID(Gtk::Stock::NO), Gtk::ICON_SIZE_MENU);
}

bool Explorer::search_focus_out(GdkEventFocus* event) {
	tool_search_icon->set(Gtk::StockID(Gtk::Stock::FIND), Gtk::ICON_SIZE_MENU);
	return false;
}

bool Explorer::search_key_press(GdkEventKey* event) {
	switch (event->keyval) {
		case GDK_Return: 
			if (event->state & GDK_CONTROL_MASK)
				append_missing(user_lang, entry3->get_text());
			else
				search(entry3->get_text());
			return true;
	}
	tool_search_icon->set(Gtk::StockID(Gtk::Stock::FIND), Gtk::ICON_SIZE_MENU);
	return false;
}

void Explorer::play() {  
	if (Gtk::TreeModel::iterator iter = treeview1->get_selection()->get_selected()) {
  		Gtk::TreeModel::Row row = *iter;
		statusbar1->push("Playing " + row[listview.filename]);
		std::string path(row[listview.path] + row[listview.filename]);
		gst->play_url(path.c_str());
		statusbar1->push("");
	}
}


bool Explorer::view_key_press(GdkEventKey* event) {
	switch (event->keyval) {
		case GDK_Return: 
			play();
			return true;
		case GDK_Left:
		case GDK_Right:
			if (treeview1->has_focus())
				treeview2->grab_focus();
			else
				treeview1->grab_focus();
			return true;
	}
	return false;
}

void Explorer::on_quitter1_activate() {  
	Gtk::Main::quit();
}

void Explorer::append_missing(Glib::ustring const lang, Glib::ustring const str) {
	try {
		Glib::ustring filename("missing_" + lang + ".txt");
		Glib::RefPtr<Gio::File> file = Gio::File::create_for_path(Glib::get_home_dir() + G_DIR_SEPARATOR_S ".swac" G_DIR_SEPARATOR_S + filename);
		Glib::RefPtr<Gio::FileOutputStream> out = file->append_to();
		out->write(str + "\n");
		out->close();
		statusbar1->push("\"" + str + "\" as been added to \"" + filename + "\"");
	}
	catch(const Glib::Exception& ex) {
		std::cerr << "Exception caught: " << ex.what () << std::endl; 
	}
}

void launch_browser(Gtk::AboutDialog& dialog, const Glib::ustring& url) {
	std::string command ("xdg-open '");
	command += url;
	command += "'";
	system(command.c_str());
}

void Explorer::on_about() {  
	Gtk::AboutDialog about;
	about.set_name("SWAC Explore");
	about.set_version(VERSION);
	about.set_logo(Gtk::Widget::render_icon(Gtk::StockID(Gtk::Stock::ABOUT), Gtk::IconSize(2)));
	about.set_website_label("Shtooka Project Home Page");
	about.set_url_hook(&launch_browser);
	about.set_website("http://shtooka.net/");
	about.set_license(
"This program is free software; you can redistribute it and/or modify\n\
it under the terms of the GNU General Public License as published by\n\
the Free Software Foundation; either version 2 of the License, or\n\
(at your option) any later version.\n\
\n\
This program is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n\
GNU General Public License for more details.\n\
\n\
You should have received a copy of the GNU General Public License\n\
along with this program; if not, write to the Free Software\n\
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA"
);
	//std::list<Glib::ustring> list_authors;
	//list_authors.push_back("Vion Nicolas");
	//about.set_authors(list_authors);
	about.set_copyright("Copyright© 2008 Vion Nicolas");
	about.set_comments("Audio collection of words (SWAC) explorer");
	about.run();
}


