/**
 * \file SwamiUISpanWin.h
 * \brief User interface piano and key/velocity spans header file
 * 
 * A GUI object that contains a piano, key/velocity range window and
 * root key pointer strip.
 */

/*
 * Swami
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Swami homepage: http://swami.sourceforge.net
 */
#ifndef __SWAMIUI_SPANWIN_H__
#define __SWAMIUI_SPANWIN_H__

#include <gtk/gtk.h>
#include <instpatch.h>

#include <libswami/SwamiMidi.h>

#include "widgets/piano.h"
#include "widgets/keyspan.h"

#include "swamidll.h"

#define SWAMIUI_SPANWIN_PIANO_LOW_NUMKEYS	17
#define SWAMIUI_SPANWIN_PIANO_HI_NUMKEYS	20
#define SWAMIUI_SPANWIN_PIANO_TOTAL_NUMKEYS	37


typedef struct _SwamiUISpanWin SwamiUISpanWin;
typedef struct _SwamiUISpanWinClass SwamiUISpanWinClass;

#define SWAMIUI_TYPE_SPANWIN   (swamiui_spanwin_get_type ())
#define SWAMIUI_SPANWIN(obj) \
  (GTK_CHECK_CAST ((obj), SWAMIUI_TYPE_SPANWIN, SwamiUISpanWin))
#define SWAMIUI_SPANWIN_CLASS(klass) \
  (GTK_CHECK_CLASS_CAST ((klass), SWAMIUI_TYPE_SPANWIN, SwamiUISpanWinClass))
#define SWAMIUI_IS_SPANWIN(obj) \
  (GTK_CHECK_TYPE ((obj), SWAMIUI_TYPE_SPANWIN))
#define SWAMIUI_IS_SPANWIN_CLASS(klass) \
  (GTK_CHECK_CLASS_TYPE ((klass), SWAMIUI_TYPE_SPANWIN))

/** spanwin mode */
typedef enum
{
  SWAMIUI_SPANWIN_PIANO,
  SWAMIUI_SPANWIN_VELOCITY
} SwamiUISpanWinMode;

/** Swami SpanWin Object */
struct _SwamiUISpanWin
{
  /* private */
  GtkHBox parent;		/* derived from GtkHBox */

  int mode;			/* current mode (SWAMIUI_SPANWIN_PIANO or
				   SWAMIUI_SPANWIN_VELOCITY) */
  IPItem *item;			/* patch item to sync to */

  SwamiMidi *midi;		/* midi object piano is connected to */
  int octave;			/* current piano octave */
  int velocity;			/* current piano velocity */
  gboolean pianokeys[128];	/* active piano keys */

  GtkWidget *ptrstrip;		/* pointer strip widget above piano */
  GtkWidget *piano;		/* piano widget */
  GtkWidget *velbar;		/* velocity bar widget */
  GtkWidget *velbar_box;	/* velocity bar box */
  GtkWidget *view_box;		/* viewport box */
  GtkWidget *span_list;		/* GTK list widget with spans */
};

struct _SwamiUISpanWinClass
{
  GtkScrolledWindowClass parent_class;

  void (*zone_select)(SwamiUISpanWin *spanwin, IPItem *item);
  void (*zone_unselect)(SwamiUISpanWin *spanwin, IPItem *item);
};

SWAMI_API extern guint swamiui_spanwin_default_keytable[];

SWAMI_API guint swamiui_spanwin_get_type (void);
SWAMI_API GtkWidget *swamiui_spanwin_new (void);
SWAMI_API void swamiui_spanwin_set_mode (SwamiUISpanWin *spanwin,
					 SwamiUISpanWinMode mode);
SWAMI_API void swamiui_spanwin_set_item (SwamiUISpanWin *spanwin,
					 IPItem *item);
SWAMI_API void swamiui_spanwin_select_keyspan (SwamiUISpanWin *spanwin,
					       IPZone *zone);
SWAMI_API void swamiui_spanwin_unselect_keyspan (SwamiUISpanWin *spanwin,
						 IPZone *zone);
SWAMI_API void swamiui_spanwin_unselect_all_keyspans (SwamiUISpanWin *spanwin);
SWAMI_API GList *
swamiui_spanwin_get_keyspan_selection (SwamiUISpanWin *spanwin);

SWAMI_API void swamiui_spanwin_piano_set_octave (SwamiUISpanWin *spanwin,
						 int octave);
SWAMI_API void swamiui_spanwin_piano_set_velocity (SwamiUISpanWin *spanwin,
						   int velocity);
SWAMI_API int swamiui_spanwin_piano_key_to_note (SwamiUISpanWin *spanwin,
						 int key);
SWAMI_API void swamiui_spanwin_piano_note_on (SwamiUISpanWin *spanwin,
					      int note);
SWAMI_API void swamiui_spanwin_piano_note_off (SwamiUISpanWin *spanwin,
					       int note);

SWAMI_API guint *swamiui_spanwin_get_keytable (SwamiUISpanWin *spanwin);
SWAMI_API void swamiui_spanwin_update_keytable (void);
SWAMI_API int swamiui_spanwin_parse_octkeys (char *str, guint **keys);
SWAMI_API char *swamiui_spanwin_encode_octkeys (guint *keyvals, int keycount);

#endif
