/**
 * \file <instpatch.h>
 * \brief Header for libinstpatch - A library for instrument patch files
 *
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 */
#ifndef __INSTPATCH_H_DEF__
#define __INSTPATCH_H_DEF__

#include <glib.h>
#include <stdarg.h>

/* for wonderful #I*$@& windows */
#if defined(INSTP_DLL_EXPORT)
#  define INSTP_API __declspec(dllexport)
#elif defined(INSTP_DLL_IMPORT)
#  define INSTP_API __declspec(dllimport)
#else
#  define INSTP_API
#endif

/* forward type declarations */

typedef struct _IPItemTypeInfo IPItemTypeInfo;
typedef struct _IPItem IPItem;
typedef struct _IPSFontVersion IPSFontVersion;
typedef struct _IPSFontInfo IPSFontInfo;
typedef struct _IPMod IPMod;
typedef union _IPGenAmount IPGenAmount;
typedef struct _IPGen IPGen;
typedef struct _IPZone IPZone;
typedef struct _IPSample IPSample;
typedef struct _IPInst IPInst;
typedef struct _IPPreset IPPreset;
typedef struct _IPFileInfo IPFileInfo;
typedef struct _IPSFont IPSFont;
typedef struct _IPGenConv IPGenConv;
typedef struct _IPGenInfo IPGenInfo;
typedef struct _IPSampleData IPSampleData;
typedef struct _IPSampleMethod IPSampleMethod;
typedef struct _IPSampleStore IPSampleStore;


#define INSTP_OK 0
#define INSTP_FAIL 1

/**
 * \defgroup IPGlobal Sound Font Interface
 * Sound Font Interface
 * \{
 */

/** sound font item (IPItem) types */
typedef enum
{
  IPITEM_NONE,			/* an sfitem "NULL" value */
  IPITEM_SFONT,			/* a IPSFont (sound font) item */
  IPITEM_PRESET,		/* a IPPreset item */
  IPITEM_INST,			/* a IPInst (instrument) item */
  IPITEM_SAMPLE,		/* a IPSample item */
  IPITEM_SAMPLE_DATA,		/* a IPSampleData item */
  IPITEM_ZONE,			/* a IPZone item */
  IPITEM_VBANK,			/* a virtual bank */
  IPITEM_VBANK_MAP,		/* a virtual bank preset map  */
  IPITEM_COUNT			/* count of item types */
}
IPItemType;

/**
 * \brief Item allocation function
 * \param type Sound font item type to allocate memory for
 * \return A pointer to the allocated space or NULL on error
 *
 * Function should allocate a zero '\0' filled memory area for an instance
 * of an item. Allows for more effecient memory allocation mechanisms or
 * allocation of additional space for custom variables.
 */
typedef IPItem * (* IPItem_AllocFunc)(IPItemType type);

/**
 * \brief Item free function
 * \param item Item to free instance memory of.
 *
 * Should free the memory allocated by the IPItem_AllocFunc function.
 */
typedef void (* IPItem_FreeFunc)(IPItem *item);

/**
 * \brief Item function for item type specific initialization
 * \param item Item to initialize
 * \return Function should return INSTP_OK on success, INSTP_FAIL otherwise
 *
 * Should do item type specific initialization only (i.e. not IPItem related
 * stuff, which has been done already when this function is called).
 */
typedef int (* IPItem_InitFunc)(IPItem *item);

/**
 * \brief Item function called when there are no more references to it
 * \param item Item without a reference.
 *
 * Responsible for freeing all resources of the item except its instance
 * structure.
 */
typedef void (* IPItem_DoneFunc)(IPItem *item);

/**
 * \brief Item destroy function
 * \param item Item to destroy
 * Function to destroy an item by unlinking it and its children.
 * Memory is not freed by this function, only when each item's
 * reference count reaches zero is its #IPItem_FreeFunc function called.
 */
typedef void (* IPItem_DestroyFunc)(IPItem *item);

/**
 * \brief Item duplicate function
 * \param item Item to duplicate
 * \return Duplicated item or NULL on error
 */
typedef IPItem * (* IPItem_DuplicateFunc)(const IPItem *item);

/** sound font item type info (one for each item type) */
struct _IPItemTypeInfo
{
  char *name;			/* descriptive name of item type */
  int instance_size;		/* item instance size */
  IPItem_AllocFunc alloc;   /* function to allocate an instance of an item */
  IPItem_FreeFunc free;		/* function to free an instance of an item */
  IPItem_InitFunc init;		/* function called for item specific
				   initialization (can be NULL). */
  IPItem_DoneFunc done;		/* function called when there are no more
				   references to item (can be NULL). */
  IPItem_DestroyFunc destroy;	/* function to destroy an item (or NULL) */
  IPItem_DuplicateFunc duplicate; /* function to duplicate item (or NULL) */
};

/** base object that Presets, Instruments, Samples and Zones are
 *  "derived" from
 */
struct _IPItem
{
  guint8 type;			/* #IPItemType */
  guint8 flags;			/* reserved flags */
  guint8 user_flags;		/* user definable flags */
  guint8 reserved;		/* reserved and should be 0 */
  guint32 ref_count;		/* reference count */
  void *userptr;		/* user definable pointer */
  IPItem *next;			/* next item in this list */
  IPItem *prev;			/* previous item in this list */
  IPItem *parent;		/* parent of this item */
  void *mutex;			/* multi-thread locking variable */
};

/** Flags an item as floating. An item is flagged as floating when it is
    created and it is given a ref_count of 1. When the item is parented,
    #instp_item_sink is called to clear this flag and decrement the ref_count.
    Prevents accidental freeing of item after creation. */
#define IPITEM_FLAG_FLOAT	1 << 0

/** sound font version info (for #IPINFO_VERSION and #IPINFO_ROM_VERSION) */
struct _IPSFontVersion
{
  guint16 major;		/* major version number */
  guint16 minor;		/* minor version number */
};

/** Info item */
struct _IPSFontInfo
{
  char id[4];			/* 4 character ID (not zero terminated!) */
  char *val;			/* value of this info item */
  IPSFontInfo *next;		/* next info item in list */
};

/** Modulator item, real time control of a generator (#IPGen) */
struct _IPMod
{
  guint16 src;			/* source modulator (MIDI controller, etc) */
  guint16 dest;			/* destination generator */
  gint16 amount;		/* degree of modulation */
  guint16 amtsrc;		/* second source controls amount of first */
  guint16 trans;		/* transform function applied to source */
  IPMod *next;		/* next modulator in list */
};

/** Generator amount (effect parameter amount) */
union _IPGenAmount
{
  gint16 sword;			/* signed 16 bit value */
  guint16 uword;		/* unsigned 16 bit value */
  struct
  {
    guint8 low;			/* low value of range */
    guint8 high;		/* high value of range */
  }
  range;			/* range values, low - high */
};

/** Generator (effect parameter) */
struct _IPGen
{
  guint16 id;			/* generator #IPGenType ID */
  IPGenAmount amount;	/* generator value */
  IPGen *next;		/* next generator in list */
};

/** Zone item for Presets or Instruments (#IPPreset or #IPInst) */
struct _IPZone
{
  IPItem sfitem;		/* "derived" from #IPItem */
  IPItem *refitem;		/* ref'd item (#IPInst *,
				   #IPSample *, or NULL) */
  IPMod *mod;		/* list of modulators */
  IPGen *gen;		/* list of generators */
};

/** Sample item */
struct _IPSample
{
  IPItem sfitem;		/* "derived" from #IPItem */
  IPSampleData *sampledata;	/* sample data object */
  char *name;			/* name of sample */
  guint32 loopstart;		/* loop start offset (in samples) */
  guint32 loopend;		/* loop end offset (in samples, first sample
				   AFTER loop actually) */
  guint32 samplerate;		/* sample rate */
  guint8 origpitch;		/* root midi key number */
  gint8 pitchadj;		/* pitch correction in cents */
  guint16 sampletype;		/* INSTP_SAMPLETYPE_* flags (bit 1 mono,
				   2 right, 4 left, 8 linked, 0x8000 ROM) */
  IPSample *linked;		/* linked sample pointer or NULL */
};

/** Instrument item */
struct _IPInst
{
  IPItem sfitem;		/* "derived" from #IPItem */
  char *name;			/* name of instrument */
  IPZone *zone;		/* list of instrument zones */
};

/** Preset item */
struct _IPPreset
{
  IPItem sfitem;		/* "derived" from IPItem */
  char *name;			/* name of preset */
  guint16 psetnum;		/* MIDI preset map number */
  guint16 bank;			/* MIDI bank map number */
  IPZone *zone;		/* list of preset zones */

  guint32 library;		/* Not used (preserved) */
  guint32 genre;		/* Not used (preserved) */
  guint32 morphology;		/* Not used (preserved) */
};

/** sound font file info */
struct _IPFileInfo
{
  int ref_count;		/* reference count for this file handle */
  int fd;			/* sound font file descriptor or -1 if
				     not open */
  guint32 sample_pos;		/* position in file of the sample data */
  guint32 sample_size;		/* sample data chunk size (in samples) */
};

/** Sound font data object */
struct _IPSFont
{
  IPItem sfitem;		/* "derived" from IPItem */
  IPSFontVersion version;	/* sound font version */
  IPSFontVersion rom_version;	/* ROM version */

  guint32 flag_changed   : 1;	/* Sound font changed flag */
  guint32 flag_saved     : 1;	/* Sound font been saved ever? */

  IPFileInfo *file;	/* sound font file related information
				     (should never be NULL) */
  char *file_name;		/* current file name or NULL if not set */
  IPSFontInfo *info;		/* list of #IPSFontInfo structures */
  IPPreset *preset;	/* list of #IPPreset structures */
  IPInst *inst;		/* list of #IPInst structures */
  IPSample *sample;	/* list of #IPSample structures */
};

/** generator unit conversion info structure */
struct _IPGenConv
{
  int (*user2sf) (float val);	/* function for user units -> sfont units */
  float (*sf2user) (IPGenAmount val); /* function for
					      sfont units -> user units */
  int (*ofs2sf) (float val);	/* function for user ofs units -> sfont
				   units */
  float (*sf2ofs) (IPGenAmount val); /* function for sfont units -> user
					     ofs units */
  int digits;			/* digits to display after decimal point */
  char *unit_text;		/* user unit label text */
  char *ofs_text;		/* user unit offset label text */
};

/** generator info and constraints structure */
struct _IPGenInfo
{
  gint16 min;			/* minimum value allowed */
  gint16 max;			/* maximum value allowed */
  gint16 def;			/* default value */
  gint16 unit;			/* #IPUnit type */
  char *label;			/* descriptive label */
};

/* type casting macros (FIXME: check type as well as cast) */

/** cast to #IPItem * */
#define INSTP_ITEM(obj)  ((IPItem *)(obj))
/** cast to #IPSFont * */
#define INSTP_SFONT(obj)  ((IPSFont *)(obj))
/** cast to #IPPreset * */
#define INSTP_PRESET(obj)  ((IPPreset *)(obj))
/** cast to #IPInst * */
#define INSTP_INST(obj)  ((IPInst *)(obj))
/** cast to #IPSample * */
#define INSTP_SAMPLE(obj)  ((IPSample *)(obj))
/** cast to #IPZone */
#define INSTP_ZONE(obj)  ((IPZone *)(obj))
/** cast to #IPSampleData */
#define INSTP_SAMPLE_DATA(obj)  ((IPSampleData *)(obj))

/** the user defined pointer in an #IPItem object */
#define INSTP_USERPTR(item) (INSTP_ITEM (item)->userptr)

/** checks if an item is a particular type */
#define INSTP_CHECK_TYPE(item, itemtype) (INSTP_ITEM (item)->type == itemtype)

/** checks if an item has a parent and that it is of a particular type,
    will issue a warning if item has no parent (FIXME) */
#define INSTP_CHECK_PARENT_TYPE(item, parenttype) \
  (INSTP_ITEM (item)->parent && INSTP_ITEM (item)->parent->type == parenttype)

/** check if item is of type IPITEM_SFONT */
#define INSTP_IS_SFONT(item) (INSTP_CHECK_TYPE (item, IPITEM_SFONT))
/** check if item is of type IPITEM_PRESET */
#define INSTP_IS_PRESET(item) (INSTP_CHECK_TYPE (item, IPITEM_PRESET))
/** check if item is of type IPITEM_INST */
#define INSTP_IS_INST(item) (INSTP_CHECK_TYPE (item, IPITEM_INST))
/** check if item is of type IPITEM_SAMPLE */
#define INSTP_IS_SAMPLE(item) (INSTP_CHECK_TYPE (item, IPITEM_SAMPLE))
/** check if item is of type IPITEM_ZONE */
#define INSTP_IS_ZONE(item) (INSTP_CHECK_TYPE (item, IPITEM_ZONE))
/** check if item is of type IPITEM_SAMPLE_DATA */
#define INSTP_IS_SAMPLE_DATA(item) (INSTP_CHECK_TYPE (item,IPITEM_SAMPLE_DATA))

/** check if item's parent is of type IPITEM_SFONT */
#define INSTP_IS_PARENT_SFONT(item) \
    (INSTP_CHECK_PARENT_TYPE (item, IPITEM_SFONT))
/** check if item's parent is of type IPITEM_PRESET */
#define INSTP_IS_PARENT_PRESET(item) \
    (INSTP_CHECK_PARENT_TYPE (item, IPITEM_PRESET))
/** check if item's parent is of type IPITEM_INST */
#define INSTP_IS_PARENT_INST(item) \
    (INSTP_CHECK_PARENT_TYPE (item, IPITEM_INST))

/** Compare two #IPMod structures to see if they are identical and
    can be combined (all fields except amount must be identical).
    Returns TRUE if identical, FALSE otherwise */
#define IPMOD_ARE_IDENTICAL(a, b)   \
   (a->src == b->src && a->dest == b->dest && a->amtsrc == b->amtsrc  \
    && a->trans == b->trans)

/** known sound font file chunk ID enumerations */
typedef enum
{
  IPCHUNK_UNKNOWN = 0,		/* unknown chunk ("NULL" value) */
  IPCHUNK_RIFF = 1,		/* "RIFF" chunk */
  IPCHUNK_LIST = 2,		/* "LIST" chunk */

  IPCHUNK_SFBK = 3,		/* "sfbk" sound font chunk */

  /* 3 main sound font chunks */
  IPCHUNK_INFO = 4,		/* "INFO" chunk */
  IPCHUNK_SDTA = 5,		/* "sdta" sample DATA chunk */
  IPCHUNK_PDTA = 6,		/* "pdta" Preset/Instrument/Sample parameters */

  IPCHUNK_SMPL = 7,		/* "smpl" sample data subchunk in SDTA */

  /* preset IDs */
  IPCHUNK_PHDR = 8,		/* "phdr" preset header (#IPFilePHDR) */
  IPCHUNK_PBAG = 9,		/* "pbag" preset bag (zone) (#IPFileBag) */
  IPCHUNK_PMOD = 10,		/* "pmod" preset modulator (#IPFileMod) */
  IPCHUNK_PGEN = 11,		/* "pgen" preset generator (#IPFileGen) */

  /* instrument IDs */
  IPCHUNK_IHDR = 12,		/* "inst" instrument header (#IPFileIHDR) */
  IPCHUNK_IBAG = 13,		/* "ibag" instrument bag (zone) (#IPFileBag) */
  IPCHUNK_IMOD = 14,		/* "imod" instrument modulator (#IPFileMod) */
  IPCHUNK_IGEN = 15,		/* "igen" instrument generator (#IPFileGen) */

  IPCHUNK_SHDR = 16,		/* "shdr" sample header */

  /* sound font INFO sub chunk IDs */
  IPINFO_VERSION = 17,		/* "ifil" sound font version */
  IPINFO_ENGINE = 18,		/* "isng" target sound engine */
  IPINFO_NAME = 19,		/* "INAM" sound font name */
  IPINFO_ROM_NAME = 20,		/* "irom" ROM name */
  IPINFO_ROM_VERSION = 21,	/* "iver" ROM version */
  IPINFO_DATE = 22,		/* "ICRD" creation date */
  IPINFO_AUTHOR = 23,		/* "IENG" sound designers and engineers */
  IPINFO_PRODUCT = 24,		/* "IPRD" product sound font is intended for */
  IPINFO_COPYRIGHT = 25,	/* "ICOP" copyright message */
  IPINFO_COMMENT = 26,		/* "ICMT" comments */
  IPINFO_SOFTWARE = 27,		/* "IIPT" software used to "create:modify" */

  IPCHUNK_COUNT			/* count of chunk types */
}IPChunkType;

#define IPINFO_FIRST IPINFO_VERSION /* first info chunk ID enumeration */
#define IPINFO_LAST IPINFO_SOFTWARE /* last info chunk ID enumeration */

/** generator (effect parameter) types */
typedef enum
{
  IPGEN_START_ADDR_OFS = 0,	/* sample start address offset */
  IPGEN_END_ADDR_OFS = 1,	/* sample end address offset */
  IPGEN_START_LOOP_ADDR_OFS = 2,/* sample loop start address offset */
  IPGEN_END_LOOP_ADDR_OFS = 3,	/* sample loop end address offset */
  IPGEN_START_ADDR_COARSE_OFS=4,/* sample start coarse offset, 32k samples */
  IPGEN_MOD_LFO_TO_PITCH = 5,	/* modulation LFO to pitch */
  IPGEN_VIB_LFO_TO_PITCH = 6,	/* vibrato LFO to pitch */
  IPGEN_MOD_ENV_TO_PITCH = 7,	/* modulation envelope to pitch */
  IPGEN_FILTER_FC = 8,		/* initial filter cutoff */
  IPGEN_FILTER_Q = 9,		/* filter Q */
  IPGEN_MOD_LFO_TO_FILTER_FC = 10, /* modulation LFO to filter cutoff */
  IPGEN_MOD_ENV_TO_FILTER_FC = 11, /* modulation envelope to filter cutoff */
  IPGEN_END_ADDR_COARSE_OFS = 12, /* sample end course offset (32k samples) */
  IPGEN_MOD_LFO_TO_VOL = 13,	/* modulation LFO to volume */
  IPGEN_UNUSED1 = 14,
  IPGEN_CHORUS_SEND = 15,	/* chorus */
  IPGEN_REVERB_SEND = 16,	/* reverb */
  IPGEN_PAN = 17,		/* panning */
  IPGEN_UNUSED2 = 18,
  IPGEN_UNUSED3 = 19,
  IPGEN_UNUSED4 = 20,
  IPGEN_MOD_LFO_DELAY = 21,	/* modulation LFO delay */
  IPGEN_MOD_LFO_FREQ = 22,	/* modulation LFO frequency */
  IPGEN_VIB_LFO_DELAY = 23,	/* vibrato LFO delay */
  IPGEN_VIB_LFO_FREQ = 24,	/* vibrato LFO frequency */
  IPGEN_MOD_ENV_DELAY = 25,	/* modulation envelope delay */
  IPGEN_MOD_ENV_ATTACK = 26,	/* modulation envelope attack */
  IPGEN_MOD_ENV_HOLD = 27,	/* modulation envelope hold */
  IPGEN_MOD_ENV_DECAY = 28,	/* modulation envelope decay */
  IPGEN_MOD_ENV_SUSTAIN = 29,	/* modulation envelope sustain */
  IPGEN_MOD_ENV_RELEASE = 30,	/* modulation envelope release */
  IPGEN_KEY_TO_MOD_ENV_HOLD = 31, /* MIDI key to modulation envelope hold */
  IPGEN_KEY_TO_MOD_ENV_DECAY = 32, /* MIDI key to modulation envelope decay */
  IPGEN_VOL_ENV_DELAY = 33,	/* volume envelope delay */
  IPGEN_VOL_ENV_ATTACK = 34,	/* volume envelope attack */
  IPGEN_VOL_ENV_HOLD = 35,	/* volume envelope hold */
  IPGEN_VOL_ENV_DECAY = 36,	/* volume envelope decay */
  IPGEN_VOL_ENV_SUSTAIN = 37,	/* volume envelope sustain */
  IPGEN_VOL_ENV_RELEASE = 38,	/* volume envelope release */
  IPGEN_KEY_TO_VOL_ENV_HOLD = 39, /* MIDI key to volume envelope hold */
  IPGEN_KEY_TO_VOL_ENV_DECAY = 40, /* MIDI key to volume envelope decay */
  IPGEN_INSTRUMENT_ID = 41,	/* instrument ID */
  IPGEN_RESERVED1 = 42,
  IPGEN_KEY_RANGE = 43,		/* note on key range */
  IPGEN_VELOCITY_RANGE = 44,	/* note on velocity range */
  IPGEN_START_LOOP_ADDR_COARSE_OFS = 45, /* sample loop start coarse ofs (32k samples) */
  IPGEN_KEY_NUM = 46,		/* MIDI key override */
  IPGEN_VELOCITY = 47,		/* MIDI velocity override */
  IPGEN_ATTENUATION = 48,	/* initial volume attenuation */
  IPGEN_RESERVED2 = 49,
  IPGEN_END_LOOP_ADDR_COARSE_OFS = 50,	/* sample end loop course ofs (32k samples)  */
  IPGEN_COARSE_TUNE = 51,	/* course tuning */
  IPGEN_FINE_TUNE = 52,		/* fine tuning */
  IPGEN_SAMPLE_ID = 53,		/* sample ID */
  IPGEN_SAMPLE_MODES = 54,	/* sample flags (looping mode) */
  IPGEN_RESERVED3 = 55,
  IPGEN_SCALE_TUNE = 56,	/* scale tuning (tuning per MIDI key) */
  IPGEN_EXCLUSIVE_CLASS = 57,	/* exclusive class (only 1 at a time) */
  IPGEN_OVERRIDE_ROOT_KEY = 58,	/* root key override */
  IPGEN_COUNT			/* count of generators (not a valid ID) */
} IPGenType;

/** generator (effect parameter) unit types */
typedef enum
{
  IPUNIT_NONE = 0,		/* no unit type ("NULL" value) */
  IPUNIT_SMPLS = 1,		/* in samples */
  IPUNIT_32KSMPLS = 2,		/* in 32k samples */
  IPUNIT_CENT = 3,		/* in cents (1/100th of a semitone) */
  IPUNIT_HZCENT = 4,		/* in Hz Cents */
  IPUNIT_TCENT = 5,		/* in Time Cents */
  IPUNIT_CB = 6,		/* in centibels (1/100th of a decibel) */
  IPUNIT_PERCENT = 7,		/* in percent */
  IPUNIT_SEMITONE = 8,		/* in semitones */
  IPUNIT_RANGE = 9		/* a range of values */
}
IPUnitType;

#define INSTP_DEFAULT_INAM "untitled" /* default sound font name (#IPINFO_NAME) */
#define INSTP_DEFAULT_ISNG "EMU8000" /* default target sound engine (#IPINFO_ENGINE) */
#define INSTP_DEFAULT_ISFT "libInstPatch:" /* default software string (#IPINFO_SOFTWARE) */


/* modulator flags */

/** Modulator source mask for the controller index field */
#define IPMOD_INDEX_MASK	0x007F
/** Modulator source MIDI continuous controller flag */
#define IPMOD_CC_FLAG		0x0080
/** Modulator source controller direction flag */
#define IPMOD_DIRECTION_FLAG	0x0100
/** Modulator source controller polarity flag */
#define IPMOD_POLARITY_FLAG 	0x0200
/** Modulator source mask for controller map function type */
#define IPMOD_TYPE_MASK		0xFC00

/** Modulator source shift value for the controller index field */
#define IPMOD_INDEX_SHIFT	0
/** Modulator source shift value for MIDI continuous controller flag */
#define IPMOD_CC_SHIFT		7
/** Modulator source shift value for controller direction flag */
#define IPMOD_DIRECTION_SHIFT	8
/** Modulator source shift value for controller polarity flag */
#define IPMOD_POLARITY_SHIFT	9
/** Modulator source shift value for controller map function type field */
#define IPMOD_TYPE_SHIFT	10

/** Modulator control type with no controller source (outputs 1) */
#define IPMOD_CONTROL_NONE		0
/** Modulator note on velocity control source */
#define IPMOD_CONTROL_NOTE_ON_VELOCITY	2
/** Modulator MIDI key number control source */
#define IPMOD_CONTROL_KEY_NUMBER	3
/** Modulator MIDI poly-pressure control source */
#define IPMOD_CONTROL_POLY_PRESSURE	10
/** Modulator MIDI channel-pressure control source */
#define IPMOD_CONTROL_CHAN_PRESSURE	13
/** Modulator MIDI pitch wheel control source */
#define IPMOD_CONTROL_PITCH_WHEEL	14
/** Modulator MIDI pitch bend range control source */
#define IPMOD_CONTROL_BEND_RANGE	16

/** Value for #IPMOD_CC_FLAG to select the General Controller Palette */
#define IPMOD_CC_GENERAL	(0 << IPMOD_CC_SHIFT)
/** Value for #IPMOD_CC_FLAG to select the MIDI Custom Controller Palette */
#define IPMOD_CC_MIDI		(1 << IPMOD_CC_SHIFT)

/** Value for #IPMOD_DIRECTION_FLAG to use controller value directly */
#define IPMOD_DIRECTION_POSITIVE (0 << IPMOD_DIRECTION_SHIFT)
/** Value for #IPMOD_DIRECTION_FLAG to invert controller value */
#define IPMOD_DIRECTION_NEGATIVE (1 << IPMOD_DIRECTION_SHIFT)

/** Value for #IPMOD_POLARITY_FLAG for unipolar */
#define IPMOD_POLARITY_UNIPOLAR	(0 << IPMOD_POLARITY_SHIFT)
/** Value for #IPMOD_POLARITY_FLAG for bipolar */
#define IPMOD_POLARITY_BIPOLAR	(1 << IPMOD_POLARITY_SHIFT)

/** Value for #IPMOD_TYPE_MASK field for linear mapping */
#define IPMOD_TYPE_LINEAR	(0 << IPMOD_TYPE_SHIFT)
/** Value for #IPMOD_TYPE_MASK field for concave mapping */
#define IPMOD_TYPE_CONCAVE	(1 << IPMOD_TYPE_SHIFT)
/** Value for #IPMOD_TYPE_MASK field for convex mapping */
#define IPMOD_TYPE_CONVEX	(2 << IPMOD_TYPE_SHIFT)
/** Value for #IPMOD_TYPE_MASK field for switch mapping */
#define IPMOD_TYPE_SWITCH	(3 << IPMOD_TYPE_SHIFT)

/** Value for the 'trans' variable in an IPMod structure to use a linear
    transform function (currently the only value defined for SF v2.01) */
#define IPMOD_TRANSFORM_LINEAR	0


/* sampletype flag defines */

#define IPSAMPLE_TYPE_MONO	(1 << 0) /* sample is mono (#IPSample) */
#define IPSAMPLE_TYPE_RIGHT	(1 << 1) /* sample is right side of stereo (#IPSample) */
#define IPSAMPLE_TYPE_LEFT	(1 << 2) /* sample is left side of stereo (#IPSample) */
#define IPSAMPLE_TYPE_LINKED	(1 << 3) /* sample is linked (#IPSample) */
#define IPSAMPLE_TYPE_ROM	0x8000 /* sample is in ROM (#IPSample) */

#define IPSAMPLE_LOOP_MASK      0x3  /* mask for loop mode bits */
#define IPSAMPLE_LOOP		0x1  /* sample loop mode */
#define IPSAMPLE_LOOP_UNROLL	0x3  /* sample loop till release */

/* sample rate and length constraints */

#define IPSAMPLE_RATE_MIN	400 /* minimum sample rate (by the standard) */
#define IPSAMPLE_RATE_MAX	50000 /* maximum sample rate (by the standard) */
#define IPSAMPLE_LENGTH_MIN	32 /* minimum sample length (by the standard) */

/** #instp_sample_data_find_store flag to find fastest store */
#define IPSAMPLE_STORE_FIND_FASTEST	(1 << 0)
/** #instp_sample_data_find_store flag to find readable store */
#define IPSAMPLE_STORE_FIND_READABLE	(1 << 1)
/** #instp_sample_data_find_store flag to find writable store */
#define IPSAMPLE_STORE_FIND_WRITABLE	(1 << 2)
/** #instp_sample_data_find_store flag to find a store that is readable and
    writable */
#define IPSAMPLE_STORE_FIND_READWRITE \
    (IPSAMPLE_STORE_FIND_READABLE | IPSAMPLE_STORE_FIND_WRITABLE)


/* instp_load() load_flags */

#define IPLOAD_INFO	(1 << 0) /* load info (#instp_load) */
#define IPLOAD_PHDRS	(1 << 1) /* load preset headers (#instp_load) */
#define IPLOAD_PMODS	(1 << 2) /* load preset modulators (#instp_load) */
#define IPLOAD_PGENS	(1 << 3) /* load preset generators (#instp_load) */
#define IPLOAD_IHDRS	(1 << 4) /* load instrument headers (#instp_load) */
#define IPLOAD_IMODS	(1 << 5) /* load instrument modulators (#instp_load) */
#define IPLOAD_IGENS	(1 << 6) /* load instrument generators (#instp_load) */
#define IPLOAD_SHDRS	(1 << 7) /* load sample headers (#instp_load) */

#define IPLOAD_ALL	(1 << 15) /* load all parts of sound font (#instp_load) */

#define IPLOAD_PARTS_MASK 0xFF	/* mask for individual sound font part flags */

/** private sound font file context, use ipfile_* routines to access */
typedef struct _IPFileContext IPFileContext;

/**
 * \brief #instp_load sample data callback function type
 * \param ctx Sound font file context
 * \param samples Number of samples available
 * \param userptr User defined pointer from #instp_load
 * \return User should return TRUE to continue loading file, FALSE to abort
 */
typedef gboolean (* IPLoadSampleCallback)(IPFileContext *ctx, int samples,
					void *userptr);

/**
 * \brief #instp_item_foreach_voice callback function type
 * \param item The item that #instp_item_foreach_voice was called with
 * \param sample Sample for this voice
 * \param gen_array Array of generator values for this voice
 * \param mods Modulator list for this voice
 * \param data User defined data passed to #instp_item_foreach_voice
 * \return TRUE to continue processing voices, FALSE to abort
 */
typedef gboolean (* IPItemForeachVoiceFunc)(IPItem *item, IPSample *sample,
					  IPGenAmount *gen_array, IPMod *mods,
					  void *data);

INSTP_API extern IPGenConv instp_gen_conv[]; /* generator unit conversion table */
INSTP_API extern IPGenInfo instp_gen_info[]; /* generator constraint table */


/* function definitions */

INSTP_API int instp_init (void);

INSTP_API IPItemTypeInfo *instp_get_item_type_info (IPItemType type);
INSTP_API char *instp_get_item_type_name (IPItemType type);
INSTP_API int instp_get_item_type_size (IPItemType type);

INSTP_API IPItem *instp_item_next (const IPItem *item);

/** Same as #instp_item_next but uses #IPPreset items */
#define instp_preset_next(pset) INSTP_PRESET(instp_item_next(INSTP_ITEM(pset)))
/** Same as #instp_item_next but uses #IPInst items */
#define instp_inst_next(inst) INSTP_INST (instp_item_next (INSTP_ITEM (inst)))
/** Same as #instp_item_next but uses #IPSample items */
#define instp_sample_next(sam) INSTP_SAMPLE(instp_item_next(INSTP_ITEM(sam)))
/** Same as #instp_item_next but uses #IPSampleData items */
#define instp_sample_data_next(sampledata) \
    INSTP_SAMPLE_DATA(instp_item_next(INSTP_ITEM(sampledata)))
/** Same as #instp_item_next but uses #IPZone items */
#define instp_zone_next(zone) INSTP_ZONE(instp_item_next(INSTP_ITEM(zone)))

INSTP_API IPItem *instp_item_prev (const IPItem *item);

/** Same as #instp_item_prev but uses #IPPreset items */
#define instp_preset_prev(pset) INSTP_PRESET(instp_item_prev(INSTP_ITEM(pset)))
/** Same as #instp_item_prev but uses #IPInst items */
#define instp_inst_prev(inst) INSTP_INST(instp_item_prev(INSTP_ITEM (inst)))
/** Same as #instp_item_prev but uses #IPSample items */
#define instp_sample_prev(sam) INSTP_SAMPLE(instp_item_prev(INSTP_ITEM(sam)))
/** Same as #instp_item_prev but uses #IPZone items */
#define instp_zone_prev(zone) INSTP_ZONE(instp_item_prev(INSTP_ITEM(zone)))

INSTP_API IPItem *instp_item_parent (const IPItem *item);

INSTP_API void instp_item_ref (IPItem *item);
INSTP_API void instp_item_unref (IPItem *item);
INSTP_API void instp_item_sink (IPItem *item);

INSTP_API int instp_item_count (const IPItem *item);

INSTP_API IPItem *instp_item_new (IPItemType type);

/** Create a new sound font object */
#define instp_sfont_new() (INSTP_SFONT (instp_item_new (IPITEM_SFONT)))
/** Create a new preset object */
#define instp_preset_new() (INSTP_PRESET (instp_item_new (IPITEM_PRESET)))
/** Create a new instrument object */
#define instp_inst_new() (INSTP_INST (instp_item_new (IPITEM_INST)))
/** Create a new sample object */
#define instp_sample_new() (INSTP_SAMPLE (instp_item_new (IPITEM_SAMPLE)))
/** Create a new sample data object */
#define instp_sample_data_new() \
    (INSTP_SAMPLE_DATA (instp_item_new (IPITEM_SAMPLE_DATA)))
/** Create a new zone object */
#define instp_zone_new() (INSTP_ZONE (instp_item_new (IPITEM_ZONE)))

INSTP_API IPItem *instp_item_alloc (IPItemType type);
INSTP_API void instp_item_free (IPItem *item);
INSTP_API void instp_item_destroy (IPItem *item);
INSTP_API IPItem *instp_item_duplicate (const IPItem *item);

/** Duplicate a preset */
#define instp_preset_duplicate(item) \
   (INSTP_PRESET (instp_item_duplicate (INSTP_ITEM (item))))
/** Duplicate an instrument */
#define instp_inst_duplicate(item) \
   (INSTP_INST (instp_item_duplicate (INSTP_ITEM (item))))
/** Duplicate a sample */
#define instp_sample_duplicate(item) \
   (INSTP_SAMPLE (instp_item_duplicate (INSTP_ITEM (item))))
/** Duplicate a zone */
#define instp_zone_duplicate(item) \
   (INSTP_ZONE (instp_item_duplicate (INSTP_ITEM (item))))

INSTP_API void instp_item_insert (IPItem *parent, IPItem *item, int pos);
INSTP_API void instp_item_insert_before (IPItem *parent, IPItem *item,
					 IPItem *sibling);
INSTP_API IPItem *instp_item_list_insert (IPItem *list_root, IPItem *item,
					  int pos);
INSTP_API IPItem *instp_item_list_insert_before (IPItem *list_root,
						 IPItem *item,
						 IPItem *sibling);
INSTP_API void instp_item_unlink (IPItem *item);

/** Convenience function to append an item in a list */
#define instp_item_append(parent, item) instp_item_insert (parent, item, -1)
/** Convenience function to prepend an item in a list */
#define instp_item_prepend(parent, item) instp_item_insert (parent, item, 0)

INSTP_API void instp_item_set_userptr (IPItem *item, const void *ptr);
INSTP_API void *instp_item_get_userptr (const IPItem *item);
INSTP_API IPItem *instp_item_find_parent_by_type (const IPItem *item,
						  IPItemType type);
INSTP_API IPItem *instp_item_find_root (const IPItem *item);
INSTP_API IPItem **instp_item_get_child_list (IPItem *item,
					      IPItemType childtype);
INSTP_API gboolean instp_item_is_parent_ok (IPItemType parent,
					    IPItemType child);
INSTP_API void instp_item_set_root_changed_flag (IPItem *item, gboolean val);
INSTP_API gboolean instp_item_foreach_voice (IPItem *item, int note,
					     int velocity,
					     IPItemForeachVoiceFunc func,
					     void *data);

INSTP_API IPSFont *instp_sfont_load (int fhandle, int load_flags,
				     IPLoadSampleCallback *smpl_cb,
				     const void *userptr);

INSTP_API void instp_set_file_name (IPSFont *sf, const char *filename);
INSTP_API void instp_set_file_info (IPSFont *sf, IPFileInfo *file_info);

INSTP_API char *instp_get_info (const IPSFont *sf, IPChunkType id);
INSTP_API char *instp_get_info_custom (const IPSFont *sf, const char id[4]);
INSTP_API int instp_set_info (IPSFont *sf, IPChunkType id, const char *val);
INSTP_API int instp_set_info_custom (IPSFont *sf, const char id[4],
				     const char *val);
INSTP_API int instp_info_max_size (IPChunkType infotype);

INSTP_API IPSFontInfo *instp_first_info (const IPSFont *sf);
INSTP_API IPSFontInfo *instp_info_next (const IPSFontInfo *info);
INSTP_API IPSFontInfo *instp_info_alloc (void);
INSTP_API void instp_info_destroy (IPSFontInfo *info);

INSTP_API IPPreset *instp_first_preset (const IPSFont *sf);
INSTP_API IPPreset *instp_create_preset (IPSFont *sf, const char *name,
					 guint16 bank, guint16 psetnum);
INSTP_API void instp_add_preset_sorted (IPSFont *sf, IPPreset *preset);
INSTP_API void instp_insert_preset (IPSFont *sf, IPPreset *preset, int pos);

INSTP_API IPInst *instp_first_inst (const IPSFont *sf);
INSTP_API IPInst *instp_create_inst (IPSFont *sf, const char *name);
INSTP_API void instp_add_inst (IPSFont *sf, IPInst *inst);
INSTP_API void instp_insert_inst (IPSFont *sf, IPInst *inst, int pos);

INSTP_API IPSample *instp_first_sample (const IPSFont *sf);
INSTP_API IPSample *instp_create_sample (IPSFont *sf, const char *name);
INSTP_API void instp_add_sample (IPSFont *sf, IPSample *sample);
INSTP_API void instp_insert_sample (IPSFont *sf, IPSample *sample, int pos);

INSTP_API int instp_find_free_preset (const IPSFont *sf, int *bank,
				      int *psetnum);
INSTP_API IPPreset *instp_find_preset (const IPSFont *sf, const char *name,
				       int bank, int psetnum,
				       const IPPreset *exclude);
INSTP_API IPInst *instp_find_inst (const IPSFont *sf, const char *name,
				   const IPInst *exclude);
INSTP_API IPSample *instp_find_sample (const IPSFont *sf, const char *name,
				       const IPSample *exclude);

INSTP_API IPFileInfo *instp_file_info_new (void);
INSTP_API void instp_file_info_ref (IPFileInfo *info);
INSTP_API void instp_file_info_unref (IPFileInfo *info);

INSTP_API int instp_preset_compare (const IPPreset *p1, const IPPreset *p2);
INSTP_API int instp_preset_set_name (IPPreset *preset, const char *name);
INSTP_API IPZone *instp_preset_first_zone (const IPPreset *preset);
INSTP_API IPZone *instp_preset_create_zone (IPPreset *preset,
					    const IPInst *zone_inst);
INSTP_API void instp_preset_add_zone (IPPreset *preset, IPZone *zone);
INSTP_API void instp_preset_insert_zone (IPPreset *preset, IPZone *zone,
					 int pos);

INSTP_API int instp_inst_set_name (IPInst *inst, const char *name);
INSTP_API IPZone *instp_inst_first_zone (const IPInst *inst);
INSTP_API IPZone *instp_inst_create_zone (IPInst *inst,
					  const IPSample *zone_sample);
INSTP_API void instp_inst_add_zone (IPInst *inst, IPZone *zone);
INSTP_API void instp_inst_insert_zone (IPInst *inst, IPZone *zone, int pos);

INSTP_API int instp_sample_set_name (IPSample *sample, const char *name);
INSTP_API void instp_sample_set_sample_data (IPSample *sample,
					     IPSampleData *sampledata);
INSTP_API IPSampleData *instp_sample_get_sample_data (IPSample *sample);
INSTP_API int instp_sample_set_blank_data (IPSample *sample);
INSTP_API int instp_sample_get_size (IPSample *sample);

INSTP_API void instp_sample_data_set_size (IPSampleData *sampledata,
					   guint size);
INSTP_API IPSampleStore *
instp_sample_data_first_store (IPSampleData *sampledata);
INSTP_API IPSampleStore *
instp_sample_data_find_store (IPSampleData *sampledata, int method_type,
			      int find_flags);
INSTP_API IPSampleData *instp_sample_data_get_blank (void);

INSTP_API IPGen *instp_zone_first_gen (const IPZone *zone);
INSTP_API IPMod *instp_zone_first_mod (const IPZone *zone);
INSTP_API void instp_zone_set_refitem (IPZone *zone, IPItem *refitem);
INSTP_API gboolean instp_zone_in_range (IPZone *zone, int note, int velocity);

INSTP_API void instp_zone_insert_gen (IPZone *zone, IPGen *gen, int pos);
INSTP_API gboolean instp_zone_get_gen (const IPZone *zone, guint16 genid,
				       IPGenAmount *out_amt);
INSTP_API int instp_zone_set_gen (IPZone *zone, guint16 genid,
				  IPGenAmount amt);
INSTP_API void instp_zone_unset_gen (IPZone *zone, guint16 genid);
INSTP_API int instp_zone_gen_count (IPZone *zone);

INSTP_API void instp_zone_insert_mod (IPZone *zone, IPMod *mod, int pos);
INSTP_API void instp_zone_remove_mod_by_ptr (IPZone *zone, IPMod *mod);
INSTP_API int instp_zone_mod_count (IPZone *zone);

INSTP_API IPGen *instp_gen_next (const IPGen *gen);
INSTP_API IPGen *instp_gen_new (void);
INSTP_API void instp_gen_free (IPGen *gen);

INSTP_API gboolean instp_genid_is_valid (int genid, gboolean ispreset);
INSTP_API IPGenAmount instp_genid_default_value (guint16 genid,
						 gboolean ispreset);
INSTP_API gboolean instp_genid_offset (int genid, IPGenAmount *dst,
				       IPGenAmount ofs);

INSTP_API IPGenAmount *instp_gen_array_alloc (void);
INSTP_API void instp_gen_array_init (IPGenAmount *genarr, gboolean offset);
INSTP_API void instp_gen_array_copy (IPGenAmount *dest,
				     const IPGenAmount *src);
INSTP_API void instp_gen_array_process_zone (IPGenAmount *genarr,
					     const IPZone *zone);
INSTP_API void instp_gen_array_offset (IPGenAmount *abs,
				       const IPGenAmount *ofs);

INSTP_API IPMod *instp_mod_next (const IPMod *mod);
INSTP_API IPMod *instp_mod_new (void);
INSTP_API IPMod * instp_mod_duplicate (IPMod *mod);
INSTP_API void instp_mod_free (IPMod *mod);
INSTP_API IPMod *instp_mod_list_insert (IPMod *list, IPMod *mod, int pos);
INSTP_API IPMod *instp_mod_list_remove_by_ptr (IPMod *list, IPMod *mod);
INSTP_API IPMod *instp_mod_list_copy (IPMod *list);
INSTP_API IPMod *instp_mod_list_combine (IPMod *amods, IPMod *bmods);
INSTP_API IPMod *instp_mod_list_offset (IPMod *amods, IPMod *bmods);

INSTP_API void instp_mod_list_free (IPMod *mod);
INSTP_API void instp_mod_list_apply_global (IPMod *globalist);
INSTP_API IPMod *instp_mod_list_default (void);
INSTP_API void instp_units_sfont_to_user (guint16 genid, IPGenAmount amt,
					  float *out_userval,
					  gboolean ispreset);
INSTP_API void instp_units_sfont_to_user_str (guint16 genid, IPGenAmount amt,
					      float *out_userval,
					      char **out_userstr,
					      gboolean ispreset);
INSTP_API int instp_units_user_to_sfont (guint16 genid, float userval,
					 gboolean ispreset, gboolean clamp);
INSTP_API void instp_units_clamp (guint16 genid, int *sfval,
				  gboolean ispreset);
INSTP_API IPGenAmount instp_units_range_intersect (const IPGenAmount a,
						   const IPGenAmount b);

INSTP_API int instp_float2int (float f);
INSTP_API float instp_int2float (IPGenAmount val);
INSTP_API int instp_hz2cents (float hz);
INSTP_API float instp_cents2hz (IPGenAmount cents);
INSTP_API int instp_db2cb (float db);
INSTP_API float instp_cb2db (IPGenAmount cb);
INSTP_API int instp_sec2tcent (float sec);
INSTP_API float instp_tcent2sec (IPGenAmount tcent);
INSTP_API int instp_perc2tperc (float perc);
INSTP_API float instp_tperc2perc (IPGenAmount tperc);

/** \} */


/**
 * \defgroup IPSampleData Sound Font Sample Data Interface
 * Sound Font Sample Data Interface
 * \{
 */

/** Type of sample data storage */
typedef enum
{
  IPSAMPLE_METHOD_NONE = 0,	/* A "NULL" value for IPSampleMethodType */
  IPSAMPLE_METHOD_SFONT,	/* Sample stored in sound font file */
  IPSAMPLE_METHOD_BUFFER,	/* Sample stored in disk sample buffer */
  IPSAMPLE_METHOD_RAM,		/* Sample stored in memory */
  IPSAMPLE_METHOD_ROM,		/* Sample is a ROM sample
				     (just retains ROM start location) */
  IPSAMPLE_METHOD_COUNT,	/* Count of built in sample methods */
  IPSAMPLE_METHOD_USER = 10	/* Start of user defined sample methods */
}
IPSampleMethodType;

/**
 * \brief Sample method function to initialize a sample store
 * \param sampledata Sample data object
 * \param store Sample store to initialize.
 * \return Should return INSTP_OK on success, INSTP_FAIL otherwise
 *
 * Responsible for storage method specific initialization of a sample store.
 * When function is called the store has not yet been added to the sample
 * data object's storage list, but has been initialized with the method
 * pointer. Should this function return an error code, the store will not
 * be created.
 */
typedef int (* IPSampleInitFunc)(IPSampleData *sampledata,
				 IPSampleStore *store);
/**
 * \brief Sample method function called before a store is destroyed
 * \param sampledata Sample data object
 * \param store Sample store in destruction phase.
 *
 * This function is responsible for cleaning up the storage method specific
 * data in the variable area of a sample store.
 */
typedef void (* IPSampleDoneFunc)(IPSampleData *sampledata,
				  IPSampleStore *store);
/**
 * \brief Sample method function called to allocate sample storage
 * \param sampledata Sample data object
 * \param store Sample store to allocate storage space for.
 * \return Should return INSTP_OK on success, INSTP_FAIL otherwise
 *
 * A function type that is responsible for allocating space for sample data
 * in a store for its method type.
 */
typedef int (* IPSampleAllocFunc)(IPSampleData *sampledata,
				  IPSampleStore *store);
/**
 * \brief Sample method function called to free sample storage
 * \param sampledata Sample data object
 * \param store Sample store to free storage of.
 *
 * A function type that is responsible for freeing space for sample data
 * for its method type.
 */
typedef void (* IPSampleFreeFunc)(IPSampleData *sampledata,
				  IPSampleStore *store);
/**
 * \brief Sample method function called to read data from a sample store
 * \param sampledata Sample data object
 * \param store Sample store to read from.
 * \param offset Offset into sample to start read from (in sample units)
 * \param size Number of samples to load
 * \param buf Buffer to store loaded samples into
 * \return INSTP_OK on success, INSTP_FAIL on error or short read
 */
typedef int (* IPSampleReadFunc)(IPSampleData *sampledata,
				 IPSampleStore *store,
				 int offset, int size, void *buf);
/**
 * \brief Sample method function called to write data to a sample store
 * \param sampledata Sample data object
 * \param store Sample store to write to.
 * \param offset Offset into sample to start write to (in sample units)
 * \param size Number of samples to save
 * \param buf Buffer to read sample from
 * \return INSTP_OK on success, INSTP_FAIL on error or short read
 */
typedef int (* IPSampleWriteFunc)(IPSampleData *sampledata,
				  IPSampleStore *store,
				  int offset, int size, const void *buf);

/** Sample storage method class (RAM, sound font file,
    sample disk buffer, etc) */
struct _IPSampleMethod
{
  guint8 type;			/* Type of storage (#IPSampleMethodType and
				   user defined) */
  guint8 var_size;		/* size in bytes of method's variable area */
  guint8 speed_rating;		/* a rating from 1-10 of how speedy the
				   storage method is 3 = hard disk, 9 = RAM,
				   etc. For prioritizing when there are
				   multiple stores for a given sample data. */
  guint8 reserved;		/* reserved (pad to 4 byte boundary) */
  IPSampleInitFunc init;	/* Called when IPSampleData initialized */
  IPSampleDoneFunc done;	/* Called when IPSampleData destroyed */
  IPSampleAllocFunc alloc;	/* Function to allocate space for new
				   sample data or NULL */
  IPSampleFreeFunc free;	/* Function to free sample data or NULL */
  IPSampleReadFunc read;	/* Function to read sample data or NULL */
  IPSampleWriteFunc write;	/* Function to write sample data or NULL */
};

/** An instance of sample data storage. Structure is followed by a
    variable size area (determined by #IPSampleMethod ->var_size) for
    the method's variables. */
struct _IPSampleStore
{
  IPSampleMethod *method;	/* sample storage method for this store */
  IPSampleStore *next;		/* next data store in list */
};

/** Check if a sample store is readable */
#define IPSAMPLE_STORE_IS_READABLE(store)  (store->method->read != NULL)
/** Check if a sample store is writable */
#define IPSAMPLE_STORE_IS_WRITABLE(store)  (store->method->write != NULL)

/** Sample data info, possibly shared with multiple samples (#IPSample) */
struct _IPSampleData
{
  IPItem sfitem;		/* derived from sound font item */
  guint32 size;		/* size of sample data (in samples) */
  IPSampleStore *storage;	/* list of stores for sample data */
};

/* sample data function prototypes */

INSTP_API IPSampleMethod *instp_sample_method_new (void);
INSTP_API int instp_sample_method_register (IPSampleMethod *method);
INSTP_API IPSampleMethod *instp_sample_method_get_by_type (int method_type);

INSTP_API IPSampleStore *instp_sample_store_new (IPSampleData *sampledata,
						 int method_type);
INSTP_API IPSampleStore *instp_sample_store_next (const IPSampleStore *store);
INSTP_API void instp_sample_store_destroy (IPSampleData *sampledata,
					   IPSampleStore *store);

INSTP_API int instp_sample_store_alloc (IPSampleData *sampledata,
					IPSampleStore *store);
INSTP_API void instp_sample_store_free (IPSampleData *sampledata,
					IPSampleStore *store);
INSTP_API int instp_sample_store_read (IPSampleData *sampledata,
				       IPSampleStore *store,
				       int offset, int size, void *buf);
INSTP_API int instp_sample_store_write (IPSampleData *sampledata,
					IPSampleStore *store,
					int offset, int size, const void *buf);
INSTP_API int instp_sample_store_copy (IPSampleData *dest_sampledata,
				       IPSampleStore *dest_store,
				       IPSampleData *src_sampledata,
				       IPSampleStore *src_store);
INSTP_API IPSampleStore *
instp_sample_store_duplicate (IPSampleData *sampledata, IPSampleStore *store,
			      int method_type);
INSTP_API void *instp_sample_store_get_varptr (IPSampleStore *store);

INSTP_API void instp_sample_method_SFONT_set_params (IPSampleData *sampledata,
						     IPSampleStore *store,
						     IPFileInfo *file,
						     guint start);
INSTP_API void instp_sample_method_SFONT_get_params (IPSampleData *sampledata,
						     IPSampleStore *store,
						     IPFileInfo **info,
						     guint *start);
INSTP_API void instp_sample_method_RAM_set_pointer (IPSampleData *sampledata,
						    IPSampleStore *store,
						    void *data);
INSTP_API void *instp_sample_method_RAM_get_pointer (IPSampleData *sampledata,
						     IPSampleStore *store);
INSTP_API void instp_sample_method_ROM_set_start (IPSampleData *sampledata,
						  IPSampleStore *store,
						  guint start);
INSTP_API guint instp_sample_method_ROM_get_start (IPSampleData *sampledata,
						   IPSampleStore *store);

/** \} */


/**
 * \defgroup IPFile Sound Font File Interface
 * Sound Font File Interface
 * \{
 */

/* private SoundFont save handle, use instp_sfont_save_* routines to access */
typedef struct _IPSFontSaveHandle IPSFontSaveHandle;

/**
 * IPSFontSaveSampleDataCallback:
 * @handle: SoundFont save handle (as returned from
 *  instp_sfont_save_new_handle()).
 * @data: Custom data to pass to callback function.
 *
 * Callback function type for instp_sfont_save() to override the default
 * sample data save function.
 *
 * Returns: Should return INSTP_OK on success, INSTP_FAIL otherwise
 */
typedef int (* IPSFontSaveSampleDataCallback)(IPSFontSaveHandle *handle,
					      gpointer data);

INSTP_API IPSFontSaveHandle *instp_sfont_save_new_handle (void);
INSTP_API void instp_sfont_save_close_handle (IPSFontSaveHandle *handle);
INSTP_API void instp_sfont_save_set_sfont (IPSFontSaveHandle *handle,
					   IPSFont *sf);
INSTP_API IPSFont *instp_sfont_save_get_sfont (IPSFontSaveHandle *handle);
INSTP_API void instp_sfont_save_set_fhandle (IPSFontSaveHandle *handle,
					     int file_handle);
INSTP_API int instp_sfont_save_get_fhandle (IPSFontSaveHandle *handle);
INSTP_API void instp_sfont_save_set_sample_callback (IPSFontSaveHandle *handle,
				IPSFontSaveSampleDataCallback callback,
						     gpointer callback_data);
INSTP_API IPFileContext *
instp_sfont_save_get_file_context (IPSFontSaveHandle *handle);
INSTP_API int instp_sfont_save (IPSFontSaveHandle *handle);
INSTP_API int instp_sfont_save_migrate_samples (IPSFontSaveHandle *handle);
INSTP_API int instp_sfont_save_report_sample_offset (IPSFontSaveHandle *handle,
						     IPSample *sample,
						     guint start);


typedef struct _IPFileChunk IPFileChunk;
typedef struct _IPFilePHDR IPFilePHDR;
typedef struct _IPFileIHDR IPFileIHDR;
typedef struct _IPFileSHDR IPFileSHDR;
typedef struct _IPFileBag IPFileBag;
typedef struct _IPFileMod IPFileMod;
typedef struct _IPFileGen IPFileGen;

/* sfont file chunk sizes */
#define IPFILE_CHUNK_SIZE	8 /* file IPChunk size (#IPFileChunk) */
#define IPFILE_VERSION_SIZE     4 /* file version info size (#IPSFontVersion) */
#define IPFILE_PHDR_SIZE	38 /* file preset header size (#IPFilePHDR) */
#define IPFILE_IHDR_SIZE	22 /* file instrument header size (#IPFileIHDR */
#define IPFILE_SHDR_SIZE	46 /* file sample header size (#IPFileSHDR) */
#define IPFILE_BAG_SIZE		4 /* file bag (zone) size (#IPFileBag) */
#define IPFILE_MOD_SIZE		10 /* file modulator size (#IPFileMod) */
#define IPFILE_GEN_SIZE		4 /* file generator size (#IPFileGen) */
#define IPFILE_NAME_SIZE	20 /* name string size (Preset, Instrument, Sample) */

/** largest structure in sound font */
#define IPFILE_LARGEST_CHUNK	IPFILE_SHDR_SIZE

/** sound font RIFF file chunk */
struct _IPFileChunk
{
  char id[4];			/* 4 character chunk id (#IPChunkType) */
  guint32 size;		/* size of the following chunk */
};

/** sound font file preset header */
struct _IPFilePHDR
{
  guint8 name[20];		/* preset name */
  guint16 psetnum;		/* preset number */
  guint16 bank;		/* MIDI bank number */
  guint16 pbagndx;		/* index into preset bag (#IPFileBag) */
  guint32 library;		/* Not used (preserved) */
  guint32 genre;		/* Not used (preserved) */
  guint32 morphology;		/* Not used (preserved) */
};

/** sound font file instrument header */
struct _IPFileIHDR
{
  char name[20];		/* name of instrument */
  guint16 ibagndx;		/* instrument bag index (#IPFileBag) */
};

/** sound font file sample header */
struct _IPFileSHDR
{
  char name[20];		/* sample name */
  guint32 start;		/* offset to start of sample */
  guint32 end;			/* offset to end of sample */
  guint32 loopstart;		/* offset to start of loop */
  guint32 loopend;		/* offset to end of loop */
  guint32 samplerate;		/* sample rate recorded at */
  guint8 origpitch;		/* root midi key number */
  gint8 pitchadj;		/* pitch correction in cents */
  guint16 samplelink;		/* linked sample index for stereo samples */
  guint16 sampletype;		/* type of sample (see INSTP_SAMPLETYPE_*) */
};

/** sound font file bag (zone), indexes for zone's generators and modulators */
struct _IPFileBag
{
  guint16 modndx;		/* index into modulator list */
  guint16 genndx;		/* index into generator list */
};

/** sound font file modulator */
struct _IPFileMod
{
  guint16 src;			/* source modulator (MIDI controller, etc) */
  guint16 dest;		/* destination generator */
  gint16 amount;		/* degree of modulation */
  guint16 amtsrc;		/* second source controls amount of first */
  guint16 trans;		/* transform function applied to source */
};

/** sound font file generator */
struct _IPFileGen
{
  guint16 id;			/* generator ID (#IPGenType) */
  IPGenAmount amount;		/* generator value */
};

/** sound font file error sub codes */
typedef enum
{
  IPFILE_NOERR,
  IPFILE_CHUNK_NOT_FOUND,
  IPFILE_CHUNK_BAD_SIZE
}
IPFileErrCode;

/**
 * IPFileCallback:
 * @context: File context
 * @userptr: Custom user data passed to callback.
 *
 * Callback function type for low level chunk based patch file processing.
 *
 * Returns: Should return INSTP_OK on success, INSTP_FAIL otherwise
 */
typedef int (IPFileCallback)(IPFileContext *context, void *userptr);

/**
 * IPFileReadFunc:
 * @ctx: File context
 * @buf: Buffer to read data into.
 * @count: Number of bytes to read into @buf.
 *
 * A function type that can be used to override the default read function.
 *
 * Returns: Number of bytes read or -1 on error (function should log a message
 *   on error or end of file).
 */
typedef guint (*IPFileReadFunc)(IPFileContext *ctx, void *buf, guint count);

/**
 * IPFileWriteFunc:
 * @ctx: File context
 * @buf: Buffer of data to write.
 * @count: Number of bytes in @buf to write.
 *
 * A function type that can be used to override the default write function.
 *
 * Returns: Number of bytes written or -1 on error (function should log an
 *   error message).
 */
typedef guint (*IPFileWriteFunc)(IPFileContext *ctx, const void *buf,
				 guint count);

/**
 * IPFileSeekFunc:
 * @ctx: File context
 * @offset: Offset in bytes to seek (acts like SEEK_CUR with lseek(2)).
 *
 * A function type that can be used to override the default seek function.
 *
 * Returns: New position in file or -1 on error (function should log an error
 *   message).
 */
typedef guint (*IPFileSeekFunc)(IPFileContext *ctx, gint offset);


/* function definitions */

INSTP_API IPChunkType ipfile_chunk_str_to_enum (const char id[4]);
INSTP_API char *ipfile_chunk_enum_to_str (IPChunkType type);

INSTP_API int ipfile_chunk_size (IPChunkType chunktype);

INSTP_API IPFileContext *ipfile_context_new (char *mode);
INSTP_API void ipfile_context_free (IPFileContext *ctx);
INSTP_API void ipfile_set_fhandle (IPFileContext *ctx, int fhandle);
INSTP_API int ipfile_get_fhandle (IPFileContext *ctx);
INSTP_API void ipfile_set_callback (IPFileContext *ctx,
				    IPFileCallback *callback,
				    const void *userptr);
INSTP_API void ipfile_get_callback (IPFileContext *ctx,
				    IPFileCallback **callback,
				    void **userptr);
INSTP_API void ipfile_set_userptr (IPFileContext *ctx, void *userptr);
INSTP_API void *ipfile_get_userptr (IPFileContext *ctx);
INSTP_API void ipfile_set_io_functions (IPFileContext *ctx,
					IPFileReadFunc read,
					IPFileWriteFunc write,
					IPFileSeekFunc seek);
INSTP_API IPChunkType ipfile_get_chunk_type (IPFileContext *ctx);
INSTP_API int ipfile_get_chunk_level_count (IPFileContext *ctx);
INSTP_API int ipfile_get_chunk_state (IPFileContext *ctx, int level,
				      IPChunkType *type,
				      guint *size, guint *pos);
INSTP_API int ipfile_get_file_size (IPFileContext *ctx);
INSTP_API int ipfile_get_file_position (IPFileContext *ctx);
INSTP_API int ipfile_get_items_left (IPFileContext *ctx);
INSTP_API int ipfile_start_transfer (IPFileContext *ctx);

INSTP_API int ipfile_load_info_chunk (IPFileContext *ctx, char *id, int *size);
INSTP_API int ipfile_load_info_data (IPFileContext *ctx, char *buf, int size);
INSTP_API int ipfile_load_info_version (IPFileContext *ctx,
					IPSFontVersion *version);
INSTP_API int ipfile_load_sample_data (IPFileContext *ctx, gint16 *buf,
				       int count);
INSTP_API int ipfile_load_phdr (IPFileContext *ctx, IPFilePHDR **phdr);
INSTP_API int ipfile_load_ihdr (IPFileContext *ctx, IPFileIHDR **ihdr);
INSTP_API int ipfile_load_shdr (IPFileContext *ctx, IPFileSHDR **shdr);
INSTP_API int ipfile_load_bag (IPFileContext *ctx, IPFileBag **bag);
INSTP_API int ipfile_load_mod (IPFileContext *ctx, IPFileMod **mod);
INSTP_API int ipfile_load_gen (IPFileContext *ctx, IPFileGen **gen);

INSTP_API int ipfile_save_info (IPFileContext *ctx, const char *id,
				const char *val, int size);
INSTP_API int ipfile_save_info_version (IPFileContext *ctx, const char *id,
					IPSFontVersion *version);
INSTP_API int ipfile_save_sample_data (IPFileContext *ctx, gint16 *buf,
				       int count);
INSTP_API void ipfile_save_sample_dummy (IPFileContext *ctx, guint size);
INSTP_API int ipfile_save_sample (IPFileContext *ctx, IPSample *sample);
INSTP_API int ipfile_save_phdr (IPFileContext *ctx, const IPFilePHDR *phdr);
INSTP_API int ipfile_save_ihdr (IPFileContext *ctx, const IPFileIHDR *ihdr);
INSTP_API int ipfile_save_shdr (IPFileContext *ctx, const IPFileSHDR *shdr);
INSTP_API int ipfile_save_bag (IPFileContext *ctx, const IPFileBag *bag);
INSTP_API int ipfile_save_mod (IPFileContext *ctx, const IPFileMod *mod);
INSTP_API int ipfile_save_gen (IPFileContext *ctx, const IPFileGen *gen);

INSTP_API int ipfile_ftell (IPFileContext *ctx);

/** \} */

#endif
