#include <stdlib.h>
#include <string.h>
#ifndef WIN32
#include "config.h"
#endif

#ifdef ENABLE_NLS
#include <libintl.h>
#endif

#define         _ISOC9X_SOURCE  1
#define         _ISOC99_SOURCE  1
#define         __USE_ISOC99    1
#define         __USE_ISOC9X    1

#include <math.h>

#include "ladspa.h"

#ifdef WIN32
#define _WINDOWS_DLL_EXPORT_ __declspec(dllexport)
int bIsFirstTime = 1; 
void _init(); // forward declaration
#else
#define _WINDOWS_DLL_EXPORT_ 
#endif

#line 9 "morph_1917.xml"

#include "config.h"

#ifdef FFTW3

#include <fftw3.h>

typedef fftwf_plan fft_plan;
typedef float fftw_real;

#else

#ifdef EXPLICIT_S
#include <srfftw.h>
#else
#include <rfftw.h>
#endif //EXPLICIT_S

typedef rfftw_plan fft_plan;

#endif //FFTW3

#include "ladspa-util.h"

#define FFT_LENGTH 1024
#define OVER_SAMP  4

#define MORPH_MORPH                    0
#define MORPH_LATENCY                  1
#define MORPH_INPUT_A                  2
#define MORPH_INPUT_B                  3
#define MORPH_OUTPUT                   4

static LADSPA_Descriptor *morphDescriptor = NULL;

typedef struct {
	LADSPA_Data *morph;
	LADSPA_Data *latency;
	LADSPA_Data *input_a;
	LADSPA_Data *input_b;
	LADSPA_Data *output;
	fftw_real *  comp_a;
	fftw_real *  comp_b;
	long         fifo_pos;
	LADSPA_Data *in_fifo_a;
	LADSPA_Data *in_fifo_b;
	LADSPA_Data *out_accum;
	LADSPA_Data *out_fifo;
	fft_plan     plan_cr;
	fft_plan     plan_rc_a;
	fft_plan     plan_rc_b;
	fftw_real *  real_a;
	fftw_real *  real_b;
	float *      window;
	LADSPA_Data run_adding_gain;
} Morph;

_WINDOWS_DLL_EXPORT_
const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {

#ifdef WIN32
	if (bIsFirstTime) {
		_init();
		bIsFirstTime = 0;
	}
#endif
	switch (index) {
	case 0:
		return morphDescriptor;
	default:
		return NULL;
	}
}

static void activateMorph(LADSPA_Handle instance) {
	Morph *plugin_data = (Morph *)instance;
	fftw_real *comp_a = plugin_data->comp_a;
	fftw_real *comp_b = plugin_data->comp_b;
	long fifo_pos = plugin_data->fifo_pos;
	LADSPA_Data *in_fifo_a = plugin_data->in_fifo_a;
	LADSPA_Data *in_fifo_b = plugin_data->in_fifo_b;
	LADSPA_Data *out_accum = plugin_data->out_accum;
	LADSPA_Data *out_fifo = plugin_data->out_fifo;
	fft_plan plan_cr = plugin_data->plan_cr;
	fft_plan plan_rc_a = plugin_data->plan_rc_a;
	fft_plan plan_rc_b = plugin_data->plan_rc_b;
	fftw_real *real_a = plugin_data->real_a;
	fftw_real *real_b = plugin_data->real_b;
	float *window = plugin_data->window;
#line 74 "morph_1917.xml"
	fifo_pos = 0;
	plugin_data->comp_a = comp_a;
	plugin_data->comp_b = comp_b;
	plugin_data->fifo_pos = fifo_pos;
	plugin_data->in_fifo_a = in_fifo_a;
	plugin_data->in_fifo_b = in_fifo_b;
	plugin_data->out_accum = out_accum;
	plugin_data->out_fifo = out_fifo;
	plugin_data->plan_cr = plan_cr;
	plugin_data->plan_rc_a = plan_rc_a;
	plugin_data->plan_rc_b = plan_rc_b;
	plugin_data->real_a = real_a;
	plugin_data->real_b = real_b;
	plugin_data->window = window;

}

static void cleanupMorph(LADSPA_Handle instance) {
#line 78 "morph_1917.xml"
	Morph *plugin_data = (Morph *)instance;
	free(plugin_data->in_fifo_a);
	free(plugin_data->in_fifo_b);
	free(plugin_data->out_fifo);
	free(plugin_data->out_accum);
	free(plugin_data->real_a);
	free(plugin_data->real_b);
	free(plugin_data->comp_a);
	free(plugin_data->comp_b);
	free(plugin_data->window);
	free(instance);
}

static void connectPortMorph(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Morph *plugin;

	plugin = (Morph *)instance;
	switch (port) {
	case MORPH_MORPH:
		plugin->morph = data;
		break;
	case MORPH_LATENCY:
		plugin->latency = data;
		break;
	case MORPH_INPUT_A:
		plugin->input_a = data;
		break;
	case MORPH_INPUT_B:
		plugin->input_b = data;
		break;
	case MORPH_OUTPUT:
		plugin->output = data;
		break;
	}
}

static LADSPA_Handle instantiateMorph(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Morph *plugin_data = (Morph *)malloc(sizeof(Morph));
	fftw_real *comp_a = NULL;
	fftw_real *comp_b = NULL;
	long fifo_pos;
	LADSPA_Data *in_fifo_a = NULL;
	LADSPA_Data *in_fifo_b = NULL;
	LADSPA_Data *out_accum = NULL;
	LADSPA_Data *out_fifo = NULL;
	fft_plan plan_cr;
	fft_plan plan_rc_a;
	fft_plan plan_rc_b;
	fftw_real *real_a = NULL;
	fftw_real *real_b = NULL;
	float *window = NULL;

#line 43 "morph_1917.xml"
	int i;
	
	in_fifo_a = calloc(FFT_LENGTH, sizeof(LADSPA_Data));
	in_fifo_b = calloc(FFT_LENGTH, sizeof(LADSPA_Data));
	out_fifo = calloc(FFT_LENGTH, sizeof(LADSPA_Data));
	out_accum = calloc(FFT_LENGTH * 2, sizeof(LADSPA_Data));
	real_a = calloc(FFT_LENGTH, sizeof(fftw_real));
	real_b = calloc(FFT_LENGTH, sizeof(fftw_real));
	comp_a = calloc(FFT_LENGTH, sizeof(fftw_real));
	comp_b = calloc(FFT_LENGTH, sizeof(fftw_real));
	window = calloc(FFT_LENGTH, sizeof(float));
	fifo_pos = 0;
	
	#ifdef FFTW3
	plan_rc_a = fftwf_plan_r2r_1d(FFT_LENGTH, real_a, comp_a, FFTW_R2HC, FFTW_MEASURE);
	plan_rc_b = fftwf_plan_r2r_1d(FFT_LENGTH, real_b, comp_b, FFTW_R2HC, FFTW_MEASURE);
	plan_cr = fftwf_plan_r2r_1d(FFT_LENGTH, comp_a, real_a, FFTW_HC2R, FFTW_MEASURE);
	#else
	plan_rc_a = rfftw_create_plan(FFT_LENGTH, FFTW_REAL_TO_COMPLEX, FFTW_ESTIMATE);
	plan_rc_b = NULL;
	plan_cr = rfftw_create_plan(FFT_LENGTH, FFTW_COMPLEX_TO_REAL, FFTW_ESTIMATE);
	#endif
	
	// Create raised cosine window table
	for (i=0; i < FFT_LENGTH; i++) {
	        window[i] = -0.5f*cos(2.0f*M_PI*(double)i/(double)FFT_LENGTH)+0.5f;
	        window[i] *= 2.0f;
	}

	plugin_data->comp_a = comp_a;
	plugin_data->comp_b = comp_b;
	plugin_data->fifo_pos = fifo_pos;
	plugin_data->in_fifo_a = in_fifo_a;
	plugin_data->in_fifo_b = in_fifo_b;
	plugin_data->out_accum = out_accum;
	plugin_data->out_fifo = out_fifo;
	plugin_data->plan_cr = plan_cr;
	plugin_data->plan_rc_a = plan_rc_a;
	plugin_data->plan_rc_b = plan_rc_b;
	plugin_data->real_a = real_a;
	plugin_data->real_b = real_b;
	plugin_data->window = window;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

static void runMorph(LADSPA_Handle instance, unsigned long sample_count) {
	Morph *plugin_data = (Morph *)instance;

	/* Morph (float value) */
	const LADSPA_Data morph = *(plugin_data->morph);

	/* Input A (array of floats of length sample_count) */
	const LADSPA_Data * const input_a = plugin_data->input_a;

	/* Input A (array of floats of length sample_count) */
	const LADSPA_Data * const input_b = plugin_data->input_b;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data * const output = plugin_data->output;
	fftw_real * comp_a = plugin_data->comp_a;
	fftw_real * comp_b = plugin_data->comp_b;
	long fifo_pos = plugin_data->fifo_pos;
	LADSPA_Data * in_fifo_a = plugin_data->in_fifo_a;
	LADSPA_Data * in_fifo_b = plugin_data->in_fifo_b;
	LADSPA_Data * out_accum = plugin_data->out_accum;
	LADSPA_Data * out_fifo = plugin_data->out_fifo;
	fft_plan plan_cr = plugin_data->plan_cr;
	fft_plan plan_rc_a = plugin_data->plan_rc_a;
	fft_plan plan_rc_b = plugin_data->plan_rc_b;
	fftw_real * real_a = plugin_data->real_a;
	fftw_real * real_b = plugin_data->real_b;
	float * window = plugin_data->window;

#line 90 "morph_1917.xml"
	int i;
	unsigned long pos;
	
	int step_size = FFT_LENGTH / OVER_SAMP;
	int fft_latency = FFT_LENGTH - step_size;
	
	if (fifo_pos == 0) {
	        fifo_pos = fft_latency;
	}
	
	for (pos = 0; pos < sample_count; pos++) {
	        in_fifo_a[fifo_pos] = input_a[pos];
	        in_fifo_b[fifo_pos] = input_b[pos];
	        buffer_write(output[pos], out_fifo[fifo_pos-fft_latency]);
	        fifo_pos++;
	
	        // If the FIFO is full
	        if (fifo_pos >= FFT_LENGTH) {
	                fifo_pos = fft_latency;
	
	                // Window input FIFO
	                for (i=0; i < FFT_LENGTH; i++) {
	                        real_a[i] = in_fifo_a[i] * window[i];
	                        real_b[i] = in_fifo_b[i] * window[i];
	                }
	
	                // Run the real->complex transform
	#ifdef FFTW3
	                fftwf_execute(plan_rc_a);
	                fftwf_execute(plan_rc_b);
	#else
	                rfftw_one(plan_rc_a, real_a, comp_a);
	                rfftw_one(plan_rc_a, real_b, comp_b);
	#endif
	
	                // Multiply the bins magnitudes by the coeficients
	                for (i = 0; i < FFT_LENGTH/2; i++) {
	                        comp_a[i] = LIN_INTERP(morph, comp_a[i], comp_b[i]);
	                        comp_b[FFT_LENGTH-i] = LIN_INTERP(morph,
	                                 comp_a[FFT_LENGTH-i], comp_b[FFT_LENGTH-i]);
	                }
	
	                // Run the complex->real transform
	#ifdef FFTW3
	                fftwf_execute(plan_cr);
	#else
	                rfftw_one(plan_cr, comp_a, real_a);
	#endif
	
	                // Window into the output accumulator
	                for (i = 0; i < FFT_LENGTH; i++) {
	                        out_accum[i] += 0.4f * window[i] * real_a[i]/(FFT_LENGTH * OVER_SAMP);
	                }
	                for (i = 0; i < step_size; i++) {
	                        out_fifo[i] = out_accum[i];
	                }
	
	                // Shift output accumulator
	                memmove(out_accum, out_accum + step_size, FFT_LENGTH*sizeof(LADSPA_Data));
	
	                // Shift input fifo
	                for (i = 0; i < fft_latency; i++) {
	                        in_fifo_a[i] = in_fifo_a[i+step_size];
	                        in_fifo_b[i] = in_fifo_b[i+step_size];
	                }
	        }
	}
	
	// Store the fifo_position
	plugin_data->fifo_pos = fifo_pos;
	
	*(plugin_data->latency) = fft_latency;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

static void setRunAddingGainMorph(LADSPA_Handle instance, LADSPA_Data gain) {
	((Morph *)instance)->run_adding_gain = gain;
}

static void runAddingMorph(LADSPA_Handle instance, unsigned long sample_count) {
	Morph *plugin_data = (Morph *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Morph (float value) */
	const LADSPA_Data morph = *(plugin_data->morph);

	/* Input A (array of floats of length sample_count) */
	const LADSPA_Data * const input_a = plugin_data->input_a;

	/* Input A (array of floats of length sample_count) */
	const LADSPA_Data * const input_b = plugin_data->input_b;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data * const output = plugin_data->output;
	fftw_real * comp_a = plugin_data->comp_a;
	fftw_real * comp_b = plugin_data->comp_b;
	long fifo_pos = plugin_data->fifo_pos;
	LADSPA_Data * in_fifo_a = plugin_data->in_fifo_a;
	LADSPA_Data * in_fifo_b = plugin_data->in_fifo_b;
	LADSPA_Data * out_accum = plugin_data->out_accum;
	LADSPA_Data * out_fifo = plugin_data->out_fifo;
	fft_plan plan_cr = plugin_data->plan_cr;
	fft_plan plan_rc_a = plugin_data->plan_rc_a;
	fft_plan plan_rc_b = plugin_data->plan_rc_b;
	fftw_real * real_a = plugin_data->real_a;
	fftw_real * real_b = plugin_data->real_b;
	float * window = plugin_data->window;

#line 90 "morph_1917.xml"
	int i;
	unsigned long pos;
	
	int step_size = FFT_LENGTH / OVER_SAMP;
	int fft_latency = FFT_LENGTH - step_size;
	
	if (fifo_pos == 0) {
	        fifo_pos = fft_latency;
	}
	
	for (pos = 0; pos < sample_count; pos++) {
	        in_fifo_a[fifo_pos] = input_a[pos];
	        in_fifo_b[fifo_pos] = input_b[pos];
	        buffer_write(output[pos], out_fifo[fifo_pos-fft_latency]);
	        fifo_pos++;
	
	        // If the FIFO is full
	        if (fifo_pos >= FFT_LENGTH) {
	                fifo_pos = fft_latency;
	
	                // Window input FIFO
	                for (i=0; i < FFT_LENGTH; i++) {
	                        real_a[i] = in_fifo_a[i] * window[i];
	                        real_b[i] = in_fifo_b[i] * window[i];
	                }
	
	                // Run the real->complex transform
	#ifdef FFTW3
	                fftwf_execute(plan_rc_a);
	                fftwf_execute(plan_rc_b);
	#else
	                rfftw_one(plan_rc_a, real_a, comp_a);
	                rfftw_one(plan_rc_a, real_b, comp_b);
	#endif
	
	                // Multiply the bins magnitudes by the coeficients
	                for (i = 0; i < FFT_LENGTH/2; i++) {
	                        comp_a[i] = LIN_INTERP(morph, comp_a[i], comp_b[i]);
	                        comp_b[FFT_LENGTH-i] = LIN_INTERP(morph,
	                                 comp_a[FFT_LENGTH-i], comp_b[FFT_LENGTH-i]);
	                }
	
	                // Run the complex->real transform
	#ifdef FFTW3
	                fftwf_execute(plan_cr);
	#else
	                rfftw_one(plan_cr, comp_a, real_a);
	#endif
	
	                // Window into the output accumulator
	                for (i = 0; i < FFT_LENGTH; i++) {
	                        out_accum[i] += 0.4f * window[i] * real_a[i]/(FFT_LENGTH * OVER_SAMP);
	                }
	                for (i = 0; i < step_size; i++) {
	                        out_fifo[i] = out_accum[i];
	                }
	
	                // Shift output accumulator
	                memmove(out_accum, out_accum + step_size, FFT_LENGTH*sizeof(LADSPA_Data));
	
	                // Shift input fifo
	                for (i = 0; i < fft_latency; i++) {
	                        in_fifo_a[i] = in_fifo_a[i+step_size];
	                        in_fifo_b[i] = in_fifo_b[i+step_size];
	                }
	        }
	}
	
	// Store the fifo_position
	plugin_data->fifo_pos = fifo_pos;
	
	*(plugin_data->latency) = fft_latency;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

#ifdef ENABLE_NLS
#define D_(s) dgettext(PACKAGE, s)
	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, PACKAGE_LOCALE_DIR);
#else
#define D_(s) (s)
#endif


	morphDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (morphDescriptor) {
		morphDescriptor->UniqueID = 1917;
		morphDescriptor->Label = "morph";
		morphDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		morphDescriptor->Name =
		 D_("Spectral Morph");
		morphDescriptor->Maker =
		 "Steve Harris <steve@plugin.org.uk>";
		morphDescriptor->Copyright =
		 "GPL";
		morphDescriptor->PortCount = 5;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(5,
		 sizeof(LADSPA_PortDescriptor));
		morphDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(5,
		 sizeof(LADSPA_PortRangeHint));
		morphDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(5, sizeof(char*));
		morphDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Morph */
		port_descriptors[MORPH_MORPH] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[MORPH_MORPH] =
		 D_("Morph");
		port_range_hints[MORPH_MORPH].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
		port_range_hints[MORPH_MORPH].LowerBound = 0;
		port_range_hints[MORPH_MORPH].UpperBound = 1;

		/* Parameters for Latency */
		port_descriptors[MORPH_LATENCY] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_CONTROL;
		port_names[MORPH_LATENCY] =
		 D_("Latency");
		port_range_hints[MORPH_LATENCY].HintDescriptor = 0;

		/* Parameters for Input A */
		port_descriptors[MORPH_INPUT_A] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[MORPH_INPUT_A] =
		 D_("Input A");
		port_range_hints[MORPH_INPUT_A].HintDescriptor = 0;

		/* Parameters for Input A */
		port_descriptors[MORPH_INPUT_B] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[MORPH_INPUT_B] =
		 D_("Input A");
		port_range_hints[MORPH_INPUT_B].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[MORPH_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[MORPH_OUTPUT] =
		 D_("Output");
		port_range_hints[MORPH_OUTPUT].HintDescriptor = 0;

		morphDescriptor->activate = activateMorph;
		morphDescriptor->cleanup = cleanupMorph;
		morphDescriptor->connect_port = connectPortMorph;
		morphDescriptor->deactivate = NULL;
		morphDescriptor->instantiate = instantiateMorph;
		morphDescriptor->run = runMorph;
		morphDescriptor->run_adding = runAddingMorph;
		morphDescriptor->set_run_adding_gain = setRunAddingGainMorph;
	}
}

void _fini() {
	if (morphDescriptor) {
		free((LADSPA_PortDescriptor *)morphDescriptor->PortDescriptors);
		free((char **)morphDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)morphDescriptor->PortRangeHints);
		free(morphDescriptor);
	}

}
