/*
   SwingWT
   Copyright(c)2003-2005, R. Rawson-Tetley

   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.

   Contact me by electronic mail: bobintetley@users.sourceforge.net

   $Log: Font.java,v $
   Revision 1.16  2005/03/08 14:03:32  bobintetley
   Font.deriveFont

   Revision 1.15  2005/01/25 10:07:07  bobintetley
   Font family support and GraphicsEnvironment implementation

   Revision 1.14  2005/01/05 09:22:25  bobintetley
   Updated copyright year on source

   Revision 1.13  2005/01/05 08:37:11  bobintetley
   Many compatibility fixes from David Barron

   Revision 1.12  2004/10/30 20:11:54  bobintetley
   Code cleanup

   Revision 1.11  2004/05/06 12:35:21  bobintetley
   Parity with Swing constants for Binary Compatibility + fixes to JDesktopPane

   Revision 1.10  2004/05/05 12:43:19  bobintetley
   Patches/new files from Laurent Martell

   Revision 1.9  2004/04/16 10:19:06  dannaab
   Misc bug fixes, InputMap implementation, preliminary undo support

   Revision 1.8  2004/01/27 12:10:15  bobintetley
   Win32 compatibility fixes

   Revision 1.7  2004/01/26 08:10:59  bobintetley
   Many bugfixes and addition of SwingSet

   Revision 1.6  2003/12/14 09:13:38  bobintetley
   Added CVS log to source headers

*/

package swingwt.awt;

import swingwtx.swing.SwingUtilities;
import swingwtx.swing.SwingWTUtils;

import org.eclipse.swt.graphics.FontData;

import java.util.*;

public class Font {
    
    private org.eclipse.swt.graphics.Font peer = null;
    
    /** Destroy the font when finalize is called. Not required for
     *  the constructor passing the font - only do this when this
     *  instance created the SWT font */
    protected boolean disposeOnFinalize = true;

    public static final int PLAIN = 0;
    public static final int BOLD = 1;
    public static final int ITALIC = 2;
    public static final int ROMAN_BASELINE = 0;
    public static final int CENTER_BASELINE = 1;
    public static final int HANGING_BASELINE = 2;
    public static final int TRUETYPE_FONT = 0;
    
    protected String name = "Dialog";
    protected int style = PLAIN;
    protected int size = 12;
    
    public Font(final String name, final int style, final int size) {
        this.name = name;
        this.style = style;
        this.size = size;
        createPeer();
    }
    
    public Font(final String name, final int style, final int size, boolean f) {
        this.name = name;
        this.style = style;
        this.size = size;
        createPeer();
    }
    
    public Font(Map attributes) {
        setAttributes(attributes);
        createPeer();
    }
    
    /** Creates a font from a matching SWT font */
    public Font(org.eclipse.swt.graphics.Font swtfont) {
        peer = swtfont;
        populateFromSWTFontData(swtfont.getFontData()[0]);
    }
    
    /** Creates a font from the SWT fontdata */
    public Font(org.eclipse.swt.graphics.FontData fd) {
        peer = new org.eclipse.swt.graphics.Font(SwingWTUtils.getDisplay(), fd);
        populateFromSWTFontData(fd);
    }
    
    public static Font getFont(Map attributes) {
        return new Font(attributes);    
    }
    
    // TODO temporary
    private static Font stubFont = null;
    public static Font getStubFont()
    {
        if (stubFont == null) {
            SwingUtilities.invokeSync( new Runnable() {
                public void run() {
                    stubFont = new Font(swingwtx.swing.SwingWTUtils.getDisplay().getSystemFont());
                }
            });
        }
        return stubFont;
    }
    
    /** Creates the SWT peer from the Font info we have */
    protected void createPeer() {
        SwingUtilities.invokeSync( new Runnable() {
            public void run() {
                // Map AWT constants to SWT
                int swtStyle = org.eclipse.swt.SWT.NONE;
                if ((style & BOLD) > 0)
                    swtStyle = swtStyle | org.eclipse.swt.SWT.BOLD;
                if ((style & ITALIC) > 0)
                    swtStyle = swtStyle | org.eclipse.swt.SWT.ITALIC;
                peer = new org.eclipse.swt.graphics.Font(swingwtx.swing.SwingWTUtils.getDisplay(), name, size, swtStyle);
                // If this class created it, it's up to it to 
                // destroy the peer.
                disposeOnFinalize = true;
            }
        }); 
    }
    
    /** Sets font properties based on a SWT font */
    protected void populateFromSWTFontData(org.eclipse.swt.graphics.FontData fd) {
        
        // Must be on dispatch thread if we have a real SWT font to throw about
        disposeOnFinalize = false;

        this.name = fd.getName();
        style = PLAIN;
        if ((fd.getStyle() & org.eclipse.swt.SWT.BOLD) > 0)
            style = style | BOLD;
        if ((fd.getStyle() & org.eclipse.swt.SWT.ITALIC) > 0)
            style = style | ITALIC;
        size = fd.getHeight();
    }
    
    public void setAttributes(Map attributes) {
        if (attributes != null) {
            String af = attributes.get("family").toString();
            if (af != null) this.name = af;
            String as = attributes.get("size").toString();
            if (as != null) this.size = Integer.parseInt(as);
            String ast = attributes.get("style").toString();
            if (ast != null) this.style = Integer.parseInt(ast);
        }   
    }
    
    public Map getAttributes() {
        Hashtable ht = new Hashtable();
        ht.put("family", this.name);
        ht.put("size", new Integer(this.size));
        ht.put("style", new Integer(this.style));
        return ht;
    }
    
    public String getName() { return name; }
    public String getFontName() { return name; }
    public String getFamily() { return name; }
    public String getFontName(Locale l) { return getFontName(); }
    public String getFamily(Locale l) { return getFamily(); }
    
    public int getStyle() { return style; }
    public int getSize() { return size; }
    public float getSize2D() { return (float) size; }
    
    public boolean isBold() { return (style & BOLD) > 0; }
    public boolean isPlain() { return style == PLAIN; }
    public boolean isItalic() { return (style & ITALIC) > 0; }
    
    public Font deriveFont(float newSize) { return new Font(name, style, (int) newSize); }    
    public Font deriveFont(int newStyle) { return new Font(name, newStyle, size); }
    public Font deriveFont(int newStyle, float newSize) { return new Font(name, newStyle, (int) newSize); }
    
    public org.eclipse.swt.graphics.Font getSWTFont() {
        return peer;    
    }
    
    public void dispose() {
        peer.dispose();    
    }
    
    protected void finalize() throws Throwable {
        if (disposeOnFinalize) dispose();
    }

   
}
