/*
   SwingWT
   Copyright(c)2003-2005, R. Rawson-Tetley

   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.

   Contact me by electronic mail: bobintetley@users.sourceforge.net

   $Log: AbstractButton.java,v $
   Revision 1.33  2005/02/25 10:21:45  bobintetley
   Compatibility deprecated methods (Paul Bolle)

   Revision 1.32  2005/01/05 09:22:30  bobintetley
   Updated copyright year on source

   Revision 1.31  2005/01/05 08:37:13  bobintetley
   Many compatibility fixes from David Barron

   Revision 1.30  2004/10/30 20:11:57  bobintetley
   Code cleanup

   Revision 1.29  2004/06/15 09:40:28  bobintetley
   ActionCommand defaults to button text if none supplied like Swing

   Revision 1.28  2004/05/06 12:35:22  bobintetley
   Parity with Swing constants for Binary Compatibility + fixes to JDesktopPane

   Revision 1.27  2004/05/05 12:43:21  bobintetley
   Patches/new files from Laurent Martell

   Revision 1.26  2004/04/30 23:18:26  dannaab
   List selection support, misc bug fixes

   Revision 1.25  2004/04/06 12:46:45  bobintetley
   ButtonDemo/ListDemo re-enabled for SwingSet2

   Revision 1.24  2004/03/30 10:42:46  bobintetley
   Many minor bug fixes, event improvements by Dan Naab. Full swing.Icon support

   Revision 1.23  2004/03/12 14:21:47  bobintetley
   Fix to threading issue and removal of debug messages

   Revision 1.22  2004/03/01 15:58:47  bobintetley
   Various little bug fixes

   Revision 1.21  2004/02/19 09:58:44  bobintetley
   Various small bug fixes and JTextArea should be much faster/lighter

   Revision 1.20  2004/02/13 15:09:23  bobintetley
   JComboBox/Abstract button non-peer selection and JTable threading fixed

   Revision 1.19  2004/01/26 08:11:00  bobintetley
   Many bugfixes and addition of SwingSet

   Revision 1.18  2004/01/23 08:04:56  bobintetley
   JComboBox fixes and better Action implementation

   Revision 1.17  2004/01/20 07:38:05  bobintetley
   Bug fixes and compatibility methods

   Revision 1.16  2004/01/05 02:50:39  djspiewak
   Added JToolBar peer functionality and commenced AWT layout manager support

   Revision 1.15  2004/01/02 10:50:50  bobintetley
   Button mnemonic/tooltip fixes

   Revision 1.14  2003/12/22 09:57:23  bobintetley
   Fixed broken ItemEvent calls in AbstractButton

   Revision 1.13  2003/12/17 16:35:52  bobintetley
   Mnemonics are no longer case sensitive

   Revision 1.12  2003/12/17 16:30:35  bobintetley
   Flowlayout fix, vertical toolbar support and cleaned up text alignment
   hierarchy.

   Revision 1.11  2003/12/16 19:04:38  bobintetley
   Fix to broken mnemonics

   Revision 1.10  2003/12/16 18:04:10  bobintetley
   Fixes to handling of mnemonics

   Revision 1.9  2003/12/16 17:46:17  bobintetley
   Additional thread safety methods

   Revision 1.8  2003/12/16 14:08:05  bobintetley
   Corrected event hierarchy for Button ActionEvents

   Revision 1.7  2003/12/16 13:14:33  bobintetley
   Use of SwingWTUtils.isSWTControlAvailable instead of null test

   Revision 1.6  2003/12/16 12:23:31  bobintetley
   Corrected handling of table selection + keyboard action events

   Revision 1.5  2003/12/14 09:13:38  bobintetley
   Added CVS log to source headers

*/

package swingwtx.swing;

import swingwt.awt.event.*;
import swingwtx.swing.event.*;
import org.eclipse.swt.widgets.*;

import java.util.*;

/**
 * Swing Abstract Button superclass for improved
 * swingy-ness (and correct ActionEvent command)
 * Written by Diane Trout, few changes and fixes up by me 
 * to provide a better hierarchy and prevent ActionEvents
 * being fired twice.
 *
 * @author Diane Trout
 * @author Robin Rawson-Tetley
 * @author Brian Sant
 */
 public abstract class AbstractButton extends JComponent implements SwingConstants {
     
     protected Button ppeer = null;
     protected String pText = null;
     protected ButtonModel pModel;
     protected ButtonGroup pGroup;
     protected char pMnemonic = ' ';
     protected boolean pSelection = false;
     protected int pHTextPosition = LEFT;
     protected int pVTextPosition = TOP;
     protected int pHAlign = LEFT;
     protected int pVAlign = TOP;
     
     protected Icon pIcon = null;
     protected Action pAction = null;
     
     protected Vector changeListeners = new Vector();
     protected Vector itemListeners = new Vector();
     
     public void setAction(Action a) {
        if (a == null || a.equals(pAction)) return;
      
        if (pAction != null) { removeActionListener(pAction); }
        if (a instanceof ActionListener) { addActionListener(a); }
      
        if (a.getValue(Action.SMALL_ICON) != null)
           setIcon((Icon) a.getValue(Action.SMALL_ICON));
        if (a.getValue(Action.NAME) != null)
           setText((String) a.getValue(Action.NAME));
        if (a.getValue(Action.SHORT_DESCRIPTION) != null)
           setToolTipText((String) a.getValue(Action.SHORT_DESCRIPTION));
        if (a.getValue(Action.MNEMONIC_KEY) != null)         
           setMnemonic(((Integer) a.getValue(Action.MNEMONIC_KEY)).intValue());
      
        // LONG_DESCRIPTION is not used in the orginal Swing implementation.
        // The javadoc mentions that the LONG_DESCRIPTION may be used in
        // an application's "context sensitive help", but Swing components
        // don't use it directly.
      
        setEnabled(a.isEnabled());
        pAction = a;      
     }
     
     /** 
      * Works like setAction.
      * @param a The action to set
      * @param addAsListener Whether or not to add the action as a listener to the button.
      *        This method is used by JToolBar with wrappers and prevents running out of
      *        stack space by getting into an endless loop of actions that fire events.
      */
     protected void setAction(Action a, boolean addAsListener) {
          setAction(a);
          removeActionListener(a);
     }
     
     public Action getAction() { return pAction; }
     
     /** If the action command is null, return the button text like Swing */
     public String getActionCommand() { 
	 if (actionCommand == null)
	     return getText();
	 else if (actionCommand.equals(""))
	     return getText();
	 else
             return actionCommand;
     }
     
     public void setIcon(Icon icon) {}
     public void setPressedIcon(Icon icon) {}
     public void setRolloverIcon(Icon icon) {}
     public void setRolloverSelectedIcon(Icon icon) {}
     public void setDisabledIcon(Icon icon) {}
     public void setSelectedIcon(Icon icon) {}
     
    public void setBorderPainted(boolean b) {}
    public void setFocusPainted(boolean b) {}
    public void setContentAreaFilled(boolean b) {}
     
     public void addChangeListener(ChangeListener l) {
        changeListeners.add(l);
     }
     
     public void removeChangeListener(ChangeListener l) {
        changeListeners.remove(l);
     }
     
    
    public void addItemListener(ItemListener l) {
        itemListeners.add(l);
    }
    
    public void removeItemListener(ItemListener l) {
        itemListeners.remove(l);
    }
    
    
    /** Overridden as we use the Button selection for action events */
    protected void registerActionEvents() {
        // Handle selection as ActionEvent
        ppeer.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
            public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
                processActionEvent(0);
            }
            public void widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent e) {}
        });    
    }
    
    public void setHorizontalAlignment(final int align) { pHAlign = align; SwingUtilities.invokeSync(new Runnable() { public void run() { if (SwingWTUtils.isSWTControlAvailable(ppeer)) ppeer.setAlignment(SwingWTUtils.translateSwingAlignmentConstant(align) | SwingWTUtils.translateSwingAlignmentConstant(pVAlign));}}); }
    public void setVerticalAlignment(final int align) { pVAlign = align; SwingUtilities.invokeSync(new Runnable() { public void run() {if (SwingWTUtils.isSWTControlAvailable(ppeer))ppeer.setAlignment(SwingWTUtils.translateSwingAlignmentConstant(align) | SwingWTUtils.translateSwingAlignmentConstant(pHAlign));}});}
    public int getHorizontalAlignment() { return pHAlign; }
    public int getVerticalAlignment() { return pVAlign; }
    public void setHorizontalTextPosition(int textpos) { setHorizontalAlignment(textpos); }
    public void setVerticalTextPosition(int textpos) { setVerticalAlignment(textpos); }
    public int getHorizontalTextPosition() { return getHorizontalAlignment(); }
    public int getVerticalTextPosition() { return getVerticalAlignment(); }
    
   /**
    *  Sends action events to listeners.
    */
     public void processActionEvent(int id) {
         
         ActionEvent ae = new ActionEvent(this, id, this.getActionCommand());
         for (int i = 0; i < actionListeners.size(); i++) {
             ActionListener al = (ActionListener) actionListeners.get(i);
             al.actionPerformed(ae);
         }
         
         // Send the actions as changes
         ChangeEvent ce = new ChangeEvent(this);
         for (int i = 0; i < changeListeners.size(); i++) {
            ChangeListener cl = (ChangeListener) changeListeners.get(i);
            cl.stateChanged(ce);
         }
         
         // Send item events too
         processItemEvent();
     }
     
    /**
     * Handles firing of Item events for when selection changes
     */
    public void processItemEvent() {
        if (itemListeners.size() == 0) return;
        boolean isSelected = pSelection;
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) isSelected = ppeer.getSelection();
        ItemEvent e = new ItemEvent(this, 0, this, (isSelected ? ItemEvent.SELECTED : ItemEvent.DESELECTED));
        
        for (int i = 0; i < itemListeners.size(); i++) {
            ItemListener il = (ItemListener) itemListeners.get(i);
            il.itemStateChanged(e);
        }
        // If the item is selected, and it is part of a group, tell the group
        if (isSelected && pGroup != null) {
            pGroup.setSelected(this.getModel(), true);
        }
    }

     public String getLabel() { return getText(); }
     public void setLabel(String text) { setText(text); }
     public String getText() { return pText; }
     public void setText(String text) {
         pText = text; 
         if (pText == null) pText = "";
         showMnemonic();
         if (SwingWTUtils.isSWTControlAvailable(ppeer)) 
             SwingUtilities.invokeSync(new Runnable() { 
                 public void run() {
                     ppeer.setText(pText);
                 }
             }); 
     }
     
    public int getMnemonic() { return pMnemonic; };
    public void setMnemonic(char mnemonic) { pMnemonic = mnemonic; showMnemonic(); }
    public void setMnemonic(int mnemonic) { pMnemonic = (char) mnemonic; showMnemonic(); }
    
    protected void showMnemonic() {
        final Object me = this;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                
                // Stop if we don't have any cached text
                if (pText == null) return;
                
                // Strip out any HTML
                if (pText.indexOf("<") != -1)
                    pText = SwingWTUtils.removeHTML(pText);
                
                // If we don't have a mnemonic, don't do anything
                if (pMnemonic == ' ') return;
                String text = ( (!SwingWTUtils.isSWTControlAvailable(ppeer)) ? pText : ppeer.getText());
                
                // Sort out the tooltip for the mnemonic if this is a push button
                // of some type (check/radio is more obviously visible)
                if (me instanceof JToggleButton || me instanceof JButton)
                    if (pToolTipText.indexOf("(ALT+") == -1) {
                        pToolTipText += " (ALT+" + new String(new char[] { pMnemonic }).toUpperCase()+ ")";
                        SwingUtilities.invokeSync(new Runnable() { 
                            public void run() {
                                if (SwingWTUtils.isSWTControlAvailable(peer)) 
                                    peer.setToolTipText(pToolTipText);
                            }
                        }); 
                    }
                
                // If there is already a mnemonic, remove it
                if (text.indexOf("&") != -1) {
                    text = SwingWTUtils.replace(text, "&", "");
                }
                
                // If there is no text, then set the text to just be the mnemonic
                if (text.equals("")) {
                    pText = "&" + new String(new char[] {pMnemonic}).toLowerCase();
                    if (SwingWTUtils.isSWTControlAvailable(ppeer)) ppeer.setText(text);
                }
                else {
                    // Convert the mnemonic and text to a lower case string to make 
                    // the match case insensitive
                    String lower = text.toLowerCase();
                    String mn = new String(new char[] { pMnemonic }).toLowerCase();
                    int pos = lower.indexOf(mn);

                    if (pos != -1) {
                        text = text.substring(0, pos) + "&" + text.substring(pos, text.length());
                        pText = text;
                        if (SwingWTUtils.isSWTControlAvailable(ppeer)) ppeer.setText(text);
                    }
                }
            }
        });
    }

     public abstract boolean isSelected();
     public abstract void setSelected(boolean b);

    /** Sets the tooltip text used on the component - overridden here,
      * so that if we have a mnemonic, we can tack it on the end. Only applies
      * to JButton and JToggleButton */
    public void setToolTipText(final String text) { 
        pToolTipText = text; 
        
        if (this instanceof JToggleButton || this instanceof JButton)
            if ( pMnemonic != ' ')
                if (pToolTipText.indexOf("(ALT+") == -1)
                    pToolTipText += " (ALT+" + new String(new char[] { pMnemonic }).toUpperCase()+ ")";
                
        SwingUtilities.invokeSync(new Runnable() { 
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(peer)) 
                    peer.setToolTipText(pToolTipText);
            }
        }); 
        
    }
     
     public void setModel(ButtonModel m) { pModel = m; }
     public ButtonModel getModel() { return pModel; }
     public void setGroup(ButtonGroup g) { pGroup = g; }

    public void doClick()
    {
        processActionEvent(0);
    }
 }
