/*
   SwingWT
   Copyright(c)2003-2005, R. Rawson-Tetley

   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.

   Contact me by electronic mail: bobintetley@users.sourceforge.net

   $Log: JLabel.java,v $
   Revision 1.31  2005/01/05 09:22:31  bobintetley
   Updated copyright year on source

   Revision 1.30  2005/01/05 08:37:13  bobintetley
   Many compatibility fixes from David Barron

   Revision 1.29  2004/10/30 20:11:57  bobintetley
   Code cleanup

   Revision 1.28  2004/05/06 12:35:22  bobintetley
   Parity with Swing constants for Binary Compatibility + fixes to JDesktopPane

   Revision 1.27  2004/05/05 12:43:21  bobintetley
   Patches/new files from Laurent Martell

   Revision 1.26  2004/04/30 23:18:26  dannaab
   List selection support, misc bug fixes

   Revision 1.25  2004/04/30 13:20:43  bobintetley
   Fix to unrealised peer preferred sizes, forwarding window events to
   content panes and fix for mouse drag events.

   Revision 1.24  2004/04/28 08:38:11  bobintetley
   Hierarchy fixes, code cleanup for base classes, additional javadocs and use of flag to identify JComponent descendants with peers

   Revision 1.23  2004/04/16 10:19:06  dannaab
   Misc bug fixes, InputMap implementation, preliminary undo support

   Revision 1.22  2004/03/30 10:42:46  bobintetley
   Many minor bug fixes, event improvements by Dan Naab. Full swing.Icon support

   Revision 1.21  2004/03/01 15:58:47  bobintetley
   Various little bug fixes

   Revision 1.20  2004/02/19 09:58:44  bobintetley
   Various small bug fixes and JTextArea should be much faster/lighter

   Revision 1.19  2004/01/26 10:57:45  bobintetley
   HTML handling (throws it away - SWT can't do anything with it)

   Revision 1.18  2004/01/26 08:11:00  bobintetley
   Many bugfixes and addition of SwingSet

   Revision 1.17  2003/12/22 14:03:22  bobintetley
   Layout updates after loading new images into labels

   Revision 1.16  2003/12/16 17:46:17  bobintetley
   Additional thread safety methods

   Revision 1.15  2003/12/16 15:47:45  bobintetley
   Thread safety added to common methods

   Revision 1.14  2003/12/16 13:14:33  bobintetley
   Use of SwingWTUtils.isSWTControlAvailable instead of null test

   Revision 1.13  2003/12/16 09:25:42  bobintetley
   Fixed label images

   Revision 1.12  2003/12/15 18:29:57  bobintetley
   Changed setParent() method to setSwingWTParent() to avoid conflicts with applications

   Revision 1.11  2003/12/15 15:52:48  bobintetley
   Alignment methods

   Revision 1.10  2003/12/14 09:13:38  bobintetley
   Added CVS log to source headers

*/


package swingwtx.swing;

import org.eclipse.swt.*;

import swingwt.awt.*;

public class JLabel extends swingwtx.swing.JComponent implements SwingConstants {

    /** The label's peer */
    protected org.eclipse.swt.widgets.Label ppeer = null;
    /** A cache of the label's text */
    protected String pText = "";
    /** A cache of the label's icon */
    protected Icon pImage = null;
    /** The label's horizontal text position */
    protected int pHTextPosition = LEFT;
    /** The label's vertical text position */
    protected int pVTextPosition = TOP;
    /** The label's horizontal alignment */
    protected int pHAlign = LEFT;
    /** The label's vertical alignment */
    protected int pVAlign = TOP;
    /** If this is the label for a particular component, then show the one it's tied to */
    protected Component labelFor = null;
    
    /** Creates a new JLabel with empty text and no icon */
    public JLabel() { }
    /** Creates a new JLabel with the specified text 
     *  @param text The text of the label
     */
    public JLabel(String text) { pText = text;  }
    /** Creates a new JLabel with the specified text and alignment 
     *  @param text The text of the label
     *  @param align The label's horizontal alignment
     */
    public JLabel(String text, int align) { pText = text; setHorizontalAlignment(align); }
    /** Creates a new JLabel with the specified icon 
     *  @param icon The label's icon
     */
    public JLabel(Icon icon) { pImage = icon; }

    public JLabel(String text, Icon icon, int align) { pText = text; pImage = icon; setHorizontalAlignment(align); }

    public JLabel(Icon icon, int alignment)
    {
        this("", icon, alignment);
    }
    /**
     * @return The text on the label
     */
    public String getText()
    {
        return pText;
    }

    /**
     * Sets the label's text
     * @param text The new text
     */
    public void setText(String text) {
        pText = text;
        if (pText == null) pText = "";
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(ppeer))
                    ppeer.setText(SwingWTUtils.removeHTML(pText));
                if (parent != null) {
                    // Invalidate parent in case our size has changed.
                    parent.invalidate();
                    repaint();
                }
            }
        });
    }
    
    /**
     * If this label is accompanying a component, the component
     * @param c The component this label is accompanying
     */
    public void setLabelFor(Component c) { labelFor = c; }
    /**
     * If this label is accompanying a component, the component
     * @return The component
     */
    public Component getLabelFor() { return labelFor; }
    
    /** NOT IMPLEMENTED */
    public void setDisplayedMnemonic(char c) {}
    /** NOT IMPLEMENTED */
    public void setDisplayedMnemonic(int c) {}
    
    /**
     * Sets the icon for this label
     * @param icon The icon to display in the label
     */
    public void setIcon(Icon icon) { 
        pImage = icon; 
        final JLabel pthis = this;
        SwingUtilities.invokeSync(new Runnable() { 
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
                    if (pImage == null)
                        ppeer.setImage(null);
                    else
                        ppeer.setImage(SwingWTUtils.getSWTImageFromSwingIcon(pthis, pImage));
                    // Tell the container to layout again in case the image is large
                    if (parent != null) {
                        parent.invalidate();
                        repaint();
                    }
                }
            }
        }); 
    }
    
    /** Gets the icon for this label
     *  @return The icon
     */
    public Icon getIcon() { return pImage; }
    
    /** NOT IMPLEMENTED */
    public Icon getDisabledIcon() { return null; }
    /** NOT IMPLEMENTED */
    public void setDisabledIcon(Icon icon) { }
    
    /**
     * Sets the horizontal alignment for this label. Use one of 
     * LEADING, LEFT, CENTER, RIGHT, TRAILING from SwingConstants
     * @param align The alignment
     */
    public void setHorizontalAlignment(final int align) { pHAlign = align; SwingUtilities.invokeAsync(new Runnable() { public void run() { if (SwingWTUtils.isSWTControlAvailable(ppeer)) ppeer.setAlignment(SwingWTUtils.translateSwingAlignmentConstant(align) | SwingWTUtils.translateSwingAlignmentConstant(pVAlign));}}); }
    /**
     * Sets the vertical alignment for this label. Use one of 
     * LEADING, LEFT, CENTER, RIGHT, TRAILING from SwingConstants
     * @param align The alignment
     */
    public void setVerticalAlignment(final int align) { pVAlign = align; SwingUtilities.invokeAsync(new Runnable() { public void run() {if (SwingWTUtils.isSWTControlAvailable(ppeer))ppeer.setAlignment(SwingWTUtils.translateSwingAlignmentConstant(align) | SwingWTUtils.translateSwingAlignmentConstant(pHAlign));}});}
    /**
     * Gets the horizontal alignment for the label
     * @return The horizontal alignment
     */
    public int getHorizontalAlignment() { return pHAlign; }
    /**
     * Gets the vertical alignment for the label
     * @return The vertical alignment
     */
    public int getVerticalAlignment() { return pVAlign; }
    /** No meaning in native, calls across to setHorizontalAlignment() */
    public void setHorizontalTextPosition(int textpos) { setHorizontalAlignment(textpos); }
    /** No meaning in native, calls across to setVerticalAlignment() */
    public void setVerticalTextPosition(int textpos) { setVerticalAlignment(textpos); }
    /** No meaning in native, calls across to getHorizontalAlignment() */
    public int getHorizontalTextPosition() { return getHorizontalAlignment(); }
    /** No meaning in native, calls across to getVerticalAlignment() */
    public int getVerticalTextPosition() { return getVerticalAlignment(); }
    
    /** Overriden to calculate non-realised
     *  preferred size.
     */
    protected swingwt.awt.Dimension calculatePreferredSize() {
        swingwt.awt.Dimension size = new swingwt.awt.Dimension( 
            SwingWTUtils.getRenderStringWidth(pText), 
            SwingWTUtils.getRenderStringHeight(pText));
        setSize(size);
        return size;
    }
    
    /**
     * Once a parent component receives an "add" call for a child, this being
     * the child, this should be called to tell us to instantiate the peer
     * and load in any cached properties.
     */
    public void setSwingWTParent(swingwt.awt.Container parent) throws Exception {
        descendantHasPeer = true;
        ppeer = new org.eclipse.swt.widgets.Label(parent.getComposite(), SWT.NONE);
        if (pText != null) ppeer.setText(SwingWTUtils.removeHTML(pText));
        ppeer.setAlignment(SwingWTUtils.translateSwingAlignmentConstant(pHAlign) | 
            SwingWTUtils.translateSwingAlignmentConstant(pVAlign));
        if (pImage != null) setIcon(pImage);
        peer = ppeer;
        this.parent = parent;
    }
}
