/*
   SwingWT
   Copyright(c)2003-2005, R. Rawson-Tetley

   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.

   Contact me by electronic mail: bobintetley@users.sourceforge.net

   $Log: JSpinner.java,v $
   Revision 1.7  2005/01/05 09:22:31  bobintetley
   Updated copyright year on source

   Revision 1.6  2004/11/02 12:50:10  bobintetley
   Fix to editorpane display and spinner column size

   Revision 1.5  2004/11/02 12:43:25  bobintetley
   Fixed bad filename and size of JSpinner

   Revision 1.4  2004/10/30 20:11:57  bobintetley
   Code cleanup

   Revision 1.3  2004/05/07 12:11:18  bobintetley
   Default layout fixes and correct behaviour for null layout

   Revision 1.2  2004/04/30 13:20:43  bobintetley
   Fix to unrealised peer preferred sizes, forwarding window events to
   content panes and fix for mouse drag events.

   Revision 1.1  2004/04/18 14:21:50  bobintetley
   JSpinner implementation


*/

package swingwtx.swing;

import swingwt.awt.*;
import swingwt.awt.event.*;
import swingwtx.swing.event.*;

import java.util.*;

/**
 * Implementation of Swing's JSpinner class. We actually use
 * a text box and a vertical scrollbar to get the same effect
 * with native widgets.
 *
 * @author  Robin Rawson-Tetley
 */
public class JSpinner extends JPanel implements ChangeListener {
    
    /** Item events */
    protected Vector changeListeners = new Vector();
    /** Model */
    protected SpinnerModel model = null;
    /** Composite Components */
    protected JTextField text = new JTextField(1);
    protected JScrollBar spin = new JScrollBar(JScrollBar.VERTICAL);
    
    protected int lastValue = 300000;
    
    public JSpinner() { this(new SpinnerNumberModel()); }
    
    public JSpinner(SpinnerModel model) { 
        
        super();
        
        setPreferredSize(new Dimension(100, 30));
        setModel(model);
        setLayout(new BorderLayout()); 
        
        // Watch for cursor keys
        text.addKeyListener(new KeyAdapter() {
            public void keyPressed(KeyEvent e) {
                if (e.getKeyCode() == 16777218) {
                    nextItem();                 // Go the next item with cursor down
                    e.consume();
                }
                else if (e.getKeyCode() == 16777217) {
                    previousItem();             // Go to the previous item with cursor up
                    e.consume();
                }
            }
        });
        
        // Watch for changes to the scroller
        spin.setMinimum(0);
        spin.setMaximum(650000);
        spin.setValue(30000);
        spin.addAdjustmentListener(new AdjustmentListener() {
            public void adjustmentValueChanged(AdjustmentEvent e) {
                if (e.getValue() > lastValue)
                    previousItem();
                else
                    nextItem();
                lastValue = e.getValue();
            }
        });
        
        // Add the components
        add(text, BorderLayout.CENTER);
        add(spin, BorderLayout.EAST);
    }
    
    protected final static int CANCELED = 0;
    protected final static int INVISIBLE = 1;
    protected final static int VISIBLE = 2;
    
    public void setModel(SpinnerModel model) {
        this.model = model;
        model.addChangeListener(this);
    }
    
    public SpinnerModel getModel() {
        return model;
    }
    
    public void addFocusListener(FocusListener l) {
        text.addFocusListener(l);    
    }
    public void removeFocusListener(FocusListener l) {
        text.removeFocusListener(l);    
    }
    public void addKeyListener(KeyListener l) {
        text.addKeyListener(l);    
    }
    public void removeKeyListener(KeyListener l) {
        text.removeKeyListener(l);    
    }
    
    public void addChangeListener(ChangeListener l) {
        changeListeners.add(l);
    }
    
    public void removeItemListener(ChangeListener l) {
        changeListeners.remove(l);    
    }
    
    public void processChangeEvent(ChangeEvent e) {
        Iterator i = changeListeners.iterator();
        while (i.hasNext()) {
            ChangeListener l = (ChangeListener) i.next();
            l.stateChanged(e);
        }
    }
    
    public void setValue(Object o) {
        model.setValue(o);
    }
    
    public Object getValue() {
        return model.getValue();
    }
    
    
    public void setEnabled(boolean b) {
        text.setEnabled(b);
        spin.setEnabled(b);
    }
    
    public boolean isEnabled() {
        return spin.isEnabled();    
    }
    
    public String getToolTipText() {
        return text.getToolTipText();
    }
    
    public void setToolTipText(String tip) {
        text.setToolTipText(tip);    
    }
    
    protected void previousItem() {  
        model.setValue( model.getPreviousValue() );
    }
    protected void nextItem() { 
        model.setValue( model.getNextValue() );
    }

    public void requestFocus() {
        text.requestFocus();
    }

    public void grabFocus() {
	text.grabFocus();
    }

    public JTextField getJTextField() {
	return text;
    }
    
    public void stateChanged(ChangeEvent e) {
        this.text.setText( model.getValue().toString());
    }
    
    /** Component displays oddly since we used a vertical scrollbar. This
     *  should lock it to no more than 30 pixels in height when LayoutManagers
     *  try to update it
     */
    public void setBounds(int x, int y, int width, int height) {
        if (height > 30) height = 30;
        super.setBounds(x, y, width, height);
    }
    
    /** Overriden to calculate non-realised
     *  preferred size.
     */
    protected swingwt.awt.Dimension calculatePreferredSize() {
        // Default 100x30
        swingwt.awt.Dimension size = new swingwt.awt.Dimension(100, 30);
        setSize(size);
        return size;
    }
    
    public void setSwingWTParent(Container parent) throws Exception {
        super.setSwingWTParent(parent);
        stateChanged(new ChangeEvent(model));
    }
    
}
