/***************************************************************************
 *   Copyright (C) 2004 by Ivan Forcada Atienza                            *
 *   ivan@forcada.info                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "frmmain.h"

extern APList * plist;

frmmain::frmmain(QWidget* parent, const char* name, WFlags fl)
                : SWScanner(parent,name,fl)
{
        char names[255];
        int num;
        qDebug("Creating frmmain...");
        plistAP = plist;
	QString interface;

	// set KDE icons
        btnRefresh->setPixmap (SmallIcon("reload"));
        buttonCancel->setPixmap (DesktopIcon ("exit"));
        buttonOk->setPixmap (DesktopIcon ("run"));
        btnStop->setPixmap (DesktopIcon ("stop"));
        btnClear->setPixmap (DesktopIcon ("eraser"));
	lstFilter->setColumnText (0, SmallIcon("filter"), lstFilter->columnText(0));
	fileOpenAction->setIconSet(BarIcon("fileopen"));
	fileSaveAction->setIconSet(BarIcon("filesave"));
	fileAddAction->setIconSet(BarIcon("edit_add"));
	fileExitAction->setIconSet(BarIcon("exit"));
	editCutAction->setIconSet(BarIcon("editcut"));
	editCopyAction->setIconSet(BarIcon("editcopy"));
	editPasteAction->setIconSet(BarIcon("editpaste"));
	toolsUtilitiesAction->setIconSet(BarIcon("wizard"));
	toolsSettingsAction->setIconSet(BarIcon("configure_toolbars"));
	helpContentsAction->setIconSet(BarIcon("help"));
	helpIndexAction->setIconSet(BarIcon("info"));
	helpAboutAction->setIconSet(DesktopIcon("swscanner"));	
	btnGPS->setIconSet(BarIcon("run"));
	
        // instantiate members
        gps = new GPSerial(this);
        shapefile = new ShapeHandler;
        log = new gpslog;
	
        // configure timer:
        timer = new QTimer( this );
        connect( timer, SIGNAL(timeout()), this, SLOT(timerDone()) );

        // configure timerStatus:
        timerStatus = new QTimer (this);
        connect( timerStatus, SIGNAL(timeout()), this, SLOT(timerStatusDone()) );
        timerStatus->start(1000);

        // configure table and header:
        header = tblScan->header();
	tblScan->setSorting(1);
	// Adjust the columns width
	for (int i=0;i < tblScan->columns(); i++)
		tblScan->adjustColumn(i);
        //connect(header, SIGNAL(clicked(int)), this, SLOT(headerClick(int)));

        // get wireless interfaces
        names[0]='\0';
        GetWirelessInterfacesNames_new(names,&num);
        QStringList strlist = QStringList::split( "#", names );

        // configure combobox and add interfaces
        cmbIface->insertStringList(strlist);

        // configure the listview (tree)
        lstFilter->setSelectionMode(QListView::Single);

        getStatus(cmbIface->currentText());

        // this->setWindowState(this->windowState() ^ WindowFullScreen);
        this->setWindowState(this->windowState() ^ WindowMaximized);

	// load configuration
	SWSconfig cfg;
	interface = cfg.getDefIface();
	
        // Dealing with commandline options:
        KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
        if (args->isSet("iface")) 
		interface = args->getOption("iface");
				
        if (args->isSet("fileopen"))
                fileOpen((char *)QString(args->getOption("fileopen")).ascii());
	else	// Only read the file argument if the --fileopen (or -f) option is not set
		if (args->count()) fileOpen((char *)QString(args->arg(0)).ascii());
	
	for(QStringList::Iterator it = strlist.begin(); it != strlist.end(); it ++)
		if ( interface == *it)
			cmbIface->setCurrentText(interface);
	
	sct = new scanThread( this, (char *)interface.ascii() );	// Initialize the thread with the aproppriate interface name.
	
	if (args->isSet("on") || cfg.getStartScanning())
		start();
	else
		stopScan();
}

frmmain::~frmmain()
{
        qDebug("Deleting frmmain...");
        if (gps->running()) {
                cout << "Waiting for 'GPScanner' thread..." << endl;
                gps->terminate();
                gps->wait();
        }
	if (sct->running()) {
                cout << "Waiting for 'scanThread' thread..." << endl;
                sct->terminate();
                sct->wait();
        }
        delete timer;
        delete timerStatus;
        delete gps;
        delete shapefile;
        delete log;
	delete sct;
        //delete pMacsViewed;

}

/*$SPECIALIZATION$*/

void frmmain::refreshCombo()
{
        char names[256];
        int num;

        // get wireless interfaces
        names[0]='\0';
        GetWirelessInterfacesNames_new(names,&num);
        QStringList strlist = QStringList::split( "#", names );
        // configure combobox and add interfaces
        cmbIface->clear();
        cmbIface->insertStringList(strlist);
}

void frmmain::showPopupMenu(QListViewItem * it, const QPoint & pos, int col)
{
        if (it == 0) return;
	QString essid = it->text(getCol(i18n("ESSID")));
        QString mac = it->text(getCol(i18n("MAC")));
        QPopupMenu * cfgPopup = new QPopupMenu(this, "config");
	
	col=col;	//avoid the warning

        frmconfig * dlgCfg = new frmconfig;	//call to constructor->configure the connections
        if (dlgCfg->isConfigured(essid, mac)) {
                cfgPopup->insertItem( BarIcon("connect_creating"), i18n("&Asociate %1").arg(cmbIface->currentText()),  this, SLOT(asociate()), CTRL+Key_A );
                cfgPopup->insertItem( BarIcon("edit"), i18n("&Change configuration"),  this, SLOT(startConfig()), CTRL+Key_C );
                cfgPopup->insertItem( BarIcon("eraser"), i18n("&Remove configuration"),  this, SLOT(removeConfig()), CTRL+Key_R );
        } else
                cfgPopup->insertItem( BarIcon("edit"), i18n("&Configure"),  this, SLOT(startConfig()), CTRL+Key_C );

        cfgPopup->exec(pos);

        delete dlgCfg;
}

void frmmain::headerClick(int section)
{  // this function is actually not used.
	section=section;	// avoid the warning
}

void frmmain::changeInterface()
{
	sct->setIface( (char *) cmbIface->currentText().ascii() );
}

void frmmain::changeInterval()
{
        //this function is called when interval or interface ar changed
        timer->stop();
        timer->start(spInterval->value()*1000);
}

void frmmain::stopScan()
{
//         QFile file("./scanning_stop.png");
//         KIconLoader *loader = KGlobal::iconLoader();
//         QIconSet icon;
// 	
//         if (file.exists())
//                 icon = QPixmap("./scanning_stop.png");
//         else
//                 icon = loader->loadIcon("scanning_stop", KIcon::User);

        tblScan->setColumnText(0, SmallIcon("stop"), i18n("  (off)"));
        timer->stop();
	plistAP->lock();
        plistAP->Drop();
	plistAP->unlock();
        chkPreserve->setChecked(TRUE);
        chkPreserveClick();
        flushOldNodes();
        //chkPreserve->setEnabled(FALSE);
}

void frmmain::start()
{
//         QFile file("./scanning_start.png");
//         KIconLoader *loader = KGlobal::iconLoader();
//         QIconSet icon;
// 	
//         if (file.exists())
//                 icon = QPixmap("./scanning_start.png");
//         else
//                 icon = loader->loadIcon("scanning_start", KIcon::User);
		
        // put the card into managed mode 
        struct iwreq wrq;
        int skfd = iw_sockets_open();

        strncpy(wrq.ifr_name, cmbIface->currentText().ascii(), IFNAMSIZ);
        wrq.u.mode = 2;

        if(iw_get_ext(skfd, (char *)cmbIface->currentText().ascii(), SIOCSIWMODE, &wrq) < 0)
                fprintf(stderr, "SIOCSIWMODE: %s\n", strerror(errno));

        tblScan->setColumnText(0, SmallIcon("run"), i18n("  (on)"));
        timer->start(spInterval->value()*1000);
        //chkPreserve->setEnabled(TRUE);

	// get the iface up (necesary for the scanning process)
        struct ifreq ifr;		//structure passed as ioctl() argument
        memset(&ifr, 0, sizeof(ifr));	//reset the interface request structure
        strcpy(ifr.ifr_ifrn.ifrn_name, cmbIface->currentText().ascii());
        ifr.ifr_flags|=IFF_UP;		//add the 'up' flag to the interface
        if (ioctl(skfd, SIOCSIFFLAGS, &ifr) < 0) {
                perror ("Error in SIOCSIFFLAGS");
                QMessageBox::warning(this, "SIOCSIFFLAGS", strerror(errno));
        }
	::close(skfd);
}

void frmmain::timerDone()
{
	if (sct->running()) return;	// Only go on if the previous scan has finished
	sct->start();
        //start_scan( (char *)cmbIface->currentText().ascii(), plistAP );		// Fills the ap list with new scan data
}

void frmmain::repaintTable()
{
        int i=0,j=0;

        for (i=0 ; i < plistAP->getLast() ; i++)
                addNewAP(&plistAP->operator [ ](i));	// adds or updates the list of ap's
		
        flushOldNodes();	// Hide ap's that are not active, or put them in 'stopped mode'
	
	// Adjust the columns width
	//for (int i=0;i < tblScan->columns(); i++)
	//	tblScan->adjustColumn(i);

	//Update de label below the table
	QListViewItem * it = tblScan->firstChild();
        while ( it ) {
                if ( it->isVisible() )
                        j++;
		it=it->nextSibling();
	}
	lblAPnum->setText(i18n("shown <b>(%1)</b>, total <b>(%2)</b>").arg(QVariant(j).toString(), QVariant(tblScan->childCount()).toString()) );
}

// This function is currently (and maybe permanently) not used
void frmmain::updateMaclist(APList* pList, QStrList* maclist)
{
        char temp[32];

        for (int i=0 ; i < pList->getLast() ; i++)
                if (!isInList(maclist, pList->operator [ ](i).getMAC(temp)))
                        maclist->append(temp);
}

bool frmmain::isInList(QStrList* macList, char * mac)
{
        for ( uint i=0; i < macList->count(); ++i )
                if ( macList->at(i) )
                        if (strcmp(macList->at(i),mac)==0)
                                return true;
        return false;
}

void frmmain::timerStatusDone()
{
        getStatus(cmbIface->currentText());
}
void frmmain::asociate()
{
        QString essid = tblScan->currentItem()->text(getCol(i18n("ESSID")));
        QString mac = tblScan->currentItem()->text(getCol(i18n("MAC")));

        frmconfig * dlgCfg;
        dlgCfg = new frmconfig;
        if (dlgCfg->isConfigured(essid, mac))
                dlgCfg->loadConfiguration(essid,mac);	//now I can access the properties directly from the widgets

        QString ip = dlgCfg->txtIP1->text() + "." + dlgCfg->txtIP2->text() + "." + dlgCfg->txtIP3->text() + "." + dlgCfg->txtIP4->text();
        QString mask = dlgCfg->txtMask1->text() + "." + dlgCfg->txtMask2->text() + "." + dlgCfg->txtMask3->text() + "." + dlgCfg->txtMask4->text();
        QString gw = dlgCfg->txtGat1->text() + "." + dlgCfg->txtGat2->text() + "." + dlgCfg->txtGat3->text() + "." + dlgCfg->txtGat4->text();

        // WIRELESS CONFIGURATION
        wireless_config wconf;
        memset((char *) &wconf, 0, sizeof(struct wireless_config)); //  reset the structure
        //  configure the structure:
        strcpy(wconf.name,cmbIface->currentText().ascii());
        wconf.has_nwid = 0;
        wconf.has_freq = 0;
        wconf.has_key = 1; // Allways 1 to enter the 'key stuff' part in iw_set_basic_config()
        strcpy((char *)wconf.key,dlgCfg->txtKey->text().ascii());
        //  It's necesary to explicitly tell the driver to disable the key
        wconf.key_flags |= (dlgCfg->chkWEP->isChecked()) ? 0 : IW_ENCODE_DISABLED;
        wconf.key_size = (wconf.key_flags & IW_ENCODE_DISABLED)?0:((dlgCfg->cmbKeyLength->currentItem()) == 0 ? 5 : 13);
        wconf.has_essid = 1;
        wconf.essid_on = 1;
        strcpy(wconf.essid,essid.ascii());
        wconf.has_mode = 1;
        wconf.mode = 2;		//managed

        //commit the changes
        int skfd = iw_sockets_open();
        iw_set_basic_config(skfd, (char *)cmbIface->currentText().ascii(), &wconf);

        //INET CONFIGURATION
        //dhcp
        if (dlgCfg->frmDhcp->isChecked()) {
                startDhcp(dlgCfg->txtDhcpClient->text());
                if (dlgCfg->txtScript->text().isNull() || dlgCfg->txtScript->text().isEmpty()) {
                        ::close(skfd);
                        return;
                }
                executeScript(dlgCfg->txtScript->text());
                ::close(skfd);
                return;
        }

        //static ip
        struct ifreq ifr;	//structure passed as ioctl() argument
        struct in_addr ipaddr;	//this struct only contains a ulong
        struct sockaddr_in *sin = (struct sockaddr_in *)&ifr.ifr_ifru.ifru_addr;	//first reference to the part
        										//that configures the ip address

        inet_aton(ip.ascii(),&ipaddr);	//convert fron ip in string(dotted) mode to a network address (ulong ins a in_addr struct)
        //ipaddr.s_addr=inet_addr(ip.ascii());
        memset(&ifr, 0, sizeof(ifr));	//reset the interface request structure
        //let's start configuring the structure:
        //-->IP ADDRESS:
        strcpy(ifr.ifr_ifrn.ifrn_name, cmbIface->currentText().ascii());
        sin->sin_family = AF_INET;
        sin->sin_addr.s_addr = ipaddr.s_addr;
        sin->sin_port = 0;	//not used
        if (ioctl(skfd, SIOCSIFADDR, &ifr) < 0) {
                perror ("Error in SIOCSIFADDR");
                QMessageBox::warning(this, "SIOCSIFADDR", strerror(errno));
        }

        //-->NETMASK:
        else {
                inet_aton(mask.ascii(),&ipaddr);	   //now ipaddr has the 'dotted' netmask transformed into a network address
                //ipaddr.s_addr=inet_addr(mask.ascii());
                sin =  (struct sockaddr_in *)&ifr.ifr_ifru.ifru_netmask;  //now sin points to the structure that controls the mask
                sin->sin_family = AF_INET;
                sin ->sin_addr.s_addr = ipaddr.s_addr;
                sin->sin_port = 0;	//not used
                if (ioctl(skfd, SIOCSIFNETMASK, &ifr) < 0) {
                        perror ("Error in SIOCSIFNETMASK");
                        QMessageBox::warning(this, "SIOCSIFNETMASK", strerror(errno));
                }
        }

        //-->DEFAULT GATEWAY:
        if (set_default_gw( skfd, (char *)gw.ascii()) < 0)
                QMessageBox::warning(this, i18n("Adding default route"), strerror(errno));

        // FINAL SCRIPT:
        if (dlgCfg->txtScript->text().isNull() || dlgCfg->txtScript->text().isEmpty()) {
                ::close(skfd);
                return;
        }
        executeScript(dlgCfg->txtScript->text());

        ::close(skfd);

        delete dlgCfg;
}
void frmmain::removeConfig()
{
        QString essid = tblScan->currentItem()->text(getCol(i18n("ESSID")));
        QString mac = tblScan->currentItem()->text(getCol(i18n("MAC")));

        frmconfig * dlgCfg;
        dlgCfg = new frmconfig;	//call to constructor->configure the connections
        dlgCfg->removeConfig(essid,mac);
        delete dlgCfg;
}
void frmmain::startConfig()
{
        QString essid = tblScan->currentItem()->text(getCol(i18n("ESSID")));
        QString mac = tblScan->currentItem()->text(getCol(i18n("MAC")));
        bool wep = tblScan->currentItem()->text(getCol(i18n("WEP")))==i18n("YES")?true:false;

        frmconfig * dlgCfg;
        dlgCfg = new frmconfig;	//call to constructor->configure the connections

        dlgCfg->lblEssid->setText(essid);
        dlgCfg->lblMAC->setText(mac);
        dlgCfg->chkWEP->setChecked(wep);
        if (wep) {
                dlgCfg->txtKey->setEnabled(true);
                dlgCfg->cmbKeyLength->setEnabled(true);
        } else {
                dlgCfg->txtKey->setEnabled(false);
                dlgCfg->cmbKeyLength->setEnabled(false);
        }

        if (dlgCfg->isConfigured(essid, mac)) {
                dlgCfg->btnSave->setText(i18n("&Change"));
                dlgCfg->loadConfiguration(essid,mac);
        } else
                dlgCfg->btnSave->setText(i18n("&Save"));

        dlgCfg->setModal(true);
        dlgCfg->show();
}

void frmmain::getStatus(QString ifc)
{
        struct interface ife;
        struct sockaddr_in *pdir_inet;
        char iface[50];
        char ipaddr[16];
        int skfd=0, ch=0;

        /************ GENERAL INFO *************/

        if (ifc.isNull() || ifc.isEmpty())
                return;
        strcpy(iface,ifc.ascii());
        strncpy(ife.name, iface, IFNAMSIZ);
        cout << "if(" << ife.name << ")->";

        if (if_fetch(&ife) < 0) {
                cout << "Error fetching interface intormation(" << iface << ")";
                return;
        }

        grpStatus->setTitle(i18n("Status (%1)").arg(ifc));
        pdir_inet = (struct sockaddr_in*)&ife.addr;
        strcpy(ipaddr,inet_ntoa(pdir_inet->sin_addr));
        lblIP->setText(ipaddr);
        cout << ipaddr << "/";
        pdir_inet = (struct sockaddr_in*)&ife.netmask;
        strcpy(ipaddr,inet_ntoa(pdir_inet->sin_addr));
        lblMask->setText(ipaddr);
        cout << ipaddr;
        pdir_inet = (struct sockaddr_in*)&ife.broadaddr;
        strcpy(ipaddr,inet_ntoa(pdir_inet->sin_addr));
        lblBcast->setText(ipaddr);
        cout << "/" << ipaddr << "####";

        /************ SPECIFIC WIRELESS INFO *************/
        //wireless_config wdata;
        wireless_info wdata;
        skfd = iw_sockets_open();

        //iw_get_basic_config(skfd, ife.name, &wdata);	//extracts information with low details

        iw_get_info(skfd, ife.name, &wdata);		//more details
        // ESSID
        if (wdata.has_essid == 1)
                lblEssid->setText(wdata.essid);
        else
                lblEssid->setText("????");
        // MODE
        if (wdata.has_mode == 1)
                switch (wdata.mode) {
                case 1:
                        lblMode->setText(i18n("Ad-hoc"));
                        break;
                case 2:
                        lblMode->setText(i18n("Managed"));
                        break;
                case 3:
                        lblMode->setText(i18n("Master"));
                        break;
                default:
                        lblMode->setText(i18n("????"));
                        break;
                }
        else
                lblMode->setText(i18n("????"));
        // CHANNEL
        struct iw_range		range;
        if (wdata.has_freq == 1) {
                iw_get_range_info(skfd, ife.name, &range);
                if (wdata.freq > 1000)
                        ch = iw_freq_to_channel(wdata.freq,&range);
                else
                        ch = (int) wdata.freq;
                lblChan->setText(QVariant(ch).toString());
        } else
                lblChan->setText(i18n("?"));
        // ENCRIPTION
        if (wdata.has_key == 1)
                if ((wdata.key_flags & IW_ENCODE_DISABLED) || (wdata.key_size == 0)) {
                        lblWEP->setText(i18n("NO"));
                        lblKLen->setText(i18n("-"));
                } else {
                        lblWEP->setText(i18n("YES"));
                        //lblWEP->setText((const char *)wdata.key);	//for debugging only
                        lblKLen->setText(QVariant(wdata.key_size).toString());
                }
        //ASOCIATED AP
        char buffer[25];
        lblAP->setText(iw_pr_ether(buffer, (const unsigned char *)wdata.ap_addr.sa_data));

        // STATISTICS
        iwstats stats;
        iw_get_stats (skfd,ife.name, &stats);
        cout << "\tlevel: " << stats.qual.level - 0x100 << "  ";
        cout << "noise: " << stats.qual.noise - 0x100;
        cout << " (" << stats.qual.qual + 0 << ")" << endl;
        lblSignal->setText(QVariant(stats.qual.level - 0x100).toString());
        lblNoise->setText(QVariant(stats.qual.noise - 0x100).toString());
        lblSNR->setText(QVariant(stats.qual.qual).toString());
        pixSignal->setPixmap(getPix2(QVariant(lblSignal->text()).toInt()));

        ::close(skfd);

}

void frmmain::startDhcp(QString dhclient)
{
	SWSconfig conf;
        QProcess * proc;
	proc = new QProcess( this );
	
	if (conf.getUseTerminal())	// if it's configured to show dhclient stdout in a terminal:
	{
		proc->addArgument(conf.getTerminal());	//execute the dhcp client in the configured terminal.
		proc->addArgument("-e");
	}
	
        proc->addArgument(dhclient);
        proc->addArgument(cmbIface->currentText());

        // 	proc->addArgument(client);
        // 	proc->addArgument(cmbIface->currentText());
        //
        // 	log = new frmLog;
        //         connect( proc, SIGNAL(readyReadStderr()), this, SLOT(dhclientStdout()));
        // 	connect( proc, SIGNAL(readyReadStdout()), this, SLOT(dhclientStdout()));
        // 	connect( proc, SIGNAL(processExited()), this, SLOT(deleteLog()));

        delay(2);
        if (!proc ->launch(QString::null))
		if (conf.getUseTerminal())
                	QMessageBox::information(this, i18n("SWScanner"), i18n("Error executing") + " '" + conf.getTerminal() + QString(" -e %1 %2'").arg(dhclient,cmbIface->currentText()));
		else
			QMessageBox::information(this, i18n("SWScanner"), i18n("Error executing %1 %2").arg(dhclient,cmbIface->currentText()));
	delete proc;

}
void frmmain::deleteLog()
{
        // delay (2);
        // delete log;
}
void frmmain::dhclientStdout()
{
        //	log->show();
        // 	log->txtLog->append(proc->readLineStderr());
        // 	log->txtLog->append(proc->readLineStdout());
}

void frmmain::helpAbout()
{
	static const QString tit = i18n("Simple Wireless Scanner (SWScanner).");
	static const QString des = i18n("A wireless network organizer, scanning and wardriving tool.");
	
	KAboutDialog about(KAboutDialog::AbtAppStandard, "", KDialogBase::Close, KDialogBase::Close, this);
	about.setTitle( des );
	about.setProduct(tit, "v"VERSION, "Ivn Forcada Atienza", "2005");
	
	KAboutContainer * c1 = about.addContainerPage(i18n("About SWScanner","&About"), AlignLeft, AlignLeft );
	if( c1 != 0 )
	{ 
		c1->addWidget(new QLabel(QString("<b><center> %1 </b></center>").arg(tit), this));
		c1->addWidget(new QLabel(i18n(des), this));
		c1->addWidget(new QLabel("<hr>", this));
		c1->addWidget(new QLabel("(c) 2005, Ivn Forcada Atienza.\n", this));
		c1->addPerson(	QString::null,
				QString::null,
				QString("http://www.swscanner.org"),
				QString::null,
				true );	
	}
	KAboutContainer * c2 = about.addContainerPage(i18n("&Author"), AlignLeft, AlignLeft );
	if(c2 != 0)
	{
		c2->addWidget(new QLabel(i18n("*** Maintainer and developer ***"), this));
		c2->addWidget(new QLabel("________________________________", this));
		c2->addWidget(new QLabel("<b>Ivn Forcada Atienza</b>", this));
		//c2->addWidget(new QLabel("", this));
		c2->addPerson(	QString::null,
		QString("ivan@swscanner.org"),
				QString::null,
				QString::null,
				true );
	}
	KAboutContainer * c3 = about.addContainerPage(i18n("&Packagers"), AlignLeft, AlignLeft );
	if(c3 != 0)
	{
		c3->addWidget(new QLabel (i18n("*** Debian packagers ***"), this));
		c3->addWidget(new QLabel (i18n("__________________________"), this));
		c3->addWidget(new QLabel ("<b>Andrs Seco Hernndez</b>", this));
		c3->addPerson ( QString::null, 
				"andresSH@alamin.org", 
				QString::null,
				QString::null,
				true);
		c3->addWidget(new QLabel ("--\n", this));
		c3->addWidget(new QLabel (i18n("Also collabored:"), this));
		c3->addWidget(new QLabel ("<b>Eugenio J. Snchez</b>", this));
		c3->addPerson ( QString::null, 
				"eugenio@humaneswireless.net", 
				QString::null,
				QString::null,
				true);
	
	}
	//about.addTextPage( "C&ontributors", "", true);
	about.addLicensePage(i18n("&License agreement"), licen_long);
	about.exec();
}

void frmmain::helpContents()
{
        QMessageBox::information(this, i18n("SWScanner"), i18n("Sorry, but documentation is not available yet :-("));
}

void frmmain::helpIndex()
{}

void frmmain::editFind()
{}

void frmmain::editPaste()
{}

void frmmain::editCopy()
{}

void frmmain::editCut()
{}

void frmmain::editRedo()
{}

void frmmain::editUndo()
{}

void frmmain::fileExit()
{}

void frmmain::filePrint()
{}

void frmmain::fileSaveAs()
{}

void frmmain::fileSave()
{
        bool isActive = timer->isActive();;
        int cols = tblScan->columns();
	QListViewItem * it = tblScan->firstChild();
	
        if (isActive)
                frmmain::stopScan();

        if (tblScan->childCount() == 0) {
                QMessageBox::information( this, i18n("SWScanner"), i18n("Sorry, but I can't save an empty table."));
                return;
        }

        QString filename = QFileDialog::getSaveFileName(
                                   QDir::homeDirPath(),
                                   i18n("SWScanner files (*.sws)"),
                                   this,
                                   i18n("Save file dialog"),
                                   i18n("Choose a filename to save under"));

        if (filename == "")
                return;
        if (filename.right(4) != ".sws")
                filename.append(".sws");
        QFile file( filename );
        if ( file.open( IO_WriteOnly ) ) {
                QTextStream stream( &file );
                while ( it ) {
                        stream << "<row>" << endl;
                        for (int j=0 ; j < cols ; j++)
                                stream << "     " << it->text(j) << endl;
                        stream << "</row>" << endl;
			it=it->nextSibling();
                }
        }
        file.close();

        if(isActive)
                frmmain::start();
}

void frmmain::fileOpen()
{
        fileOpen(0);
}
void frmmain::fileAdd()
{
        fileAdd(0);
}
void frmmain::fileOpen(char * name)
{
        if (name != 0)
                printf ("Opening file: %s", name);
        clearTable();
        fileAdd(name);
}
void frmmain::fileAdd(char * name)
{
        int i,j,k;
        char mac[20];
        APEntry ap;
        QFile file("./signal_none.png");
        KIconLoader *loader = KGlobal::iconLoader();
        QPixmap icon;
        QString filename;

        if (file.exists())
                icon = QPixmap("./signal_none.png");
        else
                icon = loader->loadIcon("signal_none", KIcon::User);

        if (timer->isActive())
                frmmain::stopScan();

        if (name == 0) {
                filename = QFileDialog::getOpenFileName(
                                   QDir::homeDirPath(),
                                   i18n("SWScanner files, NetStumbler Summaries (*.sws *.nss)"),
                                   this,
                                   i18n("Open file dialog"),
                                   i18n("Choose a file to open"));
        } else {
                filename=QString(name);
        }

        file.setName( filename );
        if ( file.open( IO_ReadOnly ) ) {
                QTextStream stream( &file );
                QString line;
                i=0;
                j=0,k=0;
                while ( !stream.atEnd() ) {
                        if (filename.right(3) == "nss")	// Parse the netstumbler summary file
                        {
                                line=stream.readLine();
                                k=0;
                                if (line.left(1) != "#")
                                {
                                        QStringList data = data.split("\t",line,TRUE);
                                        for ( QStringList::Iterator it = data.begin(); it != data.end(); ++it) {
                                                //qDebug(QVariant((uint)data.count()).toString());
                                                if (data.count() != 13) {
                                                        QMessageBox::warning( this, i18n("SWScanner"),i18n("Invalid file!!"));
                                                        return;
                                                }
                                                switch (k) {
                                                case 2:
                                                        ap.setEssid((char *)(*it).mid(2,(*it).length()-4).ascii());
                                                        break;
                                                case 4:
                                                        ap.setMAC((char *)(*it).mid(2,17).upper().ascii());
                                                        break;
                                                case 3:
                                                        ap.setMode((*it)=="BSS"?0:1);
                                                        break;
                                                case 6: {
                                                                QStringList sig = sig.split(" ",(*it),TRUE);
                                                                QStringList::Iterator it2 = sig.begin();
                                                                it2+=2;
                                                                ap.setSignal(QVariant((*it2)).toInt()-149);
                                                                it2++;
                                                                ap.setNoise(QVariant((*it2)).toInt()-149);
                                                                break;
                                                        }
                                                case 8:  // Flags
                                                        {
                                                                if ((*it).toInt(0,16) & 0x10)
                                                                        ap.setWEP(TRUE);
                                                                else
                                                                        ap.setWEP(FALSE);
                                                                break;
                                                        }
                                                case 9: // ChannelBits
                                                        {
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x2)? 1 : 0 );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x4)? 2 : 0 );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x8)? 3 : 0 );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x10)? 4 : 0 );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x20)? 5 : 0 );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x40)? 6 : 0  );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x80)? 7 : 0  );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x100)? 8  : 0 );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x200)? 9  : 0 );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x400)? 10  : 0 );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x800)? 11 : 0  );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x1000)? 12 : 0  );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x2000)? 13 : 0  );
                                                                ap.setChannel( ((*it).toLong(0,16) & 0x4000)? 14 : 0  );
                                                                break;
                                                        }
                                                }
                                                k++;
                                        }
                                        if (IsInTable(ap.getMAC(mac)))
                                                updateRow(&ap);
                                        else
                                                addNewRow(&ap);
                                        i++;
                                }
                        }
                        else	// .sws 'table' files
                        {
                                line = stream.readLine(); // line of text excluding '\n'
                                if (line.stripWhiteSpace() == "<row>") {
                                        //tblScan->insertRows(i);
                                        //tblScan->setPixmap(i,0,icon"));
                                } else if (line.stripWhiteSpace() == "</row>") {
                                        if (IsInTable(ap.getMAC(mac)))
                                                updateRow(&ap);
                                        else
                                                addNewRow(&ap);
                                        i++;
                                        j=0;
                                } else {
                                        switch (j) {
                                        case 1:
                                                ap.setEssid((char *) line.stripWhiteSpace().ascii());
                                                break;
                                        case 2:
                                                ap.setMAC((char *) line.stripWhiteSpace().ascii());
                                                break;
                                        case 3:
                                                ap.setChannel(line.stripWhiteSpace().toInt());
                                                break;
                                        case 4:
                                                ap.setMode( i18n(line.stripWhiteSpace()) == i18n("master")? 0 : 1 );
                                                break;
                                        case 6:
                                                ap.setSignal(line.stripWhiteSpace().toInt());
                                                break;
                                        case 7:
                                                ap.setNoise(line.stripWhiteSpace().toInt());
                                                break;
                                        case 10:
                                                ap.setWEP( i18n(line.stripWhiteSpace()) == i18n("YES")?TRUE:FALSE );
                                                break;
                                        }
                                        //tblScan->setText(i,j,line.stripWhiteSpace());
                                        j++;
                                }
                        }
                }
                file.close();
        }
        chkPreserve->setChecked(TRUE);
        chkPreserveClick();
        flushOldNodes();	// Fills the signal cell with N.A. among other things
        
        lblAPnum->setText(i18n("shown <b>(%1)</b>, total <b>(%2)</b>" ).arg(QVariant(i).toString(),QVariant(tblScan->childCount()).toString()));
}

void frmmain::fileNew()
{}

void frmmain::clearTable()
{
        QListViewItem * item;

	tblScan->clear();
	
        item=lstFilter->findItem(i18n("MAC"),0);
        deleteChildren(item);
        item=lstFilter->findItem(i18n("ESSID"),0);
        deleteChildren(item);
        item=lstFilter->findItem(i18n("CHANNEL"),0);
        deleteChildren(item);
	item=lstFilter->findItem(i18n("WEP"),0);
	deleteChildren(item);

	lblAPnum->setText(i18n("shown <b>(%1)</b>, total <b>(%2)</b>").arg("0","0"));
}

void frmmain::executeScript(QString script)
{
	SWSconfig conf;
        QStringList strL = QStringList::split(" ", script);
        QProcess * pscript;
	
	pscript = new QProcess( this );
	
	if (conf.getUseTerminal())	// if it's configured to show dhclient stdout in a terminal:
	{
		pscript->addArgument(conf.getTerminal());	//execute the script in the configured terminal.
		pscript->addArgument("-e");
	}
        //  Extract arguments and add them to the process
        for ( QStringList::Iterator it = strL.begin(); it != strL.end(); ++it )
                pscript->addArgument(*it);

        delete pscript;
	cout << "Executing '" << script.ascii() << "'  ...." << endl;
        delay(1);
        if (!pscript ->launch(QString::null))
                QMessageBox::information(this, i18n("SWScanner"), i18n("Error executing '%1").arg(script));

}

void frmmain::GPStart()
{
        int fd;
        //char temp [256];

        if (btnGPS->text()==i18n("Start &GPS")) {
                /* QString s = QFileDialog::getSaveFileName(
                	QDir::homeDirPath(),
                	"ESRI Shapefiles (*.shp *.shx *.dbf)",
                	this,
                	"save file dialog",
                	"Choose a filename to save under (extension is not needed)" );

                if ((len = s.length())==0) return;
                if ((s.right(4) == ".shp") || (s.right(4) == ".shx") || (s.right(4) == ".dbf"))
                	s.truncate(s.length()-4);
                shapefile->createShapefile((char *)s.ascii()); */
                if (log->open() == -1) {
                        //QMessageBox::warning(this,QString::null,"An error ocurred while opening log file");
                        return;
                }

                if (gps->running())
                        return;
                //configIni * conf = new configIni;	// [obsolete] I create this object only to read the config file in the constructor
		SWSconfig * conf = new SWSconfig;
                fd = gps->openPort((char *)conf->getSerialDev().ascii());
                gps->configure(fd);
                btnGPS->setText(i18n("Stop &GPS"));
                gps->start();
                delete conf;
        } else {
                gps->terminate();
                gps->wait();
                btnGPS->setText(i18n("Start &GPS"));
                GPSLed->off();
                log->close();
		lblLat->setText("0,0000000000");
		lblLong->setText("0,0000000000");
		lblSpeed->setText("0,0000000000");
        }
}


void frmmain::customEvent( QCustomEvent * e )
{
        QString aux;
        if ( e->type() == (QEvent::Type) newPosEventNum )	// It must be a newPosEvent
	{
                cout << "New GPS position event received..." << endl;
                GPSLed->toggle();
                newPosEvent* pe = (newPosEvent *)e;
                geodata pos = pe->getPos();
                /*        	txtGPS->append(QVariant(pos.lat_dms).toString() + "\n");
                		txtGPS->append(QVariant(pos.lat_ddeg).toString() + "\n");
                		txtGPS->append(QVariant(pos.lat_dmin).toString() + "\n");
                		txtGPS->append(QVariant(pos.long_dms).toString() + "\n");
                		txtGPS->append(QVariant(pos.long_ddeg).toString() + "\n");
                		txtGPS->append(QVariant(pos.long_dmin).toString() + "\n");
                		txtGPS->append(QVariant(pos.speed_knots).toString() + "\n");
                		txtGPS->append(QVariant(pos.speed_mph).toString() + "\n");
                		txtGPS->append(QVariant(pos.speed_kph).toString() + "\n");
                */		
                lblLat->setText(aux.sprintf("%.10f",pos.lat_ddeg));
                //lblLat->setText(QVariant(pos.lat_ddeg).toString().left(11));
                lblLong->setText(aux.sprintf("%.10f",pos.long_ddeg));
                //lblLong->setText(QVariant(pos.long_ddeg).toString().left(11));
                lblSpeed->setText(aux.sprintf("%.10f",pos.speed_kph));
                //lblSpeed->setText(QVariant(pos.speed_kph).toString().left(11));
                lblAPs->setText(QVariant(plistAP->getLast()).toString());
                plistAP->lock();
                for (int i=0; i<plistAP->getLast(); i++)
                        log->addLine(&(*plistAP)[i], pos.long_ddeg, pos.lat_ddeg);
			//shapefile->newData(&(*plistAP)[i],pos.long_ddeg, pos.lat_ddeg);	// Obsolete: before I writed directly the shapefile
		plistAP->unlock();
        } else if (e->type() == (QEvent::Type) scanFinishedEventNum )
	{
                cout << "New scanning event received..." << endl;
		plistAP->lock();	
		repaintTable();
		plistAP->unlock();
	}
}

void frmmain::toolsUtilities()
{
        frmUtils frmUtil;
        //frmUtil =  new frmUtils(this);
        frmUtil.lblInfo->setText("");
        frmUtil.exec();	//Show dialog in modal form
        //delete frmUtil;
}

void frmmain::toolsSettings()
{
        frmConfigIni frmSettings;
        //frmSettings = new frmConfigIni();
        frmSettings.exec();
        //delete frmSettings;
}

void frmmain::flushOldNodes ()
{
	// If the ap is not in the list (table) then updates his row if chkPreserve is checked. Otherwise it is hidden.
	QListViewItem * it = tblScan->firstChild();
	while ( it )
	{
		if ( !IsInList ( it->text(getCol(i18n("MAC")))) )	//if it isn't in the list of current ap's:
                {
			it->setText(getCol(i18n("SIGNAL")),i18n("N.A."));
			it->setText(getCol(i18n("SIGNAL")),i18n("N.A."));
			it->setPixmap(0,getPix(-99));                        
			if (chkPreserve->isChecked() && passTheFilter(it))
				it->setVisible(TRUE);	// show it if pass the filter and i want to preserve in list even non-active ap's
			else	
				it->setVisible(FALSE);	// otherwise, hide it (
                }
		it=it->nextSibling();
        }
}

//Check if a MAC is already displayed in the current table
bool frmmain::IsInTable (QString MAC)
{   
	if ( tblScan->findItem(MAC, getCol(i18n("MAC"))) == 0)
		return false;
	else
		return true;
}

// Check if a MAC is in the AP list of AP's detected in the scan
bool frmmain::IsInList (QString MAC)
{
        char temp[20];

        for (int i=0 ; i < plistAP->getLast() ; i++)
                if( QString(plistAP->operator [ ](i).getMAC(temp)) == MAC)
                        return true;

        //only reach this point if MAC is not in list
        return false;

}

void frmmain::addNewAP(APEntry* ap)
{
        char mac[20];
	
        if (IsInTable(ap->getMAC(mac)))	// If the row already exists then update it with the new signal-noise
                updateRow(ap);
        else
                addNewRow(ap);
}

/* This function updates a previously existing row, taking into account
if doesn't pass the filter, when it hides the row */
void frmmain::updateRow(APEntry* ap)
{
        char temp[20];
        int snrnew, snrold, sig;
        QString mac = ap->getMAC(temp);
        QPixmap pix  = getPix(ap->getSignal());
	QListViewItem * it = tblScan->findItem(mac,getCol(i18n("MAC")));

        //if (QSound::isAvailable()) QSound::play("./ap_update.wav");	// it doesn't work ???
	
	it->setPixmap(0,pix);

	sig = it->text(getCol(i18n("SIGNAL(MAX)"))).toInt();
	snrold = it->text(getCol(i18n("SNR(MAX)"))).toInt();
	snrnew = ap->getSignal() - ap->getNoise();		
	if (ap->getSignal() > sig)
		it->setText(getCol(i18n("SIGNAL(MAX)")),QVariant(ap->getSignal()).toString());
	if (snrnew >  snrold)
		it->setText(getCol(i18n("SNR(MAX)")),QVariant(snrnew).toString());
	it->setText(getCol(i18n("SIGNAL")),QVariant(ap->getSignal()).toString());
	it->setText(getCol(i18n("NOISE")),QVariant(ap->getNoise()).toString());
	it->setText(getCol(i18n("SNR")),QVariant(snrnew).toString());
	
		
	 if ( chkPreserve->isChecked() && passTheFilter(it) )
		it->setVisible(TRUE);
	else if ( !chkPreserve->isChecked() && passTheFilter(it) && ( it->text(getCol(i18n("SIGNAL")))!=i18n("N.A.")) )
		it->setVisible(TRUE);
	else
		it->setVisible(FALSE);

}

void frmmain::addNewRow(APEntry* ap)
{
	bool filt;
        char temp[255];
        char mac[20];
	QString wep;
        QPixmap pix = getPix(ap->getSignal());
        QListViewItem * it;

        // if (QSound::isAvailable()) QSound::play("./ap_new.wav");

	it = new QListViewItem( tblScan );
	
	it->setPixmap( 0, pix );
        it->setText(getCol(i18n("ESSID")),ap->getEssid(temp));
        it->setText(getCol(i18n("MAC")),ap->getMAC(mac));
        it->setText(getCol(i18n("CHANNEL")),QVariant(ap->getChannel()).toString());
        it->setText(getCol(i18n("SIGNAL(MAX)")),QVariant(ap->getSignal()).toString());
        it->setText(getCol(i18n("MODE")),ap->getModeString());
        it->setText(getCol(i18n("SIGNAL")),QVariant(ap->getSignal()).toString());
        it->setText(getCol(i18n("NOISE")),QVariant(ap->getNoise()).toString());
        it->setText(getCol(i18n("SNR")),QVariant(ap->getSignal()-ap->getNoise()).toString());
        it->setText(getCol(i18n("SNR(MAX)")),QVariant(ap->getSignal()-ap->getNoise()).toString());
	it->setText(getCol(i18n("WEP")), ap->getWEP()?i18n("YES"):i18n("NO"));

	filt = passTheFilter(it);
        if ( chkPreserve->isChecked() && filt )
                it->setVisible(TRUE);
        else if ( !chkPreserve->isChecked() && filt && (it->text(getCol(i18n("SIGNAL")))!=i18n("N.A.")) )
                it->setVisible(TRUE);
        else
                it->setVisible(FALSE);

        //add the mac'x to the filter tree
        it=lstFilter->findItem(i18n("MAC"),0);
        if (!IsInTree(it,QString(ap->getMAC(temp))))
                new QCheckListItem(it,mac);
        //add the channel to the filter, only if it isn't there yet
        it=lstFilter->findItem(i18n("CHANNEL"),0);
        if (!IsInTree(it,QVariant(ap->getChannel()).toString()))
                new QCheckListItem(it,QVariant(ap->getChannel()).toString());
        //the same with essid:
        it=lstFilter->findItem(i18n("ESSID"),0);
        if (!IsInTree(it,QString(ap->getEssid(temp))))
                new QCheckListItem(it,QString(temp));
	//and again with wep:
        it=lstFilter->findItem(i18n("WEP"),0);
        if (!IsInTree(it, wep = ap->getWEP()?i18n("YES"):i18n("NO")))
                new QCheckListItem(it,QString(wep));

}

// Returns the column index of a field name(header)
int frmmain::getCol(QString tit)
{
        QHeader* head = tblScan->header();

        for (int i=0 ; i < head->count() ; i++)
                if (tit == head->label(i))
                        return i;

        return -1;
}

int frmmain::getRow(QString mac)
{
	mac = mac;  // this function is not actually used since I use QListView instead of QTable, and QListView has the findItem member function

/*	// The next loop replace the findItem QListView member function if QTable (previously used)
	for (int i=0; i < tblScan->numRows(); i++)
		if (mac == tblScan->text(i, getCol(i18n("MAC"))))
			return i;
*/	return -1;
	
}

void frmmain::deleteChildren (QListViewItem * parentNode)
{
        QListViewItem * myChild = parentNode->firstChild();
        QListViewItem * aux;

        while (myChild) {
                aux = myChild;
                myChild = myChild->nextSibling();
                delete aux;
        }
}

//Search in the parentNode's childs, but not in the children's children
bool frmmain::IsInTree (QListViewItem * parentNode, QString strToFind)
{
        QListViewItem * myChild = parentNode->firstChild();
        while( myChild ) {
                if (myChild->key(0,true) == strToFind)
                        return true;
                myChild = myChild->nextSibling();
        }
        return false;
}

void frmmain::applyFilter(QListViewItem * node)
{
        int j=0;
        node=node;	// To avoid the warning --> not used
	QListViewItem * it = tblScan->firstChild();
	
	while ( it ) {
                if ( chkPreserve->isChecked() && passTheFilter(it) )
		{
                        it->setVisible(TRUE);
			j++;
		}
                else if ( !chkPreserve->isChecked() && passTheFilter(it) && (it->text(getCol(i18n("SIGNAL")))!=i18n("N.A.")) )
		{
                        it->setVisible(TRUE);
			j++;
		}
                else
                        it->setVisible(FALSE);
		it=it->nextSibling();
        }

        lblAPnum->setText(i18n("shown <b>(%1)</b>, total <b>(%2)</b>" ).arg(QVariant(j).toString(),QVariant(tblScan->childCount()).toString()));
}

bool frmmain::passTheFilter(QListViewItem * it)
{
        QListViewItem * sel = lstFilter->selectedItem();


        if (sel == 0)
                return true;	// There's no filter selected

	// Top-level nodes doesn't filter anything
        if (sel->key(0,true) == i18n("MAC"))
                return true;
        if (sel->key(0,true) == i18n("ESSID"))
                return true;
        if (sel->key(0,true) == i18n("CHANNEL"))
                return true;
        if (sel->key(0,true) == i18n("WEP"))
                return true;		
        if (sel->key(0,true) == QString("/"))
                return true;


        if (sel->parent()->key(0,true) == i18n("MAC"))   //filter by MAC
        {
                if (sel->key(0,true) == it->text(getCol(i18n("MAC"))))
                        return true;	//this mac is the one filtered
        } else if (sel->parent()->key(0,true) == i18n("ESSID"))   //filter by Essid
        {
                if (sel->key(0,true) == it->text(getCol(i18n("ESSID"))))
                        return true;	//this essid is the one filtered
        } else if (sel->parent()->key(0,true) == i18n("CHANNEL"))   //filter by channel
        {
                if (sel->key(0,true) == it->text(getCol(i18n("CHANNEL"))))
                        return true;	//this channel is the one filtered
        } else if (sel->parent()->key(0,true) == i18n("WEP"))   //filter by wep
        {
                if (sel->key(0,true) == it->text(getCol(i18n("WEP"))))
                        return true;	//this wep mode is the one filtered
        }

        return false;
}

void frmmain::chkPreserveClick()
{
        int j=0;
	QListViewItem * it = tblScan->firstChild();
        
	if (chkPreserve->isChecked()) {
                chkPreserve->setText(i18n("Show ALL nodes"));
		while ( it ) {
                        if (passTheFilter(it))
			{
                                it->setVisible(TRUE);
				j++;
			}
                        else
                                it->setVisible(FALSE);
			it=it->nextSibling();
                }
        } else {
                chkPreserve->setText(i18n("Show CURRENT nodes"));
		while ( it ) {
                        if (passTheFilter(it) && it->text(getCol(i18n("SIGNAL")))!=i18n("N.A."))
			{
                                it->setVisible(TRUE);
				j++;
			}
                        else
                                it->setVisible(FALSE);
			it=it->nextSibling();
                }
        }

	lblAPnum->setText(i18n("shown <b>(%1)</b>, total <b>(%2)</b>").arg(QVariant(j).toString(), QVariant(tblScan->childCount()).toString()) );
}

QPixmap getPix(int signal)
{
        QFile file;
        KIconLoader *loader = KGlobal::iconLoader();
        QPixmap icon;

        if ( (signal < -90) || ((signal > 0) && (signal < 5)) ) {
                file.setName("./signal_none.png");
                if (file.exists())
                        icon = QPixmap("./signal_none.png");
                else
                        icon = loader->loadIcon("signal_none", KIcon::User);
        } else if ( ((signal >= -90) && (signal < -80)) || ((signal >= 5) && (signal < 25)) ) {
                file.setName("./signal_low.png");
                if (file.exists())
                        icon = QPixmap("./signal_low.png");
                else
                        icon = loader->loadIcon("./signal_low", KIcon::User);
        } else if ( ((signal >= -80) && (signal < -70)) || ((signal >= 25) && (signal < 45)) ) {
                file.setName("./signal_medium.png");
                if (file.exists())
                        icon = QPixmap("./signal_medium.png");
                else
                        icon = loader->loadIcon("./signal_medium", KIcon::User);
        } else if ( (signal >= -70) || (signal >= 40) ) {
                file.setName("./signal_good.png");
                if (file.exists())
                        icon = QPixmap("./signal_good.png");
                else
                        icon = loader->loadIcon("./signal_good", KIcon::User);
        }
        return icon;
}

QPixmap getPix2(int signal)
{
        QFile file;
        KIconLoader *loader = KGlobal::iconLoader();
        QPixmap icon;

        if ((signal > -100) && (signal < -90)) {
                file.setName("./minimum.png");
                if (file.exists())
                        icon = QPixmap("./minimum.png");
                else
                        icon = loader->loadIcon("minimum", KIcon::User);
        } else if ((signal >= -90) && (signal < -80)) {
                file.setName("./marginal.png");
                if (file.exists())
                        icon = QPixmap("./marginal.png");
                else
                        icon = loader->loadIcon("marginal", KIcon::User);
        } else if ((signal >= -80) && (signal < -70)) {
                file.setName("./good.png");
                if (file.exists())
                        icon = QPixmap("./good.png");
                else
                        icon = loader->loadIcon("good", KIcon::User);
        } else if ( (signal >= -70) || (signal < -200) ) {
                file.setName("./excellent.png");
                if (file.exists())
                        icon = QPixmap("./excellent.png");
                else
                        icon = loader->loadIcon("excellent", KIcon::User);
        } else {
                file.setName("./offline.png");
                if (file.exists())
                        icon = QPixmap("./offline.png");
                else
                        icon = loader->loadIcon("offline", KIcon::User);
        }
        return icon;
}

void delay (int sec)
{
        time_t start;
        time_t current;

        time(&start);
        do {
                time(&current);
        } while(difftime(current,start) < (double)sec);
}

#include "frmmain.moc"

