/* TABLIX, PGA general timetable solver                              */
/* Copyright (C) 2002-2005 Tomaz Solc                                      */

/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; either version 2 of the License, or       */
/* (at your option) any later version.                                     */

/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */

/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software             */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/* $Id: sametimeas.c,v 1.1.2.4 2005/09/02 19:30:38 avian Exp $ */

/** @module
 *
 * @author Tomaz Solc
 * @author-email tomaz.solc@siol.net
 *
 * @credits 
 * Module ported to 0.2.x kernel and extended by 
 * Nick Robinson <npr@bottlehall.co.uk>
 *
 * @brief Adds a weight whenever two events are not scheduled at the same time
 * when they have been identified as needing to be.
 *
 * @ingroup School scheduling
 */

/** @tuple-restriction same-time-as
 *
 * <restriction type="same-time-as">event name</restriction>
 *
 * This restriction specifies that the current event needs to be scheduled
 * at the same time as the event identified in the restriction.
 *
 * <event name="a" repeats="4">
 * 	...
 * 	<restriction type="same-time-as">b</restriction>
 * <event>
 * <event name="b" repeats="5">
 * 	...
 * <event>
 *
 * In this case, four events "a" are scheduled at the same time as four events
 * "b". The fifth event "b" can be scheduled at any time.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "module.h"

struct sametime {
	int tupleid1;
	int tupleid2;
};

static struct sametime *sa;
static int numsa;

int event_used(int tupleid2) {
	int n;

	for(n=0;n<numsa;n++) {
		if(sa[n].tupleid2==tupleid2) return(1);
	}
	return(0);
}

int getevent(char *restriction, char *cont, tupleinfo *tuple)
{
	int tupleid;
	int tupleid1,tupleid2;

	int found;

	tupleid1=tuple->tupleid;
	tupleid2=-1;

	found=0;
	for(tupleid=0;tupleid<dat_tuplenum;tupleid++) {
		if(!strcmp(dat_tuplemap[tupleid].name,cont)) {
			found=1;
			if(!event_used(tupleid)) {
				tupleid2=tupleid;
				break;
			}
		}
	}
	if(!found) {
		error(_("No events match name '%s' "
			"in 'same-time-as' restriction"), cont);
		return(-1);
	}
	if(tupleid2==-1) {
		error(_("Repeats for this event must be less or equal "
			"than the target event '%s' in 'same-time-as' "
			"restriction"), cont);
		return(-1);
	}
	if(tupleid1==tupleid2) {
		error(_("Source and target events for 'same-time-as' "
			"restriction are the same event"));
	}

	sa[numsa].tupleid1=tupleid1;
	sa[numsa].tupleid2=tupleid2;

	numsa++;

	return 0;
}

int module_fitness(chromo **c, ext **e, slist **s)
{
	int n;
	int sum;
	chromo *time;
	int tupleid1, tupleid2;

	time=c[0];

	sum=0;
	for(n=0;n<numsa;n++) {
		tupleid1=sa[n].tupleid1;
		tupleid2=sa[n].tupleid2;

		if(time->gen[tupleid1]!=time->gen[tupleid2]) sum++;
	}

	return(sum);
}

int module_init(moduleoption *opt) 
{
	fitnessfunc *fitness;

	sa=malloc(sizeof(*sa)*dat_tuplenum);
	if(sa==NULL) {
		error(_("Can't allocate memory"));
		return -1;
	}
	numsa=0;

	handler_tup_new("same-time-as", getevent);

	fitness=fitness_new("force same time", 
			option_int(opt, "weight"), 
			option_int(opt, "mandatory"), 
			module_fitness);
	if(fitness==NULL) return -1;

	if(fitness_request_chromo(fitness, "time")) return -1;

	return(0);
}
