/* TABLIX, PGA general timetable solver                                    */
/* Copyright (C) 2002-2006 Tomaz Solc                                      */

/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; either version 2 of the License, or       */
/* (at your option) any later version.                                     */

/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */

/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software             */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/* $Id: depend.c,v 1.2 2006-02-04 14:16:12 avian Exp $ */

#include <stdio.h>
#include <stdlib.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "data.h"
#include "depend.h"
#include "gettext.h"
#include "assert.h"

/** @file 
 *  @brief Support for updater functions for dependent events. */

/** @brief Linked list of all updater functions for dependent events. */
static updaterfunc *dep_updaterlist=NULL;

/** @brief Checks if the source event of the updater function itself depends
 * on another event or not.
 *
 * Helper function for updater_reorder().
 *
 * This function walks the \a dat_updatelist linked list and searches for an
 * updater function that has a destination event equal to the source event
 * of the updater function in the argument. 
 *
 * @param updater Pointer to the updaterfunc structure to be checked.
 * @return 0 if the source event is independent or -1 if it is dependent. */
static int updater_dependent(updaterfunc *updater) {
	updaterfunc *cur;

	assert(updater!=NULL);

	cur=dep_updaterlist;
	while(cur!=NULL) {
		if(cur->dst_tupleid==updater->src_tupleid) return -1;
		cur=cur->next;
	}

	return 0;
}

/** @brief Moves an updaterfunc structure to the end of a linked list.
 *
 * Helper function for updater_reorder().
 *
 * @param list Pointer to a linked list of updaterfunc structures.
 * @param updater Pointer to the updaterfunc structure to be moved to the
 * end of the list. */
static void updater_movetoend(updaterfunc **list, updaterfunc *updater) {
	updaterfunc *cur;

	assert(list!=NULL);
	assert(updater!=NULL);

	updater->next=NULL;

	if(*list==NULL) {
		*list=updater;
	} else {
		cur=*list;
		while(cur->next!=NULL) cur=cur->next;

		cur->next=updater;
	}
}

/** @brief Reorders \a dep_updaterlist linked list.
 *
 * A source (independent) event for an updater function can infact be a 
 * destination (dependent) event of another updater function. This means that
 * updater functions must be called in the correct order.
 *
 * Example: Function A has a source event 2 and destination event 3. Function
 * B has a source event 1 and destination event 2. In this case 
 * updater_reorder() puts function B in front of A.
 *
 * @return 0 on success or -1 on error (there was a circular dependency and
 * the correct order of calling can not be determined. */
int updater_reorder() 
{
	updaterfunc *cur, *prev, *next;
	updaterfunc *newlist=NULL;

	int circular=0;

	while(dep_updaterlist!=NULL) {
		prev=NULL;
		cur=dep_updaterlist;
		circular=1;
		while(cur!=NULL) {
			next=cur->next;
			if(!updater_dependent(cur)) {
				circular=0;
				if(prev!=NULL) {
					prev->next=next;
				} else {
					dep_updaterlist=next;
				}
				updater_movetoend(&newlist, cur);

				prev=prev;
			} else {
				prev=cur;
			}
			cur=next;
		}
		if(circular) {
			debug("updater_reorder: circular dependency\n");
			break;
		}
	}

	if(circular) {
		if(newlist!=NULL) {
			cur=newlist;
			while(cur!=NULL) {
				error(_("Event '%s' depends on event '%s'"), 
					dat_tuplemap[cur->src_tupleid].name,
					dat_tuplemap[cur->dst_tupleid].name);
				cur=cur->next;
			}
			
			cur=newlist;
			while(cur->next!=NULL) cur=cur->next;
			cur->next=dep_updaterlist;
		}
		return -1;
	} else {
		dep_updaterlist=newlist;
		return 0;
	}
}

/** @brief Checks if an updater function can be registered for a dependent 
 * event.
 *
 * @param dst Tuple ID of the dependent event.
 * @return 0 if an updater function can be registered or -1 if an updater
 * function can not be registered for this event. */
int updater_check(int dst) 
{
	assert(dst<dat_tuplenum&&dst>=0);

	if(dat_tuplemap[dst].dependent) {
		return -1;
	} else {
		return 0;
	}
}

/** @brief Register a new updater function.
 *
 * It is recommended that updater_check() is called before calling this
 * function to verify if there isn't another updater function registered for
 * this dependent event.
 *
 * @param src Tuple ID of the independent event.
 * @param dst Tuple ID of the dependent event.
 * @param func Pointer to the updater function.
 * @return Pointer to the updaterfunc structure or NULL on error. */
updaterfunc *updater_new(int src, int dst, updater_f func) 
{
	updaterfunc *cur;

	assert(func!=NULL);
	assert(src<dat_tuplenum&&src>=0);
	assert(dst<dat_tuplenum&&dst>=0);

	if(dat_tuplemap[dst].dependent) return NULL;

	cur=malloc(sizeof(*cur));
	if(cur==NULL) return NULL;

	cur->src_tupleid=src;
	cur->dst_tupleid=dst;
	cur->func=func;

	cur->next=dep_updaterlist;

	dat_tuplemap[dst].dependent=1;

	dep_updaterlist=cur;

	return cur;
}

/** @brief Calls all registered updater functions 
 *
 * Functions are called in the order of the linked list. updater_reorder() 
 * must be called before the first call to updater_call() to properly reorder 
 * the linked list.
 * 
 * @param tab Pointer to the timetable structure. */
void updater_call(table *tab)
{
	updaterfunc *cur;

	assert(tab!=NULL);

	cur=dep_updaterlist;

	while(cur!=NULL) {
		(*cur->func)(cur->src_tupleid, cur->dst_tupleid, tab);
		cur=cur->next;
	}
}
