#ifndef TAG_H
#define TAG_H

#pragma interface

#include <string>

class Facet;
class FacetImpl;
class TagImpl;
class TagSet;
class FacetSet;

/**
 * Representation of a tag.
 * Tagcoll::Tag represents a Tag with all its informations.  It is implemented
 * via a reference-counted smart pointer, so it can be passed around freely and
 * efficiently without worrying for memory management issues.
 * The class is normally instantiated via Facet::obtainTag:
 * @code
 * 	Tag mytag = facet.obtainTag("printable");
 * @endcode
 * Tag can contain a "false" value, in which case using any of its methonds
 * produce a null-pointer dereference segfault.  The "null" tags are useful as
 * "none" return values:
 * @code
 *    Tag mytag = facet.getTag("printable");
 *    if (!mytag)
 *       throw ConsistencyCheckException("tag \"printable\" has not been defined");
 * @endcode
 */
class Tag
{
protected:
	TagImpl* impl;

	Tag(FacetImpl* facet, const std::string& name) throw ();

public:
	Tag() throw ();
	Tag(const Tag& f) throw ();
	~Tag() throw ();
	Tag& operator=(const Tag& f) throw ();

	bool operator==(const Tag& f) const throw ();
	bool operator<(const Tag& f) const throw ();

	operator bool() const throw ();

	Facet facet() const throw ();

	const std::string& name() const throw ();
	std::string fullname() const throw ();

	friend class FacetImpl;
	friend class TagSet;
};

class Facet
{
protected:
	FacetImpl* impl;

	Facet(FacetImpl* impl) throw ();

public:
	explicit Facet(const std::string& name) throw ();
	Facet() throw ();
	Facet(const Facet& f) throw ();
	~Facet() throw ();
	Facet& operator=(const Facet& f) throw ();

	bool operator==(const Facet& f) const throw ();
	bool operator<(const Facet& f) const throw ();

	operator bool() const throw ();

	const std::string& name() const throw ();

	Tag obtainTag(const std::string& name) throw ();

	bool hasTag(const std::string& name) const throw ();
	Tag getTag(const std::string& name) const throw ();
	const TagSet& tags() const throw ();

	friend class Tag;
	friend class FacetSet;
};

// vim:set ts=3 sw=3:
#endif
