/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with self library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "tpa-handle.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION
#define HOLD FALSE

#include <tapioca/base/tpa-debug.h>

#include <tapioca/base/tpa-connection-bindings.h>
#include <tapioca/base/tpa-signals-marshal.h>

struct _TpaHandlePrivate {
    DBusGProxy *proxy;
    guint id;
    gchar *name;
    TpaHandleType type;
    gboolean hold;
    gboolean disposed;
};

static guint count = 0;

G_DEFINE_TYPE(TpaHandle, tpa_handle, TPA_TYPE_OBJECT)

static void
tpa_handle_dispose (GObject *object)
{
    TpaHandle *self = TPA_HANDLE (object);
    GArray *handles = NULL;
    GError *error = NULL;

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    /* Release handle if it is on hold */
    if (self->priv->hold) {
        handles = g_array_new (FALSE, TRUE, 1);
        g_array_append_val (handles, self->priv->id);
        if (!org_freedesktop_Telepathy_Connection_release_handles (self->priv->proxy, self->priv->type, handles, &error) ||
            error) {
            ERROR ("%s", error->message);
        }
        g_array_free (handles, TRUE);
    }

    g_free (self->priv->name);

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    G_OBJECT_CLASS (tpa_handle_parent_class)->dispose (object);
}

static void
tpa_handle_class_init (TpaHandleClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_handle_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaHandlePrivate));

    gobject_class->dispose = tpa_handle_dispose;
}

static void
tpa_handle_init (TpaHandle *self)
{
    self->priv = TPA_HANDLE_GET_PRIVATE (self);
    self->priv->disposed = FALSE;
    self->priv->hold = FALSE;
    self->priv->name = NULL;
}

/**
 * tpa_handle_new:
 * @object: #TpaObject instance.
 * @type: #TpaHandleType type.
 * @name: Handle string name.
 */
TpaHandle *
tpa_handle_new (TpaObject *object,
                TpaHandleType type,
                const gchar *name)
{
    TpaHandle *self = NULL;
    GError *error = NULL;
    gchar *names[2] = { NULL, NULL };
    GArray *ids = NULL;
    DBusGProxy *proxy;

    VERBOSE ("(%d, %s)", type, name);
    g_return_val_if_fail (name != NULL, NULL);

    names[0] = g_strdup (name);
    proxy = tpa_object_get_proxy (object, TPA_INTERFACE_CONNECTION);

    if (!org_freedesktop_Telepathy_Connection_request_handles (proxy, type, (const char **)names, &ids, &error) ||
        error) {
        ERROR ("%s", error->message);
        return NULL;
    }

    g_return_val_if_fail (ids->len > 0, NULL);

    self = tpa_handle_new_by_id (object, type, g_array_index (ids, guint, 0));

    g_array_free (ids, TRUE);
    g_object_unref (proxy);

    VERBOSE ("return %p", self);
    return self;
}

/**
 * tpa_handle_new_with_id:
 * @object: #TpaObject instance.
 * @type: #TpaHandleType type.
 * @id: Handle id.
 */
TpaHandle *
tpa_handle_new_by_id (TpaObject *object,
                      TpaHandleType handle_type,
                      guint id)
{
    TpaHandle *self;
    GError *error = NULL;
    GArray *handles = g_array_new (FALSE, FALSE, sizeof(guint));
    DBusGProxy *proxy;

    self = TPA_HANDLE (g_object_new (TPA_TYPE_HANDLE, "object", object, NULL));
    proxy = tpa_object_get_proxy (object, TPA_INTERFACE_CONNECTION);

    VERBOSE ("(%d, %d)", handle_type, id);

    g_array_append_val (handles, id);

    if (HOLD) {
        if (!org_freedesktop_Telepathy_Connection_hold_handles (proxy, handle_type, handles, &error) ||
            error) {
            ERROR ("%s", error->message);
            goto OUT;
        }
        self->priv->hold = TRUE;
    }

    self->priv->id = id;
    self->priv->type = handle_type;
    self->priv->proxy = proxy;

OUT:
    g_array_free (handles, TRUE);
    VERBOSE ("return %p", self);
    return self;
}

/**
 * tpa_handle_get_name:
 * @self: #TpaHandle instance.
 * @returns:
 */
const gchar *
tpa_handle_get_name (TpaHandle *self)
{
    gchar **names;
    GArray *handles;
    GError *error = NULL;

    /**
     * Handle system really worth inspecting
     * every single string?
     */
    count++;
    VERBOSE ("(%p, %d)", self, count);
    g_assert (self);

    if (!self->priv->hold && !self->priv->name) {
        handles = g_array_new (FALSE, FALSE, sizeof (guint));
        g_array_append_val (handles, self->priv->id);
        if (!org_freedesktop_Telepathy_Connection_inspect_handles (self->priv->proxy, self->priv->type, handles, &names, &error) ||
            error) {
            ERROR ("%s", error->message);
        }
        g_array_free (handles, TRUE);
        self->priv->name = g_strdup (names[0]);
        g_strfreev (names);
    }

    VERBOSE ("return %s", self->priv->name);
    return self->priv->name;
}

/**
 * tpa_handle_get_id:
 * @self: #TpaHandle instance.
 */
guint
tpa_handle_get_id (TpaHandle *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    VERBOSE ("return %d", self->priv->id);
    return self->priv->id;
}

/**
 * tpa_handle_get_handle_type:
 * @self: #TpaHandle instance.
 */
TpaHandleType
tpa_handle_get_handle_type (TpaHandle *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    VERBOSE ("return %d", self->priv->type);
    return self->priv->type;
}

/**
 * tpa_handle_hold:
 * @self: #TpaHandle instance.
 */
void
tpa_handle_hold (TpaHandle *self)
{
    GArray *handles = NULL;
    GError *error = NULL;

    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!self->priv->hold) {
        handles = g_array_new (FALSE, TRUE, 1);
        g_array_append_val (handles, self->priv->id);
        if (!org_freedesktop_Telepathy_Connection_hold_handles (self->priv->proxy, self->priv->type, handles, &error) ||
            error) {
            ERROR ("%s", error->message);
        }
        self->priv->hold = TRUE;
        g_array_free (handles, TRUE);
    }
    VERBOSE ("return");
}
