/*
 * Tapioca library
 * Copyright (C) 2007 INdT.
 * @author  Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "tpa-stream.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-signals-marshal.h>

struct _TpaStreamPrivate {
    TpaChannel *channel;
    gboolean disposed;
    guint id;
    guint contact;
    guint type;
    guint state;
    guint direction;
    guint flags;
};

enum
{
    ARG_0,
    ARG_CHANNEL,
    ARG_ID,
    ARG_CONTACT,
    ARG_TYPE,
    ARG_STATE,
    ARG_DIRECTION,
    ARG_FLAGS
};

G_DEFINE_TYPE(TpaStream, tpa_stream, TPA_TYPE_OBJECT)

/* Signals */
enum {
    ERROR,
    STATE_CHANGED,
    LAST_SIGNAL
};

static guint tpa_stream_signals[LAST_SIGNAL] = { 0 };

static void
proxy_stream_state_changed (DBusGProxy *proxy,
                            guint stream_id,
                            guint state,
                            gpointer user_data)
{
    TpaStream* self = TPA_STREAM (user_data);

    VERBOSE ("(%p, %d, %d, %p)", proxy, stream_id, state, user_data);
    g_assert (self);

    if (self->priv->id == stream_id) {
        INFO ("[stream-state-changed] stream:%d state:%d", stream_id, state);

        self->priv->state = state;
        g_signal_emit (self, tpa_stream_signals[STATE_CHANGED], 0, state);
    }

    VERBOSE ("return");
}

static void
proxy_stream_error (DBusGProxy *proxy,
                    guint stream_id,
                    guint error,
                    const gchar *msg,
                    gpointer user_data)
{
    TpaStream* self = TPA_STREAM (user_data);

    VERBOSE ("(%p, %d, %d, %s, %p)", proxy, stream_id, error, msg, user_data);
    g_assert (self);

    if (self->priv->id == stream_id) {
        INFO ("[stream-error] stream:%d error:%d msg:%s", stream_id, error, msg);

        g_signal_emit (self, tpa_stream_signals[ERROR], 0, error, msg);
    }

    VERBOSE ("return");
}

static void
tpa_stream_set_property (GObject *object,
                         guint prop_id,
                         const GValue *value,
                         GParamSpec *pspec)
{
    TpaStream *self = TPA_STREAM (object);

    switch (prop_id) {
        case ARG_CHANNEL:
            self->priv->channel = g_value_get_object (value);
            break;
        case ARG_ID:
            self->priv->id= g_value_get_uint (value);
            break;
        case ARG_CONTACT:
            self->priv->contact = g_value_get_uint (value);
            break;
        case ARG_TYPE:
            self->priv->type = g_value_get_uint (value);
            break;
        case ARG_STATE:
            self->priv->state = g_value_get_uint (value);
            break;
        case ARG_DIRECTION:
            self->priv->direction = g_value_get_uint (value);
            break;
        case ARG_FLAGS:
            self->priv->flags = g_value_get_uint (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
tpa_stream_get_property (GObject *object,
                         guint prop_id,
                         GValue *value,
                         GParamSpec *pspec)
{
    TpaStream *self = TPA_STREAM (object);

    switch (prop_id) {
        case ARG_CHANNEL:
            g_value_set_pointer (value, self->priv->channel);
            break;
        case ARG_ID:
            g_value_set_uint (value, self->priv->id);
            break;
        case ARG_CONTACT:
            g_value_set_uint (value, self->priv->contact);
            break;
        case ARG_TYPE:
            g_value_set_uint (value, self->priv->type);
            break;
        case ARG_STATE:
            g_value_set_uint (value, self->priv->state);
            break;
        case ARG_DIRECTION:
            g_value_set_uint (value, self->priv->direction);
            break;
        case ARG_FLAGS:
            g_value_set_uint (value, self->priv->flags);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static GObject*
tpa_stream_constructor (GType type,
                        guint n_construct_params,
                        GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaStream *self;

    object = G_OBJECT_CLASS (tpa_stream_parent_class)->constructor
                            (type, n_construct_params, construct_params);
    self = TPA_STREAM (object);

    if (self->priv->channel) {
        tpa_object_connect_signal (TPA_OBJECT (self->priv->channel),
                                   TPA_INTERFACE_STREAMED_MEDIA,
                                   "StreamStateChanged",
                                   G_CALLBACK (proxy_stream_state_changed),
                                   self);
        tpa_object_connect_signal (TPA_OBJECT (self->priv->channel),
                                   TPA_INTERFACE_STREAMED_MEDIA,
                                   "StreamError",
                                   G_CALLBACK (proxy_stream_error),
                                   self);
    }

    return object;
}

static void
tpa_stream_dispose (GObject *object)
{
    TpaStream *self = TPA_STREAM (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    G_OBJECT_CLASS (tpa_stream_parent_class)->dispose (object);
}

static void
tpa_stream_class_init (TpaStreamClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_stream_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaStreamPrivate));

    gobject_class->dispose = tpa_stream_dispose;
    gobject_class->constructor = tpa_stream_constructor;
    gobject_class->get_property = tpa_stream_get_property;
    gobject_class->set_property = tpa_stream_set_property;

    g_object_class_install_property (gobject_class,
                                     ARG_CHANNEL,
                                     g_param_spec_object ("channel",
                                     "channel",
                                     "channel",
                                     TPA_TYPE_CHANNEL,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_ID,
                                     g_param_spec_uint ("id",
                                     "id",
                                     "id",
                                     0,
                                     G_MAXUINT,
                                     0,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_CONTACT,
                                     g_param_spec_uint ("contact",
                                     "contact",
                                     "contact",
                                     0,
                                     G_MAXUINT,
                                     0,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_TYPE,
                                     g_param_spec_uint ("type",
                                     "type",
                                     "type",
                                     TPA_STREAM_TYPE_AUDIO,
                                     TPA_STREAM_TYPE_VIDEO,
                                     TPA_STREAM_TYPE_AUDIO,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_STATE,
                                     g_param_spec_uint ("state",
                                     "state",
                                     "state",
                                     TPA_STREAM_STATE_CONNECTING,
                                     TPA_STREAM_STATE_PLAYING,
                                     TPA_STREAM_STATE_CONNECTING,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_DIRECTION,
                                     g_param_spec_uint ("direction",
                                     "direction",
                                     "direction",
                                     0,
                                     G_MAXUINT,
                                     0,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_FLAGS,
                                     g_param_spec_uint ("flags",
                                     "flags",
                                     "flags",
                                     0,
                                     G_MAXUINT,
                                     0,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    tpa_stream_signals[STATE_CHANGED] = g_signal_new ("state-changed",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__UINT,
        G_TYPE_NONE,
        1,
        G_TYPE_UINT);

    tpa_stream_signals[ERROR] = g_signal_new ("error",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        tpa_marshal_VOID__UINT_STRING,
        G_TYPE_NONE,
        2,
        G_TYPE_UINT,
        G_TYPE_STRING);
}

static void
tpa_stream_init (TpaStream *self)
{
    self->priv = TPA_STREAM_GET_PRIVATE (self);
    self->priv->id = 0;
    self->priv->contact = 0;
    self->priv->state = 0;
    self->priv->type  = 0;
    self->priv->direction = 0;
    self->priv->flags = 0;
    self->priv->channel = NULL;
    self->priv->disposed = FALSE;
}

/**
 * tpa_stream_pause:
 * @self: #TpaStream instance.
 */
void
tpa_stream_pause (TpaStream *self,
                  gboolean value)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

//    self->priv->playing = FALSE;
    INFO ("paused")
//    g_signal_emit (self, tpa_stream_signals[PLAYING], 0, FALSE);

    VERBOSE ("return");
}

TpaStreamState
tpa_stream_get_state (TpaStream *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    VERBOSE ("return");

    return (TpaStreamState) self->priv->state;
}

TpaStreamType
tpa_stream_get_stream_type  (TpaStream *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    VERBOSE ("return");

    return (TpaStreamType) self->priv->type;
}

guint
tpa_stream_get_id (TpaStream *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    VERBOSE ("return");

    return self->priv->id;
}

TpaChannel *
tpa_stream_get_channel (TpaStream *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    VERBOSE ("return");

    return g_object_ref (self->priv->channel);
}
