/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "tpa-user-contact-priv.h"
#include "tpa-channel-target-priv.h"
#include "tpa-contact-base-priv.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-connection-bindings.h>
#include <tapioca/base/tpa-signals-marshal.h>

struct _TpaUserContactPrivate {
    DBusGProxy *proxy_presence;
    DBusGProxy *proxy_aliasing;
    DBusGProxy *proxy_avatars;
    DBusGProxy *proxy_capabilities;
    TpaContactPresence presence;
    TpaAvatar *avatar;
    gchar *message;
    gboolean disposed;
};

static const gchar *presence_status[TPA_PRESENCE_BUSY + 1] =
    { "", "offline", "available", "away", "extended away", "hidden", "dnd" };

G_DEFINE_TYPE(TpaUserContact, tpa_user_contact, TPA_TYPE_CONTACT_BASE)

static GObject*
tpa_user_contact_constructor (GType type,
                              guint n_construct_params,
                              GObjectConstructParam *construct_params)
{
    GObject *object = NULL;

    object = G_OBJECT_CLASS (tpa_user_contact_parent_class)->constructor (type, n_construct_params, construct_params);

    return object;
}

static void
tpa_user_contact_dispose (GObject *object)
{
    TpaUserContact *self = TPA_USER_CONTACT (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    if (self->priv->proxy_presence)
        g_object_unref (self->priv->proxy_presence);
    if (self->priv->proxy_presence)
        g_object_unref (self->priv->proxy_aliasing);
    if (self->priv->proxy_presence)
        g_object_unref (self->priv->proxy_avatars);
    if (self->priv->proxy_presence)
        g_object_unref (self->priv->proxy_capabilities);

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    G_OBJECT_CLASS (tpa_user_contact_parent_class)->dispose (object);
}

static void
tpa_user_contact_class_init (TpaUserContactClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_user_contact_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaUserContactPrivate));

    gobject_class->constructor = tpa_user_contact_constructor;
    gobject_class->dispose = tpa_user_contact_dispose;
}

static void
tpa_user_contact_init (TpaUserContact *self)
{
    self->priv = TPA_USER_CONTACT_GET_PRIVATE (self);
    self->priv->disposed = FALSE;
    self->priv->proxy_presence = NULL;
    self->priv->proxy_aliasing = NULL;
    self->priv->proxy_avatars = NULL;
    self->priv->proxy_capabilities = NULL;
}

/**
 * tpa_user_contact_new:
 * @handle: #TpaHandle instance.
 * @object: #TpaObject instance.
 * @returns: #TpaUserContact instance.
 */
TpaUserContact *
tpa_user_contact_new (TpaHandle *handle,
                      TpaObject *object)
{
    TpaUserContact *self;

    VERBOSE ("(%p, %p)", handle, object);
    g_return_val_if_fail (handle != NULL, NULL);
    g_return_val_if_fail (object != NULL, NULL);

    self = TPA_USER_CONTACT (g_object_new (TPA_TYPE_USER_CONTACT,
                             "handle", handle, "object", object, NULL));

    self->priv->proxy_presence = tpa_object_get_proxy (object, TPA_INTERFACE_PRESENCE);
    self->priv->proxy_avatars = tpa_object_get_proxy (object, TPA_INTERFACE_AVATARS);
    self->priv->proxy_capabilities = tpa_object_get_proxy (object, TPA_INTERFACE_CAPABILITIES);
    self->priv->proxy_aliasing = tpa_object_get_proxy (object, TPA_INTERFACE_ALIASING);
    self->priv->message = g_strdup ("");

    VERBOSE ("return %p", self);
    return self;
}

/**
 * tpa_user_contact_set_avatar:
 * @self: #TpaUserContact instance.
 * @avatar: #TpaAvatar instance.
 * @returns: TRUE if avatar could be set.
 */
gboolean
tpa_user_contact_set_avatar (TpaUserContact *self,
                             TpaAvatar *avatar)
{
    GError *error = NULL;
    const GArray *image;
    const gchar *mimetype;
    gchar *token;
    gboolean ret = TRUE;

    VERBOSE ("(%p, %p)", self, avatar);
    g_assert (self);
    g_return_val_if_fail (self->priv->proxy_avatars != NULL, FALSE);

    image = tpa_avatar_get_image (avatar);
    mimetype = tpa_avatar_get_mime_type (avatar);
    token = (gchar *) tpa_avatar_get_token (avatar);

    if (token) {
         g_free (token);
         token = NULL;
    }

    INFO ("setting avatar");
    if (!org_freedesktop_Telepathy_Connection_Interface_Avatars_set_avatar (self->priv->proxy_avatars, image, mimetype, &token, &error)
        || error) {
        ERROR ("error: %s", error->message);
        g_error_free (error);
        ret = FALSE;
    }

    VERBOSE ("return %s", ret ? "TRUE" : "FALSE");
    return ret;
}

/**
 * tpa_user_contact_set_presence:
 * @self: #TpaUserContact instance.
 * @presence: #TpaContactPresence status.
 * @returns: TRUE if the message could be sent.
 */
gboolean
tpa_user_contact_set_presence (TpaUserContact *self,
                               TpaContactPresence presence)
{
    VERBOSE ("(%p, %d)", self, presence);

    return tpa_user_contact_set_presence_with_message (self,
                                                       presence,
                                                       self->priv->message);
}

/**
 * tpa_user_contact_set_presence_message:
 * @self: #TpaUserContact instance.
 * @message: String message.
 * @returns: TRUE if the message could be sent.
 */
gboolean
tpa_user_contact_set_presence_message (TpaUserContact *self,
                                       const gchar *message)
{
    VERBOSE ("(%p, %s)", self, message);

    return tpa_user_contact_set_presence_with_message (self,
                                                       self->priv->presence,
                                                       message);
}

/**
 * tpa_user_contact_set_presence_with_message:
 * @self: #TpaUserContact instance.
 * @presence: #TpaContactPresence status.
 * @message: String message.
 * @returns: TRUE if the message could be sent.
 */
gboolean
tpa_user_contact_set_presence_with_message (TpaUserContact *self,
                                            TpaContactPresence presence,
                                            const gchar *message)
{
    GHashTable *status = NULL;
    GHashTable *dic = NULL;
    GError *error = NULL;
    gboolean ret = TRUE;

    VERBOSE ("(%p, %d, %s)", self, presence, message);
    g_assert (self);
    g_return_val_if_fail (self->priv->proxy_presence != NULL, FALSE);

    if (message && (self->priv->message != message) &&
        !g_str_equal (self->priv->message, message)) {
        g_free (self->priv->message);
        self->priv->message = g_strdup (message);
    }

    if (presence != self->priv->presence)
        self->priv->presence = presence;

    INFO ("setting presence status: %s - %s", presence_status[presence], message);
    status = g_hash_table_new (g_str_hash, g_str_equal);
    dic = g_hash_table_new (g_str_hash, g_str_equal);
    if (message) {
        GValue *value = g_new0 (GValue, 1);

        g_value_init (value, G_TYPE_STRING);
        g_value_set_string (value, message);
        g_hash_table_insert (dic, "message", value);
    }
    g_hash_table_insert (status, (gpointer) presence_status[presence], dic);

    if (!org_freedesktop_Telepathy_Connection_Interface_Presence_set_status (self->priv->proxy_presence, status, &error) ||
        error) {
        ERROR ("%s", error->message);
        g_error_free (error);
        ret = FALSE;
    }

    VERBOSE ("return %s", ret ? "TRUE" : "FALSE");
    return ret;
}

/**
 * tpa_user_contact_set_alias:
 * @self: #TpaUserContact instance
 * @alias: String alias.
 * @returns: TRUE if alias could be set.
 */
gboolean
tpa_user_contact_set_alias (TpaUserContact *self,
                            const gchar *alias)
{
    TpaHandle *handle = NULL;
    GHashTable *aliases = NULL;
    GError *error = NULL;
    gboolean ret = TRUE;
    guint id;

    VERBOSE ("(%p, %s)", self, alias);
    g_assert (self);
    g_return_val_if_fail (self->priv->proxy_aliasing != NULL, FALSE);

    aliases = g_hash_table_new (g_direct_hash, g_direct_equal);
    handle = tpa_channel_target_get_handle (TPA_CHANNEL_TARGET (self));
    id = tpa_handle_get_id (handle);
    g_hash_table_insert (aliases, GUINT_TO_POINTER (id), (gpointer) alias);

    INFO ("setting alias: %s", alias);

    if (!org_freedesktop_Telepathy_Connection_Interface_Aliasing_set_aliases (self->priv->proxy_aliasing, aliases, &error) ||
        error) {
        ERROR ("%s", error->message);
        g_error_free (error);
        ret = FALSE;
    }

    g_hash_table_destroy (aliases);
    VERBOSE ("return %s", ret ? "TRUE" : "FALSE");
    return ret;
}

/**
 * tpa_user_contact_set_capabilities:
 * @self: #TpaUserContact instance.
 * @capability: TpaCapability capability.
 * @returns: TRUE if capability could be set.
 */
gboolean
tpa_user_contact_set_capabilities (TpaUserContact *self,
                                   TpaCapability capability)
{
    GError *error = NULL;
    GPtrArray *add;
    GPtrArray *ret = NULL;
    GValueArray *cap;
    guint flags = 0;
    const gchar *empty[1] = { "" };

    VERBOSE ("(%p, %d)", self, capability);
    g_assert (self);

    if (!self->priv->proxy_capabilities)
        return FALSE;

    add = g_ptr_array_new ();
    INFO ("setting user capabilities: %d", capability);

    if (capability & TPA_CAPABILITY_TEXT) {
        cap = g_value_array_new (2);

        g_value_array_append (cap, NULL);
        g_value_init (g_value_array_get_nth (cap, 0), G_TYPE_STRING);
        g_value_set_string (g_value_array_get_nth (cap, 0), TPA_INTERFACE_TEXT);

        g_value_array_append (cap, NULL);
        g_value_init (g_value_array_get_nth (cap, 1), G_TYPE_UINT);
        g_value_set_uint (g_value_array_get_nth (cap, 1), 3);

        g_ptr_array_add (add, cap);
    }

    if (capability & TPA_CAPABILITY_AUDIO ||
        capability & TPA_CAPABILITY_VIDEO) {
        cap = g_value_array_new (2);

        g_value_array_append (cap, NULL);
        g_value_init (g_value_array_get_nth (cap, 0), G_TYPE_STRING);
        g_value_set_string (g_value_array_get_nth (cap, 0), TPA_INTERFACE_STREAMED_MEDIA);

        if (capability & TPA_CAPABILITY_AUDIO)
            flags |= 1;
        if (capability & TPA_CAPABILITY_VIDEO)
            flags |= 2;

        g_value_array_append (cap, NULL);
        g_value_init (g_value_array_get_nth (cap, 1), G_TYPE_UINT);
        g_value_set_uint (g_value_array_get_nth (cap, 1), flags);

        g_ptr_array_add (add, cap);
    }

    if (!org_freedesktop_Telepathy_Connection_Interface_Capabilities_advertise_capabilities (self->priv->proxy_capabilities, add, empty, &ret, &error) ||
        error) {
        ERROR ("%s", error->message);
        g_error_free (error);
        VERBOSE ("return FALSE");
        return FALSE;
    }
    else {
        guint i;

        for (i = 0; i < ret->len; i++)
            g_value_array_free (g_ptr_array_index (ret, i));
    }
    g_ptr_array_free (add, TRUE);
    tpa_contact_base_capabilities_changed (TPA_CONTACT_BASE (self), capability);
    VERBOSE ("return TRUE");
    return TRUE;
}

