/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <glib.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-account.h"
#include "tpa-session.h"
#include "tpa-ifaces.h"

#include "tpa-connection-private.h"
#include "tpa-aliasing-private.h"
#include "tpa-avatars-private.h"
#include "tpa-capabilities-private.h"
#include "tpa-contact-info-private.h"
#include "tpa-forwarding-private.h"
#include "tpa-presence-private.h"
#include "tpa-properties-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION

#include <tapioca/base/tpa-connection-glue.h>
#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-enums.h>
#include <tapioca/base/tpa-errors.h>

#define _DBUS_PATH "/org/freedesktop/Telepathy/Connection"

G_DEFINE_TYPE_WITH_CODE(TpaAccount, tpa_account, G_TYPE_OBJECT, \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_ICONNECTION, tpa_connection_init) \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_IALIASING, tpa_aliasing_init) \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_IAVATARS, tpa_avatars_init) \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_ICAPABILITIES, tpa_capabilities_init) \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_ICONTACT_INFO, tpa_contact_info_init) \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_IFORWARDING, tpa_forwarding_init) \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_IPRESENCE, tpa_presence_init) \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_IPROPERTIES, tpa_properties_init))

struct _TpaAccountPrivate
{
    DBusGConnection *dbus;
    GHashTable *parameters;

    gboolean disposed;
    gboolean registered;
};

#define TPA_ACCOUNT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), TPA_TYPE_ACCOUNT, TpaAccountPrivate))

static void tpa_account_dispose (GObject *object);
static void tpa_account_finalize (GObject *object);

static void
tpa_account_class_init (TpaAccountClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    VERBOSE ("(%p)", klass);

    g_type_class_add_private (klass, sizeof (TpaAccountPrivate));

    object_class->dispose = tpa_account_dispose;
    object_class->finalize = tpa_account_finalize;

    dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (klass), &dbus_glib_connection_object_info);
}

static void
tpa_account_init (TpaAccount *self)
{
    self->priv = TPA_ACCOUNT_GET_PRIVATE (self);

    VERBOSE ("(%p)", self);

    self->manager = NULL;
    self->priv->parameters = NULL;
    self->priv->registered = FALSE;
}

void
tpa_account_dispose (GObject *object)
{
    TpaAccount *self = TPA_ACCOUNT (object);

    VERBOSE ("(%p)", self);

    if (self->priv->disposed) {
        /* If dispose did already run, return. */
        return;
    }

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    tpa_aliasing_finalize (object);
    tpa_avatars_finalize (object);
    tpa_capabilities_finalize (object);
    tpa_contact_info_finalize (object);
    tpa_forwarding_finalize (object);
    tpa_presence_finalize (object);
    tpa_properties_finalize (object);

    G_OBJECT_CLASS (tpa_account_parent_class)->dispose (object);
}

void
tpa_account_finalize (GObject *object)
{
    G_OBJECT_CLASS (tpa_account_parent_class)->finalize (object);
}

/**
 * tpa_connnection_new
 *
 * Create a new connection, if a it will try to register it under
 * org.freedesktop.Telepathy.Account.(manager).(protocol).(account)
 */
TpaAccount*
tpa_account_new (GType object_type,
                 TpaManager *manager,
                 GHashTable *parameters,
                 const gchar *protocol,
                 const gchar *account)
{
    TpaIConnectionManager *iface = NULL;
    TpaAccount *self = NULL;

    VERBOSE ("(%p, %p, %s, %s)", manager, parameters, protocol, account);

    self = TPA_ACCOUNT (g_object_new (object_type, NULL));

    if (self && self->priv) {
        self->priv->parameters = parameters;
        self->manager = manager;
        if (manager && TPA_IS_ICONNECTION_MANAGER (manager)) {
            iface = TPA_ICONNECTION_MANAGER (manager);
            tpa_account_register (self, iface->name, protocol, account);
            tpa_manager_add_account (manager, G_OBJECT (self));
        }
        else
            tpa_account_register (self, "manager", protocol, account);
    }

    return self;
}

/**
 * tpa_account_unref
 *
 * Delete connection created via tpa_account_new
 */
void
tpa_account_unref (TpaAccount *self)
{
    VERBOSE ("(%p)", self);

    if (self->manager)
        tpa_manager_remove_account (self->manager, G_OBJECT (self));

    g_object_unref (self);
}

/**
 * tpa_account_register
 *
 * Register connection managers on dbus, the DBUS path always
 * begins with "/org/freedesktop/Telepathy/Connection".
 */
gboolean
tpa_account_register (TpaAccount *self,
                      const gchar *manager,
                      const gchar *protocol,
                      const gchar *account)
{
    DBusGProxy* driver_proxy = NULL;
    gchar *dbus_path = NULL;
    gchar *dbus_iface = NULL;
    gboolean ok = FALSE;
    GError* error = NULL;
    int request_ret = 0;
    gchar *dbus_account = NULL;
    TpaIConnection *connection = NULL;

    g_assert (TPA_IS_ACCOUNT (self));

    if (self->priv->dbus == NULL) {
        self->priv->dbus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);

        if (self->priv->dbus == NULL) {
            VERBOSE ("Unable to connect to dbus: %s", error->message);
            g_error_free (error);
        }
    }

    VERBOSE ("(%p, %s, %s, %s)", self, manager, protocol, account);
    g_return_val_if_fail (manager != NULL, FALSE);
    g_return_val_if_fail (g_str_equal (manager, "") != FALSE, FALSE);
    g_return_val_if_fail (protocol != NULL, FALSE);
    g_return_val_if_fail (g_str_equal (protocol, "") != FALSE, FALSE);
    g_return_val_if_fail (account != NULL, FALSE);
    g_return_val_if_fail (g_str_equal (account, "") != FALSE, FALSE);

    dbus_account = g_strdup (account);
    g_strdelimit (dbus_account, ":@.", '_');
    dbus_path = g_strdup_printf ("%s/%s/%s/%s", DBUS_CONNECTION_PATH, manager, protocol, dbus_account);

    dbus_iface = g_strdup_printf ("%s.%s.%s.%s", DBUS_CONNECTION_IFACE, manager, protocol, dbus_account);

    /* Register DBUS path */
    dbus_g_connection_register_g_object (self->priv->dbus,
        dbus_path,
        G_OBJECT (self));

    /* Register the service name, the constants here are defined in dbus-glib.h */
    driver_proxy = dbus_g_proxy_new_for_name (self->priv->dbus,
            DBUS_SERVICE_DBUS,
            DBUS_PATH_DBUS,
            DBUS_INTERFACE_DBUS);


    ok = dbus_g_proxy_call (driver_proxy, "RequestName", &error,
                            G_TYPE_STRING, dbus_iface,
                            G_TYPE_UINT, DBUS_NAME_FLAG_DO_NOT_QUEUE,
                            G_TYPE_INVALID,
                            G_TYPE_UINT, &request_ret, /* See tutorial for more infos about these */
                            G_TYPE_INVALID);

    if(!ok) {
        VERBOSE ("(%p) Unable to register service: %s\n", self, error->message);
        g_error_free (error);
        return FALSE;
    }
    else if (request_ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
        switch (request_ret) {
            case DBUS_REQUEST_NAME_REPLY_IN_QUEUE:
              g_warning ("Request has been queued, though we request non-queueing.");
              break;
            case DBUS_REQUEST_NAME_REPLY_EXISTS:
              g_warning ("A manager already has this busname.");
              break;
            case DBUS_REQUEST_NAME_REPLY_ALREADY_OWNER:
              g_warning ("Manager already has a connection to this account.");
              break;
            default:
              g_warning ("Unknown error return from ReleaseName");
        }
        return FALSE;
    }
    else {
        VERBOSE ("(%p) interface registered: %s", self, dbus_iface);
        connection = TPA_ICONNECTION (self);
        connection->protocol = g_strdup (protocol);
        connection->handle = tpa_connection_get_handle (G_OBJECT (self),
            TPA_CONNECTION_HANDLE_TYPE_CONTACT, account);
        self->priv->registered = ok;
    }

    g_free (dbus_account);
    g_free (dbus_path);
    g_free (dbus_iface);
    g_object_unref (driver_proxy);

    return TRUE;
}

/**
 * tpa_connection_add_session
 *
 */
void
tpa_account_add_session (TpaAccount *self,
                         GObject *session)
{
    TpaIConnection *connection = NULL;

    g_assert (TPA_IS_ACCOUNT (self));
    g_assert (TPA_IS_SESSION (session));

    VERBOSE ("(%p, %p)", self, session);

    connection = TPA_ICONNECTION (self);
    if (connection->sessions) {
        g_ptr_array_add (connection->sessions, session);
        VERBOSE ("session %p added", session);
    }
}

/**
 * tpa_connection_remove_session
 *
 */
void
tpa_account_remove_session (TpaAccount *self,
                            GObject *session)
{
    TpaIConnection *connection = NULL;

    g_assert (TPA_IS_ACCOUNT (self));
    g_assert (TPA_IS_SESSION (session));

    VERBOSE ("(%p, %p)", self, session);

    connection = TPA_ICONNECTION (self);
    if (connection->sessions) {
        g_ptr_array_remove (connection->sessions, session);
        VERBOSE ("session %p removed", session);
    }
}
