/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib-lowlevel.h>
#include <glib.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-channel.h"
#include "tpa-ifaces.h"

#include "tpa-channel-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>

/* signal enum */
enum
{
    CLOSED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

/* we need to define the get_type function */
GType
tpa_channel_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIChannel),
            NULL,   /* base init */
            NULL,   /* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaIChannel",
                &object_info, 0);
    }
    return object_type;
}


void
tpa_channel_init (TpaIChannel *iface,
                  gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->close = NULL;
    iface->get_interfaces = NULL;

    signals[CLOSED] =
        g_signal_new ("closed",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

    iface->closed = FALSE;
    iface->type = -1;
    iface->handle = NULL;
}

void
tpa_channel_finalize (GObject *obj)
{
    TpaIChannel *iface = TPA_ICHANNEL (obj);

    VERBOSE ("(%p)", obj);

    if (!iface->closed)
        tpa_channel_signal_closed (obj);
}

/**
 * tpa_channel_close
 *
 * Implements DBus method Close
 * on interface org.freedesktop.Telepathy.Channel
 *
 * error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_channel_close (GObject *obj,
                   GError **error)
{
    TpaIChannel *iface = TPA_ICHANNEL (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->close != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    if (!iface->closed)
        error_code = iface->close (obj);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_channel_get_channel_type
 *
 * Implements DBus method ChannelType
 * on interface org.freedesktop.Telepathy.Channel
 *
 * error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_channel_get_channel_type (GObject *obj,
                              const gchar **ret,
                              GError **error)
{
    TpaIChannel *iface = TPA_ICHANNEL (obj);

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    if (iface->type != -1 && ret) {
        VERBOSE ("Type: %d", iface->type);
        switch (iface->type) {
            case TPA_CHANNEL_TYPE_CONTACTLIST:
                *ret = g_strdup (DBUS_TYPE_CONTACTLIST_IFACE);
                break;
            case TPA_CHANNEL_TYPE_CONTACTSEARCH:
                *ret = g_strdup (DBUS_TYPE_CONTACTSEARCH_IFACE);
                break;
            case TPA_CHANNEL_TYPE_STREAMEDMEDIA:
                *ret = g_strdup (DBUS_TYPE_STREAMEDMEDIA_IFACE);
                break;
            case TPA_CHANNEL_TYPE_ROOMLIST:
                *ret = g_strdup (DBUS_TYPE_ROOMLIST_IFACE);
                break;
            case TPA_CHANNEL_TYPE_TEXT:
                *ret = g_strdup (DBUS_TYPE_TEXT_IFACE);
                break;
            default:
                *ret = g_strdup ("");
                VERBOSE ("Invalid type");
        }
    }

    return TRUE;
}

/**
 * tpa_channel_get_handle
 *
 * Implements DBus method GetHandle
 * on interface org.freedesktop.Telepathy.Channel
 *
 * error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_channel_get_handle (GObject *obj,
                        guint *ret,
                        guint *ret1,
                        GError **error)
{
    TpaIChannel *iface = TPA_ICHANNEL (obj);

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p, %p)", obj, ret, ret1);

    if (ret && ret1 && iface->handle) {
        *ret = iface->handle->type;
        *ret1 = iface->handle->id;
    }

    return TRUE;
}

/**
 * tpa_channel_get_interfaces
 *
 * Implements DBus method GetInterfaces
 * on interface org.freedesktop.Telepathy.Channel
 *
 * error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_channel_get_interfaces (GObject *obj,
                            gchar ***ret,
                            GError **error)
{
    TpaIChannel *iface = TPA_ICHANNEL (obj);
    TpaError error_code = TPA_ERROR_NONE;
    const gchar *interfaces[] = { NULL };

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    /* Check if CM implements GetInterfaces */
    if (iface->get_interfaces)
        error_code = iface->get_interfaces (obj, ret);
    else if (ret) {
        *ret = g_strdupv ((gchar **) interfaces);
    }

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_channel_signal_closed
 *
 * Implements DBus signal Closed
 * on interface org.freedesktop.Telepathy.Channel
 */
void
tpa_channel_signal_closed (GObject *obj)
{
    TpaIChannel *iface = TPA_ICHANNEL (obj);

    g_assert (TPA_IS_ICHANNEL (iface));
    VERBOSE ("(%p)", obj);

    if (!iface->closed)
        iface->closed = TRUE;

    g_signal_emit (obj, signals[CLOSED], 0);
}
