/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-ice.h"

#include "tpa-ice-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>

/* signal enum */
enum
{
    NEW_ICE_SESSION_HANDLER,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

struct _TpaIcePrivate {
    GObject *object;
};

/* we need to define the get_type function */
GType
tpa_ice_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIIce),
            NULL,   /* base init */
            NULL,   /* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaIIce",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_ice_init (TpaIIce *iface,
              gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->get_session_handler = NULL;

    /* Interface signals */
    signals[NEW_ICE_SESSION_HANDLER] =
        g_signal_new ("new-ice-session-handler",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      tpa_marshal_VOID__STRING_STRING,
                      G_TYPE_NONE, 2, DBUS_TYPE_G_OBJECT_PATH, G_TYPE_STRING);
    iface->priv = g_new0 (TpaIIcePrivate, 1);
    VERBOSE ("private members %p", iface->priv);
}

void
tpa_ice_finalize (GObject *obj)
{
    TpaIIce *iface = TPA_IICE (obj);
    VERBOSE ("(%p)", obj);

    if (iface->priv)
        g_free (iface->priv);
}

/**
 * tpa_ice_signalling_get_session_handlers
 *
 * Implements DBus method GetSessionHandlers
 * on interface org.freedesktop.Telepathy.Channel.Interface.IceSignalling
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_ice_signalling_get_session_handlers (GObject *obj,
                                         GPtrArray *ret,
                                         GError **error)
{
    TpaIIce *iface = TPA_IICE (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    g_return_error_val_if_fail (iface->get_session_handler != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->get_session_handler (obj, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_ice_signal_new_ice_session_handler
 *
 * Implements DBus signal NewIceSessionHandler
 * on interface org.freedesktop.Telepathy.Channel.Interface.Ice
 */
void
tpa_ice_signal_new_ice_session_handler (GObject *obj,
                                        const gchar *session_handler,
                                        const gchar *type)
{
    g_assert (TPA_IS_IICE (obj));

    VERBOSE ("(%p, %s, %s)", obj, session_handler, type);

    g_signal_emit (obj, signals[NEW_ICE_SESSION_HANDLER], 0, session_handler, type);
}
