/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib-lowlevel.h>
#include <glib.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-session.h"
#include "tpa-ifaces.h"

#include "tpa-channel-private.h"
#include "tpa-group-private.h"
#include "tpa-ice-private.h"
#include "tpa-media-private.h"
#include "tpa-room-list-private.h"
#include "tpa-text-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-channel-glue.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>

#define _DBUS_PATH "/org/freedesktop/Telepathy/Channel"

#define TPA_CHANNEL_LIST_TYPE (dbus_g_type_get_struct ("GValueArray", \
      DBUS_TYPE_G_OBJECT_PATH, G_TYPE_STRING, G_TYPE_UINT, G_TYPE_UINT, \
      G_TYPE_INVALID))

G_DEFINE_TYPE_WITH_CODE(TpaSession, tpa_session, G_TYPE_OBJECT, \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_ICHANNEL, tpa_channel_init) \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_ITEXT, tpa_text_init) \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_IMEDIA, tpa_media_init) \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_IROOM_LIST, tpa_room_list_init) \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_IGROUP, tpa_group_init) \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_IICE, tpa_ice_init))

/* signal enum */
enum
{
    CLOSED,
    LAST_SIGNAL
};

struct _TpaSessionPrivate
{
    DBusGConnection *dbus;

    TpaHandle *handle;
    gchar *manager;
    gchar *name;
    gboolean registered;
    gboolean disposed;
};

#define TPA_SESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), TPA_TYPE_SESSION, TpaSessionPrivate))

void
tpa_session_dispose (GObject *object)
{
    TpaSession *self = TPA_SESSION (object);

    g_debug ("(%p) %s", self, G_STRFUNC);

    if (self->priv->disposed) {
        /* If dispose did already run, return. */
        return;
    }

    if (self->priv->manager)
        g_free (self->priv->manager);

    if (self->priv->name)
        g_free (self->priv->name);

    if (self->account)
        tpa_account_remove_session (self->account, G_OBJECT (self));

    tpa_channel_signal_closed (G_OBJECT (self));

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    tpa_media_finalize (object);
    tpa_ice_finalize (object);
    tpa_group_finalize (object);

    G_OBJECT_CLASS (tpa_session_parent_class)->dispose (object);
}

void
tpa_session_finalize (GObject *object)
{
  G_OBJECT_CLASS (tpa_session_parent_class)->finalize (object);
}

static void
tpa_session_class_init (TpaSessionClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (TpaSessionPrivate));

    object_class->dispose = tpa_session_dispose;
    object_class->finalize = tpa_session_finalize;

    dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (klass), &dbus_glib_channel_object_info);
}

static void
tpa_session_init (TpaSession *self)
{
    self->priv = TPA_SESSION_GET_PRIVATE (self);

    VERBOSE ("(%p)", self);

    self->priv->handle = NULL;
    self->priv->manager = NULL;
    self->priv->name = NULL;
    self->priv->registered = FALSE;
}

/**
 * tpa_session_new
 *
 * Create a new channel, if a it will try to register it under
 * org.freedesktop.Telepathy.Channel.(manager).(protocol).(account)
 */
TpaSession*
tpa_session_new (GType object_type,
                 TpaAccount *account,
                 TpaChannelType type,
                 TpaHandle *handle,
                 const gchar *manager,
                 const gchar *name)
{
    TpaSession *self = NULL;
    TpaIChannel *iface = NULL;

    VERBOSE ("(%p, %d, %s, %s)", account, type, manager, name);

    self = TPA_SESSION (g_object_new (object_type, NULL));

    if (self && self->priv) {
        iface = TPA_ICHANNEL (self);
        iface->type = type;
        iface->handle = handle;
        tpa_session_register (self, manager, name);
        self->account = account;

        if (account)
            tpa_account_add_session (account, G_OBJECT (self));
    }

    return self;
}

/**
 * tpa_session_unref
 *
 * Delete session created via #tpa_session_new
 */
void
tpa_session_unref (TpaSession *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    g_object_unref (self);
}

/**
 * tpa_session_register
 *
 * Register connection managers on dbus, the DBUS path always
 * begins with "/org/freedesktop/Telepathy/Channel".
 */
gboolean
tpa_session_register (TpaSession *self,
                      const gchar *manager,
                      const gchar *name)
{
    DBusGProxy* driver_proxy = NULL;
    gchar *dbus_path = NULL;
    gchar *dbus_iface = NULL;
    gboolean ok = FALSE;
    GError* error = NULL;
    int request_ret = 0;

    g_assert (TPA_IS_SESSION (self));

    if (self->priv->dbus == NULL) {
        self->priv->dbus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);

        if (self->priv->dbus == NULL) {
            VERBOSE ("Unable to connect to dbus: %s", error->message);
            g_error_free (error);
        }
    }

    VERBOSE ("(%p, %s, %s)", self, manager, name);
    g_return_val_if_fail (manager != NULL, FALSE);
    g_return_val_if_fail (g_str_equal (manager, "") != FALSE, FALSE);
    g_return_val_if_fail (name != NULL, FALSE);
    g_return_val_if_fail (g_str_equal (name, "") != FALSE, FALSE);

    dbus_path = g_strdup_printf ("%s/%s/%s", DBUS_CHANNEL_PATH, manager, name);
    dbus_iface = g_strdup_printf ("%s.%s.%s", DBUS_CHANNEL_IFACE, manager, name);

    /* Register DBUS path */
    dbus_g_connection_register_g_object (self->priv->dbus,
        dbus_path,
        G_OBJECT (self));

    /* Register the service name, the constants here are defined in dbus-glib.h */
    driver_proxy = dbus_g_proxy_new_for_name (self->priv->dbus,
            DBUS_SERVICE_DBUS,
            DBUS_PATH_DBUS,
            DBUS_INTERFACE_DBUS);

    ok = dbus_g_proxy_call (driver_proxy, "RequestName", &error,
                            G_TYPE_STRING, dbus_iface,
                            G_TYPE_UINT, 0,
                            G_TYPE_INVALID,
                            G_TYPE_UINT, &request_ret, /* See tutorial for more infos about these */
                            G_TYPE_INVALID);

    if(!ok) {
        VERBOSE ("(%p) Unable to register service: %s\n", self, error->message);
        g_error_free (error);
    }
    else {
        VERBOSE ("(%p) interface registered: %s", self, dbus_iface);
        /* TODO: Generate self handle */
        self->priv->manager = g_strdup (manager);
        self->priv->name = g_strdup (name);
        self->priv->registered = ok;
    }

    g_free (dbus_path);
    g_free (dbus_iface);
    g_object_unref (driver_proxy);
    return ok;
}

/**
 * tpa_session_list
 *
 * Append channel data to list.
 * Used when ListChannels are called.
 */
void
tpa_session_list (TpaSession *self,
                  GPtrArray *list)
{
    gchar *path = NULL;
    gchar *type = NULL;
    GValue entry = { 0, };

    g_assert (TPA_IS_SESSION (self));

    VERBOSE ("(%p, %p)", self, list);

    g_value_init (&entry, TPA_CHANNEL_LIST_TYPE);
    g_value_take_boxed (&entry, dbus_g_type_specialized_construct
        (TPA_CHANNEL_LIST_TYPE));

    if (self->priv->handle && self->priv->manager && self->priv->name) {
        path = g_strdup_printf ("%s/%s/%s", _DBUS_PATH, self->priv->manager, self->priv->name);
        tpa_channel_get_channel_type (G_OBJECT (self), (const gchar**) &type, NULL);

        dbus_g_type_struct_set (&entry,
            0, path,
            1, type,
            2, self->priv->handle->type,
            3, self->priv->handle->id,
            G_MAXUINT);

        g_ptr_array_add (list, g_value_get_boxed (&entry));

        g_free (path);
        g_free (type);
    }
}
