/* $Id: //depot/Teapop/0.3/teapop/pop_maildir.c#7 $ */

/*
 * Copyright (c) 2000-2001 ToonTown Consulting
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the company nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <dirent.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>

#include "config.h"

#ifdef HAVE_MD5_H
#include <md5.h>
#else
#include "md5.h"
#endif /* HAVE_MD5 */

#include "teapop.h"
#include "pop_maildir.h"

/*
 * TODO: Like other maildir programs the name of file can be used to define the 
 * UIDL of message, because message filename is unique
 * Here is a problem because the routine that calculate UIDL also count the 
 * number of lines that is need in pop_dnld.c
 * the need for caching UIDL is increasing because users does not like the 
 * delay when connecting to the pop server
 */

/*
 * Temporary storage for work in add_maildir_message
 * This structure has the big fields and the necessary
 * fields to be used between calls to add_maildir_message
 */
typedef struct _maildir_info
{
	POP_MSG *lastmsg;
	char buf[1024];
	char realname[256];
/*
 * Used to name the message files with unique name
 */
	char hostname[128];
	int pid;
	int msgcount;
}
MAILDIR_INFO;

/*
 * This routine checks a message and adds it to user messages
 * It received the directory and filename as separate to make it
 * easy to make checks directly in filename
 */
int
add_maildir_message(minfo, pinfo, dir, fname)
MAILDIR_INFO *minfo;
POP_INFO *pinfo;
char *dir;
char *fname;
{
	int counter, finduidl, retval;
	char *ptr;
	unsigned char digest[16];
	FILE *fd;
	MD5_CTX ctx;
	POP_MSG *curmsg;
	struct stat sb;

	/*
	 * Define the full name of message file
	 */
	strncpy(minfo->realname, dir, sizeof(minfo->realname));
	strncat(minfo->realname, fname,
	    sizeof(minfo->realname) - strlen(minfo->realname));
	minfo->realname[sizeof(minfo->realname)] = '\0';

	retval = lstat(minfo->realname, &sb);
	if (retval != 0 || !S_ISREG(sb.st_mode))
		return 0;

	/*
	 * XXX - Add a sanity check so the filename
	 * is in the correct format, or ignore the
	 * file.
	 */
	/*
	 * Maildir specification say to ignore files beginning with '.'
	 */
	if (fname[0] == '.')
		return 0;

	if ((curmsg = malloc(sizeof(POP_MSG))) == NULL) {
		syslog(LOG_ERR,
		    "Error allocating memory in add_maildir_message : %d",
		    errno);
		return 1;
	}
	bzero(curmsg, sizeof(POP_MSG));
	if ((ptr = malloc((strlen(minfo->realname) + 1) *
		    sizeof(char))) == NULL) {
		free(curmsg);
		syslog(LOG_ERR,
		    "Error allocating memory in add_maildir_message : %d",
		    errno);
		return 1;
	}
	(void) strcpy(ptr, minfo->realname);
	curmsg->file = ptr;

	if ((fd = fopen(curmsg->file, "r")) == NULL) {
		syslog(LOG_ERR, "Error opening message file : %s %d",
		    curmsg->file, errno);
		free(ptr);
		free(curmsg);
		return 0;
	}
	finduidl = pinfo->useuidl;
	MD5Init(&ctx);
/*
 * When FAST_MAILDIR is defined it will calculate the UIDL from filename
 * instead of using the data on file
 */	
#ifdef FAST_MAILDIR
	MD5Update(&ctx, (unsigned char *) curmsg->file, strlen(curmsg->file));
	(void) fseek(fd, 0, SEEK_END);
	curmsg->size = ftell(fd);
	curmsg->lines = 99999999; 
#else
	while (fgets(minfo->buf, sizeof(minfo->buf), fd)) {
		if (minfo->buf[strlen(minfo->buf) - 1] == '\n') {
			curmsg->size++;
			curmsg->lines++;
		} else if (strlen(minfo->buf) != sizeof(minfo->buf))
			minfo->buf[strlen(minfo->buf)] = '?';
		if (finduidl == 1 && !strncasecmp(minfo->buf, "X-UIDL: ", 8)) {
			minfo->buf[strlen(minfo->buf) - 1] = '\0';
			strncpy(curmsg->uidl, &minfo->buf[8],
			    sizeof(curmsg->uidl));
			curmsg->uidl[sizeof(curmsg->uidl) - 1] = '\0';
			finduidl = 0;
		}
		curmsg->size += strlen(minfo->buf);
		MD5Update(&ctx, (unsigned char *) minfo->buf,
		    strlen(minfo->buf));
	}
#endif
	(void) fclose(fd);

	MD5Final(digest, &ctx);
	if (!(finduidl == 0 && pinfo->useuidl == 1)) {
		ptr = curmsg->uidl;
		for (counter = 0; counter < (int) sizeof(digest);
		    counter++, ptr += 2)
			sprintf(ptr, "%02x", digest[counter] & 0xff);
		*ptr = '\0';
	}

	if (pinfo->firstmsg == NULL)
		pinfo->firstmsg = minfo->lastmsg = curmsg;
	else
		minfo->lastmsg = minfo->lastmsg->nextmsg = curmsg;

	return 0;
}

int
pop_maildir_get_status(pinfo)
POP_INFO *pinfo;
{
	int i;
	char procdir[10];

	DIR *dirp;
	MAILDIR_INFO minfo;
	struct dirent *dp;

	/*
	 * Initialize the MAILDIR_INFO
	 */
	minfo.pid = getpid();
	gethostname((char *) &minfo.hostname, sizeof(minfo.hostname));
	minfo.msgcount = 0;
	/*
	 * This is not really needed, but it makes -Wall happy, AND
	 * it's there if Teapop ever in the future would support a
	 * user with several mailboxes.
	 */
	if (pinfo->firstmsg == NULL)
		minfo.lastmsg = NULL;
	else {
		minfo.lastmsg = pinfo->firstmsg;
		while (minfo.lastmsg->nextmsg != NULL)
			minfo.lastmsg = minfo.lastmsg->nextmsg;
	}

	for (i = 0; i < 2; i++) {
		if (i == 0) {
			dirp = opendir("cur");
			(void) strncpy(procdir, "cur/", sizeof(procdir));
		} else if (i == 1) {
			dirp = opendir("new");
			(void) strncpy(procdir, "new/", sizeof(procdir));
		} else
			return 1;

		if (dirp == NULL) {
			syslog(LOG_ERR, "can't open maildir/%d %s (%d)", i,
			    procdir, errno);
			continue;
		}

		while ((dp = readdir(dirp)) != NULL) {

			if (add_maildir_message(&minfo, pinfo, procdir,
				dp->d_name) == 1) {
				break;
			}
		}
		(void) closedir(dirp);
	}
	return 0;
}
