/*
 *  Copyright (C) 2006, Raphaël Slinckx <raphael@slinckx.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id$
 */
 
#include "tp-blue-sms-controller.h"
#include "tp-blue-at-controller.h"
#include <string.h>
#include <stdio.h>

#define TP_BLUE_SMS_CONTROLLER_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), TP_BLUE_TYPE_SMS_CONTROLLER, TpBlueSmsControllerPrivate))


/* Type = sms submit */
#define PDU_TYPE "11" /*#"01"*/
/* Let the phone handle this */
#define MESSAGE_REFERENCE "00"
/* Type of phone number (= international) */
#define TYPE_OF_ADDRESS "A1" /*#"91"*/
/* PID  = 00 */
#define TP_PID "00"
/* 7bit encoding */
#define TP_DCS "00"
/* Validity Period */
#define TP_VALIDITY "FF"

struct _TpBlueSmsControllerPrivate
{
	TpBlueAtController *at_controller;
	gchar *bt_address;
	guint bt_channel;
};

enum {
	PROP0,
	BT_ADDRESS,
	BT_CHANNEL,
};

enum
{
	SMS_RECEIVED,
	DISCONNECTED,
	LAST_SIGNAL
};
static guint signals[LAST_SIGNAL] = { 0 };


G_DEFINE_TYPE(TpBlueSmsController, tp_blue_sms_controller, G_TYPE_OBJECT)

static void tp_blue_sms_controller_class_init		(TpBlueSmsControllerClass *klass);
static void tp_blue_sms_controller_init		(TpBlueSmsController *obj);
static void tp_blue_sms_controller_finalize		(GObject *object);
static void tp_blue_sms_controller_set_property (GObject      *object,
                        guint         property_id,
                        const GValue *value,
                        GParamSpec   *pspec);
static void tp_blue_sms_controller_get_property (GObject      *object,
                        guint         property_id,
                        GValue       *value,
                        GParamSpec   *pspec);

static void
tp_blue_sms_controller_class_init (TpBlueSmsControllerClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);
	object_class->finalize = tp_blue_sms_controller_finalize;
	object_class->set_property = tp_blue_sms_controller_set_property;
	object_class->get_property = tp_blue_sms_controller_get_property;
  
	/* Install properties and signals here */
	signals[SMS_RECEIVED] =
		g_signal_new ("sms-received",
				G_OBJECT_CLASS_TYPE (object_class),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (TpBlueSmsControllerClass, sms_received),
				NULL, NULL,
				g_cclosure_marshal_VOID__STRING,
				G_TYPE_NONE,
				1,
				G_TYPE_STRING);
				
	signals[DISCONNECTED] =
		g_signal_new ("disconnected",
				G_OBJECT_CLASS_TYPE (object_class),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (TpBlueSmsControllerClass, disconnected),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID,
				G_TYPE_NONE,
				0);
				
	g_object_class_install_property (object_class,
				BT_ADDRESS,
				g_param_spec_string ("bt-address",
					"Bluetooth Device Address",
					"Set the bluetooth device address to connect to",
					NULL,
					G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));
					
	g_object_class_install_property (object_class,
				BT_CHANNEL,
				g_param_spec_uint ("bt-channel",
					"Bluetooth RFCOMM Channel",
					"Set the bluetooth rfcomm channel to connect to",
					0, 30, 0,
					G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));
                                   
	g_type_class_add_private (object_class, sizeof (TpBlueSmsControllerPrivate));
}


static void
tp_blue_sms_controller_finalize (GObject *object)
{
	/*TpBlueSmsController *obj = TP_BLUE_SMS_CONTROLLER (object);
	TpBlueSmsControllerPrivate *priv = obj->priv;*/
	
	G_OBJECT_CLASS (tp_blue_sms_controller_parent_class)->finalize (object);
}

static void
tp_blue_sms_controller_set_property (GObject      *object,
                        guint         property_id,
                        const GValue *value,
                        GParamSpec   *pspec)
{
	TpBlueSmsController *self = TP_BLUE_SMS_CONTROLLER (object);

	switch (property_id) {
		case BT_ADDRESS:
			g_free (self->priv->bt_address);
			self->priv->bt_address = g_value_dup_string (value);
			break;
		case BT_CHANNEL:
			self->priv->bt_channel = g_value_get_uint (value);
			break;
		default:
			/* We don't have any other property... */
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object,property_id,pspec);
			break;
	}
}

static void
tp_blue_sms_controller_get_property (GObject      *object,
                        guint         property_id,
                        GValue       *value,
                        GParamSpec   *pspec)
{
	TpBlueSmsController *self = TP_BLUE_SMS_CONTROLLER (object);

	switch (property_id) {
		case BT_ADDRESS:
			g_value_set_string (value, self->priv->bt_address);
			break;
		case BT_CHANNEL:
			g_value_set_uint (value, self->priv->bt_channel);
			break;
		default:
			/* We don't have any other property... */
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object,property_id,pspec);
			break;
	}
}


TpBlueSmsController *
tp_blue_sms_controller_new (gchar *address, guint channel)
{
	return TP_BLUE_SMS_CONTROLLER (
		g_object_new (TP_BLUE_TYPE_SMS_CONTROLLER,
			"bt-address", address,
			"bt-channel", channel,
			NULL));
}

static gchar *
tp_blue_sms_controller_encode_number (TpBlueSmsController *controller, gchar *number)
{
	gchar *bcd;
	gsize number_length, bcd_length;
	gint i, j;
	g_message ("Encoding number: %s", number);
	
	number_length = strlen (number);
	bcd_length = number_length + (number_length % 2) + 1;
	bcd = g_new0(gchar, bcd_length);
	
	j=0;
	for (i=0; i < number_length; i += 2)
	{
		if (i+1 < number_length)
		{
			bcd[j++] = number[i+1];
			bcd[j++] = number[i];
		}
		else
		{
			bcd[j++] = 'F';
			bcd[j++] = number[i];
		}
	}
			
	return bcd;
}

static gchar *
tp_blue_sms_controller_encode_message (TpBlueSmsController *controller, gchar *message)
{
	gint length;
	gint j, i, bb;
	GString *dp;
	gchar *c;
	g_message ("Encoding message in 7bits: %s", message);
	
	length = strlen(message);
	dp = g_string_new ("");
	
	j = 0;
	while (length > 0)
	{
		bb = message[j];
		i = 7;
		while (i > 0 && length > 0)
		{
			bb = bb | message[j+8-i] << i;
			
			c = g_strdup_printf ("%02x", 0xff & bb);
			g_string_append(dp, c);
			g_free (c);
			
			bb = bb >> 8;
			length = length - 1;
			i = i - 1;
		}
		length = length - 1;
		j = j + 8;
	}
	
	return g_string_free (dp, FALSE);
}

static gchar *
tp_blue_sms_controller_decode_message (TpBlueSmsController *controller, gchar *message, guint length)
{
	GString *dp;
	guchar *c, *packed;
	guchar j, i;
	guchar bb;
	guint msg_length = strlen (message);
	g_message ("decoding message: %s %d", message, msg_length);
	
	packed = g_new0 (guchar, msg_length/2);
	for (i = 0; i<msg_length; i+=2)
	{
		c = g_strdup_printf ("%c%c", message[i], message[i+1]);
		packed[i/2] = g_ascii_strtoull(c, NULL, 16);
		g_message ("Adding %c%c %d %u", message[i], message[i+1], i, packed[i/2]);
		g_free (c);
	}
	
	dp = g_string_new ("");
	j = 0;
	while (length > 0)
	{
		i = 0;
		bb = 0;
		while (i < 7 && length > 0)
		{
			bb = bb | packed[i+j] << i;
			
			c = g_strdup_printf ("%c", bb & 0x7f);
			g_message ("Appending: %s", c);
			g_string_append(dp, c);
			g_free (c);
			
			bb = bb >> 7;

			i = i + 1;
			length = length - 1;
		}
		length = length - 1;
		if (length > 0)
		{
			c = g_strdup_printf ("%c", bb);
			g_string_append(dp, c);
			g_free (c);
		}
		j = j + 7;
	}

	return g_string_free (dp, FALSE);
}

static gchar *
tp_blue_sms_controller_encode_pdu (TpBlueSmsController *self,
										gchar *number,
										gchar *message)
{
	gchar *ADDRESS_LENGTH, *PHONE_NUMBER, *TP_USER_DATA_LENGTH, *TP_USER_DATA, *PDU, *uPDU;
	gchar *converted_message;
	GError *err = NULL;
	
	if (number == NULL || number[0] != '+')
	{
		g_warning ("Number does not start with +");
		return NULL;
	}
	number++;
	
	converted_message = g_convert (message, strlen (message), "utf-8", "iso-8859-1", NULL, NULL, &err);
	if (err != NULL) {
		g_warning ("Conversion error: %d %s", err->code, err->message);
		g_error_free (err);
		converted_message = g_strdup (message);
	}
		
	
	ADDRESS_LENGTH = g_strdup_printf ("%02x", strlen (number));
	PHONE_NUMBER = tp_blue_sms_controller_encode_number (self, number);
	TP_USER_DATA_LENGTH = g_strdup_printf ("%02x", strlen (converted_message));
	TP_USER_DATA = tp_blue_sms_controller_encode_message (self, converted_message);
	
	g_message ("%s:%s:%s:%s:%s:%s:%s:%s:%s:%s:%s",
		"00", PDU_TYPE, MESSAGE_REFERENCE, ADDRESS_LENGTH,
		TYPE_OF_ADDRESS, PHONE_NUMBER, TP_PID, TP_DCS,
		TP_VALIDITY, TP_USER_DATA_LENGTH, TP_USER_DATA);
		
	PDU = g_strconcat ("00", PDU_TYPE, MESSAGE_REFERENCE,
				ADDRESS_LENGTH, TYPE_OF_ADDRESS,
				PHONE_NUMBER, TP_PID, TP_DCS,
				TP_VALIDITY, TP_USER_DATA_LENGTH, TP_USER_DATA, "\x1A\r", NULL);
	
	uPDU = g_ascii_strup (PDU, -1);
	g_free (PDU);
	g_free (ADDRESS_LENGTH);
	g_free (TP_USER_DATA_LENGTH);
	g_free (TP_USER_DATA);
	g_free (converted_message);
	
	return uPDU;
}

static void
tp_blue_sms_controller_decode_sms (TpBlueSmsController *self, gchar *sms)
{
#if 0	
	gchar *length, *type_of_address;
	n = sscanf(message, "%2s%2s", &length, &type_of_address);
#endif
}

static void
tp_blue_sms_controller_fetch_sms (TpBlueSmsController *self, gchar *mem, gchar *index)
{
#if 0
	gboolean match;
	gchar *tmp, *sms = NULL;
	g_message ("Mem: '%s' Index: '%s'", mem, index);
	
	tmp = g_strdup_printf ("AT+CPMS=\"%s\"\r", mem);
	match = tp_blue_at_controller_send (self->priv->at_controller, tmp, NULL, TRUE, NULL);
	match = tp_blue_at_controller_send (self->priv->at_controller, NULL, "OK", TRUE, NULL) && match;
	if (!match)
	{
		g_message ("Unable to set memory with %s, ignoring.", tmp);
	}
	g_free (tmp);
	

	
	tmp = g_strdup_printf ("AT+CMGR=%s\r", index);
	/* +CMGR: 0,,33 */
	match = tp_blue_at_controller_send (self->priv->at_controller, tmp, NULL, TRUE, NULL);
	/* PDU SMS */
	match = tp_blue_at_controller_send (self->priv->at_controller, tmp, NULL, TRUE, &sms) && match;
	/* OK Code */
	match = tp_blue_at_controller_send (self->priv->at_controller, NULL, "OK", TRUE, NULL) && match;
	if (!match)
	{
		g_message ("Unable to retreive SMS from memory with %s.", tmp);
	}
	else
	{
		tp_blue_sms_controller_decode_sms (self, sms);
	}
	g_free (tmp);
#endif
}

static void
tp_blue_sms_controller_response_received (TpBlueAtController *controller, gchar *message, TpBlueSmsController *self)
{
	gchar *mem = NULL, *index = NULL;
	gint n;
	
	g_message ("*** SMS Response received, probably a new SMS: %s", message);
	if (g_str_has_prefix (message, "+CMTI: "))
	{
		n = sscanf(message, "+CMTI: \"%3s\",%4s", &mem, &index);
		if (n == 2)
			tp_blue_sms_controller_fetch_sms (self, mem, index);

		g_free (mem);
		g_free (index);
	}
}

static void
tp_blue_sms_controller_error_received (TpBlueAtController *controller, TpBlueSmsController *self)
{
	g_message ("Error Received in SMS");
}

static void
tp_blue_sms_controller_disconnected (TpBlueAtController *controller, TpBlueSmsController *self)
{
	g_signal_emit_by_name (self, "disconnected");
}

static void
tp_blue_sms_controller_init (TpBlueSmsController *self)
{
	TpBlueSmsControllerPrivate *priv;
	
	priv = self->priv = TP_BLUE_SMS_CONTROLLER_GET_PRIVATE (self);
	priv->at_controller = tp_blue_at_controller_new ();
	g_signal_connect (priv->at_controller, "response-received", G_CALLBACK (tp_blue_sms_controller_response_received), self);
	g_signal_connect (priv->at_controller, "error-received", G_CALLBACK (tp_blue_sms_controller_error_received), self);
	g_signal_connect (priv->at_controller, "disconnected", G_CALLBACK (tp_blue_sms_controller_disconnected), self);
	
	g_message ("Decoding: '%s'", tp_blue_sms_controller_decode_message(self, "C8329BFD065DDF72361904028140", 16));
}

gboolean
tp_blue_sms_controller_connect_bt (TpBlueSmsController *self)
{
	gboolean connected;
	
	connected = tp_blue_at_controller_connect (self->priv->at_controller, self->priv->bt_address, self->priv->bt_channel);
	if (!connected)
		return connected;
	
	connected = tp_blue_at_controller_send (self->priv->at_controller, "AT+CNMI=2,1\r", "OK", TRUE, NULL);
	if (!connected)
		return connected;
	
	connected = tp_blue_at_controller_send (self->priv->at_controller, "AT+CMGF=0\r", "OK", TRUE, NULL);
	return connected;
}

void
tp_blue_sms_controller_disconnect_bt (TpBlueSmsController *self)
{
	tp_blue_at_controller_disconnect (self->priv->at_controller);
}

gboolean
tp_blue_sms_controller_send_sms (TpBlueSmsController *self, gchar *number, gchar *message)
{
	gboolean match;
	gchar *pdu, *at_str;
	gsize pdu_length;
	
	if (number == NULL || number[0] != '+')
	{
		g_warning ("Number does not start with +");
		return FALSE;
	}
	
	pdu = tp_blue_sms_controller_encode_pdu (self, number, message);
	pdu_length = (strlen(pdu)-3)/2;
	at_str = g_strdup_printf ("AT+CMGS=%d\r", pdu_length);
	
	g_message("PDU_LGTH: '%s': '%s'", at_str, pdu);
	match = tp_blue_at_controller_send (self->priv->at_controller, at_str, ">", FALSE, NULL);
	g_free (at_str);
	
	/* FIXME: do something*/
	if (!match)
	{
		g_warning ("Unable to receive prompt for CMGS");
		return FALSE;
	}

	g_message ("PDU: '%s'", pdu);
	match = tp_blue_at_controller_send (self->priv->at_controller, pdu, "+CMGS: 0", TRUE, NULL);
	match = tp_blue_at_controller_send (self->priv->at_controller, NULL, "OK", TRUE, NULL) && match;
	if (match)
		g_message ("Message Sent");
	
	g_free (pdu);
	
	return match;
}
