/*

    File: partsun.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <ctype.h>      /* toupper, tolower */
#include "types.h"
#include "common.h"
#include "testdisk.h"
#include "fnctdsk.h"
//#include "analyse.h"
#include "lang.h"
#include "intrf.h"
#include "chgtype.h"
#include "sun.h"
#include "swap.h"
#include "bsd.h"
#include "fat.h"
#include "ntfs.h"
#include "ext2.h"
#include "rfs.h"
#include "lvm.h"
#include "md.h"
#include "jfs_superblock.h"
#include "jfs.h"
#include "savehdr.h"
#include "cramfs.h"
#include "xfs.h"
#include "ufs.h"

static int check_part_sun(t_param_disk *disk_car, const int debug,t_partition *partition,const int saveheader);
static int get_geometry_from_sunmbr(const unsigned char *buffer, const int debug, t_CHS *geometry);
static t_list_part *read_part_sun(t_param_disk *disk_car, const int debug, const int saveheader);
static int write_part_sun(t_param_disk *disk_car, const t_list_part *list_part, const int ro , const int debug, const int align);
static t_list_part *init_part_order_sun(const t_param_disk *disk_car, t_list_part *list_part);
static int write_MBR_code_sun(unsigned char *buffer);
static t_list_part *add_partition_sun(t_param_disk *disk_car,t_list_part *list_part, const int debug);
static void set_next_status_sun(const t_param_disk *disk_car, t_partition *partition);
static int test_structure_sun(t_list_part *list_part);
static int is_part_type_ok_sun(unsigned int part_type);
static void init_structure_sun(const t_param_disk *disk_car,t_list_part *list_part, const int debug);
static void erase_list_part_sun(unsigned char *buffer);

static const struct systypes sun_sys_types[] = {
  /* 0            */  {"\x00" "Empty"        },
  /* 1            */  {"\x01" "Boot"         },
  /* 2            */  {"\x02" "SunOS root"   },
  /* SUNOS_SWAP   */  {"\x03" "SunOS swap"   },
  /* 4            */  {"\x04" "SunOS usr"    },
  /* WHOLE_DISK   */  {"\x05" "Whole disk"   },
  /* 6            */  {"\x06" "SunOS stand"  },
  /* 7            */  {"\x07" "SunOS var"    },
  /* 8            */  {"\x08" "SunOS home"   },
  /* 9            */  {"\x09" "SunOS alt."   },
  /* 10           */  {"\x0A" "SunOS cachefs"},
  /* LINUX_SWAP   */  {"\x82" "Linux swap"   },
  /* LINUX_NATIVE */  {"\x83" "Linux native" },
  /* 0x8e         */  {"\x8e" "Linux LVM"    },
  /* New (2.2.x) raid partition with autodetect using persistent superblock */
  /* 0xfd         */  {"\xfd" "Linux raid autodetect" },
                      { NULL }
};

t_arch_fnct arch_sun=
{
  .part_name="Partition table type : Sun",
  .parttype_name_table=sun_sys_types,
  .read_part=read_part_sun,
  .write_part=write_part_sun,
  .init_part_order=init_part_order_sun,
  .get_geometry_from_mbr=get_geometry_from_sunmbr,
  .check_part=check_part_sun,
  .write_MBR_code=write_MBR_code_sun,
  .add_partition=add_partition_sun,
  .set_prev_status=set_next_status_sun,
  .set_next_status=set_next_status_sun,
  .test_structure=test_structure_sun,
  .is_part_type_ok=is_part_type_ok_sun,
  .init_structure=init_structure_sun,
  .erase_list_part=erase_list_part_sun
};

int get_geometry_from_sunmbr(const unsigned char *buffer, const int debug, t_CHS *geometry)
{
  const sun_partition *sunlabel=(const sun_partition*)buffer;
  if(debug>1)
  {
    ecrit_rapport("get_geometry_from_sunmbr\n");
  }
  geometry->cylinder=0;
  geometry->head=be16(sunlabel->ntrks)-1;
  geometry->sector=be16(sunlabel->nsect);
  if(geometry->sector>0)
  {
    ecrit_rapport("Geometry from SUN MBR: head=%u sector=%u\n",geometry->head+1,geometry->sector);
  }
  return 0;
}

t_list_part *read_part_sun(t_param_disk *disk_car, const int debug, const int saveheader)
{
  unsigned char buffer[DEFAULT_SECTOR_SIZE];
  sun_partition *sunlabel=(sun_partition*)&buffer;
  t_list_part *new_list_part=NULL;
  unsigned int i;
  aff_buffer(BUFFER_RESET,"Q");
  if(read_MBR(disk_car,&buffer)!=0)
    return NULL;
  if (be16(sunlabel->magic) != SUN_LABEL_MAGIC)
  {
    aff_buffer(BUFFER_ADD,"Bad SUN partition\n");
    return NULL;
  }
  for(i=0;i<8;i++)
  {
     if (sunlabel->partitions[i].num_sectors > 0
	 && sunlabel->infos[i].id > 0
	 && sunlabel->infos[i].id != WHOLE_DISK)
     {
       t_partition *new_partition=partition_new();
       partition_reset(new_partition);
       new_partition->order=i;
       new_partition->part_type=sunlabel->infos[i].id;
       new_partition->part_offset=be32(sunlabel->partitions[i].start_cylinder)*be16(sunlabel->ntrks)*be16(sunlabel->nsect)*disk_car->sector_size;
       new_partition->part_size=(uint64_t)be32(sunlabel->partitions[i].num_sectors)*disk_car->sector_size;
       new_partition->status=STATUS_PRIM;
       disk_car->arch->check_part(disk_car,debug,new_partition,saveheader);
       new_list_part=insert_new_partition(new_list_part,new_partition);
       aff_part_buffer(AFF_PART_ORDER,disk_car,new_partition);
     }
  }
  return new_list_part;
}

static int write_part_sun(t_param_disk *disk_car, const t_list_part *list_part, const int ro, const int debug, const int align)
{ /* TODO: Implement it */
  if(ro==0)
  {
    not_implemented("write_part_sun");
  }
  return 0;
}

static t_list_part *init_part_order_sun(const t_param_disk *disk_car, t_list_part *list_part)
{
  int nbr_prim=0;
  t_partition *new_partition;
  t_list_part *element;
  for(element=list_part;element!=NULL;element=element->next)
  {
    switch(element->part->status)
    {
      case STATUS_PRIM:
      case STATUS_PRIM_BOOT:
	if(nbr_prim==2)
	  nbr_prim++;
	element->part->order=nbr_prim++;
	break;
      default:
	ecrit_rapport("init_part_order_sun: severe error\n");
	break;
    }
  }
  new_partition=partition_new();
  new_partition->part_offset=0;
  new_partition->part_size=disk_car->disk_size;
  new_partition->status=STATUS_PRIM;
  new_partition->part_type=WHOLE_DISK;
  new_partition->order=2;
  return insert_new_partition(list_part, new_partition);
}

static int write_MBR_code_sun(unsigned char *buffer)
{ /* TODO: Implement it */
  not_implemented("write_MBR_code_sun");
  return 0;
}

static t_list_part *add_partition_sun(t_param_disk *disk_car,t_list_part *list_part, const int debug)
{
  t_CHS start,end;
  t_partition *new_partition=partition_new();
  int position=0;
  start.cylinder=0;
  start.head=0;
  start.sector=1;
  end.cylinder=disk_car->CHS.cylinder;
  end.head=disk_car->CHS.head;
  end.sector=disk_car->CHS.sector;
  {
    int done = FALSE;
    while (done==FALSE) {
      int command;
      static struct MenuItem menuGeometry[]=
      {
	{ 'c', "Cylinders", 	"Change starting cylinder" },
	{ 'C', "Cylinders", 	"Change ending cylinder" },
	{ 'T' ,"Type",		"Change partition type"},
	{ 'd', "Done", "" },
	{ 0, NULL, NULL }
      };
      aff_copy(stdscr);
      wmove(stdscr,4,0);
      wdoprintf(stdscr,"%s",disk_car->description(disk_car));
      new_partition->part_offset=CHS2offset(disk_car,&start);
      new_partition->part_size=CHS2offset(disk_car,&end) - new_partition->part_offset + disk_car->sector_size;
      wmove(stdscr,10, 0);
      wclrtoeol(stdscr);
      aff_part(stdscr,AFF_PART_SHORT,disk_car,new_partition);
      wmove(stdscr,COMMAND_LINE_Y, COMMAND_LINE_X);
      wclrtoeol(stdscr);
      wrefresh(stdscr);
      command=wmenuSimple(stdscr,menuGeometry, position);
      switch (command) {
	case 'c':
	  start.cylinder=ask_number(start.cylinder,0,disk_car->CHS.cylinder,"Enter the starting cylinder ");
	  position=1;
	  break;
	case 'C':
	  end.cylinder=ask_number(end.cylinder,start.cylinder,disk_car->CHS.cylinder,"Enter the ending cylinder ");
	  position=2;
	  break;
	case 'T':
	case 't':
	  change_part_type(disk_car,new_partition);
	  position=3;
	  break;
	case key_ESC:
	case 'd':
	case 'D':
	case 'q':
	case 'Q':
	  done = TRUE;
	  break;
      }
    }
  }
  if((CHS2offset(disk_car,&end)>new_partition->part_offset)&& new_partition->part_type>0)
  {
    t_list_part *new_list_part=insert_new_partition(list_part, new_partition);
    new_partition->status=STATUS_PRIM;
    if(test_structure_sun(list_part)!=0)
    {
      new_partition->status=STATUS_DELETED;
    }
    return new_list_part;
  }
  FREE(new_partition);
  return list_part;
}

static void set_next_status_sun(const t_param_disk *disk_car, t_partition *partition)
{
  if(partition->status==STATUS_DELETED)
    partition->status=STATUS_PRIM;
  else
    partition->status=STATUS_DELETED;
}

static int test_structure_sun(t_list_part *list_part)
{ /* Return 1 if bad*/
  t_list_part *new_list_part=NULL;
  t_list_part *element;
  t_list_part *new_element;
  int res=0;
  /* Sort list_part in new_list_part */
  for(element=list_part;element!=NULL;element=element->next)
  {
    if(element->part->status!=STATUS_DELETED)
      new_list_part=insert_new_partition(new_list_part,element->part);
  }
  /* Test overlapping */
  for(element=new_list_part;element!=NULL;element=element->next)
  {
    if(((element->prev!=NULL) && (element->part->part_offset<=element->prev->part->part_offset+element->prev->part->part_size-1)) ||
	((element->next!=NULL) && (element->part->part_offset+element->part->part_size-1>=element->next->part->part_offset)))
    {
      res=1;
    }
  }
  for(element=new_list_part;element!=NULL;element=new_element)
  {
    new_element=element->next;
    FREE(element);
  }
  return res;
}

static int is_part_type_ok_sun(unsigned int part_type)
{
  return (part_type>0 && part_type <= 255);
}

static void init_structure_sun(const t_param_disk *disk_car,t_list_part *list_part, const int debug)
{
  t_list_part *element;
  t_list_part *new_list_part=NULL;
  /* Create new list */
  for(element=list_part;element!=NULL;element=element->next)
    element->to_be_removed=0;
  for(element=list_part;element!=NULL;element=element->next)
  {
    int to_be_removed=0;
    t_list_part *element2;
    for(element2=element->next;element2!=NULL;element2=element2->next)
      if(element->part->part_offset+element->part->part_size-1 >= element2->part->part_offset)
      {
	to_be_removed=1;
	element2->to_be_removed=1;
      }
    if(to_be_removed)
      element->to_be_removed=1;
    if(element->to_be_removed==0)
      new_list_part=insert_new_partition_aux(new_list_part,element_new(element->part),1);
  }
#ifdef DEBUG
  check_list_part(new_list_part);
#endif
    for(element=new_list_part;element!=NULL;element=element->next)
      element->part->status=STATUS_PRIM;
  if(disk_car->arch->test_structure(new_list_part))
  {
    for(element=new_list_part;element!=NULL;element=element->next)
      element->part->status=STATUS_DELETED;
  }
  {
    /* free */
    t_list_part *new_element;
    for(element=new_list_part;element!=NULL;element=new_element)
    {
      new_element=element->next;
      FREE(element);
    }
  }
#ifdef DEBUG
  check_list_part(list_part);
#endif
}

static void erase_list_part_sun(unsigned char *buffer)
{ /* TODO: Implement it */
  not_implemented("erase_list_part_sun");
}

static int check_part_sun(t_param_disk *disk_car,const int debug,t_partition *partition, const int saveheader)
{
  int ret=0;
  switch(partition->part_type)
  {
    case PSUN_BOOT:
    case PSUN_ROOT:
    case PSUN_USR:
    case PSUN_STAND:
    case PSUN_VAR:
    case PSUN_HOME:
    case PSUN_ALT:
      ret=check_ufs(disk_car,partition,debug);
      break;
    case PSUN_LINUX:
      ret=check_JFS(disk_car,partition,debug);
      if(ret!=0)
      {
	ret=check_rfs(disk_car,partition,debug);
      }
      if(ret!=0)
      {
	ret=check_EXT2(disk_car,partition,debug);
      }
      if(ret!=0)
      {
	ret=check_cramfs(disk_car,partition,debug);
      }
      if(ret!=0)
      {
	ret=check_xfs(disk_car,partition,debug);
      }
      if(ret!=0)
      { aff_buffer(BUFFER_ADD,"No EXT2, JFS, Reiser, cramfs or XFS marker\n"); }
      break;

    case PSUN_LINSWAP:
      ret=check_Linux_SWAP(disk_car,partition,debug);
      break;
    default:
      if(debug>0)
      {
	ecrit_rapport("check_part_sun %u type %02X: no test\n",partition->order,partition->part_type);
      }
      break;
  }
  if(ret!=0)
  {
    ecrit_rapport("check_part_sun failed for partition type %02X\n", partition->part_type);
    aff_part_buffer(AFF_PART_ORDER,disk_car,partition);
    if(saveheader>0)
    {
      save_header(disk_car,partition,debug);
    }
  }
  return ret;
}

