/*

    File: testdisk.c

    Copyright (C) 1998-2006 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 59
    Temple Place - Suite 330, Boston MA 02111-1307, USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include <stdarg.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>	/* dup2 */
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <ctype.h>      /* tolower */
#ifdef HAVE_LOCALE_H
#include <locale.h>	/* setlocale */
#endif
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif
#include <errno.h>
#include "types.h"
#include "common.h"
#include "testdisk.h"
#include "lang.h"
#include "intrf.h"
#include "intrface.h"
#include "godmode.h"
#include "fnctdsk.h"
#include "ext2_dir.h"
#include "rfs_dir.h"
#include "ntfs_dir.h"
#include "hdcache.h"
extern const t_arch_fnct arch_i386;
extern const t_arch_fnct arch_mac;
extern const t_arch_fnct arch_sun;

static FILE* init_log(const char*filename,int argc, char**argv);

static FILE *f_rapport=NULL;
static int f_status=0;

void ecrit_rapport_string(const char *string,const int max_length)
{
  int i;
  for(i=0;(string[i]!='\0')&&(i<max_length);i++)
    ecrit_rapport("%c",string[i]);
}

int ecrit_rapport(const char *_format, ...)
{
  int res=0;
  if(f_rapport!=NULL)
  {
    va_list ap;
    va_start(ap,_format);
    res=vfprintf(f_rapport,_format,ap);
    va_end(ap);
    if(res<0)
    {
      f_status=1;
    }
    fflush(stderr);
    if(fflush(f_rapport))
    {
      f_status=1;
    }
  }
  return res;
}

static FILE* init_log(const char*filename,int argc, char**argv)
{
  FILE*f_file=fopen(filename,"a");
  if(f_file==NULL)
  {
    printf("\nWarning: Can not create testdisk.log: %s\n",strerror(errno));
    printf("Press the enter key to continue.\n");
    getchar();
  }
  else
  {
	int i;
	time_t my_time;
	my_time=time(NULL);
	fprintf(f_file,"\n\n%s",ctime(&my_time));
	fprintf(f_file,"TestDisk command line :");
	for(i=1;i<argc;i++)
	  fprintf(f_file," %s", argv[i]);
	fprintf(f_file,"\n");
	fflush(f_file);
#ifdef HAVE_DUP2
	dup2(fileno(f_file),2);
#endif
  }
  return f_file;
}

#ifdef HAVE_SIGACTION
void sighup_hdlr(int shup)
{
  ecrit_rapport("SIGHUP detected! TestDisk has been killed.\n");
  if(f_rapport!=NULL)
  {
    fclose(f_rapport);
  }
  exit(1);
}
#endif

void aff_copy(WINDOW *window)
{
  wclear(window);
  keypad(window, TRUE); /* Need it to get arrow key */
  wmove(window,0,0);
  wdoprintf(window, "TestDisk %s, Data Recovery Utility, %s",VERSION,TESTDISKDATE);
  wmove(window,1,0);
  wdoprintf(window,"Christophe GRENIER <grenier@cgsecurity.org>");
  wmove(window,2,0);
  wdoprintf(window,"http://www.cgsecurity.org");
}

int main( int argc, char **argv )
{
  int i;
  int help=0, create_log=0, debug=0, dump_ind=0;
  int do_list=0;
  int write_used;
  int saveheader=0;
  int create_backup=0;
  int testdisk_mode=TESTDISK_O_RDWR|TESTDISK_O_READAHEAD_8K;
  t_list_disk *list_disk=NULL;
  t_list_disk *element_disk;
  const char *cmd_device=NULL;
  char *cmd_run=NULL;
#ifdef TARGET_SOLARIS
  const t_arch_fnct *arch=&arch_sun;
#elif defined __APPLE__
  const t_arch_fnct *arch=&arch_mac;
#else
  const t_arch_fnct *arch=&arch_i386;
#endif
#ifdef HAVE_SIGACTION
  struct sigaction action;
#endif
#ifdef TESTING
  srand(1);
#endif
#ifdef HAVE_SIGACTION
  /* set up the signal handler for SIGHUP */
  action.sa_handler  = sighup_hdlr;
  action.sa_flags = 0;
  if(sigaction(SIGHUP, &action, NULL)==-1)
  {
    printf("Error on SIGACTION call\n");
    return -1;
  }
#endif
#if defined(__CYGWIN__) || defined(__MINGW32__) || defined(__APPLE__)
  /* Average Windows user doesn't know how to specify a parameter */
  if(argc==1)
  {
    create_log=1;
    debug=1;
  }
#endif
  for(i=1;i<argc;i++)
  {
    if((strcmp(argv[i],"/dump")==0) || (strcmp(argv[i],"-dump")==0))
      dump_ind=1;
    else if((strcmp(argv[i],"/log")==0) ||(strcmp(argv[i],"-log")==0))
    {
      create_log=1;
    }
    else if((strcmp(argv[i],"/debug")==0) || (strcmp(argv[i],"-debug")==0))
    {
      debug++;
      create_log=1;
    }
    else if((strcmp(argv[i],"/list")==0) || (strcmp(argv[i],"-list")==0))
      do_list=1;
    else if((strcmp(argv[i],"/saveheader")==0) || (strcmp(argv[i],"-saveheader")==0))
      saveheader=1;
    else if((strcmp(argv[i],"/backup")==0) || (strcmp(argv[i],"-backup")==0))
      create_backup=1;
    else if((strcmp(argv[i],"/direct")==0) || (strcmp(argv[i],"-direct")==0))
      testdisk_mode|=TESTDISK_O_DIRECT;
    else if((strcmp(argv[i],"/help")==0) || (strcmp(argv[i],"-help")==0) || (strcmp(argv[i],"--help")==0) ||
      (strcmp(argv[i],"/h")==0) || (strcmp(argv[i],"-h")==0))
      help=1;
    else if(strcmp(argv[i],"/cmd")==0)
    {
      if(i+2>=argc)
	help=1;
      else
      {
	t_param_disk *disk_car;
	cmd_device=argv[++i];
	cmd_run=argv[++i];
	disk_car=file_test_availability(cmd_device,debug,arch,testdisk_mode);
	if(disk_car==NULL)
	{
	  printf("\nUnable to open file or device %s\n",cmd_device);
	  help=1;
	}
	else
	  list_disk=insert_new_disk(list_disk,disk_car);
      }
    }
    else
    {
      t_param_disk *disk_car=file_test_availability(argv[i],debug,arch,testdisk_mode);
      if(disk_car==NULL)
      {
	printf("\nUnable to open file or device %s\n",argv[i]);
	help=1;
      }
      else
	list_disk=insert_new_disk(list_disk,disk_car);
    }
  }
  if(cmd_device!=NULL)
  {
    create_log=1;
    if(debug<1)
      debug=1;
  }
  printf("TestDisk %s, Data Recovery Utility, %s\nChristophe GRENIER <grenier@cgsecurity.org>\nhttp://www.cgsecurity.org\n",VERSION,TESTDISKDATE);
  if(help!=0)
  {
    printf("\n" \
	"Usage: testdisk [/log] [/debug] [file or device]\n"\
	"       testdisk /list  [/log]   [file or device]\n" \
	"\n" \
	"/log          : create a testdisk.log file\n" \
	"/debug        : add debug information\n" \
	"/list         : display current partitions\n" \
	"\n" \
	"TestDisk checks and recovers lost partitions\n" \
	"It works with :\n" \
	"- BeFS (BeOS)                           - BSD disklabel (Free/Open/Net BSD)\n" \
	"- CramFS, Compressed File System        - DOS/Windows FAT12, FAT16 and FAT32\n" \
	"- HFS, HFS+, Hierarchical File System   - JFS, IBM's Journaled File System\n" \
	"- Linux Ext2 and Ext3                   - Linux Raid\n" \
	"- Linux Swap                            - LVM, LVM2, Logical Volume Manager\n" \
	"- Netware NSS                           - NTFS (Windows NT/2K/XP/2003)\n" \
	"- ReiserFS 3.5 and 3.6                  - Sun Solaris i386 disklabel\n" \
	"- UFS and UFS2 (Sun/BSD/...)            - XFS, SGI's Journaled File System\n" \
	"\n" \
	"If you have problems with TestDisk or bug reports, please contacte me.\n");
    return 0;
  }
  if(create_log==1 && f_rapport==NULL)
  {
    f_rapport=init_log("testdisk.log",argc,argv);
    ecrit_rapport("TestDisk %s, Data Recovery Utility, %s\nChristophe GRENIER <grenier@cgsecurity.org>\nhttp://www.cgsecurity.org\n",VERSION,TESTDISKDATE);
#ifdef DJGPP
    ecrit_rapport("Dos version");
#elif defined(TARGET_BSD)
    ecrit_rapport("BSD version");
#elif defined(TARGET_LINUX)
    ecrit_rapport("Linux version");
#elif defined(TARGET_SOLARIS)
    ecrit_rapport("Solaris version");
#elif defined(__CYGWIN__) || defined(__MINGW32__)
    ecrit_rapport("Windows version");
#elif defined(__APPLE__)
    ecrit_rapport("Apple version");
#else
    ecrit_rapport("Undefined OS");
#endif
    ecrit_rapport(" (ext2fs lib: %s, ntfs lib: %s, reiserfs lib: %s)\n",td_ext2fs_version(),td_ntfs_version(),td_reiserfs_version());
  }
  printf("Please wait...\n");
#ifdef HAVE_SETLOCALE
  {
    const char *locale;
    locale = setlocale (LC_ALL, "");
    if (locale==NULL) {
      locale = setlocale (LC_ALL, NULL);
      ecrit_rapport("Failed to set locale, using default '%s'.\n", locale);
    } else {
      ecrit_rapport("Using locale '%s'.\n", locale);
    }
  }
#endif
  aff_buffer(BUFFER_RESET,"Q");
  /* Scan for avaible device only if no device or image has been supplied in parameter */
  if(list_disk==NULL)
    list_disk=hd_parse(list_disk,debug,arch,testdisk_mode);
  /* Activate the cache */
  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
    element_disk->disk=new_diskcache(element_disk->disk,testdisk_mode);
#ifdef DJGPP
  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
  {
    printf("%s\n",element_disk->disk->description(element_disk->disk));
  }
#endif
  hd_update_all_geometry(list_disk,0,debug);
  /* save disk parameters to rapport */
  ecrit_rapport("Hard disk list\n");
  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
  {
    printf("%s, sector size=%u\n",element_disk->disk->description(element_disk->disk),element_disk->disk->sector_size);
    ecrit_rapport("%s, sector size=%u\n",element_disk->disk->description(element_disk->disk),element_disk->disk->sector_size);
  }
  printf("\n");
  ecrit_rapport("\n");
  if(do_list!=0)
  {
    for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
    {
      interface_list(element_disk->disk,debug,0,saveheader,create_backup);
    }
  }
  else
  {
    do_curses_testdisk(debug,dump_ind,list_disk,saveheader,argv[0],cmd_device,cmd_run);
    cmd_device=NULL;
    cmd_run=NULL;
  }
  write_used=delete_list_disk(list_disk);
  if(f_rapport!=NULL)
  {
    ecrit_rapport("TestDisk exited normally.\n");
    if(fclose(f_rapport))
    {
      f_status=1;
    }
  }
  if(f_status!=0)
  {
    printf("TestDisk: Log file corrupted!\n");
  }
  else
  {
    printf("TestDisk exited normally.\n");
  }
  if(write_used!=0)
  {
    printf("You have to reboot for the change to take effect.\n");
  }
  return 0;
}
