/*

    File: md.c

    Copyright (C) 1998-2007 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 51
    Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "md.h"
#include "fnctdsk.h"
#include "log.h"

int check_MD(disk_t *disk_car,partition_t *partition,const int debug)
{
  unsigned char *buffer=(unsigned char*)MALLOC(MD_SB_BYTES);
  uint64_t offset=MD_NEW_SIZE_SECTORS(partition->part_size/512)*512;
  if(debug>1)
  {
    log_debug("Raid offset %llu\n",(long long unsigned)offset/512);
  }
  if(disk_car->read(disk_car,MD_SB_BYTES, buffer, partition->part_offset+offset)!=0)
  {
    free(buffer);
    return 1;
  }
  if(test_MD(disk_car,(struct mdp_superblock_s*)buffer,partition,debug,0)!=0)
  {
    free(buffer);
    return 1;
  }
  set_MD_info(disk_car,(struct mdp_superblock_s*)buffer,partition,debug,0);
  free(buffer);
  return 0;
}

int set_MD_info(disk_t *disk_car, const struct mdp_superblock_s *sb,partition_t *partition,const int debug, const int dump_ind)
{
  unsigned int i;
  sprintf(partition->name,"md%u",(unsigned int)sb->md_minor);
  sprintf(partition->info,"Raid %u: devices",(unsigned int)sb->level);
  for(i=0;i<MD_SB_DISKS;i++)
  {
    if(sb->disks[i].major!=0 && sb->disks[i].minor!=0)
    {
      if(strlen(partition->info)<sizeof(partition->info)-26)
      { 
	sprintf(&partition->info[strlen(partition->info)]," %u(%u,%u)",
	    (unsigned int)sb->disks[i].number,
	    (unsigned int)sb->disks[i].major,(unsigned int)sb->disks[i].minor);
	if(sb->disks[i].major==sb->this_disk.major && sb->disks[i].minor==sb->this_disk.minor)
	  sprintf(&partition->info[strlen(partition->info)],"*");
      }
    }
  }
  if(debug>0)
    log_info("%s %s\n", partition->name, partition->info);
  return 0;
}

int recover_MD(disk_t *disk_car, const struct mdp_superblock_s *sb,partition_t *partition,const int debug, const int dump_ind)
{
  uint64_t offset;
  if(test_MD(disk_car,sb,partition,debug,dump_ind)!=0)
    return 1;
  set_MD_info(disk_car,sb,partition,debug,dump_ind);
  partition->part_type_i386=P_RAID;
  partition->part_type_sun=PSUN_RAID;
  partition->part_size=(uint64_t)(sb->size<<1)*disk_car->sector_size+MD_RESERVED_BYTES;
  offset=MD_NEW_SIZE_SECTORS(partition->part_size/512)*512;
  partition->part_offset=partition->part_offset-offset;
  return 0;
}

int test_MD(disk_t *disk_car, const struct mdp_superblock_s *sb,partition_t *partition,const int debug, const int dump_ind)
{
  if(sb->md_magic==(unsigned int)MD_SB_MAGIC)
  {
    log_info("\nRaid magic value at %u/%u/%u\n", offset2cylinder(disk_car,partition->part_offset),offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset));
    log_info("Raid apparent size: %llu sectors\n",(long long unsigned)(sb->size<<1));
    log_info("Raid chunk size: %llu bytes\n",(long long unsigned)sb->chunk_size);
    if(sb->chunk_size==0)
      return 1;
    if(dump_ind!=0)
    {
      /* There is a little offset ... */
      dump_log(sb,DEFAULT_SECTOR_SIZE);
    }
    partition->upart_type=UP_RAID;
    return 0;
  }
  return 1;
}
