/* -*- mode: c; c-file-style: "gnu" -*-
 * config.c -- configuration handling functions
 * Copyright (C) 2002, 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file config.c
 * Configuration handling functions.
 *
 * Really, this is just argp glue code and initialisation.
 */

#include "system.h"

#if defined (HAVE_ARGP_H) && defined(HAVE_ARGP_PARSE)
#include <argp.h>
#endif
#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if HAVE_STRINGS_H
#include <strings.h>
#endif
#include <sys/socket.h>
#include <unistd.h>

#include "compat/compat.h"
#include "compat/compat-regex.h"
#include "bh-libs/list.h"

#include "config.h"
#include "network.h"
#include "thy.h"
#include "types.h"

/** @internal Options set in the configuration files.
 * This will be used by thy_config_file_parse() to store temporary
 * data in.
 */
bhl_list_t *conffile_list;

/** @internal Structure to keep the state of our configuration map.
 * This structure is only used by the argp suite, to keep track of
 * which map they are working on.
 */
typedef struct
{
  config_t *global; /**< Global configuration. */
  char *curmap; /**< ID of the current map. */
} _thy_int_config_t;

/** The default port Thy should listen on.
 */
int thy_default_port = 0;

/** @internal Default listener mode.
 * Zero if HTTP, non-zero if HTTPS.
 */
thy_listener_ssl_t thy_default_tls = LISTENER_SSL_AUTO;

/** @var thy_default_config
 * @brief Default configuration for mapped stuff.
 */
static thy_mappable_config_t thy_default_config;

/** @var thy_config
 * @brief The server configuration.
 */
static config_t thy_config;

/** @var thy_read_default_config
 * @brief Flag indicating wether Thy should read a default config
 * file (if one exists).
 */
static int thy_read_default_config = 1;

/** @internal Address to send bugreports to.
 * Used by the argp suite.
 */
const char *argp_program_bug_address = "<algernon@bonehunter.rulez.org>";
/** @internal Supported command-line options.
 * Used by the argp suite.
 */
static struct argp_option options[] = {
  {"webroot", 'w', "PATH", 0, "Root of the web-accessible documents", 1},
  {"listen", 'l', "ADDR", 0, "Addresses to listen on (proto:ip/port)", 1},
  {"default-type", 'd', "TYPE", 0, "Default Content-Type to send", 1},
  {"timeout", 't', "SECONDS", 0,
   "Seconds to wait for a request to complete", 1},
  {"options", 'o', "OPTIONS...", 0, "Toggle some options.\n"
   "(See the manual for details)", 1},
  {"userdir", 'u', "PATH", 0,
   "Directory inside the users home to serve docs from", 1},
  {"indexes", 'i', "FILES", 0,
   "Name of the directory-index files", 1},
  {"uid", 'U', "UID", 0, "UID of the user thy should run as", 1},
  {"cgiexts", 'c', "EXTS", 0, "List of allowed extensions for CGIs", 1},
  {"pidfile", 'P', "PATH", 0, "File to log pid into", 1},
  {"errordoc", 'e', "CODE=PATH", 0, "Error documents to show\n"
   "(See the manual for details)", 1},
  {"add-env", 'E', "VAR=VALUE", 0,
   "Register a new environment variable", 1},
  {"add-handler", 'H', "EXT=PATH", 0, "Register a new handler", 1},
  {"add-method", 'M', "METH=PATH", 0, "Register a new method handler", 1},
  {"add-header", 'D', "HDR=VALUE", 0,
   "Register a new response header field", 1},
  {"mime-type", 'm', "EXT=TYPE", 0, "Register a new MIME type", 1},
  {"ssl", 's', "OPTION=VALUE", 0,
   "Set SSL options\n"
   "(See the manual for details)", 1},
  {"ipv4", '4', NULL, 0, "Bind only to IPv4 addresses", 1},
  {"ipv6", '6', NULL, 0, "Bind only to IPv6 addresses", 1},
  {"gzip", 'g', "OPTION=VALUE", 0,
   "Set compression options (type and level)", 1},
  {"limit", 'L', "WHAT=SIZE", 0, "Limit certain properties\n"
   "(See the manual for details)", 1},
  {"cgidirs", 'C', "DIRS...", 0, "Special CGI directories", 1},
  {"auth", 'a', "OPTIONS...", 0, "Authenticator options\n"
   "(Available options are arg, file, path and uid)", 1},
  {"etag", 'T', "OPTIONS...", 0, "ETag options\n"
   "(Available options are dirtag and etag)", 1},
  {"map", 'R', "REGEX", 0, "Localise subsequent options", 1},
  {"port", 'p', "PORT", OPTION_HIDDEN, "Default port to bind to", 1},
  {"worker", 'W', "OPTIONS...", 0, "Worker options\n"
   "(Available options are arg, path and uid)", 1},
  {"lru", 'r', "CACHE=SIZE", OPTION_HIDDEN, "LRU cache size control", 1},
  {"unmap", 'F', NULL, OPTION_HIDDEN, "Globalise subsequent options", 1},
  {"alias", 'A', "SOURCE=DEST", 0, "Add a new alias", 1},
  {"cache-control", 'X', "FIELD=VALUE", 0, "Change cache control options", 1},
  {0, 0, 0, 0, NULL, 0}
};
static error_t _config_parse_opt (int key, char *arg,
				  struct argp_state *state);

#ifndef __DOXYGEN__
#ifdef WITH_EASTER_EGG
#define EGG "\v" \
   ".       .\n" \
   "\\`-\"'\"-'/\n" \
   " } 6 6 {\n" \
   "=.  Y  ,=\n" \
   "  /^^^\\  .         Meow.\n" \
   " /     \\  )\n" \
   "(  )-(  )/ \n" \
   " \"\"   \"\"\n"
#else
#define EGG
#endif
#endif

/** @internal Function to print Thy version number and exit.
 */
static void
_thy_print_version (FILE *stream, struct argp_state *state)
{
  fprintf (stream, "%s\n", thy_servername (NULL));
  exit (0);
}
/** @internal Function to call upon --version.
 */
void (*argp_program_version_hook)
     (FILE *stream, struct argp_state * state) = _thy_print_version;

/** @internal Final definition of the argp parser.
 */
static struct argp argp =
  {options, _config_parse_opt, "[CONFIG FILES...]",
   __THY__ " -- Use thy power." EGG, NULL, NULL, NULL};

#ifndef __DOXYGEN__
enum
{
  THY_O_SUBOPT_AUTH = 0,
  THY_O_SUBOPT_NOAUTH,
  THY_O_SUBOPT_VHOST,
  THY_O_SUBOPT_NOVHOST,
  THY_O_SUBOPT_USERDIR,
  THY_O_SUBOPT_NOUSERDIR,
  THY_O_SUBOPT_CGI,
  THY_O_SUBOPT_NOCGI,
  THY_O_SUBOPT_USERCGI,
  THY_O_SUBOPT_NOUSERCGI,
  THY_O_SUBOPT_BUFFER,
  THY_O_SUBOPT_NOBUFFER,
  THY_O_SUBOPT_CASEMIME,
  THY_O_SUBOPT_NOCASEMIME,
  THY_O_SUBOPT_FOLLOWALL,
  THY_O_SUBOPT_NOFOLLOWALL,
  THY_O_SUBOPT_SERVER,
  THY_O_SUBOPT_NOSERVER,
  THY_O_SUBOPT_SSL,
  THY_O_SUBOPT_NOSSL,
  THY_O_SUBOPT_PIDFILE,
  THY_O_SUBOPT_NOPIDFILE,
  THY_O_SUBOPT_VARY,
  THY_O_SUBOPT_NOVARY,
  THY_O_SUBOPT_CHROOT,
  THY_O_SUBOPT_NOCHROOT,
  THY_O_SUBOPT_DIRINDEX,
  THY_O_SUBOPT_NODIRINDEX,
  THY_O_SUBOPT_WORKER,
  THY_O_SUBOPT_NOWORKER,
  THY_O_SUBOPT_EXPECT,
  THY_O_SUBOPT_NOEXPECT,
  THY_O_SUBOPT_HARDLIMIT,
  THY_O_SUBOPT_NOHARDLIMIT,
  THY_O_SUBOPT_STATS,
  THY_O_SUBOPT_NOSTATS,
  THY_O_SUBOPT_LAZYCGI,
  THY_O_SUBOPT_NOLAZYCGI,
  THY_O_SUBOPT_CACHE,
  THY_O_SUBOPT_NOCACHE,
  THY_O_SUBOPT_DAEMON,
  THY_O_SUBOPT_NODAEMON,
  THY_O_SUBOPT_EVENTSYSTEM,
  THY_O_SUBOPT_END
};

static char *o_subopts[] = {
  C99_INIT([THY_O_SUBOPT_AUTH]) "auth",
  C99_INIT([THY_O_SUBOPT_NOAUTH]) "noauth",
  C99_INIT([THY_O_SUBOPT_VHOST]) "vhost",
  C99_INIT([THY_O_SUBOPT_NOVHOST]) "novhost",
  C99_INIT([THY_O_SUBOPT_USERDIR]) "userdir",
  C99_INIT([THY_O_SUBOPT_NOUSERDIR]) "nouserdir",
  C99_INIT([THY_O_SUBOPT_CGI]) "cgi",
  C99_INIT([THY_O_SUBOPT_NOCGI]) "nocgi",
  C99_INIT([THY_O_SUBOPT_USERCGI]) "usercgi",
  C99_INIT([THY_O_SUBOPT_NOUSERCGI]) "nousercgi",
  C99_INIT([THY_O_SUBOPT_BUFFER]) "buffer",
  C99_INIT([THY_O_SUBOPT_NOBUFFER]) "nobuffer",
  C99_INIT([THY_O_SUBOPT_CASEMIME]) "casemime",
  C99_INIT([THY_O_SUBOPT_NOCASEMIME]) "nocasemime",
  C99_INIT([THY_O_SUBOPT_FOLLOWALL]) "followall",
  C99_INIT([THY_O_SUBOPT_NOFOLLOWALL]) "nofollowall",
  C99_INIT([THY_O_SUBOPT_SERVER]) "server",
  C99_INIT([THY_O_SUBOPT_NOSERVER]) "noserver",
  C99_INIT([THY_O_SUBOPT_SSL]) "ssl",
  C99_INIT([THY_O_SUBOPT_NOSSL]) "nossl",
  C99_INIT([THY_O_SUBOPT_PIDFILE]) "pidfile",
  C99_INIT([THY_O_SUBOPT_NOPIDFILE]) "nopidfile",
  C99_INIT([THY_O_SUBOPT_VARY]) "vary",
  C99_INIT([THY_O_SUBOPT_NOVARY]) "novary",
  C99_INIT([THY_O_SUBOPT_CHROOT]) "chroot",
  C99_INIT([THY_O_SUBOPT_NOCHROOT]) "nochroot",
  C99_INIT([THY_O_SUBOPT_DIRINDEX]) "dirindex",
  C99_INIT([THY_O_SUBOPT_NODIRINDEX]) "nodirindex",
  C99_INIT([THY_O_SUBOPT_WORKER]) "worker",
  C99_INIT([THY_O_SUBOPT_NOWORKER]) "noworker",
  C99_INIT([THY_O_SUBOPT_EXPECT]) "expect",
  C99_INIT([THY_O_SUBOPT_NOEXPECT]) "noexpect",
  C99_INIT([THY_O_SUBOPT_HARDLIMIT]) "hardlimit",
  C99_INIT([THY_O_SUBOPT_NOHARDLIMIT]) "nohardlimit",
  C99_INIT([THY_O_SUBOPT_STATS]) "stats",
  C99_INIT([THY_O_SUBOPT_NOSTATS]) "nostats",
  C99_INIT([THY_O_SUBOPT_LAZYCGI]) "lazycgi",
  C99_INIT([THY_O_SUBOPT_NOLAZYCGI]) "nolazycgi",
  C99_INIT([THY_O_SUBOPT_CACHE]) "cache",
  C99_INIT([THY_O_SUBOPT_NOCACHE]) "nocache",
  C99_INIT([THY_O_SUBOPT_DAEMON]) "daemon",
  C99_INIT([THY_O_SUBOPT_NODAEMON]) "nodaemon",
  C99_INIT([THY_O_SUBOPT_EVENTSYSTEM]) "eventsystem",
  C99_INIT([THY_O_SUBOPT_END]) NULL
};

static char *e_subopts[] = {
  C99_INIT([HTTP_STATUS_100]) "",
  C99_INIT([HTTP_STATUS_101]) "",
  C99_INIT([HTTP_STATUS_200]) "",
  C99_INIT([HTTP_STATUS_206]) "",
  C99_INIT([HTTP_STATUS_301]) "301",
  C99_INIT([HTTP_STATUS_302]) "302",
  C99_INIT([HTTP_STATUS_304]) "",
  C99_INIT([HTTP_STATUS_400]) "400",
  C99_INIT([HTTP_STATUS_401]) "401",
  C99_INIT([HTTP_STATUS_403]) "403",
  C99_INIT([HTTP_STATUS_404]) "404",
  C99_INIT([HTTP_STATUS_405]) "405",
  C99_INIT([HTTP_STATUS_408]) "408",
  C99_INIT([HTTP_STATUS_411]) "411",
  C99_INIT([HTTP_STATUS_412]) "412",
  C99_INIT([HTTP_STATUS_413]) "413",
  C99_INIT([HTTP_STATUS_416]) "416",
  C99_INIT([HTTP_STATUS_417]) "417",
  C99_INIT([HTTP_STATUS_500]) "500",
  C99_INIT([HTTP_STATUS_501]) "501",
  C99_INIT([HTTP_STATUS_503]) "503",
  C99_INIT([HTTP_STATUS_505]) "505",
  C99_INIT([HTTP_STATUS_UNKNOWN]) "all",
  C99_INIT([HTTP_STATUS_UNKNOWN+1]) NULL
};

enum
{
  THY_S_SUBOPT_CA,
  THY_S_SUBOPT_CERT,
  THY_S_SUBOPT_KEY,
  THY_S_SUBOPT_KEYRING,
  THY_S_SUBOPT_TRUSTDB,
  THY_S_SUBOPT_TYPE,
  THY_S_SUBOPT_VERIFY,
  THY_S_SUBOPT_NOVERIFY,
  THY_S_SUBOPT_SRPPASSWD,
  THY_S_SUBOPT_SRPCONF,
  THY_S_SUBOPT_END
};

static char *s_subopts[] = {
  C99_INIT([THY_S_SUBOPT_CA]) "ca",
  C99_INIT([THY_S_SUBOPT_CERT]) "cert",
  C99_INIT([THY_S_SUBOPT_KEY]) "key",
  C99_INIT([THY_S_SUBOPT_KEYRING]) "keyring",
  C99_INIT([THY_S_SUBOPT_TRUSTDB]) "trustdb",
  C99_INIT([THY_S_SUBOPT_TYPE]) "type",
  C99_INIT([THY_S_SUBOPT_VERIFY]) "verify",
  C99_INIT([THY_S_SUBOPT_NOVERIFY]) "noverify",
  C99_INIT([THY_S_SUBOPT_SRPPASSWD]) "srppasswd",
  C99_INIT([THY_S_SUBOPT_SRPCONF]) "srpconf",
  C99_INIT([THY_S_SUBOPT_END]) NULL
};

enum
{
  THY_G_SUBOPT_TYPE,
  THY_G_SUBOPT_LEVEL,
  THY_G_SUBOPT_END
};

static char *g_subopts[] = {
  C99_INIT([THY_G_SUBOPT_TYPE]) "type",
  C99_INIT([THY_G_SUBOPT_LEVEL]) "level",
  C99_INIT([THY_G_SUBOPT_END]) NULL
};

enum
{
  THY_L_SUBOPT_HEADER,
  THY_L_SUBOPT_POST_BUFFER,
  THY_L_SUBOPT_POST,
  THY_L_SUBOPT_CGIS,
  THY_L_SUBOPT_KEEPALIVE,
  THY_L_SUBOPT_MAXCLIENTS,
  THY_L_SUBOPT_MMAP,
  THY_L_SUBOPT_END
};

static char *L_subopts[] = {
  C99_INIT([THY_L_SUBOPT_HEADER]) "header",
  C99_INIT([THY_L_SUBOPT_POST_BUFFER]) "post_buffer",
  C99_INIT([THY_L_SUBOPT_POST]) "post",
  C99_INIT([THY_L_SUBOPT_CGIS]) "cgis",
  C99_INIT([THY_L_SUBOPT_KEEPALIVE]) "keepalive",
  C99_INIT([THY_L_SUBOPT_MAXCLIENTS]) "maxclients",
  C99_INIT([THY_L_SUBOPT_MMAP]) "mmap",
  C99_INIT([THY_L_SUBOPT_END]) NULL
};

enum
{
  THY_A_SUBOPT_ARG,
  THY_A_SUBOPT_PATH,
  THY_A_SUBOPT_UID,
  THY_A_SUBOPT_FILE,
  THY_A_SUBOPT_END
};

static char *a_subopts[] = {
  C99_INIT([THY_A_SUBOPT_ARG]) "arg",
  C99_INIT([THY_A_SUBOPT_PATH]) "path",
  C99_INIT([THY_A_SUBOPT_UID]) "uid",
  C99_INIT([THY_A_SUBOPT_FILE]) "file",
  C99_INIT([THY_A_SUBOPT_END]) NULL
};

enum
{
  THY_W_SUBOPT_ARG,
  THY_W_SUBOPT_PATH,
  THY_W_SUBOPT_UID,
  THY_W_SUBOPT_END
};

static char *W_subopts[] = {
  C99_INIT([THY_W_SUBOPT_ARG]) "arg",
  C99_INIT([THY_W_SUBOPT_PATH]) "path",
  C99_INIT([THY_W_SUBOPT_UID]) "uid",
  C99_INIT([THY_W_SUBOPT_END]) NULL
};

enum
{
  THY_T_SUBOPT_DIRTAG,
  THY_T_SUBOPT_NODIRTAG,
  THY_T_SUBOPT_ETAG,
  THY_T_SUBOPT_NOETAG,
  THY_T_SUBOPT_END
};

static char *T_subopts[] = {
  C99_INIT([THY_T_SUBOPT_DIRTAG]) "dirtag",
  C99_INIT([THY_T_SUBOPT_NODIRTAG]) "nodirtag",
  C99_INIT([THY_T_SUBOPT_ETAG]) "etag",
  C99_INIT([THY_T_SUBOPT_NOETAG]) "noetag",
  C99_INIT([THY_T_SUBOPT_END]) NULL
};

enum
{
  THY_R_SUBOPT_FILE,
  THY_R_SUBOPT_URL,
  THY_R_SUBOPT_FILECACHE,
  THY_R_SUBOPT_URLCACHE,
  THY_R_SUBOPT_EXPIRE,
  THY_R_SUBOPT_MAXAGE,
  THY_R_SUBOPT_END
};

static char *r_subopts[] = {
  C99_INIT([THY_R_SUBOPT_FILE]) "file",
  C99_INIT([THY_R_SUBOPT_URL]) "url",
  C99_INIT([THY_R_SUBOPT_FILECACHE]) "filecache",
  C99_INIT([THY_R_SUBOPT_URLCACHE]) "urlcache",
  C99_INIT([THY_R_SUBOPT_EXPIRE]) "expire",
  C99_INIT([THY_R_SUBOPT_MAXAGE]) "maxage",
  C99_INIT([THY_R_SUBOPT_END]) NULL
};

enum
{
  THY_t_SUBOPT_KEEPALIVE,
  THY_t_SUBOPT_END
};

static char *t_subopts[] = {
  C99_INIT([THY_t_SUBOPT_KEEPALIVE]) "keepalive",
  C99_INIT([THY_t_SUBOPT_END]) NULL
};

enum
{
  THY_X_SUBOPT_NO_CACHE,
  THY_X_SUBOPT_CACHE,
  THY_X_SUBOPT_NO_STORE,
  THY_X_SUBOPT_STORE,
  THY_X_SUBOPT_NO_TRANSFORM,
  THY_X_SUBOPT_TRANSFORM,
  THY_X_SUBOPT_MUST_REVALIDATE,
  THY_X_SUBOPT_MAX_AGE,
  THY_X_SUBOPT_NOMAX_AGE,
  THY_X_SUBOPT_EXPIRY_BASE,
  THY_X_SUBOPT_END
};

static char *X_subopts[] = {
  C99_INIT([THY_X_SUBOPT_NO_CACHE]) "no-cache",
  C99_INIT([THY_X_SUBOPT_CACHE]) "cache",
  C99_INIT([THY_X_SUBOPT_NO_STORE]) "no-store",
  C99_INIT([THY_X_SUBOPT_STORE]) "store",
  C99_INIT([THY_X_SUBOPT_NO_TRANSFORM]) "no-transform",
  C99_INIT([THY_X_SUBOPT_TRANSFORM]) "transform",
  C99_INIT([THY_X_SUBOPT_MUST_REVALIDATE]) "must-revalidate",
  C99_INIT([THY_X_SUBOPT_MAX_AGE]) "max-age",
  C99_INIT([THY_X_SUBOPT_NOMAX_AGE]) "nomax-age",
  C99_INIT([THY_X_SUBOPT_EXPIRY_BASE]) "expiry-base",
  C99_INIT([THY_X_SUBOPT_END]) NULL
};

static char *empty_subopts[] = { NULL };

static char *event_systems[] = {
#if HAVE_EPOLL_CREATE
  C99_INIT([THY_NQUEUE_WAITER_EPOLL]) "epoll",
#endif
#if HAVE_KQUEUE
  C99_INIT([THY_NQUEUE_WAITER_KQUEUE]) "kqueue",
#endif
#if defined(HAVE_SYS_DEVPOLL_H) || defined(HAVE_LINUX_DEVPOLL_H)
  C99_INIT([THY_NQUEUE_WAITER_DEVPOLL]) "devpoll",
#endif
#if HAVE_SYS_POLL_H
  C99_INIT([THY_NQUEUE_WAITER_POLL]) "poll",
#endif
#if HAVE_SYS_SELECT_H
  C99_INIT([THY_NQUEUE_WAITER_SELECT]) "select",
#endif
  C99_INIT([THY_NQUEUE_WAITER_NONE]) NULL
};
#endif /* !__DOXYGEN__ */

/** @internal Parse a HOST/IP pair.
 * Splits up a given HOST/IP pair, and converts them into structures
 * directly usable by libc routines.
 *
 * @param address is the address to resolve. It is of the format
 * [(http|https|ssl|tls):][host]/port.
 *
 * @returns A pointer to a bhl_listener_t structure, or NULL on error.
 */
static thy_listener_t *
_thy_config_listen_parse (const char *address)
{
  char *s_ip = NULL;
  char *tmp;
  int port;
  thy_listener_t *l;
  const char *addy;
  int any_addr = 0;
  struct in6_addr in6_any = IN6ADDR_ANY_INIT;

  l = (thy_listener_t *)bhc_malloc (sizeof (thy_listener_t));
  l->ssl = thy_default_tls;
  if (thy_default_tls)
    port = _THY_PORT_SSL;
  else
    port = _THY_PORT;

  addy = address;
  if (strstr (address, "http:") == address)
    {
      addy = &address[5];
      l->ssl = LISTENER_SSL_UNSET;
    }
  else if (strstr (address, "https:") == address)
    {
      addy = &address[6];
      l->ssl = LISTENER_SSL_SET;
    }
  else if ((strstr (address, "ssl:") == address) ||
	   (strstr (address, "tls:") == address))
    {
      addy = &address[4];
      l->ssl = LISTENER_SSL_SET;
    }

  tmp = strchr (addy, '/');

  if (!tmp)
    s_ip = bhc_strdup (addy);
  else if (tmp == addy)
    {
      any_addr = 1;
      s_ip = bhc_strdup ("0.0.0.0");
      port = (int)atoi (&addy[1]);
    }
  else
    {
      port = (int)atoi (&tmp[1]);
      s_ip = bhc_strndup (addy, (size_t)(tmp - addy));
    }

  if (!port)
    {
      free (s_ip);
      free (l);
      return NULL;
    }

  l->sock = bhc_malloc (sizeof (struct sockaddr_storage));
  if (thy_addr_get (s_ip, l->sock) == -1)
    {
      free (s_ip);
      free (l->sock);
      free (l);
      return NULL;
    }

  free (s_ip);

  switch (l->sock->ss_family)
    {
    case AF_INET6:
      if (any_addr)
	((struct sockaddr_in6 *)(l->sock))->sin6_addr = in6_any;
      ((struct sockaddr_in6 *)(l->sock))->sin6_port = htons (port);
      break;
    case AF_INET:
      if (any_addr)
	((struct sockaddr_in *)(l->sock))->sin_addr.s_addr =
	  htonl (INADDR_ANY);
      ((struct sockaddr_in *)(l->sock))->sin_port = htons (port);
      break;
    }

  return l;
}

/** @internal Parse a field=value pair.
 * Parses a field=value pair, and puts the result into a specified
 * list of pairs.
 *
 * @param arg is the string to parse
 * @param desc is the description of the thing we parse.
 * @param list is the destination list, where the result will be
 * stored.
 * @param state is the argp state, for reporting errors.
 *
 * @note Modifies list in-place.
 */
static void
_config_parse_pair (const char *arg, const char *desc,
		    bhl_list_t *list, struct argp_state *state)
{
  char *tmp, *subopts, *value, *field, *spec;

  tmp = subopts = bhc_strdup (arg);
  while (*subopts != '\0')
    {
      char *t;
      pair_t pair;

      bhc_getsubopt (&subopts, empty_subopts, &value);
      spec = value;

      t = strchr (value, '=');
      field = value;
      value = t + 1;
      t[0] = '\0';
      if (!field || !*field)
	{
	  argp_error (state, "malformed %s specification: `%s'",
		      desc, spec);
	  break;
	}
      if (!value || !*value)
	{
	  argp_error (state, "malformed %s specification: `%s'",
		      desc, spec);
	  break;
	}

      pair.field = bhc_strdup (field);
      pair.value = bhc_strdup (value);
      bhl_list_append (list, (void *)&pair, sizeof (pair));
    }
  free (tmp);
}

/** @internal Parse a comma separated argument list.
 * Parses a comma separated argument list, and puts the individual
 * entries into a list.
 *
 * @param arg is the comma separated argument list.
 * @param list is the destination list.
 *
 * @note Modifies all paramaters in-place.
 */
static void
_config_parse_list (char *arg, bhl_list_t *list)
{
  char *subopts = arg;
  char *value;

  while (*subopts != '\0')
    {
      bhc_getsubopt (&subopts, empty_subopts, &value);
      bhl_list_append_string (list, value);
    }
}

/** @internal Retrieve the configuration of a map.
 * Iterates through the list of maps, and returns the configuration
 * associated with the one that matches our search.
 *
 * @param config is the global configuration.
 * @param name is the map ID to search for.
 * @param create flags if the map should be created if it does not
 * exist.
 *
 * @returns The associated configuration, or the default one, if one
 * is not found. If an error occurred during the creation of the map,
 * returns NULL.
 */
static thy_mappable_config_t *
_config_get_map (config_t *config, const char *name, int create)
{
  size_t i;
  thy_config_map_t *map;

  if (name == NULL)
    return config->defconf;

  for (i = 0; i < bhl_list_size (config->maps); i++)
    {
      bhl_list_get (config->maps, i, (void **)&map);
      if (!strcmp (map->url, name))
	{
	  thy_mappable_config_t *cfg = map->config;
	  free (map);
	  return cfg;
	}
      free (map);
    }

  if (!create)
    return config->defconf;

  map = bhc_malloc (sizeof (thy_config_map_t));
  map->url = bhc_strdup (name);

  map->preg = bhc_malloc (sizeof (regex_t));
  if (regcomp ((regex_t *)map->preg, map->url, REG_EXTENDED) != 0)
    {
      free (map->preg);
      free (map->url);
      free (map);
      return NULL;
    }

  map->config = bhc_calloc (1, sizeof (thy_mappable_config_t));
  map->config->encoding.level = -1;
  map->config->indexes = bhl_list_init (0, NULL);
  map->config->cgiexts = bhl_list_init (0, NULL);
  map->config->handlers = bhl_list_init (0, NULL);
  map->config->methods = bhl_list_init (0, NULL);
  map->config->mime_types = bhl_list_init (0, NULL);
  map->config->headers = bhl_list_init (0, NULL);
  map->config->env = bhl_list_init (0, NULL);
  map->config->cgidirs = bhl_list_init (0, NULL);
  map->config->aliases = bhl_list_init (0, NULL);

  bhl_list_append (config->maps, map, sizeof (thy_config_map_t));
  return map->config;
}

/** @internal Parse one option.
 * See the argp docs for details.
 */
static error_t
_config_parse_opt (int key, char *arg, struct argp_state *state)
{
  _thy_int_config_t *inp = (_thy_int_config_t *)state->input;
  thy_mappable_config_t *cfg = inp->global->defconf;
  thy_mappable_config_t *map_cfg = _config_get_map (inp->global,
						    inp->curmap, 1);
  config_t *global = inp->global;
  char *subopts, *value;
  int i;
  thy_listener_t *listener;
  size_t wl;

  if (!map_cfg)
    argp_error (state, "Invalid map specification: %s", inp->curmap);

  switch (key)
    {
    case 'F':
      free (inp->curmap);
      inp->curmap = NULL;
      break;
    case 'R':
      free (inp->curmap);
      if (!arg || strlen (arg) == 0)
	inp->curmap = NULL;
      else
	inp->curmap = bhc_strdup (arg);
      break;
    case '4':
      global->type = PF_INET;
      break;
    case '6':
      global->type = PF_INET6;
      break;
    case 'u':
      cfg->userdir = bhc_strdup (arg);
      break;
    case 'U':
      if (bhc_atoi (arg, (long int *)&global->uid) != 0)
	argp_error (state, "invalid user id: `%s'", arg);
      break;
    case 'H':
      _config_parse_pair (arg, "handler", map_cfg->handlers, state);
      break;
    case 'M':
      _config_parse_pair (arg, "method", map_cfg->methods, state);
      break;
    case 'D':
      _config_parse_pair (arg, "header", map_cfg->headers, state);
      break;
    case 'm':
      _config_parse_pair (arg, "MIME type", map_cfg->mime_types, state);
      break;
    case 'A':
      _config_parse_pair (arg, "alias", map_cfg->aliases, state);
      break;
    case 'E':
      _config_parse_pair (arg, "environment variable",
			  map_cfg->env, state);
      break;
    case 'i':
      _config_parse_list (optarg, map_cfg->indexes);
      break;
    case 'C':
      _config_parse_list (optarg, map_cfg->cgidirs);
      break;
    case 'c':
      _config_parse_list (optarg, map_cfg->cgiexts);
      break;
    case 'p':
      if (bhc_atoi (arg, (long int *)&thy_default_port) != 0)
	argp_error (state, "invalid port: `%s'", arg);
      break;
    case 'P':
      global->pidfile = bhc_strdup (arg);
      break;
    case 'e':
      subopts = optarg;
      while (*subopts != '\0')
	{
	  i = bhc_getsubopt (&subopts, e_subopts, &value);
	  if (i < 0 || i >= HTTP_STATUS_UNKNOWN + 1)
	    {
	      argp_error (state, "unrecognised HTTP response code: `%s'",
			  value);
	      break;
	    }
	  else
	    {
	      if (*value == '\0')
		{
		  argp_error (state,
			      "the `%s' HTTP response requires a value.",
			      e_subopts[i]);
		  break;
		}
	      if (i == HTTP_STATUS_UNKNOWN)
		{
		  /* All */
		  for (i = HTTP_STATUS_100; i < HTTP_STATUS_UNKNOWN; i++)
		    map_cfg->http_status[i] = bhc_strdup (value);
		}
	      else
		map_cfg->http_status[i] = bhc_strdup (value);
	    }
	}
      break;
    case 'X':
      subopts = optarg;
      while (*subopts != '\0')
	switch (bhc_getsubopt (&subopts, X_subopts, &value))
	  {
	  case THY_X_SUBOPT_NO_CACHE:
	    map_cfg->cache.no_cache = THY_BOOL_TRUE;
	    break;
	  case THY_X_SUBOPT_CACHE:
	    map_cfg->cache.no_cache = THY_BOOL_FALSE;
	    break;
	  case THY_X_SUBOPT_NO_STORE:
	    map_cfg->cache.no_store = THY_BOOL_TRUE;
	    break;
	  case THY_X_SUBOPT_STORE:
	    map_cfg->cache.no_store = THY_BOOL_FALSE;
	    break;
	  case THY_X_SUBOPT_NO_TRANSFORM:
	    map_cfg->cache.no_transform = THY_BOOL_TRUE;
	    break;
	  case THY_X_SUBOPT_TRANSFORM:
	    map_cfg->cache.no_transform = THY_BOOL_FALSE;
	    break;
	  case THY_X_SUBOPT_MUST_REVALIDATE:
	    map_cfg->cache.must_revalidate = THY_BOOL_TRUE;
	    break;
	  case THY_X_SUBOPT_MAX_AGE:
	    if (bhc_atoi (value, (long int *)&map_cfg->cache.max_age) != 0)
	      argp_error (state, "invalid max age: `%s'", value);
	    else
	      map_cfg->cache.do_max_age = THY_BOOL_TRUE;
	    break;
	  case THY_X_SUBOPT_NOMAX_AGE:
	    map_cfg->cache.do_max_age = THY_BOOL_FALSE;
	    break;
	  case THY_X_SUBOPT_EXPIRY_BASE:
	    if (!strcasecmp (value, "now") ||
		!strcasecmp (value, "access"))
	      map_cfg->cache.expiry_base = THY_EXPIRY_BASE_NOW;
	    else if (!strcasecmp (value, "modification"))
	      map_cfg->cache.expiry_base = THY_EXPIRY_BASE_MODIFICATION;
	    else
	      argp_error (state, "unrecognised expiry base: `%s'", value);
	    break;
	  default:
	    argp_error (state, "unrecognised cache control directive: `%s'",
			value);
	    break;
	  }
      break;
    case 's':
      subopts = optarg;
      while (*subopts != '\0')
	{
	  int opt = bhc_getsubopt (&subopts, s_subopts, &value);

	  if (*value == '\0' && opt != THY_S_SUBOPT_VERIFY &&
	      opt != THY_S_SUBOPT_NOVERIFY)
	    {
	      argp_error (state, "the `%s' SSL option requires a value.",
			  s_subopts[opt]);
	      break;
	    }
	  switch (opt)
	    {
	    case THY_S_SUBOPT_SRPPASSWD:
	      global->ssl.srp.passwd = bhc_strdup (value);
	      thy_tls_enabled = THY_BOOL_TRUE;
	      break;
	    case THY_S_SUBOPT_SRPCONF:
	      global->ssl.srp.conf = bhc_strdup (value);
	      thy_tls_enabled = THY_BOOL_TRUE;
	      break;
	    case THY_S_SUBOPT_CA:
	      bhl_list_append_string (global->ssl.x509.cafiles, value);
	      thy_tls_enabled = THY_BOOL_TRUE;
	      break;
	    case THY_S_SUBOPT_CERT:
	      if (global->ssl.cert_type == THY_TLS_CERT_TYPE_OPENPGP)
		global->ssl.openpgp.certfile = bhc_strdup (value);
	      else
		bhl_list_append_string (global->ssl.x509.certfiles, value);
	      thy_tls_enabled = THY_BOOL_TRUE;
	      break;
	    case THY_S_SUBOPT_KEY:
	      if (global->ssl.cert_type == THY_TLS_CERT_TYPE_OPENPGP)
		global->ssl.openpgp.keyfile = bhc_strdup (value);
	      else
		bhl_list_append_string (global->ssl.x509.keyfiles, value);
	      thy_tls_enabled = THY_BOOL_TRUE;
	      break;
	    case THY_S_SUBOPT_KEYRING:
	      global->ssl.openpgp.keyring = bhc_strdup (value);
	      thy_tls_enabled = THY_BOOL_TRUE;
	      break;
	    case THY_S_SUBOPT_TRUSTDB:
	      global->ssl.openpgp.trustdb = bhc_strdup (value);
	      thy_tls_enabled = THY_BOOL_TRUE;
	      break;
	    case THY_S_SUBOPT_TYPE:
	      if (!strcasecmp (value, "x509"))
		global->ssl.cert_type = THY_TLS_CERT_TYPE_X509;
	      else if (!strcasecmp (value, "openpgp"))
		global->ssl.cert_type = THY_TLS_CERT_TYPE_OPENPGP;
	      else
		argp_error (state, "unrecognised certification type: `%s'",
			    value);
	      break;
	    case THY_S_SUBOPT_VERIFY:
	      if (!value || *value == '\0')
		global->ssl.verify_level = 1;
	      else
		if (bhc_atoi (value,
			      (long int *)&global->ssl.verify_level) != 0)
		  argp_error (state, "invalid verify level: `%s'", value);
	      break;
	    case THY_S_SUBOPT_NOVERIFY:
	      global->ssl.verify_level = 0;
	      break;
	    default:
	      argp_error (state, "unrecognised SSL option: `%s'", value);
	      break;
	    }
	}
      break;
    case 'g':
      subopts = optarg;
      while (*subopts != '\0')
	{
	  int opt = bhc_getsubopt (&subopts, g_subopts, &value);

	  if (*value == '\0')
	    {
	      argp_error (state, "the `%s' gzip option requires a value.",
			  g_subopts[opt]);
	      break;
	    }
	  switch (opt)
	    {
	    case THY_G_SUBOPT_TYPE:
	      if (!strcasecmp (value, "static") ||
		  !strcasecmp (value, "precomp"))
		map_cfg->encoding.type = ENCODING_TYPE_STATIC;
	      else if (!strcasecmp (value, "none"))
		map_cfg->encoding.type = ENCODING_TYPE_NONE;
	      else if (!strcasecmp (value, "dynamic") ||
		       !strcasecmp (value, "gzip"))
		map_cfg->encoding.type = ENCODING_TYPE_DYNAMIC;
	      else
		argp_error (state, "unrecognised encoding: `%s'", value);
	      break;
	    case THY_G_SUBOPT_LEVEL:
	      if (bhc_atoi (value,
			    (long int *)&map_cfg->encoding.level) != 0)
		argp_error (state, "invalid encoding level: `%s'", value);
	      else
		{
		  if (map_cfg->encoding.level < 0 ||
		      map_cfg->encoding.level > 9)
		    argp_error (state, "unsupported compression level: %d",
				map_cfg->encoding.level);
		}
	      break;
	    default:
	      argp_error (state, "unrecognised option: `%s'", value);
	      break;
	    }
	}
      break;
    case 'r':
      subopts = optarg;
      while (*subopts != '\0')
	switch (bhc_getsubopt (&subopts, r_subopts, &value))
	  {
	  case THY_R_SUBOPT_FILE:
	  case THY_R_SUBOPT_FILECACHE:
	    if (bhc_atoi (value, (long int *)&global->lru.fcache) != 0)
	      argp_error (state, "invalid file cache size: `%s'", value);
	    break;
	  case THY_R_SUBOPT_URL:
	  case THY_R_SUBOPT_URLCACHE:
	    if (bhc_atoi (value, (long int *)&global->lru.ucache) != 0)
	      argp_error (state, "invalid url cache size: `%s'", value);
	    break;
	  case THY_R_SUBOPT_EXPIRE:
	  case THY_R_SUBOPT_MAXAGE:
	    if (bhc_atoi (value, (long int *)&global->lru.expire) != 0)
	      argp_error (state, "invalid cache expiry time: `%s'", value);
	    break;
	  default:
	    if (bhc_atoi (value, (long int *)&global->lru.ucache) != 0)
	      argp_error (state, "invalid cache size: `%s'", value);
	    else
	      global->lru.fcache = global->lru.ucache;
	    break;
	  }
      break;
    case 'L':
      subopts = optarg;
      while (*subopts != '\0')
	switch (bhc_getsubopt (&subopts, L_subopts, &value))
	  {
	  case THY_L_SUBOPT_HEADER:
	    if (bhc_atoi (value, (long int *)&global->limits.header) != 0)
	      argp_error (state, "invalid header limit: `%s'", value);
	    break;
	  case THY_L_SUBOPT_POST:
	  case THY_L_SUBOPT_POST_BUFFER:
	    if (bhc_atoi (value,
			  (long int *)&global->limits.post_buffer) != 0)
	      argp_error (state, "invalid POST buffer limit: `%s", value);
	    break;
	  case THY_L_SUBOPT_CGIS:
	    if (bhc_atoi (value, (long int *)&global->limits.cgis) != 0)
	      argp_error (state, "invalid CGI limit: `%s'", value);
	    break;
	  case THY_L_SUBOPT_KEEPALIVE:
	    if (bhc_atoi (value, (long int *)&global->keep_max) != 0)
	      argp_error (state, "invalid maximum keep-alive count: `%s'",
			  value);
	    break;
	  case THY_L_SUBOPT_MAXCLIENTS:
	    if (bhc_atoi (value, (long int *)&_THY_MAXCONN) != 0)
	      argp_error (state, "invalid maximum client count: `%s'",
			  value);
	    break;
	  case THY_L_SUBOPT_MMAP:
	    if (bhc_atoi (value, (long int *)&global->limits.mmap) != 0)
	      argp_error (state, "invalid maximum mmap() limit: `%s'", value);
	    break;
	  default:
	    argp_error (state, "unrecognised limit: `%s'", value);
	    break;
	  }
      break;
    case 'a':
      subopts = optarg;
      while (*subopts != '\0')
	{
	  int opt = bhc_getsubopt (&subopts, a_subopts, &value);

	  if (*value == '\0')
	    {
	      argp_error (state,
			  "the `%s' authoriser option requires a value.",
			  a_subopts[opt]);
	      break;
	    }

	  switch (opt)
	    {
	    case THY_A_SUBOPT_ARG:
	      global->auth.args.argc++;
	      XSREALLOC (global->auth.args.argv, char *,
			 global->auth.args.argc + 1);
	      global->auth.args.argv[global->auth.args.argc] =
		bhc_strdup (value);
	      break;
	    case THY_A_SUBOPT_PATH:
	      global->auth.path = bhc_strdup (value);
	      break;
	    case THY_A_SUBOPT_UID:
	      if (bhc_atoi (value, (long int *)&global->auth.uid) != 0)
		argp_error (state, "invalid Authoriser UID: `%s'", value);
	      break;
	    case THY_A_SUBOPT_FILE:
	      global->auth.file = bhc_strdup (value);
	      break;
	    default:
	      argp_error (state, "unrecognised authenticator option: `%s'",
			  value);
	      break;
	    }
	}
      break;
    case 'W':
      subopts = optarg;
      while (*subopts != '\0')
	{
	  int opt = bhc_getsubopt (&subopts, W_subopts, &value);

	  if (*value == '\0')
	    {
	      argp_error (state, "the `%s' worker option require a value.",
			  W_subopts[opt]);
	      break;
	    }

	  switch (opt)
	    {
	    case THY_W_SUBOPT_ARG:
	      global->worker.args.argc++;
	      XSREALLOC (global->worker.args.argv, char *,
			 global->worker.args.argc + 1);
	      global->worker.args.argv[global->worker.args.argc] =
		bhc_strdup (value);
	      break;
	    case THY_W_SUBOPT_PATH:
	      global->worker.path = bhc_strdup (value);
	      break;
	    case THY_W_SUBOPT_UID:
	      if (bhc_atoi (value, (long int *)&global->worker.uid) != 0)
		argp_error (state, "invalid Worker UID: `%s'", value);
	      break;
	    default:
	      argp_error (state, "unrecognised worker option: `%s'",
			  value);
	      break;
	    }
	}
      break;
    case 'T':
      subopts = optarg;
      while (*subopts != '\0')
	switch (bhc_getsubopt (&subopts, T_subopts, &value))
	  {
	  case THY_T_SUBOPT_DIRTAG:
	    map_cfg->etag.dirtag = THY_BOOL_TRUE;
	    break;
	  case THY_T_SUBOPT_NODIRTAG:
	    map_cfg->etag.dirtag = THY_BOOL_FALSE;
	    break;
	  case THY_T_SUBOPT_ETAG:
	    map_cfg->etag.etag = THY_BOOL_TRUE;
	    break;
	  case THY_T_SUBOPT_NOETAG:
	    map_cfg->etag.etag = THY_BOOL_FALSE;
	    break;
	  default:
	    argp_error (state, "unrecognised ETag option: `%s'", value);
	    break;
	  }
      break;
    case 'o':
      subopts = optarg;
      while (*subopts != '\0')
	switch (bhc_getsubopt (&subopts, o_subopts, &value))
	  {
	  case THY_O_SUBOPT_AUTH:
	    map_cfg->options.auth = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOAUTH:
	    map_cfg->options.auth = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_VHOST:
	    map_cfg->options.vhosting = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOVHOST:
	    map_cfg->options.vhosting = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_USERDIR:
	    map_cfg->options.userdir = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOUSERDIR:
	    map_cfg->options.userdir = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_CGI:
	    map_cfg->options.cgi = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOCGI:
	    map_cfg->options.cgi = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_USERCGI:
	    map_cfg->options.usercgi = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOUSERCGI:
	    map_cfg->options.usercgi = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_BUFFER:
	    global->options.buffers = 1;
	    if (value && *value)
	      global->buffer_size = atol (value) * 1024;
	    break;
	  case THY_O_SUBOPT_NOBUFFER:
	    global->options.buffers = 0;
	    break;
	  case THY_O_SUBOPT_CASEMIME:
	    map_cfg->options.casemime = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOCASEMIME:
	    map_cfg->options.casemime = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_FOLLOWALL:
	    map_cfg->options.followall = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOFOLLOWALL:
	    map_cfg->options.followall = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_SERVER:
	    thy_servername (bhc_strdup (value));
	    break;
	  case THY_O_SUBOPT_NOSERVER:
#ifndef __LCLINT__
	    thy_servername (__THY__ "/" _THY_VERSION);
#endif
	    break;
	  case THY_O_SUBOPT_SSL:
	    thy_default_tls = LISTENER_SSL_SET;
	    thy_tls_enabled = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOSSL:
	    thy_default_tls = LISTENER_SSL_UNSET;
	    thy_tls_enabled = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_PIDFILE:
	    if (*value == '\0')
	      argp_error (state, "option `pidfile' requires a value.");
	    else
	      global->pidfile = bhc_strdup (value);
	    break;
	  case THY_O_SUBOPT_NOPIDFILE:
	    global->pidfile = NULL;
	    break;
	  case THY_O_SUBOPT_VARY:
	    map_cfg->options.vary = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOVARY:
	    map_cfg->options.vary = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_CHROOT:
	    global->options.chroot = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOCHROOT:
	    global->options.chroot = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_DIRINDEX:
	    map_cfg->options.dirindex = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NODIRINDEX:
	    map_cfg->options.dirindex = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_WORKER:
	    global->options.worker = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOWORKER:
	    global->options.worker = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_EXPECT:
	    global->options.expect = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOEXPECT:
	    global->options.expect = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_HARDLIMIT:
	    global->options.hardlimit = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOHARDLIMIT:
	    global->options.hardlimit = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_STATS:
	    if (bhc_atoi (value, (long int *)&global->options.stats) != 0)
	      argp_error (state, "invalid status report interval: `%s'",
			  value);
	    break;
	  case THY_O_SUBOPT_NOSTATS:
	    global->options.stats = 0;
	    break;
	  case THY_O_SUBOPT_LAZYCGI:
	    map_cfg->options.lazycgi = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NOLAZYCGI:
	    map_cfg->options.lazycgi = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_CACHE:
	    global->lru.fcache = _THY_LRUSIZE;
	    global->lru.ucache = _THY_LRUSIZE;
	    break;
	  case THY_O_SUBOPT_NOCACHE:
	    global->lru.fcache = 0;
	    global->lru.ucache = 0;
	    break;
	  case THY_O_SUBOPT_DAEMON:
	    global->options.daemon = THY_BOOL_TRUE;
	    break;
	  case THY_O_SUBOPT_NODAEMON:
	    global->options.daemon = THY_BOOL_FALSE;
	    break;
	  case THY_O_SUBOPT_EVENTSYSTEM:
	    if (*value == '\0')
	      argp_error (state, "option `eventsystem' requires a value.");
	    else
	      {
		for (i = 0; i < THY_NQUEUE_WAITER_NONE; i++)
		  {
		    if (!strcasecmp (event_systems[i], value))
		      {
			global->options.waiter = i;
			break;
		      }
		  }
		if (global->options.waiter ==  THY_NQUEUE_WAITER_NONE)
		  argp_error (state, "invalid event system: `%s'", value);
	      }
	    break;
	  default:
	    argp_error (state, "unrecognised option: `%s'", value);
	    break;
	  }
      break;
    case 'l':
      subopts = optarg;
      while (*subopts != '\0')
	{
	  bhc_getsubopt (&subopts, empty_subopts, &value);

	  listener = _thy_config_listen_parse (value);
	  if (listener)
	    bhl_list_append (global->ips, listener,
			     sizeof (thy_listener_t));
	  free (listener);
	}
      break;
    case 'w':
      map_cfg->webroot = bhc_strdup (arg);

      /* Workaround: If webroot contains a trailing slash, strip it */
      wl = strlen (map_cfg->webroot);
      if (wl > 1 && map_cfg->webroot[wl - 1] == '/')
	map_cfg->webroot[wl - 1] = '\0';
      break;
    case 't':
      subopts = optarg;
      while (*subopts != '\0')
	switch (bhc_getsubopt (&subopts, t_subopts, &value))
	  {
	  case THY_t_SUBOPT_KEEPALIVE:
	    if (bhc_atoi (value, (long int *)&global->ktimeout) != 0)
	      argp_error (state, "invalid timeout: `%s'", arg);
	    break;
	  default:
	    if (bhc_atoi (value, (long int *)&global->timeout) != 0)
	      argp_error (state, "invalid timeout: `%s'", arg);
	    break;
	  }
      break;
    case 'd':
      map_cfg->default_type = bhc_strdup (arg);
      break;
    case ARGP_KEY_ARG:
      thy_read_default_config = 0;
      switch (thy_config_file_parse (arg))
	{
	case 1:
	  argp_error (state, "error parsing configuration file: `%s'",
		      arg);
	  break;
	case 2:
	  argp_error (state, "configuration file does not exist: `%s'",
		      arg);
	  break;
	}

      if (bhl_list_size (conffile_list) > 1)
	{
	  int c_argc;
	  char **c_argv;
	  size_t l;
	  _thy_int_config_t cfg_int;
	  config_t *config = config_get ();
	  struct argp argp_cf =
	    {options, _config_parse_opt, "[CONFIG FILES...]",
	     __THY__ " -- Use thy power." EGG, NULL, NULL, NULL};

	  cfg_int.global = config;
	  cfg_int.curmap = NULL;

	  c_argc = (int)bhl_list_size (conffile_list);
	  c_argv = (char **)bhc_calloc (c_argc + 1, sizeof (char *));
	  for (l = 0; l < bhl_list_size (conffile_list); l++)
	    {
	      char *tmp;

	      bhl_list_get (conffile_list, l, (void **)&tmp);
	      c_argv[l] = tmp;
	    }
	  argp_parse (&argp_cf, c_argc, c_argv, 0, NULL, &cfg_int);
	}
      bhl_list_free (conffile_list);
      conffile_list = NULL;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }

  return 0;
}

/** Set up basic configuration and parse command-line options.
 *
 * @returns A pointer to the newly created configuration structure.
 */
void
config_parse (int argc, char **argv)
{
  _thy_int_config_t cfg;
  config_t *config = config_get ();
  size_t wl;

  config->defconf = &thy_default_config;

  config->defconf->indexes = bhl_list_init (1, NULL);
  config->defconf->cgiexts = bhl_list_init (1, NULL);
  config->defconf->handlers = bhl_list_init (0, NULL);
  config->defconf->methods = bhl_list_init (0, NULL);
  config->defconf->mime_types = bhl_list_init (0, NULL);
  config->defconf->headers = bhl_list_init (0, NULL);
  config->defconf->env = bhl_list_init (0, NULL);
  config->defconf->cgidirs = bhl_list_init (0, NULL);
  config->defconf->aliases = bhl_list_init (0, NULL);

  thy_default_config.webroot = _THY_WEBROOT;
  thy_default_config.default_type = _THY_DEFTYPE;
  thy_default_config.userdir = _THY_USERDIR;

  thy_default_config.options.userdir = THY_BOOL_TRUE;
  thy_default_config.options.vhosting = THY_BOOL_FALSE;
  thy_default_config.options.auth = THY_BOOL_FALSE;
  thy_default_config.options.cgi = THY_BOOL_FALSE;
  thy_default_config.options.usercgi = THY_BOOL_FALSE;
  thy_default_config.options.followall = THY_BOOL_FALSE;
  thy_default_config.options.casemime = THY_BOOL_TRUE;
  thy_default_config.options.vary = THY_BOOL_TRUE;
  thy_default_config.options.dirindex = THY_BOOL_TRUE;
  thy_default_config.options.lazycgi = THY_BOOL_FALSE;

  thy_default_config.cache.no_cache = THY_BOOL_FALSE;
  thy_default_config.cache.no_store = THY_BOOL_FALSE;
  thy_default_config.cache.no_transform = THY_BOOL_FALSE;
  thy_default_config.cache.must_revalidate = THY_BOOL_FALSE;
  thy_default_config.cache.max_age = 0;
  thy_default_config.cache.do_max_age = THY_BOOL_FALSE;
  thy_default_config.cache.expiry_base = THY_EXPIRY_BASE_NOW;

  thy_default_config.encoding.type = ENCODING_TYPE_STATIC;
  thy_default_config.encoding.level = 6;

  thy_default_config.etag.dirtag = THY_BOOL_FALSE;
  thy_default_config.etag.etag = THY_BOOL_TRUE;

  thy_default_config.http_status[HTTP_STATUS_301] =
    _THY_CONFDIR "/http301.html";
  thy_default_config.http_status[HTTP_STATUS_302] =
    _THY_CONFDIR "/http302.html";
  thy_default_config.http_status[HTTP_STATUS_400] =
    _THY_CONFDIR "/http400.html";
  thy_default_config.http_status[HTTP_STATUS_401] =
    _THY_CONFDIR "/http401.html";
  thy_default_config.http_status[HTTP_STATUS_403] =
    _THY_CONFDIR "/http403.html";
  thy_default_config.http_status[HTTP_STATUS_404] =
    _THY_CONFDIR "/http404.html";
  thy_default_config.http_status[HTTP_STATUS_405] =
    _THY_CONFDIR "/http405.html";
  thy_default_config.http_status[HTTP_STATUS_408] =
    _THY_CONFDIR "/http408.html";
  thy_default_config.http_status[HTTP_STATUS_411] =
    _THY_CONFDIR "/http411.html";
  thy_default_config.http_status[HTTP_STATUS_412] =
    _THY_CONFDIR "/http412.html";
  thy_default_config.http_status[HTTP_STATUS_413] =
    _THY_CONFDIR "/http413.html";
  thy_default_config.http_status[HTTP_STATUS_416] =
    _THY_CONFDIR "/http416.html";
  thy_default_config.http_status[HTTP_STATUS_417] =
    _THY_CONFDIR "/http417.html";
  thy_default_config.http_status[HTTP_STATUS_500] =
    _THY_CONFDIR "/http500.html";
  thy_default_config.http_status[HTTP_STATUS_501] =
    _THY_CONFDIR "/http501.html";
  thy_default_config.http_status[HTTP_STATUS_503] =
    _THY_CONFDIR "/http503.html";
  thy_default_config.http_status[HTTP_STATUS_505] =
    _THY_CONFDIR "/http505.html";

  config->maps = bhl_list_init (0, NULL);
  config->ips = bhl_list_init (1, NULL);
  config->ssl.x509.certfiles = bhl_list_init (0, NULL);
  config->ssl.x509.keyfiles = bhl_list_init (0, NULL);
  config->ssl.x509.cafiles = bhl_list_init (0, NULL);

  config->uid = _THY_UID;
  config->timeout = 180;
  config->ktimeout = 15;
  config->keep_max = 100;
  config->type = PF_UNSPEC;

  config->pidfile = _THY_PIDFILE;
  config->buffer_size = _THY_BUFSIZE;

  config->options.buffers = 1;
  config->options.daemon = THY_BOOL_TRUE;
  config->options.chroot = THY_BOOL_FALSE;
  config->options.worker = THY_BOOL_FALSE;
  config->options.expect = THY_BOOL_TRUE;
  config->options.hardlimit = THY_BOOL_FALSE;
  config->options.stats = 3600;
  config->options.waiter = THY_NQUEUE_WAITER_NONE;

  config->ssl.verify_level = 0;
  config->ssl.cert_type = THY_TLS_CERT_TYPE_X509;
  config->ssl.openpgp.keyfile = _THY_CONFDIR "/secret.asc";
  config->ssl.openpgp.certfile = _THY_CONFDIR "/public.asc";
  config->ssl.openpgp.keyring = _THY_CONFDIR "/ring.gpg";
  config->ssl.openpgp.trustdb = _THY_CONFDIR "/trustdb.gpg";
  config->ssl.srp.passwd = NULL;
  config->ssl.srp.conf = NULL;

  config->auth.path = _THY_PKGLIBDIR "/thy-auth";
  config->auth.uid = 0;
  config->auth.file = ".realm";
  config->auth.args.argc = 0;
  config->auth.args.argv = bhc_calloc (2, sizeof (char *));

  config->worker.path = _THY_PKGLIBDIR "/thy-worker";
  config->worker.uid = _THY_UID;
  config->worker.args.argc = 0;
  config->worker.args.argv = bhc_calloc (2, sizeof (char *));

  config->limits.post_buffer = _THY_BUFSIZE;
  config->limits.header = 2048;
  config->limits.cgis = 0;
  config->limits.mmap = 0;

  config->lru.fcache = _THY_LRUSIZE;
  config->lru.ucache = _THY_LRUSIZE;
  config->lru.expire = _THY_LRUEXPIRE;

  cfg.global = config;
  cfg.curmap = NULL;

  argp_parse (&argp, argc, argv, 0, NULL, &cfg);

  if (thy_read_default_config && access (_THY_CONFDIR "/thy.conf", R_OK) == 0)
    {
      thy_config_file_parse (_THY_CONFDIR "/thy.conf");

      if (bhl_list_size (conffile_list) > 1)
	{
	  int c_argc;
	  char **c_argv;
	  size_t l;
	  _thy_int_config_t cfg_int;
	  config_t *conf = config_get ();
	  struct argp argp_cf =
	    {options, _config_parse_opt, "[CONFIG FILES...]",
	     __THY__ " -- Use thy power." EGG, NULL, NULL, NULL};

	  cfg_int.global = conf;
	  cfg_int.curmap = NULL;

	  c_argc = (int)bhl_list_size (conffile_list);
	  c_argv = (char **)bhc_calloc (c_argc + 1, sizeof (char *));
	  for (l = 0; l < bhl_list_size (conffile_list); l++)
	    {
	      char *tmp;

	      bhl_list_get (conffile_list, l, (void **)&tmp);
	      c_argv[l] = tmp;
	    }
	  argp_parse (&argp_cf, c_argc, c_argv, 0, NULL, &cfg_int);
	}
      bhl_list_free (conffile_list);
      conffile_list = NULL;
    }

  if (bhl_list_size (config->defconf->indexes) == 0)
    bhl_list_append_string (config->defconf->indexes, _THY_DEFINDEX);
  if (bhl_list_size (config->defconf->cgiexts) == 0)
    bhl_list_append_string (config->defconf->cgiexts, _THY_DEFCGI);
  if (bhl_list_size (config->ssl.x509.certfiles) == 0)
    bhl_list_append_string (config->ssl.x509.certfiles,
			    _THY_CONFDIR "/cert.pem");
  if (bhl_list_size (config->ssl.x509.keyfiles) == 0)
    bhl_list_append_string (config->ssl.x509.keyfiles,
			    _THY_CONFDIR "/key.pem");

  config->auth.args.argv[0] = config->auth.path;
  XSREALLOC (config->auth.args.argv, char *,
	     config->auth.args.argc + 2);
  config->auth.args.argv[config->auth.args.argc + 1] = NULL;

  config->worker.args.argv[0] = config->worker.path;
  XSREALLOC (config->worker.args.argv, char *,
	     config->worker.args.argc + 2);
  config->worker.args.argv[config->worker.args.argc + 1] = NULL;

  if (config->ktimeout > 15)
    config->stimeout = 15;
  else
    config->stimeout = config->ktimeout / 2;

  if (_THY_MAXCONN < _THY_MAXACCEPT * 2)
    _THY_MAXCONN = _THY_MAXACCEPT * 2;

  /* Sanity check: the number of certs and keys need to match. */
  wl = bhl_list_size (config->ssl.x509.certfiles);
  if (wl != bhl_list_size (config->ssl.x509.keyfiles))
    {
      bhc_error ("%s", "The number of X.509 certificates does not "
		 "match the number of keys.");
      bhc_exit (1);
    }
}

/** Returns a pointer to the global configuration struct.
 */
config_t *
config_get (void)
{
  return &thy_config;
}

/** Returns a pointer to a mapped configuartion struct.
 * This function constructs a configuration for a given URL, by
 * finding all the maps that match it, and weeding together all their
 * options.
 *
 * @param absuri is the URL we want the configuration for.
 * @param file is the file we want the configuration for.
 *
 * @note Returns a newly allocated structure, which must be freed by
 * the caller. Also note that the elements of the structure are not
 * allocated, and should not be modified!
 *
 * @note A map either matches absuri or file, never
 * both. (Technically, it can match both, but only one will be used.)
 */
thy_mappable_config_t *
config_get_mapped (const char *absuri, const char *file)
{
  thy_mappable_config_t *cfg;
  thy_config_map_t *map;
  size_t i, j;
  const config_t *config = config_get ();

  cfg = bhc_malloc (sizeof (thy_mappable_config_t));
  memcpy (cfg, (config_get ())->defconf, sizeof (thy_mappable_config_t));

  if (!absuri)
    return cfg;

  for (i = 0; i < bhl_list_size (config->maps); i++)
    {
      int found = 0;

      bhl_list_get (config->maps, i, (void **)&map);
      if (regexec ((regex_t *)map->preg, absuri, 0,
		   NULL, 0) != REG_NOMATCH)
	found = 1;
      if (!found && file)
	{
	  if (regexec ((regex_t *)map->preg, file, 0,
		       NULL, 0) != REG_NOMATCH)
	    found = 1;
	}
      if (!found)
	{
	  free (map);
	  continue;
	}

      /* Ok, match found, set up changes... */
      if (map->config->webroot)
	cfg->webroot = map->config->webroot;
      if (map->config->default_type)
	cfg->default_type = map->config->default_type;
      if (map->config->userdir)
	cfg->userdir = map->config->userdir;
      for (j = HTTP_STATUS_100; j < HTTP_STATUS_UNKNOWN; j++)
	{
	  if (map->config->http_status[j])
	    cfg->http_status[j] = map->config->http_status[j];
	}
      if (map->config->options.vhosting != THY_BOOL_UNSET)
	cfg->options.vhosting = map->config->options.vhosting;
      if (map->config->options.auth != THY_BOOL_UNSET)
	cfg->options.auth = map->config->options.auth;
      if (map->config->options.userdir != THY_BOOL_UNSET)
	cfg->options.userdir = map->config->options.userdir;
      if (map->config->options.cgi != THY_BOOL_UNSET)
	cfg->options.cgi = map->config->options.cgi;
      if (map->config->options.usercgi != THY_BOOL_UNSET)
	cfg->options.usercgi = map->config->options.usercgi;
      if (map->config->options.casemime != THY_BOOL_UNSET)
	cfg->options.casemime = map->config->options.casemime;
      if (map->config->options.followall != THY_BOOL_UNSET)
	cfg->options.followall = map->config->options.followall;
      if (map->config->options.vary != THY_BOOL_UNSET)
	cfg->options.vary = map->config->options.vary;
      if (map->config->options.dirindex != THY_BOOL_UNSET)
	cfg->options.dirindex = map->config->options.dirindex;
      if (map->config->options.lazycgi != THY_BOOL_UNSET)
	cfg->options.lazycgi = map->config->options.lazycgi;

      if (map->config->cache.no_cache != THY_BOOL_UNSET)
	cfg->cache.no_cache = map->config->cache.no_cache;
      if (map->config->cache.no_store != THY_BOOL_UNSET)
	cfg->cache.no_store = map->config->cache.no_store;
      if (map->config->cache.no_transform != THY_BOOL_UNSET)
	cfg->cache.no_transform = map->config->cache.no_transform;
      if (map->config->cache.must_revalidate != THY_BOOL_UNSET)
	cfg->cache.must_revalidate = map->config->cache.must_revalidate;
      if (map->config->cache.do_max_age != THY_BOOL_UNSET)
	cfg->cache.do_max_age = map->config->cache.do_max_age;
      if (map->config->cache.expiry_base != THY_EXPIRY_BASE_UNKNOWN)
	cfg->cache.expiry_base = map->config->cache.expiry_base;
      cfg->cache.max_age = map->config->cache.max_age;

      if (map->config->encoding.type != ENCODING_TYPE_UNSET)
	cfg->encoding.type = map->config->encoding.type;
      if (map->config->encoding.level > -1)
	cfg->encoding.level = map->config->encoding.level;

      if (map->config->etag.dirtag != THY_BOOL_UNSET)
	cfg->etag.dirtag = map->config->etag.dirtag;
      if (map->config->etag.etag != THY_BOOL_UNSET)
	cfg->etag.etag = map->config->etag.etag;

      if (bhl_list_size (map->config->indexes) > 0)
	cfg->indexes = map->config->indexes;
      if (bhl_list_size (map->config->cgiexts) > 0)
	cfg->cgiexts = map->config->cgiexts;
      if (bhl_list_size (map->config->handlers) > 0)
	cfg->handlers = map->config->handlers;
      if (bhl_list_size (map->config->mime_types) > 0)
	cfg->mime_types = map->config->mime_types;
      if (bhl_list_size (map->config->methods) > 0)
	cfg->methods = map->config->methods;
      if (bhl_list_size (map->config->headers) > 0)
	cfg->headers = map->config->headers;
      if (bhl_list_size (map->config->env) > 0)
	cfg->env = map->config->env;
      if (bhl_list_size (map->config->cgidirs) > 0)
	cfg->cgidirs = map->config->cgidirs;
      if (bhl_list_size (map->config->aliases) > 0)
	cfg->aliases = map->config->aliases;

      free (map);
    }

  return cfg;
}
