/* -*- mode: c; c-file-style: "gnu" -*-
 * nqueue.h -- Network queue handling function prototypes.
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file nqueue.h
 * Network queue handling function prototypes.
 */

#ifndef _THY_NQUEUE_H
#define _THY_NQUEUE_H

#include "types.h"

extern long int thy_nq_max_fd;
extern long int thy_nq_act_fds;
extern int *trigger;
extern long trigger_count;
extern int *fdstates;

/** Control the properties of a FD in the network queue.
 * Controls which events are expected from a file descriptor in the
 * event queue.
 *
 * @param fd is the FD in question.
 * @param event is the event the FD is expected to present.
 * @param set controls wheter the flag is set or cleared.
 *
 * @note Only one event can be set on a single FD - we are a state
 * machine after all.
 */
void thy_nq_fd_control (int fd, thy_nq_event_t event, int set);

/** Initialise the network I/O queue.
 * @returns Zero when successful, -1 otherwise.
 */
int thy_nq_init (void);

/** Get the highest numbered FD in any of the sets.
 * @returns the highest numbered FD in any of the sets.
 */
long int thy_nq_max (void);

/** Get the number of active FDs in all our sets.
 * @returns the number of all active FDs.
 */
long int thy_nq_act (void);

/** Check if an even occurred on an FD.
 * Determines if a given event occurred on a file descriptor or not,
 * by looking at the appropriate FD set.
 *
 * @param fd is the file descriptor we're interested in.
 * @param event is the event we're looking for.
 *
 * @returns One if the event happened, zero otherwise.
 */
int thy_nq_fd_check (int fd, thy_nq_event_t event);

/** Reset the network queue.
 */
void thy_nq_reset (void);

/** Wait for events or a timeout.
 * This is basically a wrapper around whatever waiter function was
 * selected at compile time. It simply waits for any traffic on one of
 * the FDs in our sets, or times out.
 *
 * @returns whatever the waiter function would return.
 */
int thy_nq_wait (long int timeout);

/** Check if we have any pending sessions.
 * Determines if there are any sessions which are always considered to
 * have traffic.
 *
 * @returns One if there are, zero otherwise.
 */
int thy_nq_has_pending (void);
#endif
