/* -*- mode: c; c-file-style: "gnu" -*-
 * nqueue_kqueue.c - kqueue()-based event system
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "system.h"

#if defined(HAVE_SYS_EVENT_H) && defined(HAVE_KQUEUE)
#include <sys/types.h>
#include <inttypes.h>

#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <sys/event.h>
#include <sys/time.h>
#include <unistd.h>

#include "compat/compat.h"

#include "nqueue.h"
#include "nqueue_kqueue.h"
#include "types.h"
#include "thy.h"

static int kq; /**< kqueue() ID. */
static struct kevent *revents; /**< Events returned by kevent(). */
static int *rfdidx; /** FD->returned event lookup table. */

int
thy_nq_init_kqueue (void)
{
  kq = kqueue ();
  if (kq == -1)
    {
      bhc_error ("kqueue(): %s", strerror (errno));
      return -1;
    }

  revents = (struct kevent *)bhc_calloc (_THY_MAXCONN + 1,
					 sizeof (struct kevent));
  rfdidx = (int *)bhc_calloc (_THY_MAXCONN + 1, sizeof (int));

  return 0;
}

void
thy_nq_fd_control_kqueue (int fd, thy_nq_event_t event, int set)
{
  struct kevent kev;

  if (fd < 0)
    return;

  if (fdstates[fd] && thy_nq_act_fds)
    thy_nq_act_fds--;

  if (trigger[fd])
    trigger_count--;

  if (fd > thy_nq_max_fd)
    thy_nq_max_fd = fd;

  fdstates[fd] = 0;
  trigger[fd] = 0;
  kev.ident = fd;
  switch (event)
    {
    case THY_NQ_EVENT_INPUT:
      kev.filter = EVFILT_READ;
      break;
    case THY_NQ_EVENT_OUTPUT:
      kev.filter = EVFILT_WRITE;
      break;
    case THY_NQ_EVENT_TRIGGER:
      trigger_count++;
      trigger[fd] = 1;
      return;
    default:
      kev.filter = EVFILT_WRITE | EVFILT_READ;
      break;
    }

  if (set)
    {
      fdstates[fd] = 1;
      thy_nq_act_fds++;
    }

  kev.flags = (set) ? EV_ADD : EV_DELETE;

  kevent (kq, &kev, 1, NULL, 0, NULL);
}

int
thy_nq_fd_check_kqueue (int fd, thy_nq_event_t event)
{
  int ridx, rval = 0;

  if (fd > thy_nq_max_fd)
    thy_nq_max_fd = fd;

  if (event == THY_NQ_EVENT_TRIGGER && trigger[fd])
    return 1;

  ridx = rfdidx[fd];

  if ((int)revents[ridx].ident != fd)
    return 0;

  if (revents[ridx].flags & EV_ERROR)
    return 0;

  switch (event)
    {
    case THY_NQ_EVENT_INPUT:
      if (revents[ridx].filter == EVFILT_READ)
	rval = 1;
      break;
    case THY_NQ_EVENT_OUTPUT:
      if (revents[ridx].filter == EVFILT_WRITE)
	rval = 1;
      break;
    default:
      break;
    }

  return rval;
}

int
thy_nq_wait_kqueue (long int timeout)
{
  struct timespec ts;
  int i, r;

  ts.tv_sec = timeout;
  ts.tv_nsec = 0;

  errno = 0;
  r = kevent (kq, NULL, 0, revents, _THY_MAXCONN, &ts);
  if (r == -1)
    return -1;

  for (i = 0; i < r; i++)
    rfdidx[revents[i].ident] = i;

  return r;
}
#endif

/* arch-tag: c579415e-727a-4efd-ae40-e271ef8383d3 */
