/*
 * Copyright (C) 2004-2006 Jimmy Do <crispyleaves@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef GLOO_TIMER_H
#define GLOO_TIMER_H

#include <glib-object.h>
#include "gloo-enums.h"
#include "gloo-types.h"

G_BEGIN_DECLS

#define GLOO_TYPE_TIMER              (gloo_timer_get_type ())
#define GLOO_TIMER(obj)              (G_TYPE_CHECK_INSTANCE_CAST ((obj), GLOO_TYPE_TIMER, GlooTimer))
#define GLOO_TIMER_CLASS(klass)      (G_TYPE_CHECK_CLASS_CAST ((klass), GLOO_TYPE_TIMER, GlooTimerClass))
#define GLOO_IS_TIMER(obj)           (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GLOO_TYPE_TIMER))
#define GLOO_IS_TIMER_CLASS(klass)   (G_TYPE_CHECK_CLASS_TYPE ((klass), GLOO_TYPE_TIMER))
#define GLOO_TIMER_GET_CLASS(obj)    (G_TYPE_INSTANCE_GET_CLASS ((obj), GLOO_TYPE_TIMER, GlooTimerClass))

typedef struct _GlooTimer GlooTimer;
typedef struct _GlooTimerClass GlooTimerClass;
typedef struct _GlooTimerPrivate GlooTimerPrivate;

struct _GlooTimer
{
	GObject parent;
	
	GlooTimerPrivate *priv;
};

struct _GlooTimerClass
{
	GObjectClass parent;
	
	guint state_changed_signal_id;
	guint time_changed_signal_id;
};

GType gloo_timer_get_type (void);

/**
 * Sets the number of seconds that the timer
 * should count down from. The timer defaults to zero
 * seconds if this method is never invoked.
 * This may only be invoked when the timer is in the
 * IDLE state.
 */
void
gloo_timer_set_duration (GlooTimer *self, guint seconds);

/**
 * Returns the number of seconds specified by the
 * last invocation of gloo_timer_set_duration().
 * Returns zero if gloo_timer_get_duration() has
 * not been invoked.
 * This may be invoked when the timer is in any state.
 */
guint
gloo_timer_get_duration (GlooTimer *self);

/**
 * Sets an arbitrary name that can be associated
 * with the timer. The timer defaults to an empty string
 * if this method is never invoked.
 * This method creates a copy of the given name. The caller
 * is responsible for freeing the original string.
 * This may only be invoked when the timer is in the
 * IDLE state.
 */
void
gloo_timer_set_name (GlooTimer *self, const gchar *name);

/**
 * Returns the name specified by the last invocation
 * of gloo_time_set_name().
 * Returns an empty string if gloo_timer_set_name()
 * has not been invoked.
 * This method returns a pointer to the internally-allocated
 * name string. The caller should NOT free the returned string.
 * This may be invoked when the timer is in any state.
 */
const gchar *
gloo_timer_get_name (GlooTimer *self);

/**
 * If the timer is idle, this will start it counting
 * down from the number of seconds specified in the
 * last invocation of gloo_timer_set_duration().
 * If the timer is paused, this will continue counting
 * down from the remaining number of seconds.
 * This may only be invoked when the timer is in either
 * the IDLE or the PAUSED state.
 */
void
gloo_timer_start (GlooTimer *self);

/**
 * Stops the timer, keeping the current number of remaining
 * seconds. Use gloo_timer_start() to continue countdown.
 * Use gloo_timer_reset() to reset the remaining seconds back to
 * the value specified by the last call to gloo_timer_set_duration().
 * This may only be invoked when the timer is in the RUNNING state.
 */
void
gloo_timer_stop (GlooTimer *self);

/**
 * Brings the timer back to the idle state and sets the
 * remaining seconds back to the value specified by the'
 * last invocation of gloo_timer_set_duration().
 * This may be invoked when the timer is in any state.
 */
void
gloo_timer_reset (GlooTimer *self);

/**
 * Returns the timer's state.
 * This may be invoked when the timer is in any state.
 */
GlooTimerState
gloo_timer_get_state (GlooTimer *self);

/**
 * Returns the remaining time in seconds.
 * This may be invoked when the timer is in any state,
 * except IDLE.
 */
guint
gloo_timer_get_remaining_time (GlooTimer *self);

G_END_DECLS

#endif /* GLOO_TIMER_H */
