
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>

#include <tinysnmp/manager/udp.h>

#ifdef DEBUG
#include <debug/log.h>
#include <debug/hex.h>
#endif	/* #ifndef DEBUG */

/* select non-blocking I/O */
static int set_nonblock (int fd)
{
   int flags;

   if ((flags = fcntl (fd,F_GETFL)) < 0 || fcntl (fd,F_SETFL,flags | O_NONBLOCK) < 0)
	 return (-1);

   return (0);
}

/*
 * initialize udp data structure.
 *
 *     addr          address/port of destination host
 *     timeout       timeout for operations
 */
void udp_open (udp_t *udp,const struct sockaddr_in *addr,uint32_t timeout)
{
   memcpy (&udp->addr,addr,sizeof (struct sockaddr_in));
   udp->fd = -1;
   udp->timeout = timeout;
}

/*
 * clean up. this also closes the connection if you're currently
 * connected to a host.
 */
void udp_close (udp_t *udp)
{
   udp_disconnect (udp);
}

/*
 * disconnect.
 */
void udp_disconnect (udp_t *udp)
{
   if (udp->fd > 0)
	 {
		close (udp->fd);
		udp->fd = -1;
	 }
}

/*
 * connect to the host specified in udp_open(). return -1 if some
 * error occurred, 0 if successful. check errno to see what error
 * occurred.
 */
int udp_connect (udp_t *udp)
{
   int saved,result;
   socklen_t len;

   if (udp->fd > 0)
	 {
		errno = EBUSY;
		return (-1);
	 }

   /* create a socket for communication */
   if ((udp->fd = socket (PF_INET,SOCK_DGRAM,IPPROTO_UDP)) < 0)
	 return (-1);

   /* force non-blocking I/O on the socket if timeout is not zero */
   if (udp->timeout && set_nonblock (udp->fd) < 0)
	 goto error;

   /* initiate a connection on that socket */
   result = connect (udp->fd,(const struct sockaddr *) &udp->addr,sizeof (udp->addr));

   /* if timeout is not zero and errno == EINPROGRESS, we have to poll for completion using select() */
   if (udp->timeout && result < 0 && errno == EINPROGRESS)
	 {
		fd_set wfds;
		struct timeval tv;
		int errval;

		/* watch fd to see whether the socket is writable */
		FD_ZERO (&wfds);
		FD_SET (udp->fd,&wfds);

		/* set timeout */
		tv.tv_sec = udp->timeout / 1000;
		tv.tv_usec = (udp->timeout % 1000) * 1000;

		if ((result = select (udp->fd + 1,NULL,&wfds,NULL,&tv)) < 0)
		  goto error;

		if (result && FD_ISSET (udp->fd,&wfds))
		  {
			 len = sizeof (errval);
			 if (getsockopt (udp->fd,SOL_SOCKET,SO_ERROR,(void *) &errval,&len) < 0)
			   goto error;
			 if (!errval) return (0);
			 errno = errval, result = -1;
		  }
		else errno = ETIMEDOUT, result = -1;
	 }

   if (!result) return (0);

error:
   saved = errno;
   udp_disconnect (udp);
   errno = saved;
   return (-1);
}

static uint32_t elapsed (const struct timeval *then)
{
   struct timeval now;

   if (gettimeofday (&now,NULL) < 0) return (0xffffffff);

   now.tv_sec -= then->tv_sec;
   now.tv_usec -= then->tv_usec;

   while (now.tv_usec < 0) now.tv_sec--, now.tv_usec += 1000000;

   if (now.tv_sec < 0) return (0xffffffff);

   return (now.tv_sec / 1000 + now.tv_usec * 1000);
}

/*
 * send a single packet.
 *
 *     buf   where the packet data is stored
 *     len   length of packet (in bytes)
 *
 * return -1 if an error occurred, 0 if successful. check errno
 * to see what error occurred.
 */
int udp_send (const udp_t *udp,const uint8_t *buf,uint16_t len)
{
   int retval,flags = 0;
   struct timeval tv,now;
   fd_set wfds;
   int64_t timeout;

   if (udp->fd <= 0)
	 {
		errno = EINVAL;
		return (-1);
	 }

#ifdef MSG_NOSIGNAL
   flags |= MSG_NOSIGNAL;
#endif	/* #ifdef MSG_NOSIGNAL */

#if defined(MSG_CONFIRM) && !defined(COMPAT22)
   flags |= MSG_CONFIRM;	/* this is good, but only Linux 2.3+ supports it (see send(2)) */
#endif	/* #if defined(MSG_CONFIRM) && !defined(COMPAT22) */

   retval = send (udp->fd,buf,len,flags);

   /* if timeout is not zero and errno == EAGAIN, we have to poll for completion using select() */
   if (udp->timeout && retval < 0 && (errno == EAGAIN || errno == EWOULDBLOCK))
	 {
		timeout = udp->timeout;
		while (retval < 0 && (errno == EAGAIN || errno == EWOULDBLOCK))
		  {
			 /* watch fd to see whether the socket is writable */
			 FD_ZERO (&wfds);
			 FD_SET (udp->fd,&wfds);

			 /* remember the time */
			 if (gettimeofday (&now,NULL) < 0) return (-1);

			 /* set timeout */
			 tv.tv_sec = timeout / 1000;
			 tv.tv_usec = (timeout % 1000) * 1000;

			 if ((retval = select (udp->fd + 1,NULL,&wfds,NULL,&tv)) < 0)
			   return (-1);

			 if (!retval || !FD_ISSET (udp->fd,&wfds))
			   {
				  errno = ETIMEDOUT;
				  return (-1);
			   }

			 if ((retval = send (udp->fd,buf,len,flags)) < 0)
			   {
				  timeout -= elapsed (&now);
				  if (timeout <= 0)
					{
					   errno = ETIMEDOUT;
					   return (-1);
					}
			   }
		  }
	 }

   if (retval < 0) return (-1);

   if (retval != len)
	 {
		errno = EIO;
		return (-1);
	 }

#ifdef DEBUG
   log_printf (LOG_DEBUG,"Send:\n");
   hexdump (LOG_DEBUG,buf,len);
#endif	/* #ifdef DEBUG */

   return (0);
}

/*
 * receive a single packet.
 *
 *     buf   where the packet data is stored
 *     len   when the function is called len should be set to
 *           the size of the buffer (in bytes). when the function
 *           returns, len will be adjusted to the size of received
 *           packet (in bytes).
 *
 * return -1 if an error occurred, 0 if successful. check errno
 * to see what error occurred.
 */
int udp_receive (const udp_t *udp,uint8_t *buf,uint16_t *len)
{
   int retval,flags = MSG_WAITALL;
   struct timeval tv,now;
   fd_set rfds;
   int64_t timeout;

   if (udp->fd <= 0)
	 {
		errno = EINVAL;
		return (-1);
	 }

#ifdef MSG_NOSIGNAL
   flags |= MSG_NOSIGNAL;
#endif	/* #ifdef MSG_NOSIGNAL */

   retval = recv (udp->fd,buf,*len,flags);

   /* if timeout is not zero and errno == EAGAIN, we have to poll for completion using select() */
   if (udp->timeout && retval < 0 && (errno == EAGAIN || errno == EWOULDBLOCK))
	 {
		timeout = udp->timeout;
		while (retval < 0 && (errno == EAGAIN || errno == EWOULDBLOCK))
		  {
			 /* watch fd to see whether the socket is readable */
			 FD_ZERO (&rfds);
			 FD_SET (udp->fd,&rfds);

			 /* remember the time */
			 if (gettimeofday (&now,NULL) < 0) return (-1);

			 /* set timeout */
			 tv.tv_sec = timeout / 1000;
			 tv.tv_usec = (timeout % 1000) * 1000;

			 if ((retval = select (udp->fd + 1,&rfds,NULL,NULL,&tv)) < 0)
			   return (-1);

			 if (!retval || !FD_ISSET (udp->fd,&rfds))
			   {
				  errno = ETIMEDOUT;
				  return (-1);
			   }

			 if ((retval = recv (udp->fd,buf,*len,flags)) < 0)
			   {
				  timeout -= elapsed (&now);
				  if (timeout <= 0)
					{
					   errno = ETIMEDOUT;
					   return (-1);
					}
			   }
		  }
	 }

   if (retval < 0) return (-1);

   if (retval > *len)
	 {
		errno = EIO;
		return (-1);
	 }

   *len = retval;

#ifdef DEBUG
   log_printf (LOG_DEBUG,"Receive:\n");
   hexdump (LOG_DEBUG,buf,*len);
#endif	/* #ifdef DEBUG */

   return (0);
}

