
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <ctype.h>

#include <debug/log.h>
#include <ber/ber.h>
#include <abz/typedefs.h>
#include <tinysnmp/tinysnmp.h>

#include "show.h"

#define PRINTABLE(x) (isprint(x) || isspace(x))

#define priv_printf(fmt,args...) log_printf_stub(filename,line,function,level,fmt,##args)
#define priv_puts(s) log_puts_stub(filename,line,function,level,s)
#define priv_putc(c) log_putc_stub(filename,line,function,level,c)

void show_stub (const char *filename,int line,const char *function,const uint32_t *oid,uint8_t type,const snmp_data_t *data)
{
   uint32_t i;
   int raw = 0,level = type == BER_NULL ? LOG_WARNING : LOG_NORMAL;

   priv_printf ("%" PRIu32 ".%" PRIu32,oid[1] / 40,oid[1] % 40);
   for (i = 2; i <= oid[0]; i++) priv_printf (".%" PRIu32,oid[i]);

   priv_puts (" = ");

   switch (type)
	 {
	  case BER_INTEGER:
		priv_printf ("INTEGER %" PRId32 "\n",data->INTEGER);
		break;
	  case BER_Counter32:
		priv_printf ("Counter32 %" PRIu32 "\n",data->Counter32);
		break;
	  case BER_Gauge32:
		priv_printf ("Gauge32 %" PRIu32 "\n",data->Gauge32);
		break;
	  case BER_TimeTicks:
		priv_printf ("TimeTicks %" PRIu32 "\n",data->TimeTicks);
		break;
	  case BER_Counter64:
		priv_printf ("INTEGER %" PRIu64 "\n",data->Counter64);
		break;
	  case BER_OID:
		priv_printf ("OBJECT IDENTIFIER %" PRIu32 ".%" PRIu32,data->OID[1] / 40,data->OID[1] % 40);
		for (i = 2; i <= data->OID[0]; i++) priv_printf (".%" PRIu32,data->OID[i]);
		priv_putc ('\n');
		break;
	  case BER_OCTET_STRING:
		priv_puts ("OCTET STRING");
		for (i = 0; i < data->OCTET_STRING.len && !raw; i++)
		  if (!PRINTABLE (data->OCTET_STRING.buf[i]))
			raw = 1;
		if (!raw)
		  {
			 priv_putc (' ');
			 for (i = 0; i < data->OCTET_STRING.len; i++) priv_putc (data->OCTET_STRING.buf[i]);
		  }
		else
		  {
			 for (i = 0; i < data->OCTET_STRING.len; i++)
			   priv_printf (" %.2x",data->OCTET_STRING.buf[i]);
		  }
		priv_putc ('\n');
		break;
	  case BER_NULL:
		priv_puts ("No Such Name\n");
		break;
	  case BER_IpAddress:
		priv_printf ("IpAddress %u.%u.%u.%u\n",NIPQUAD (data->IpAddress));
		break;
	  default:
		priv_puts ("(unknown)\n");
	 }
}

