# $Id: login.tcl,v 1.55 2003/12/13 21:35:26 aleksey Exp $


if {![info exists use_tls]} {
    set use_tls 1
}
if {$use_tls && [catch { package require tls 1.4 }]} {
    debugmsg login "unable to load the TLS package, so no SSL support!

The TLS package is available at http://tls.sf.net/"

    set use_tls 0
}
if {!$use_tls} {
    set loginconf(usessl) 0
}

set loginconf(usesasl) $jlib::lib(have_sasl)

custom::defgroup Warnings [::msgcat::mc "Warning display options."] \
    -group Tkabber

custom::defvar tls_warnings 1 [::msgcat::mc "Display SSL warnings."] \
    -group Warnings -type boolean

custom::defgroup Login \
    [::msgcat::mc "Login options."] \
    -group Tkabber

custom::defvar loginconf(user) "" \
    [::msgcat::mc "User name."] \
    -group Login -type string
custom::defvar loginconf(password) "" \
    [::msgcat::mc "Password."] \
    -group Login -type password
custom::defvar loginconf(usedigest) 1 \
    [::msgcat::mc "Use hashed password transmission."] \
    -group Login -type boolean
custom::defvar loginconf(resource) "tkabber" \
    [::msgcat::mc "Resource."] \
    -group Login -type string
custom::defvar loginconf(server) "localhost" \
    [::msgcat::mc "Server name."] \
    -group Login -type string
custom::defvar loginconf(port) "5222" \
    [::msgcat::mc "Server port."] \
    -group Login -type string
custom::defvar loginconf(priority) "8" \
    [::msgcat::mc "Priority."] \
    -group Login -type integer
custom::defvar loginconf(connect_forever) 0 \
    [::msgcat::mc "Retry to connect forever."] \
    -group Login -type boolean
custom::defvar loginconf(usessl) $loginconf(usessl) \
    [::msgcat::mc "Use SSL to connect to server."] \
    -group Login -type boolean
custom::defvar loginconf(sslcertfile) "" \
    [::msgcat::mc "SSL certificate file (optional)."] \
    -group Login -type string
custom::defvar loginconf(sslcafile) "" \
    [::msgcat::mc "SSL CA file (optional)."] \
    -group Login -type string
#custom::defvar loginconf(sslkeyfile) "" \
#    [::msgcat::mc "SSL private key file (optional)."] \
#    -group Login -type string
custom::defvar loginconf(sslport) "5223" \
    [::msgcat::mc "SSL port."] \
    -group Login -type string
custom::defvar loginconf(useproxy) 0 \
    [::msgcat::mc "Use HTTP proxy to connect."] \
    -group Login -type boolean
custom::defvar loginconf(httpproxy) "localhost" \
    [::msgcat::mc "HTTP proxy address."] \
    -group Login -type string
custom::defvar loginconf(httpproxyport) 3128 \
    [::msgcat::mc "HTTP proxy port."] \
    -group Login -type string
custom::defvar loginconf(httplogin) "" \
    [::msgcat::mc "HTTP proxy username."] \
    -group Login -type string
custom::defvar loginconf(httppassword) "" \
    [::msgcat::mc "HTTP proxy password."] \
    -group Login -type password
custom::defvar loginconf(usealtserver) 0 \
    [::msgcat::mc "Use explicitly-specified server address."] \
    -group Login -type boolean
custom::defvar loginconf(altserver) "" \
    [::msgcat::mc "Server name or IP-address."] \
    -group Login -type string
custom::defvar loginconf(replace_opened) 1 \
    [::msgcat::mc "Replace opened connections."] \
    -group Login -type boolean
custom::defvar loginconf(usehttppoll) 0 \
    [::msgcat::mc "Use HTTP poll connection method."] \
    -group Login -type boolean
custom::defvar loginconf(pollurl) "http://connect.jabber.cz/" \
    [::msgcat::mc "URL to connect to."] \
    -group Login -type string
custom::defvar loginconf(usepollkeys) 0 \
    [::msgcat::mc "Use HTTP poll client security keys (recommended)."] \
    -group Login -type boolean
custom::defvar loginconf(pollmin) 3000 \
    [::msgcat::mc "Minimum Poll Interval."] \
    -group Login -type integer
custom::defvar loginconf(pollmax) 30000 \
    [::msgcat::mc "Maximum Poll Interval."] \
    -group Login -type integer
custom::defvar loginconf(usesasl) $loginconf(usesasl) \
    [::msgcat::mc "Use SASL authentification."] \
    -group Login -type boolean

custom::defvar reasonlist {} [::msgcat::mc "List of logout reasons."] \
	-group Hidden

package require http

proc login {} {
    global loginconf
    global login_after_time
    after cancel login
    debugmsg login "Starting login"
    if {[catch {login_connect} connid] > 0} {
	# Nasty thing has happened.
	debugmsg login "Failed to connect: $connid"
	if {$loginconf(connect_forever)} {
	    login_retry
	} else {
	    set res [MessageDlg .connect_err -aspect 50000 -icon error \
		-message [format [::msgcat::mc "Failed to connect: %s"] $connid] \
		-type user -buttons [list abort [::msgcat::mc "Keep trying"]] \
		-default 0 -cancel 0]
	    if {$res} {
		set loginconf(connect_forever) 1
		login_retry
	    }
	}
	return
    }
    # OK, connected.
    debugmsg login "Connect successful"
    set login_after_time 15000
    login_login $connid
}

proc login_retry {} {
    global login_after_time
    if {![info exists login_after_time]} {set login_after_time 15000}
    if {$login_after_time < 1800000} {
	# 1800000 == 30 * 60 * 1000 == 30min
	# the sequence goes: 30s, 1min, 2min, 4min, 8min, 16min, 32min, 32min...
	set login_after_time [expr {$login_after_time * 2}]
    }
    debugmsg login "Scheduling connect retry in ${login_after_time}ms"
    after $login_after_time login
}

proc tls_callback {sock args} {
    global tls_result tls_warnings
    global ssl_certificate_fields
    global tls_warning_info

    switch -- [lindex $args 0] {
	info {
	    set_status [join [lrange $args 2 end] " "]
	}

	verify {
	    if {[cequal [set reason [lindex $args 5]] ""]} {
		return 1
	    }
	    set info [::msgcat::mc [string totitle $reason 0 0]]
	    append tls_warning_info($sock) "$info\n"
	    if {!$tls_warnings} {
		return 1
	    }
	    append info [::msgcat::mc ". Proceed?\n\n"]
	    foreach {k v} [lindex $args 3] {
		if {![cequal $v ""] && [info exists ssl_certificate_fields($k)]} {
		    append info [format "%s: %s\n" $ssl_certificate_fields($k) $v]
		}
	    }

	    set blocking [fconfigure [set fd [lindex $args 1]] -blocking]
	    fconfigure $fd -blocking 1
	    set readable [fileevent $fd readable]
	    fileevent $fd readable {}

	    set res [MessageDlg .tls_callback -aspect 50000 -icon warning \
			        -type user -buttons {yes no} -default 1 \
			        -cancel 1 \
			        -message [string trim $info]]

	    fileevent $fd readable $readable
	    fconfigure $fd -blocking $blocking

	    if {$res} {
		set res 0
	    } else {
		set res 1
	    }
	    return $res
	}

	error {
	    set tls_result [join [lrange $args 2 end] " "]
	}

	default {
	}
    }
}

proc tls_handshake {fd} {
    global tls_count
    global tls_failures
    global tls_result

    incr tls_count
    if {[eof $fd]} {
	set tls_result "EOF during TLS handshake"
	fileevent $fd readable {}
	fileevent $fd writable {}
	return
    }
    flush $fd
    set thrown [catch { tls::handshake $fd } shook]
    if {(!$thrown) && ($shook)} {
	set tls_result ""
    } elseif {[incr tls_failures] > 15} {
	set tls_result $shook
    }
}

proc login_connect {} {
    global use_tls
    global loginconf
    global tls_warning_info

    if {!$loginconf(usehttppoll)} {
	if {!$loginconf(useproxy)} {
	    if {$loginconf(usealtserver)} {
		set server $loginconf(altserver)
	    } else {
		set server $loginconf(server)
	    }
	    if {$use_tls && $loginconf(usessl)} {
		global tls_count
		global tls_failures
		global tls_result

	        set sock [socket $server $loginconf(sslport)]
		set tls_warning_info($sock) {}
		set args [list -command [list [namespace current]::tls_callback $sock] \
			       -ssl2    false                                          \
			       -ssl3    true                                           \
			       -tls1    true                                           \
			       -request true                                           \
			       -require false                                          \
			       -server  false]
		if {![cequal $loginconf(sslcertfile) ""]} {
		    if {[cequal $loginconf(sslcafile) ""]} {
		        lappend args -cafile $loginconf(sslcertfile)
		    } else {
		      lappend args -cafile $loginconf(sslcafile)
		    }
		}
		eval [list tls::import $sock] $args

		fconfigure $sock -encoding binary -translation binary

		fileevent $sock readable \
			  [list [namespace current]::tls_handshake $sock]
		fileevent $sock writable {}

		set tls_count 0
		set tls_failures 0
		catch { unset tls_result }
		tls_handshake $sock
		while {![info exists tls_result]} {
		    vwait tls_count
		}

		fileevent $sock readable {}

		if {![cequal $tls_result ""]} {
		    catch { close $sock }
		    error $tls_result
		}
	    } else {
	        set sock [socket $server $loginconf(port)]
	    }
	} else {
	    set sock [connect_httpproxy]
	}

	return [jlib::connect $sock $loginconf(server) \
			-newconnection [expr {!$loginconf(replace_opened)}] \
			-user $loginconf(user) \
			-usesasl $loginconf(usesasl)]
    } else {
	if $loginconf(useproxy) {
	    set Proxy(use)      1
	    ::http::config -proxyhost $loginconf(httpproxy) -proxyport $loginconf(httpproxyport)

	    if {$loginconf(httplogin) != ""} {
		set auth [base64::encode \
                              [encoding convertto "$loginconf(httplogin):$loginconf(httppassword)"]]
		set Proxy(auth) [list "Proxy-Authorization" "Basic $auth"]
	    } else {
		set Proxy(auth) {}
	    }
	} else {
	    set Proxy(use)      0
	    set Proxy(auth)     {}
	}
	
	return [jlib::connect {} $loginconf(server) \
			-newconnection [expr {!$loginconf(replace_opened)}] \
			-user $loginconf(user) \
			-httppoll 1 \
			-pollint $loginconf(pollmin) \
			-pollmin $loginconf(pollmin) \
			-pollmax $loginconf(pollmax) \
			-pollurl $loginconf(pollurl) \
			-proxy $Proxy(use) \
			-proxyauth $Proxy(auth) \
			-usesasl $loginconf(usesasl)]
    }
}
    
proc login_login {connid} {
    global loginconf
    global loginconf_hist_$connid
    global gr_nick gr_server gra_server
    global auth_result

    set gr_nick $loginconf(user)
    set gr_server conference.$loginconf(server)
    set gra_server conference.$loginconf(server)

    array set loginconf_hist_$connid [array get loginconf]

    jlib::wait_for_stream $connid

    if {$loginconf(usedigest)} {
	set autht digest
    } else {
	set autht plain
    }

    lassign [jlib::get_authtypes $loginconf(user) $connid] res data

    switch -- $res {
	ERR {
	    clear_status
	    recv_auth_result $connid ERR $data
	    return
	}
	OK {
	    set authtype ""
	    foreach at [list $autht digest plain] {
		if {[lcontain $data $at]} {
		    set authtype $at
		    break
		}
	    }

	    if {[cequal $authtype ""]} {
		MessageDlg .auth_err -aspect 50000 -icon error \
		    -message [::msgcat::mc "Can't authenticate: Remote server doesn't support\nplain or digest authentication method"] \
		    -type user -buttons ok -default 0 -cancel 0
		return
	    }
	    if {[cequal $authtype plain] && [cequal $autht digest]} {
		set res [MessageDlg .auth_err -aspect 50000 -icon warning \
		    -message [::msgcat::mc "Warning: Remote server doesn't support\nhashed password authentication.\n\nProceed with PLAINTEXT authentication?"] \
		    -type user -buttons {yes no} -default 0 -cancel 1]
		if {$res} {
		    return
		}
	    }
	}
	SASL {
	    set authtype ""
	}
    }

    jlib::send_auth \
	$loginconf(user) $loginconf(password) $loginconf(resource) \
	[list recv_auth_result $connid] $authtype $connid
    vwait auth_result($connid)

    if {$auth_result($connid) == "OK"} {
        connected $connid
    }
}

set reconnect_retries 0

proc logout {{connid {}}} {
    global reconnect_retries

    after cancel login

    jlib::disconnect $connid
    if {$connid == {}} {
	roster::clean
    } else {
	roster::clean_connection $connid
    }

    disconnected $connid

    set reconnect_retries 0
}

proc client:disconnect {connid} {
    logout $connid
}

# TODO
proc client:reconnect {connid} {
    global reconnect_retries

    puts "RECONNECT $connid"
    roster::clean_connection $connid

    if {[jlib::connections] == {}} {
	set_status "Disconnected"
    }

    disconnected $connid
    if {[incr reconnect_retries] <= 3} {
        after 1000 \
	    "array set loginconf \[array get loginconf_hist_$connid\]; login"
    }
}

proc connected {connid} {
    global use_tls
    global loginconf
    global userstatus textstatus
    global ssl_ind

    if {$use_tls} {
	set ::use_ssl($connid) $loginconf(usessl)
	update_ssl_ind
    }

    set ::main_window_title \
	"$loginconf(user)@$loginconf(server)/$loginconf(resource) - Tkabber"
    wm title . $::main_window_title
    wm iconname . $::main_window_title

    jlib::roster_get -command client:roster_cmd -connection $connid
    .presence.button configure -state normal
    #set textstatus ""
    #if {[cequal $userstatus unavailable]} {
    #    set userstatus available
    #}
    #set userstatus $userstatus
    send_first_presence $connid
    hook::run connected_hook $connid
}

# TODO
proc disconnected {connid} {
    global use_tls
    global curuserstatus userstatusdesc
    global ssl_ind

    if {$use_tls} {
	catch { unset ::use_ssl($connid) }
	update_ssl_ind
    }

    set ::main_window_title Tkabber
    wm title . $::main_window_title
    wm iconname . $::main_window_title

    if {[jlib::connections] == {}} {
	set curuserstatus unavailable
	set userstatusdesc [::msgcat::mc "Not logged in"]
	.presence.button configure -state disabled
	hook::run change_our_presence_post_hook unavailable
    }
    hook::run disconnected_hook $connid
}

proc update_login_entries {l {i 0}} {
    global ltmp

    if {$i} {
	array set ltmp [array get ::loginconf$i]
    }
    foreach ent {username server port password resource priority \
	    altserver sslport httpproxy httpproxyport httplogin httppassword \
	    sslcertfile pollurl} {
	if {[winfo exists $l.$ent]} {
	    catch { $l.$ent icursor end }
	}
    }
    foreach {check enable disable} { \
	    usehttppoll {lpollurl pollurl} {usessl} \
	    usessl {sslport lsslport} {} \
	    usealtserver {altserver laltserver} {} \
	    useproxy {httpproxy httpproxyport httplogin httppassword \
	    lhttpproxy lhttpproxyport lhttplogin lhttppassword} {} \
	    } {
	if {![info exists ltmp($check)] || ![winfo exists $l.$check]} {
	    continue
	}

	if {$ltmp($check) && ![cequal [$l.$check cget -state] disabled]} {
	    set state1 normal
	    set state2 disabled
	} else {
	    set state1 disabled
	    set state2 normal
	}
	foreach ent $enable {
	    if {[winfo exists $l.$ent]} {
		$l.$ent configure -state $state1
		if {[cequal [focus] $l.$ent] && [cequal $state "disabled"]} {
		    focus [Widget::focusPrev $l.$ent]
		}
	    }
	}
	foreach ent $disable {
	    if {[winfo exists $l.$ent]} {
		$l.$ent configure -state $state2
		if {[cequal [focus] $l.$ent] && [cequal $state "disabled"]} {
		    focus [Widget::focusPrev $l.$ent]
		}
	    }
	}
    }
}

proc show_login_dialog {} {
    global use_tls
    global loginconf
    global ltmp
    #global tmpusername tmppassword tmpresource tmpserver tmpport tmppriority
    #global tmpusessl tmpsslport tmpuseproxy tmphttpproxy tmphttpproxyport

    if {[winfo exists .login]} {
	focus -force .login
	return
    }

    array set ltmp [array get loginconf]

    Dialog .login -title [::msgcat::mc "Login"] \
	-separator 1 -anchor e -default 0 -cancel 1

    wm resizable .login 0 0

    set l [.login getframe]

    set n 1
    while {[info exists ::loginconf$n]} {incr n}
    incr n -1

    if {$n} {
	menubutton $l.profiles -text [::msgcat::mc Profiles] -relief raised \
	    -menu $l.profiles.menu
	set m [menu $l.profiles.menu -tearoff 0]
	for {set i 1} {$i <= $n} {incr i} {
	    if {[info exists ::loginconf${i}(profile)]} {
		set lab [set ::loginconf${i}(profile)]
	    } else {
		set lab "[::msgcat::mc Profile] $i"
	    }
	    if {$i <= 10} {
		set j [expr {$i % 10}]
		$m add command -label $lab -accelerator "Ctrl-$j" \
		    -command [list update_login_entries $l $i]
		bind .login <Control-Key-$j> [list update_login_entries $l $i]
	    } else {
		$m add command -label $lab \
		    -command [list update_login_entries $l $i]
	    }
	}

	grid $l.profiles -row 0 -column 0 -sticky e
    }

    set nb [NoteBook $l.nb]

    set account_page [$nb insert end account_page -text [::msgcat::mc "Account"]]

    label $l.lusername -text [::msgcat::mc "Username:"]
    entry $l.username -textvariable ltmp(user)
    label $l.lserver -text [::msgcat::mc "Server:"]
    entry $l.server -textvariable ltmp(server)
    label $l.lpassword -text [::msgcat::mc "Password:"]
    entry $l.password -show * -textvariable ltmp(password)
    checkbutton $l.usedigest -text [::msgcat::mc "Use hashed password"] \
	-variable ltmp(usedigest)
    label $l.lresource -text [::msgcat::mc "Resource:"]
    entry $l.resource -textvariable ltmp(resource)
    label $l.lpriority -text [::msgcat::mc "Priority:"]
    Spinbox $l.priority -1000 1000 1 ltmp(priority)

    grid $l.lusername -row 0 -column 0 -sticky e -in $account_page
    grid $l.username  -row 0 -column 1 -sticky ew -in $account_page
    grid $l.lserver   -row 0 -column 2 -sticky e -in $account_page
    grid $l.server    -row 0 -column 3 -sticky ew -in $account_page
    grid $l.lpassword -row 1 -column 0 -sticky e -in $account_page
    grid $l.password  -row 1 -column 1 -sticky ew -in $account_page
    grid $l.usedigest -row 1 -column 3 -sticky w -in $account_page
    grid $l.lresource -row 2 -column 0 -sticky e -in $account_page
    grid $l.resource  -row 2 -column 1 -sticky ew -in $account_page
    grid $l.lpriority -row 2 -column 2 -sticky e -in $account_page
    grid $l.priority  -row 2 -column 3 -sticky ew -in $account_page

    grid columnconfigure $account_page 1 -weight 3
    grid columnconfigure $account_page 2 -weight 1
    grid columnconfigure $account_page 3 -weight 3

    set connection_page [$nb insert end connection_page -text [::msgcat::mc "Connection"]]

    label $l.lport -text [::msgcat::mc "Server Port:"]
    Spinbox $l.port 0 65535 1 ltmp(port)
    checkbutton $l.usealtserver -text [::msgcat::mc "Connect via alternate server"] \
	-variable ltmp(usealtserver) -command [list update_login_entries $l]
    label $l.laltserver -text [::msgcat::mc "Server:"]
    entry $l.altserver -textvariable ltmp(altserver)

    grid $l.lport     -row 0 -column 0 -sticky e -in $connection_page
    grid $l.port      -row 0 -column 1 -sticky w -in $connection_page
    grid $l.usealtserver -row 1 -column 0 -sticky w -columnspan 2 -in $connection_page
    grid $l.laltserver -row 1 -column 2 -sticky e -in $connection_page
    grid $l.altserver -row 1 -column 3 -sticky ew -in $connection_page

    checkbutton $l.replace -text [::msgcat::mc "Replace opened connections"] \
	-variable ltmp(replace_opened)
    grid $l.replace   -row 3 -column 0 -sticky w -columnspan 3 -in $connection_page

    grid columnconfigure $connection_page 1 -weight 1
    grid columnconfigure $connection_page 3 -weight 2

    if {$use_tls} {
	set ssl_page [$nb insert end ssl_page -text [::msgcat::mc "SSL"]]

	checkbutton $l.usessl -text [::msgcat::mc "Use SSL"] \
	    -variable ltmp(usessl) -command [list update_login_entries $l]
	label $l.lsslport -text [::msgcat::mc "SSL Port:"]
	Spinbox $l.sslport 0 65535 1 ltmp(sslport)
	label $l.lsslcertfile -text [::msgcat::mc "SSL Certificate:"] -state disabled
	entry $l.sslcertfile -textvariable ltmp(sslcertfile) -state disabled
	button $l.bsslcertfile -text [::msgcat::mc "Browse..."] -state disabled \
	    -command [list eval set ltmp(sslcertfile) {[tk_getOpenFile]}]


	grid $l.usessl       -row 0 -column 0 -sticky w -columnspan 3 -in $ssl_page
	grid $l.lsslport     -row 1 -column 0 -sticky e -in $ssl_page
	grid $l.sslport      -row 1 -column 1 -sticky w -in $ssl_page
	grid $l.lsslcertfile -row 2 -column 0 -sticky e -in $ssl_page
	grid $l.sslcertfile  -row 2 -column 1 -sticky ew -in $ssl_page
	grid $l.bsslcertfile -row 2 -column 2 -sticky w -in $ssl_page

	grid columnconfigure $ssl_page 0 -weight 1
	grid columnconfigure $ssl_page 1 -weight 6
	grid columnconfigure $ssl_page 2 -weight 1
    }

    set proxy_page [$nb insert end proxy_page -text [::msgcat::mc "Proxy"]]

    checkbutton $l.useproxy -text [::msgcat::mc "Use Proxy"] \
	-variable ltmp(useproxy) -command [list update_login_entries $l]
    grid $l.useproxy -row 0 -column 0 -sticky w -columnspan 3 -in $proxy_page

    label $l.lhttpproxy -text [::msgcat::mc "Proxy Server:"]
    entry $l.httpproxy -textvariable ltmp(httpproxy)
    label $l.lhttpproxyport -text [::msgcat::mc "Proxy Port:"]
    Spinbox $l.httpproxyport 0 65535 1 ltmp(httpproxyport)

    grid $l.lhttpproxy     -row 1 -column 0 -sticky e -in $proxy_page
    grid $l.httpproxy      -row 1 -column 1 -sticky ew -in $proxy_page
    grid $l.lhttpproxyport -row 1 -column 2 -sticky e -in $proxy_page
    grid $l.httpproxyport  -row 1 -column 3 -sticky ew -in $proxy_page

    label $l.lhttplogin -text [::msgcat::mc "Proxy Login:"]
    ecursor_entry [entry $l.httplogin -textvariable ltmp(httplogin)]
    label $l.lhttppassword -text [::msgcat::mc "Proxy Password:"]
    ecursor_entry [entry $l.httppassword -show * -textvariable ltmp(httppassword)]

    grid $l.lhttplogin    -row 2 -column 0 -sticky e -in $proxy_page
    grid $l.httplogin     -row 2 -column 1 -sticky ew -in $proxy_page
    grid $l.lhttppassword -row 2 -column 2 -sticky e -in $proxy_page
    grid $l.httppassword  -row 2 -column 3 -sticky ew -in $proxy_page

    set httppoll_page [$nb insert end httpoll_page -text [::msgcat::mc "HTTP Poll"]]

    checkbutton $l.usehttppoll -text [::msgcat::mc "Connect via HTTP polling"] \
	-variable ltmp(usehttppoll) -command [list update_login_entries $l]
    label $l.lpollurl -text [::msgcat::mc "URL to poll:"]
    entry $l.pollurl -textvariable ltmp(pollurl)
    checkbutton $l.usepollkeys -text [::msgcat::mc "Use client security keys"] \
	-state disabled \
	-variable ltmp(usepollkeys) -command [list update_login_entries $l]
    
    grid $l.usehttppoll -row 0 -column 0 -sticky w -columnspan 3 -in $httppoll_page
    grid $l.lpollurl -row 1 -column 0 -sticky e -in $httppoll_page
    grid $l.pollurl -row 1 -column 1 -sticky ew -in $httppoll_page
    grid $l.usepollkeys -row 2 -column 0 -sticky w -columnspan 3 -in $httppoll_page

    grid columnconfigure $httppoll_page 1 -weight 1

    if {$jlib::lib(have_sasl)} {
	set sasl_page [$nb insert end sasl_page -text [::msgcat::mc "SASL"]]

	checkbutton $l.usesasl \
	    -text [::msgcat::mc "Use SASL authentification"] \
	    -variable ltmp(usesasl) -command [list update_login_entries $l]
	#label $l.lsaslport -text [::msgcat::mc "SASL Port:"]
	#Spinbox $l.saslport 0 65535 1 ltmp(saslport)
	#label $l.lsaslcertfile -text [::msgcat::mc "SASL Certificate:"] -state disabled
	#entry $l.saslcertfile -textvariable ltmp(saslcertfile) -state disabled
	#button $l.bsaslcertfile -text [::msgcat::mc "Browse..."] -state disabled \
	#    -command [list eval set ltmp(saslcertfile) {[tk_getOpenFile]}]


	grid $l.usesasl       -row 0 -column 0 -sticky w -columnspan 3 -in $sasl_page
	#grid $l.lsaslport     -row 1 -column 0 -sticky e -in $sasl_page
	#grid $l.saslport      -row 1 -column 1 -sticky w -in $sasl_page
	#grid $l.lsaslcertfile -row 2 -column 0 -sticky e -in $sasl_page
	#grid $l.saslcertfile  -row 2 -column 1 -sticky ew -in $sasl_page
	#grid $l.bsaslcertfile -row 2 -column 2 -sticky w -in $sasl_page

	grid columnconfigure $sasl_page 0 -weight 1
	grid columnconfigure $sasl_page 1 -weight 6
	grid columnconfigure $sasl_page 2 -weight 1
    }


    $nb compute_size
    $nb raise account_page
    bind .login <Control-Prior> [list tab_move $nb -1]
    bind .login <Control-Next> [list tab_move $nb 1]
    grid $nb -row 1 -column 0

    .login add -text [::msgcat::mc "Log in"] -command {
	array set loginconf [array get ltmp]
	destroy .login
	if {$loginconf(replace_opened)} {
	    logout
	}
	update
	login
    }
    .login add -text [::msgcat::mc "Cancel"] -command {destroy .login}

    update_login_entries $l

    .login draw $l.password
}

proc logout_dialog {} {
    global logout_conn

    switch -- [llength [jlib::connections]] {
	0 {
	    return
	}
	1 {
	    logout
	    return
	}
    }

    set w .logout
    if {[winfo exists $w]} {
	focus -force $w
	return
    }

    set lnames {}
    foreach connid [jlib::connections] {
	lappend lnames $connid [jlib::connection_jid $connid]
    }

    CbDialog $w [::msgcat::mc "Logout"] \
	[list [::msgcat::mc "Log out"] [list logout_dialog_logout $w] \
	      [::msgcat::mc "Cancel"] [list destroy $w]] \
	logout_conn $lnames {}
}

proc logout_dialog_logout {w} {
    global logout_conn

    foreach connid [array names logout_conn] {
	if {[lcontain [jlib::connections] $connid] && $logout_conn($connid)} {
	    logout $connid
	}
    }
    destroy $w
}

package require base64

proc connect_httpproxy {} {
    global use_tls
    global loginconf


    set sock [socket $loginconf(httpproxy) $loginconf(httpproxyport)]
    fconfigure $sock -buffering line

    if {$loginconf(usealtserver)} {
	set server $loginconf(altserver)
    } else {
	set server $loginconf(server)
    }
    if {$use_tls && $loginconf(usessl)} {
	puts $sock \
	    "CONNECT $server:${loginconf(sslport)} HTTP/1.0"
    } else {
	puts $sock "CONNECT $server:${loginconf(port)} HTTP/1.0"
    }
    
    if {$loginconf(httplogin) != ""} {
	set auth [base64::encode \
		      [encoding convertto "$loginconf(httplogin):$loginconf(httppassword)"]]
	puts $sock "Proxy-Authorization: Basic $auth"
    }
    puts $sock ""

    fileevent $sock readable {set proxy_readable ""}
    global proxy_readable
    vwait proxy_readable
    fileevent $sock readable {}

    set result [gets $sock]

    set code [lindex [split $result { }] 1]

    #debugmsg login $code
    if {$code >= 200 && $code < 300} {
	gets $sock
	if {$use_tls && $loginconf(usessl)} {
	    if {[cequal $loginconf(sslcertfile) ""]} {
		tls::import $sock
	    } else {
		tls::import $sock -cafile $loginconf(sslcafile)
	    }
	}
	return $sock
    } else {
	error "proxy return: $result"
    }
}

proc recv_auth_result {connid res args} {
    global auth_result
    global loginconf

    if {$res == "OK"} {
	set auth_result($connid) OK
    } else {
	set auth_result($connid) ERR
	set res [MessageDlg .auth_err -aspect 50000 -icon error \
		     -message [format \
				   [::msgcat::mc "Authentication failed: %s\nCreate new account?"] \
				   [error_to_string [lindex $args 0]]] \
		     -type user -buttons {yes no} -default 0 -cancel 1]
	if {!$res} {
	    jlib::send_iq set \
		[jlib::wrapper:createtag query \
		     -vars {xmlns jabber:iq:register} \
		     -subtags [list [jlib::wrapper:createtag username \
					 -chdata $loginconf(user)] \
				   [jlib::wrapper:createtag password \
					-chdata $loginconf(password)]]] \
		-command recv_register_result
	}
    }
}

proc recv_register_result {res args} {
    if {$res == "OK"} {
	jlib::disconnect
	login
    } else {
	MessageDlg .auth_err -aspect 50000 -icon error \
	    -message [format [::msgcat::mc "Registration failed: %s"] [error_to_string [lindex $args 0]]] \
	    -type user -buttons ok -default 0 -cancel 0
    }
}

proc change_password_dialog {} {
    global oldpassword newpassword password

    set oldpassword ""
    set newpassword ""
    set password ""

    if {[winfo exists .passwordchange]} {
	destroy .passwordchange
    }
    
    Dialog .passwordchange -title [::msgcat::mc "Change password"] \
	-separator 1 -anchor e -default 0 -cancel 1

    .passwordchange add -text [::msgcat::mc "OK"] -command {
	destroy .passwordchange
	send_change_password
    }
    .passwordchange add -text [::msgcat::mc "Cancel"] -command [list destroy .passwordchange]


    set p [.passwordchange getframe]
    
    label $p.loldpass -text [::msgcat::mc "Old password:"]
    ecursor_entry [entry $p.oldpass -show * -textvariable oldpassword]
    label $p.lnewpass -text [::msgcat::mc "New password:"]
    ecursor_entry [entry $p.newpass -show * -textvariable newpassword]
    label $p.lpassword -text [::msgcat::mc "Repeat new password:"]
    ecursor_entry [entry $p.password -show * -textvariable password]

    grid $p.loldpass  -row 0 -column 0 -sticky e
    grid $p.oldpass   -row 0 -column 1 -sticky ew
    grid $p.lnewpass  -row 1 -column 0 -sticky e
    grid $p.newpass   -row 1 -column 1 -sticky ew
    grid $p.lpassword -row 2 -column 0 -sticky e
    grid $p.password  -row 2 -column 1 -sticky ew

    focus $p.oldpass
    .passwordchange draw

}

proc send_change_password {} {
    global loginconf
    global oldpassword newpassword password

    if {$oldpassword != $loginconf(password)} {
	MessageDlg .auth_err -aspect 50000 -icon error \
	        -message [::msgcat::mc "Old password is incorrect"] \
		-type user -buttons ok -default 0 -cancel 0
	return
    }
    if {$newpassword != $password} {
	MessageDlg .auth_err -aspect 50000 -icon error \
	        -message [::msgcat::mc "New passwords do not match"] \
		-type user -buttons ok -default 0 -cancel 0
	return
    }

    jlib::send_iq set \
	    [jlib::wrapper:createtag query \
		    -vars {xmlns jabber:iq:register} \
		    -subtags [list [jlib::wrapper:createtag username \
					-chdata $loginconf(user)] \
				   [jlib::wrapper:createtag password \
					-chdata $password]]] \
	    -to $loginconf(server) \
	    -command recv_change_password_result
}

proc recv_change_password_result {res args} {
    global loginconf
    global newpassword

    if {$res == "OK"} {
	MessageDlg .shpasswd_result -aspect 50000 -icon info \
		-message [::msgcat::mc "Password is changed"] \
		-type user -buttons ok -default 0 -cancel 0
	for {set i 1} {[info exists ::loginconf$i]} {incr i} {
	    if {!([info exists ::loginconf${i}(user)] && \
		    [info exists ::loginconf${i}(server)] && \
		    [info exists ::loginconf${i}(password)])} {
		continue
	    }
	    upvar ::loginconf${i}(user) user
	    upvar ::loginconf${i}(server) server
	    upvar ::loginconf${i}(password) password
	    if {($user == $loginconf(user)) && \
		    ($server == $loginconf(server)) && \
		    ($password == $loginconf(password))} {
		set password $newpassword
	    }
	}
	set loginconf(password) $newpassword
    } else {
	MessageDlg .shpasswd_result -aspect 50000 -icon error \
	    -message [format [::msgcat::mc "Password change failed: %s"] [error_to_string [lindex $args 0]]] \
	    -type user -buttons ok -default 0 -cancel 0
    }
}

proc show_logout_dialog {} {
    global reason reasonlist

    set lw .logout

    if {![winfo exists $lw]} {
        Dialog $lw -title [::msgcat::mc "Logout with reason"] \
	    -separator 1 -anchor e -default 0 -cancel 1

        set lf [$lw getframe]
        grid columnconfigure $lf 1 -weight 1

	if {[llength $reasonlist]} {set reason [lindex $reasonlist 0]}

        label $lf.lreason   -text    [::msgcat::mc "Reason:"]
        ecursor_entry [ComboBox $lf.reason -textvariable reason \
		-values $reasonlist -width 35].e
        label $lf.lpriority -text    [::msgcat::mc "Priority:"]
        ecursor_entry [entry $lf.priority -textvariable loginconf(priority)]

        grid $lf.lreason   -row 0 -column 0 -sticky e
        grid $lf.reason    -row 0 -column 1 -sticky ew
        grid $lf.lpriority -row 1 -column 0 -sticky e
        grid $lf.priority  -row 1 -column 1 -sticky ew

        $lw add -text [::msgcat::mc "Log out"] -command logout_reason
        $lw add -text [::msgcat::mc "Cancel"] -command "$lw withdraw"
    } else {
        set lf [$lw getframe]
    }

    $lw draw $lf.reason
}

proc logout_reason {} {
    global userstatus textstatus reason reasonlist

    set reasonlist [update_combo_list $reasonlist $reason 10]
    set custom::saved(::::reasonlist) $reasonlist
    custom::store

    set lw .logout
    $lw withdraw

    set textstatus $reason
    set userstatus unavailable

    logout

    destroy $lw
}


