# $Id: logger.tcl,v 1.12 2003/10/07 20:03:20 aleksey Exp $

namespace eval ::logger {
    custom::defgroup Logging [::msgcat::mc "Logging options."] -group Chat

    custom::defvar options(logdir) ~/.tkabber/logs \
	    [::msgcat::mc "Directory to store logs."] \
	-type string -group Logging

    custom::defvar options(log_chat) 1 \
	    [::msgcat::mc "Store private chats logs."] \
	-type boolean -group Logging

    custom::defvar options(log_groupchat) 1 \
	    [::msgcat::mc "Store group chats logs."] \
	-type boolean -group Logging

    array set search_mode {}

    if {![file exists $options(logdir)]} {
	file mkdir $options(logdir)
    }
}


proc ::logger::log_file {jid} {
    variable options

    regsub -all @|/ $jid _ filename 
    return ${options(logdir)}/$filename
}

proc ::logger::log_message {chatid from type body x} {
    variable options

    if {$type == "chat" && !$options(log_chat)} return
    if {$type == "groupchat" && !$options(log_groupchat)} return

    set connid [chat::get_connid $chatid]
    set jid [chat::get_jid $chatid]
    set nas [node_and_server_from_jid $jid]
    if {$type == "chat" && ![chat::is_groupchat [chat::chatid $connid $nas]]} {
	set jid $nas
    }

    set logfile [log_file $jid]
    set nick [chat::get_nick $from $type]

    set seconds [clock seconds]
    foreach xelem $x {
	jlib::wrapper:splitxml $xelem tag vars isempty chdata children
	
	if {[cequal [jlib::wrapper:getattr $vars xmlns] jabber:x:delay]} {
	    set seconds [clock scan [jlib::wrapper:getattr $vars stamp] -gmt 1]
	}
    }
    set ts [clock format $seconds -format "%Y%m%dT%H%M%S"]

    set fd [open $logfile a]
    fconfigure $fd -encoding utf-8
    puts $fd [list [list timestamp $ts nick ${nick} body $body]]
    close $fd
}

hook::add draw_message_hook ::logger::log_message 15


proc ::logger::winid {name} {
    set allowed_name [jid_to_tag $name]
    return .log_$allowed_name
}


proc ::logger::do_search {txt entr jid} {
	variable search_mode
	
	set searchpattern [ $entr get]
	if ![string length $searchpattern] {return 0}

	if {$search_mode($jid,back)} {
		set search_from sel_start
		set search_to   0.0
		set search_dir  -backwards
	} else {
		set search_from sel_end
		set search_to	end
		set search_dir  -forwards
	}

	if {$search_mode($jid,case)} {
		set case ""
	} else {
		set case -nocase
	}

	if {$search_mode($jid,exact)} {
		set exact -regexp
	} else {
		set exact -exact
	}


#	set index [eval "$txt search $search_dir $case $exact -- [list $searchpattern] $search_from $search_to"]
	set index [eval "$txt search $search_dir $case $exact -- [list $searchpattern] $search_from"]

	if ![string length $index] {                                                    
 		return 0                                                                       
	} else {
		$txt tag remove sel 0.0 end
		if {$exact == "-regexp"} {
			eval "regexp $case --" [list $searchpattern [$txt get "$index linestart"\
   			"$index lineend"] match]
			$txt tag add sel $index "$index + [string length $match] chars"
		} else {
			$txt tag add sel $index "$index + [string length $searchpattern] chars"
		}
		$txt mark set sel_start sel.first
		$txt mark set sel_end sel.last
		$txt see $index
	}
}

proc ::logger::switch_button {jid bn var mode1 mode2} {
	global searchicon
	variable search_mode

	set search_mode($jid,$var) [expr !$search_mode($jid,$var)]

	set lw	[winid $jid]
	set sbox $lw.controls.searchbox
	
	if {$search_mode($jid,$var)} {
		eval "$sbox itemconfigure $bn $mode1"
	} else {
		eval "$sbox itemconfigure $bn $mode2"
	}
}

proc ::logger::show_log {jid} {
    global font
    global tcl_platform
    global defaultnick
    global loginconf
    global searchicon

    variable search_mode

    set search_mode($jid,back) 1
    set search_mode($jid,case) 0
    set search_mode($jid,exact) 0

    set nas [node_and_server_from_jid $jid]
    if {![chat::is_groupchat $nas]} {
	set jid $nas
    }

    set logfile [log_file $jid]
    if {[file exists $logfile]} {
	set lw [winid $jid]
	debugmsg plugins "LOGGER: $lw"
	if {[winfo exists $lw]} {
	    focus -force $lw
	    return
	}

	set fd [open $logfile r]
	fconfigure $fd -encoding utf-8
	set hist [read $fd]
	close $fd

	set subdirs current
	foreach subdir [glob -nocomplain -directory [file dirname $logfile] */] {
	    lappend subdirs [file tail $subdir]
	}
	set subdirs [lsort -decreasing $subdirs]

	set mynick [get_group_nick $jid $loginconf(user)]

	if { [string equal $tcl_platform(platform) "unix"] } {
	    set re raised
	    set bd 1
	} else {
	    set re flat
	    set bd 0
	}


	toplevel $lw -relief $re -borderwidth $bd -class Chat
	wm withdraw $lw
	set title [format [::msgcat::mc "History for %s"] $jid]
	wm title $lw $title
	wm iconname $lw $title

	frame $lw.controls

	set sentry [entry $lw.controls.search]
	pack $sentry -padx 2m -pady 2m -side left -anchor w

	set sbox [ButtonBox $lw.controls.searchbox -spacing 0 -padx 0 -default 0]
	$sbox add -text [::msgcat::mc "Go"] \
    	-relief raised -borderwidth 1 \
    	-helptext [::msgcat::mc "Do search"]

	$sbox add -image $searchicon(case) \
    	-relief raised -borderwidth 1 \
    	-helptext [::msgcat::mc "Match case..."] \
		-command [list logger::switch_button $jid 1 case {-relief sunken} {-relief raised} ]

	$sbox add -image $searchicon(exact) \
    	-relief raised -borderwidth 1 \
    	-helptext [::msgcat::mc "Match by regexp"] \
		-command  [list logger::switch_button $jid 2 exact {-relief sunken} {-relief raised} ]

	$sbox add -image $searchicon(back) \
    	-relief raised -borderwidth 1 \
    	-helptext [::msgcat::mc "Search direction"] \
		-command  [list logger::switch_button $jid 3 back {-image $searchicon(back)} {-image $searchicon(forward)}]

	pack $sbox -padx 2m -pady 2m -anchor e -side left
	bind $lw <Return> "ButtonBox::invoke $sbox default"
	
	set bbox [ButtonBox $lw.controls.bbox -spacing 10 -padx 10 -default 1]
	$bbox add -text [::msgcat::mc "Export to XHTML"] \
	    -command [list [namespace current]::export $lw.sw.log $hist $mynick]
	$bbox add -text [::msgcat::mc "Close"] -command [list destroy $lw]
	$bbox setfocus 1
	pack $bbox -padx 2m -pady 2m -anchor e -side right
	bind $lw <Escape> "ButtonBox::invoke $bbox default"
	pack $lw.controls -side bottom -fill x

	set sep [Separator::create $lw.sep -orient horizontal]
	pack $sep -pady 1m -fill x -side bottom

	set mf [frame $lw.mf]
	pack $mf -side top -fill x -expand no -padx 2m -pady 1m
	set mlabel [label $mf.mlabel -text [::msgcat::mc "Month:"]]
	pack $mlabel -side left

	set lf [ScrolledWindow $lw.sw]
	pack $lf -padx 1m -pady 2m -fill both -expand yes

	text $lf.log -font $font -wrap word -takefocus 1
	set l $lf.log
	$lf setwidget $lf.log

	$l mark set sel_start end
	$l mark set sel_end 0.0
	$sbox itemconfigure 0 -command [list logger::do_search $l $sentry $jid]


	$lf setwidget $l
	focus $l

	$l tag configure they -foreground [option get $lw theyforeground Chat]
	$l tag configure me -foreground [option get $lw meforeground Chat]
	$l tag configure server_lab \
	    -foreground [option get $lw serverlabelforeground Chat]
	$l tag configure server \
	    -foreground [option get $lw serverforeground Chat]

	set mcombo [ComboBox $mf.mcombo \
			-editable no \
			-values $subdirs \
			-text current \
			-modifycmd [list [namespace current]::change_month \
					$mf.mcombo $logfile $l $mynick]]
	pack $mcombo -side left


	if {![lempty $hist]} {
	    set vars [lindex $hist 0]
	    array set tmp $vars
	    if {[info exists tmp(timestamp)]} {
		set seconds [clock scan $tmp(timestamp) -gmt 0]
		set ym [clock format $seconds -format %Y%m]
		set curym [clock format [clock seconds] -format %Y%m]
		if {$ym < $curym} {
		    set hist [filter_old_history $logfile $hist]
		}
	    }
	}

	draw_messages $l $hist $mynick

	$lf.log see end
	$lf.log configure -state disabled

	wm deiconify $lw
	
    } else {
	
    }
}


proc ::logger::draw_messages {l hist mynick} {
    $l configure -state normal
    $l delete 0.0 end

    foreach vars $hist {
	array unset tmp
	array set tmp $vars
	if {[info exists tmp(timestamp)]} {
	    set seconds [clock scan $tmp(timestamp) -gmt 0]
	    $l insert end [clock format $seconds -format {[%Y-%m-%d %X]}]
	}
	if {[info exists tmp(nick)] && $tmp(nick) != ""} {
	    if {$tmp(nick) == $mynick} {
		set tag me
	    } else {
		set tag they
	    }
	    if {[info exists tmp(body)] && [regsub {^/me } $tmp(body) {} body]} {
		$l insert end "*$tmp(nick) $body" $tag
		unset tmp(body)
	    } else {
		$l insert end "<$tmp(nick)>" $tag
	    }
	    set servertag ""
	} else {
	    $l insert end "---" server_lab
	    set servertag server
	}
	if {[info exists tmp(body)]} {
	    $l insert end " $tmp(body)" $servertag
	}
	if {![$l compare "end -1 chars linestart" == "end -1 chars"]} {
	    $l insert end "\n"
	}
    }
    $l see end
    $l configure -state disabled
}


proc ::logger::change_month {mcombo logfile l mynick} {
    set month [$mcombo cget -text]

    if {$month == "current"} {
	set filename $logfile
    } else {
	set filename [file join \
			  [file dirname $logfile] \
			  $month \
			  [file tail $logfile]]
	if {![file exists $filename]} {
	    set filename $logfile
	}
    }

    set fd [open $filename r]
    fconfigure $fd -encoding utf-8
    set hist [read $fd]
    close $fd

    draw_messages $l $hist $mynick
}


proc ::logger::filter_old_history {logfile hist} {
    set newhist {}
    set curym [clock format [clock seconds] -format %Y-%m]
    foreach vars $hist {
	array unset tmp
	array set tmp $vars
	if {[info exists tmp(timestamp)]} {
	    set seconds [clock scan $tmp(timestamp) -gmt 0]
	    set ym [clock format $seconds -format %Y-%m]
	    if {$ym < $curym} {
		lappend oldhist($ym) $vars
	    } else {
		lappend newhist $vars
	    }
	}
    }

    foreach ym [array names oldhist] {
	set dir [file join [file dirname $logfile] $ym]
	set oldlog [file join $dir [file tail $logfile]]
	file mkdir $dir
	
	set fd [open $oldlog a]
	fconfigure $fd -encoding utf-8
	foreach vars $oldhist($ym) {
	    puts $fd [list $vars]
	}
	close $fd
    }

    set fd [open $logfile w]
    fconfigure $fd -encoding utf-8
    foreach vars $newhist {
	puts $fd [list $vars]
    }
    close $fd

    return $newhist
}


proc ::logger::export {lw hist mynick} {
    set filename [tk_getSaveFile -defaultextension .html]
    if {$filename == ""} return
    set fd [open $filename w]
    fconfigure $fd -encoding utf-8

    puts $fd {<?xml version="1.0" encoding="UTF-8"?>}
    puts $fd {<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "DTD/xhtml1-strict.dtd">}
    puts $fd {<html xmlns="http://www.w3.org/1999/xhtml">}
    set head [jlib::wrapper:createtag head \
		  -subtags [list [jlib::wrapper:createtag link \
				      -vars {
					  rel stylesheet
					  type text/css
					  href tkabber-logs.css
				      }]]]
    puts $fd [jlib::wrapper:createxml $head]


    foreach vars $hist {
	array unset tmp
	array set tmp $vars
	set subtags {}
	if {[info exists tmp(timestamp)]} {
	    set seconds [clock scan $tmp(timestamp) -gmt 0]
	    set timestamp [clock format $seconds -format {[%Y-%m-%d %X]}]
	    lappend subtags [jlib::wrapper:createtag span \
				 -vars {class timestamp} \
				 -chdata $timestamp]
	}
	if {[info exists tmp(nick)] && $tmp(nick) != ""} {
	    if {$tmp(nick) == $mynick} {
	        set tag me
	    } else {
	        set tag they
	    }
	    if {[info exists tmp(body)] && [regsub {^/me } $tmp(body) {} body]} {
		set nick "*$tmp(nick) $body"
		unset tmp(body)
	    } else {
		set nick "<$tmp(nick)> "
	    }
	    lappend subtags [jlib::wrapper:createtag span \
				 -vars [list class $tag] \
				 -chdata $nick]

	    if {[info exists tmp(body)]} {
		lappend subtags [jlib::wrapper:createtag span \
				     -vars [list class body] \
				     -chdata "$tmp(body)"]
	    }
	} else {
	    if {[info exists tmp(body)]} {
		lappend subtags [jlib::wrapper:createtag span \
				     -vars [list class server] \
				     -chdata "--- $tmp(body)"]
	    }
	}
	#if {![$l compare "end -1 chars linestart" == "end -1 chars"]} {
	#    puts "\n"
	#}
	set msg [jlib::wrapper:createtag div \
		     -vars [list class message] \
		     -subtags $subtags]

	puts $fd [jlib::wrapper:createxml $msg]
    }
    
    puts $fd {</html>}
    close $fd

    write_css $lw [file join [file dirname $filename] tkabber-logs.css]
}

proc ::logger::write_css {lw filename} {
    set fd [open $filename w]

    puts $fd "
html body {
    background-color: [$lw cget -background];
    color: [$lw cget -foreground];
}

.me {
    color: [$lw tag cget me -foreground];
}

.they {
    color: [$lw tag cget they -foreground];
}

.server {
    color: [$lw tag cget server -foreground];
}
"

    close $fd
}


