#!/usr/bin/env python
#------------------------------------------------------------------------------------
#
# TopShelf - A 'Currently Important Files' Applet
#
# Copyright (C) 2007  Alon Zakai ('Kripken')
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
#-------------------------------------------------------------------------------------

import sys
import pickle
import os.path
import subprocess
import mimetypes
import urllib

import gobject
import gtk, gtk.gdk
import pygtk
pygtk.require('2.0')

import gnomeapplet

## Global constants

OPEN_APP = "gnome-open"
HELP_APP = "yelp"

HELP_FILE = "/usr/share/doc/topshelf/topshelf.docbook"

## Global utilities

def popup_warning(window, message):
	dlg = gtk.MessageDialog(parent         = window,
	                        flags          = gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
	                        buttons        = gtk.BUTTONS_OK,
	                        message_format = message,
	                        type           = gtk.MESSAGE_WARNING)

	window.present() # Ensure we are on top, for our losing keep_above to not hide us
	window.set_keep_above(False)
	result = dlg.run()
	dlg.destroy()
	window.set_keep_above(True)

def popup_question(window, message):
	dlg = gtk.MessageDialog(parent         = window,
	                        flags          = gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
	                        buttons        = gtk.BUTTONS_YES_NO,
	                        message_format = message,
	                        type           = gtk.MESSAGE_QUESTION)

	window.present() # Ensure we are on top, for our losing keep_above to not hide us
	window.set_keep_above(False)
	result = dlg.run()
	dlg.destroy()
	window.set_keep_above(True)

	if result == gtk.RESPONSE_YES:
		return True
	elif result == gtk.RESPONSE_NO:
		return False
	elif result == gtk.RESPONSE_DELETE_EVENT:
		return False
	else:
		return False

def try_load_icon(icon_theme, icon_name, icon_size):
	try:
		return icon_theme.load_icon(icon_name, icon_size, 0)
	except gobject.GError:
		return None

def guess_mime_icon(filename):
	ICON_SIZE = 24

	icon_theme = gtk.icon_theme_get_default()

	if os.path.isdir(to_filename(filename)):
		ret = try_load_icon(icon_theme, "gnome-fs-directory", ICON_SIZE)
		if ret is None:
			ret = try_load_icon(icon_theme, "gtk-directory", ICON_SIZE)
		return ret

	filename = mimetypes.guess_type(os.path.basename(filename))[0]

	if filename is None:
		return try_load_icon(icon_theme, "unknown", ICON_SIZE)

	ret = try_load_icon(icon_theme, filename, ICON_SIZE)

	filename = filename.replace("/", "-")

	if ret is None:
		ret = try_load_icon(icon_theme, filename, ICON_SIZE)

	if ret is None:
		ret = try_load_icon(icon_theme, "gnome-mime-" + filename, ICON_SIZE)

	if ret is None:
		ret = try_load_icon(icon_theme, "gnome-mime-" + filename[0:filename.index("-")], ICON_SIZE)

	if ret is None:
		ret = try_load_icon(icon_theme, "unknown", ICON_SIZE)

	return ret

def to_filename(uri_or_something):
	return uri_or_something.replace("file://", "")


### Persistent data

PERSISTENT_FILE = os.path.join(os.path.expanduser("~"), ".topshelf.conf")

# Tests whether persistent data is 0.1, and hence needs migration
def test_point1(data):
	try:
		temp = data.data
		return False
	except AttributeError:
		return True
	
# Migrates persistent data from 0.1 to 0.2
def migrate_point1_to_point2(old):
	new = ModernPersistentData()
	new.set(current_files, old.current_files)
	new.set('w',           old.w)
	new.set('h',           old.h)
	return new

class PersistentData: ## Legacy!
	def __init__(self):
		self.current_files = []
		self.w = -1
		self.h = -1

current_files = "current_files" # A constant, really...

DEFAULT_VALUES = { current_files: [],
                   'x': 400,   # For standalone app
                   'y': 300,   # For standalone app
                   'w': 600,
                   'h': 350  }

class ModernPersistentData:
	def __init__(self):
		self.data = { }

	def get(self, key):
		try:
			return self.data[key]
		except KeyError:
			try:
				self.data[key] = DEFAULT_VALUES[key]
				return self.data[key]
			except KeyError:
				print "Fatal error: Key with no default value in ModernPersistentData"

	def set(self, key, value):
		self.data[key] = value


### Data for a single file

NORMAL_FG_COLOR  = "black"
DISABLE_FG_COLOR = "#FF0000"

class FileData:
	def __init__(self, filename, position, short_filename=None, mime_icon=None, color=None, strikethrough=None):
		self.filename  = filename
		self.position  = position

	def listify(self):
		return (self.filename, self.position, os.path.basename(self.filename), guess_mime_icon(self.filename), NORMAL_FG_COLOR, False)


### Main class

# full-path, position, short-filename, mime-type-icon, color
FILE_STORE_FULLPATH      = 0
FILE_STORE_POSITION      = 1
FILE_STORE_SHORTNAME     = 2
FILE_STORE_MIME_ICON     = 3
FILE_STORE_COLOR         = 4
FILE_STORE_STRIKETHROUGH = 5

class TopShelf:
	def __init__(self, applet, standalone_window=None):
		self.applet = applet
		self.applet.ts = self # Needed for standalone

		self.standalone_window = standalone_window

		self.window = None

		self.pressed  = False
		self.resizing = False

		self.panel_pb     = gtk.gdk.pixbuf_new_from_file("/usr/share/pixmaps/topshelf-24.png")
		self.panel_off_pb = gtk.gdk.pixbuf_new_from_file("/usr/share/pixmaps/topshelf-24-off.png")

		self.panel_image = gtk.Image()
		self.panel_image.set_from_pixbuf(self.panel_pb)
		self.panel_image.set_tooltip_text("TopShelf")
		self.eventbox = gtk.EventBox()
		self.eventbox.connect("button-press-event", self.do_button_press)
		self.eventbox.drag_dest_set(gtk.DEST_DEFAULT_ALL, [('text/uri-list', 0, 81)], gtk.gdk.ACTION_LINK)
		self.eventbox.connect("drag-data-received", self.do_drag_data_received)
		self.eventbox.connect("key-press-event", self.do_key_press)

		self.eventbox.add(self.panel_image)
		self.applet.add(self.eventbox)
		self.applet.show_all()

		# full-path, position, short-filename, mime-type-icon, color, strikethrough
		self.files_store = gtk.ListStore(str, int, str, gtk.gdk.Pixbuf, str, bool)

		try:
			self.unpickle()
		except IOError:
			self.persistent_data = ModernPersistentData()

		if self.standalone_window is not None:
			self.standalone_window.move(self.persistent_data.get('x'), self.persistent_data.get('y'))
			self.standalone_window.connect("configure-event", self.standalone_window_configure_event)
			self.standalone_window.set_keep_above(True)

	# Events

	def do_button_toggle(self, param1=None):
		if not self.pressed:
			self.show_window()
		else:
			self.hide_window()

	def do_button_press(self, btn, event):
		if event.type == gtk.gdk.BUTTON_PRESS and event.button == 1:
			if not self.resizing:
				self.do_button_toggle()
			else:
				self.do_stop_resizing()
		elif event.type == gtk.gdk.BUTTON_PRESS and event.button == 3:
			btn.emit_stop_by_name("button_press_event")
			propxml="""
				<popup name="button3">
				<menuitem name="Item 3" verb="Help"  label="_Help" pixtype="stock" pixname="gtk-help"/>
				<menuitem name="Item 4" verb="About" label="_About" pixtype="stock" pixname="gtk-about"/>
				</popup>"""
			verbs = [("About", self.do_about),
			         ("Help",  self.do_help)  ]
			self.applet.setup_menu(propxml, verbs, None)

	def do_files_button_press(self, btn, event):
		if event.button == 3:
			model, iter = self.get_selected()
			if iter is not None:
				self.popup_menu.popup(None, None, None, event.button, event.time)

		return False

	def standalone_window_configure_event(self, param1=None, param2=None):
		x, y = self.standalone_window.get_position()
		if x != self.persistent_data.get('x') or y != self.persistent_data.get('y'):
			self.persistent_data.set('x', x)
			self.persistent_data.set('y', y)
			self.pickle() # This might thrash, if we get many many events at once. TODO.

	def window_state_event(self, widget, event):
		if event.changed_mask & gtk.gdk.WINDOW_STATE_ICONIFIED:
			self.hide_window()
			self.window.deiconify()

	def configure_event(self, widget, event):
		w, h = self.window.get_size()

		if w != self.persistent_data.get('w') or h != self.persistent_data.get('h'):
			self.persistent_data.set('w', w)
			self.persistent_data.set('h', h)

			self.pickle()

	def do_cursor_change(self, param1):
		model, iter = self.get_selected()
		self.check_file(iter)

	# Actions

	def do_about(self, param1=None, param2=None):
		self.window.set_keep_above(False) # So we are not concealed!

		logo_pb = gtk.gdk.pixbuf_new_from_file("/usr/share/pixmaps/topshelf-64.png")

		dlg = gtk.AboutDialog()
		dlg.set_icon(logo_pb)

		dlg.set_name("TopShelf")
		dlg.set_program_name("TopShelf")
		dlg.set_version("0.2.1")
		dlg.set_comments("Currently Important Files Applet")
		dlg.set_authors(["Alon Zakai (Kripken)"])
		dlg.set_license("TopShelf is free software; you can redistribute it and/or\nmodify it under the terms of the GNU General Public License\nas published by the Free Software Foundation; either version\n2 of the License, or (at your option) any later version.")
		dlg.set_copyright("(c) 2007 Alon Zakai ('Kripken')")
		dlg.set_website("http://launchpad.net/topshelf")
		dlg.set_logo(logo_pb)

		dlg.run()
		dlg.destroy()

		self.window.set_keep_above(True)

		return False

	def do_open(self, param1=None, param2=None, param3=None):
		model, iter = self.get_selected()
		if iter is not None:
			self.check_file(iter) # Update color depending on existence of file. Not needed if a thread does this!

			filename = model.get_value(iter, 0)
			ret = subprocess.call(OPEN_APP + " '" + filename + "'", shell=True)
			self.hide_window() # After opening, we hide... for convenience
			if ret != 0:
				popup_warning(self.window, "The file '" + filename + "' could not be opened due to an error.")

		return False

	def do_open_folder(self, param1=None, param2=None, param3=None):
		model, iter = self.get_selected()
		if iter is not None:
			self.check_file(iter) # Update color depending on existence of file. Not needed if a thread does this!

			dirname = os.path.dirname(model.get_value(iter, 0))
			ret = subprocess.call(OPEN_APP + " '" + dirname + "'", shell=True)
			self.hide_window() # After opening, we hide... for convenience
			if ret != 0:
				popup_warning(self.window, "The directory '" + dirname + "' could not be opened due to an error.")

		return False

	def do_move_up(self, param1=None):
		model, iter = self.get_selected()
		if iter is not None:
			old_val = model.get(iter, 1)[0]
			new_val = old_val - 1
			self.swap_file_position(model, iter, old_val, new_val)

		return False

	def do_move_down(self, param1=None):
		model, iter = self.get_selected()
		if iter is not None:
			old_val = model.get(iter, 1)[0]
			new_val = old_val + 1
			self.swap_file_position(model, iter, old_val, new_val)

		return False

	def do_add(self, wgt):
		self.window.set_keep_above(False) # So we are not concealed
		chooser = gtk.FileChooserDialog("Select a file to add to TopShelf:",
		                                self.window,
		                                gtk.FILE_CHOOSER_ACTION_OPEN,
		                                buttons=(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_OPEN,gtk.RESPONSE_OK))

		response = chooser.run()
		if response == gtk.RESPONSE_OK:
			filename = chooser.get_filenames()
		else:
			response == None

		chooser.destroy()

		self.window.set_keep_above(True)

		if filename is not None:
			self.add_file(filename[0])

		return False

	def do_remove(self, wgt):
		model, iter = self.get_selected()
		if iter is not None:
			if popup_question(self.window,
			                  "Are you sure you want to remove the file '" + model.get(iter, 2)[0] +
			                  "' from the list? (The actual file will not be touched)"):
				old_position = model.get(iter, 1)[0]
				self.files_store.remove(iter)
				self.remove_file_position(model, old_position)

				self.pickle()

		return False

	def do_drag_data_received(self, widget, drag_context, x, y, selection_data, info, timestamp):
		uris = selection_data.data.strip().split()
		for uri in uris:
			filename = urllib.url2pathname(uri)
			self.add_file(filename)

	def do_package(self, param1=None):
		if self.get_n_files() == 0:
			return

		chooser = gtk.FileChooserDialog("Select a filename to save the zipped files as:",
		                                self.window,
		                                gtk.FILE_CHOOSER_ACTION_SAVE,
		                                buttons=(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_SAVE,gtk.RESPONSE_OK))

		response = chooser.run()
		if response == gtk.RESPONSE_OK:
			filename = chooser.get_filenames()[0]
		else:
			response == None
		chooser.destroy()

		if package_name is None:
			return

		file_names = ""
		iter = self.files_store.get_iter_first()
		while iter is not None:
			file_names = file_names + " '" + urllib.url2pathname(self.files_store.get(iter, 0)[0]) + "' "
			iter = self.files_store.iter_next(iter)

		file_names = to_filename(file_names)

		command = "zip '" + package_name + "' " + file_names
		subprocess.call(command, shell=True)

		return False

	def do_help(self, param1=None, param2=None, param3=None):
		self.hide_window()
		ret = subprocess.call(HELP_APP + " '" + HELP_FILE + "'", shell=True)
		if ret != 0:
			popup_warning(self.window, "The help file could not be opened due to an error.")

		return False

	def do_key_press(self, widget, event):
		if event.keyval == 65307:
			self.hide_window()
		elif event.keyval == 65535:
			self.do_remove(None)


	# Persistency

	def unpickle(self):
		pickle_file = open(PERSISTENT_FILE, 'rb')
		self.persistent_data = pickle.load(pickle_file)
		pickle_file.close()

		# Migrate, if needed
		if test_point1(self.persistent_data):
			self.persistent_data = migrate_point1_to_point2(self.persistent_data)

		for file_data in self.persistent_data.get(current_files):
			self.files_store.insert(0, file_data.listify())

	def pickle(self):
		self.persistent_data.set(current_files, [])

		iter = self.files_store.get_iter_first()
		while iter is not None:
			self.persistent_data.get(current_files).append(FileData(*self.files_store.get(iter, *range(self.files_store.get_n_columns()))))
			iter = self.files_store.iter_next(iter)

		pickle_file = open(PERSISTENT_FILE, 'wb')
		pickle.dump(self.persistent_data, pickle_file)
		pickle_file.close()

	# File management

	def add_file(self, filename):
		self.files_store.insert(0, FileData(filename, self.get_n_files()).listify())

		self.pickle()

	def get_file_iter_by_filename(self, filename):
		class Ret:
			def __init__(self):
				self.val = None

			def search(self, m, p, i, filename):
				if m.get(i, 0)[0] == filename:
					self.val = i

		ret = Ret()
		self.files_store.foreach(ret.search, (filename))
		return ret.val

	# Window functions

	def show_window(self):
		if self.window is None:
			self.create_window()

		if not self.resizing:
			self.panel_image.set_from_pixbuf(self.panel_off_pb)
		self.pressed = True

		# These four commands - present, keep-above, present, position - are very odd. But they work, in THIS order.
		self.window.present()
		self.window.set_keep_above(True) # Force it!
		self.window.present()
		self.position_window() # Needed here because of odd behavior otherwise (after adding, hide, show, it moved)

		self.check_files_thread() # One manual operation, then later as a background thread
#		gobject.timeout_add(2000, self.check_files_thread) No thread for now, since it wastes CPU. Later, add an option to do this.

	def hide_window(self, param1=False, param2=False):
		if self.window is None:
			return True

		self.window.hide()

		while gtk.events_pending():
			gtk.main_iteration() # Ensure the hide actually happens before we show e.g. Help windows

		self.panel_image.set_from_pixbuf(self.panel_pb)
		self.pressed = False

		return True # Stop delete events

	def create_window(self):
		self.window = gtk.Window()

		self.window.set_title("TopShelf")
		self.window.connect("delete-event",       self.hide_window)
		self.window.connect("window-state-event", self.window_state_event)
		self.window.connect("configure-event",    self.configure_event)
		self.window.connect("key-press-event",    self.do_key_press)

		self.window.set_keep_above(True)
		self.window.resize(self.persistent_data.get('w'), self.persistent_data.get('h'))

		self.window.drag_dest_set(gtk.DEST_DEFAULT_ALL, [('text/uri-list', 0, 81)], gtk.gdk.ACTION_LINK)
		self.window.connect("drag-data-received", self.do_drag_data_received)

		# Main view

		self.treeview = gtk.TreeView()
		self.treeview.set_model(self.files_store)
		col = self.append_column(self.treeview, gtk.TreeViewColumn("#",    gtk.CellRendererText(),   text  =1), 1)
		tmp = self.append_column(self.treeview, gtk.TreeViewColumn("",     gtk.CellRendererPixbuf(), pixbuf=3))
		tmp = self.append_column(self.treeview, gtk.TreeViewColumn("File", gtk.CellRendererText(),   text  =2), 2,
		                         foreground_id    = FILE_STORE_COLOR,
		                         strikethrough_id = FILE_STORE_STRIKETHROUGH)

		col.clicked() # Set default sort
		self.treeview.connect("row-activated" ,     self.do_open)
		self.treeview.connect("cursor-changed",     self.do_cursor_change)
		self.treeview.connect("button-press-event", self.do_files_button_press)

		scrolled_window = gtk.ScrolledWindow()
		scrolled_window.add(self.treeview)
		scrolled_window.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)

		# Toolbar & buttons on bottom

		toolbar = gtk.Toolbar()

		self.add_stock_button    (toolbar, gtk.STOCK_OPEN,                            self.do_open       , "Open file")
		self.add_altstock_button (toolbar, gtk.STOCK_DIRECTORY,      "Open Folder",   self.do_open_folder, "Open folder containing file")
		toolbar.add(gtk.SeparatorToolItem())
		self.add_stock_button    (toolbar, gtk.STOCK_GO_UP,                           self.do_move_up    , "Move file up in order")
		self.add_stock_button    (toolbar, gtk.STOCK_GO_DOWN,                         self.do_move_down  , "Move file down in order")
		toolbar.add(gtk.SeparatorToolItem())
		self.add_stock_button    (toolbar, gtk.STOCK_ADD,                             self.do_add        , "Add a file to the list")
		self.add_stock_button    (toolbar, gtk.STOCK_REMOVE,                          self.do_remove     , "Remove file from the list")
		toolbar.add(gtk.SeparatorToolItem())
		self.add_stock_button    (toolbar, gtk.STOCK_HELP,                            self.do_help       , "Show help")
		self.add_stock_button    (toolbar, gtk.STOCK_ABOUT,                           self.do_about      , "About TopShelf")
#		self.add_pixbuf_button   (toolbar, guess_mime_icon("a.zip"), "Package Files", self.do_package)

		# Main vbox

		vbox = gtk.VBox()
		vbox.add(scrolled_window)
		sep = gtk.HSeparator()
		vbox.add(sep)
		vbox.child_set_property(sep, "expand", False)
		vbox.add(toolbar)
		vbox.child_set_property(toolbar, "expand", False)

		# Popup menu

		self.popup_menu = gtk.Menu()
		self.add_menuitem   (self.popup_menu, gtk.STOCK_OPEN,                     self.do_open)
		self.add_altmenuitem(self.popup_menu, gtk.STOCK_DIRECTORY, "Open Folder", self.do_open_folder)
		self.popup_menu.add(gtk.SeparatorMenuItem())
		self.add_menuitem   (self.popup_menu, gtk.STOCK_GO_UP,                    self.do_move_up)
		self.add_menuitem   (self.popup_menu, gtk.STOCK_GO_DOWN,                  self.do_move_down)
		self.popup_menu.add(gtk.SeparatorMenuItem())
		self.add_menuitem   (self.popup_menu, gtk.STOCK_REMOVE,                   self.do_remove)
		self.popup_menu.show_all()

		# Finalize main window

		self.window.add(vbox)
		self.position_window()
		self.window.show_all()

	def add_altstock_button(self, toolbar, stock_id, alt_text, callback, tooltip):
		icon = gtk.Image()
		icon.set_from_stock(stock_id, gtk.ICON_SIZE_LARGE_TOOLBAR)

		btn = gtk.ToolButton()
		btn.set_icon_widget(icon)
		btn.set_label(alt_text)
		btn.connect("clicked", callback)
		btn.set_tooltip_text(tooltip)
		toolbar.insert(btn, -1)

	def add_stock_button(self, toolbar, stock_id, callback, tooltip):
		btn = gtk.ToolButton(stock_id)
		btn.connect("clicked", callback)
		btn.set_tooltip_text(tooltip)
		toolbar.insert(btn, -1)

	def add_menuitem(self, menu, stock_id, callback):
		item = gtk.ImageMenuItem(stock_id)
		item.connect("activate", callback)
		menu.append(item)

	def add_altmenuitem(self, menu, stock_id, alt_text, callback):
		icon = gtk.Image()
		icon.set_from_stock(stock_id, gtk.ICON_SIZE_LARGE_TOOLBAR)

		item = gtk.ImageMenuItem()
		item.set_image(icon)
		item.add(gtk.Label(alt_text)) # Weird, that GTK forces us to do this...
		item.connect("activate", callback)
		menu.append(item)

	def append_column(self, treeview, column, sort_id=None, foreground_id=None, strikethrough_id=None):
		renderer = column.get_cell_renderers()[0]
		if sort_id is not None:
			column.set_sort_column_id(sort_id)
		if foreground_id is not None:
			column.add_attribute(renderer, "foreground",    foreground_id)
		if strikethrough_id is not None:
			column.add_attribute(renderer, "strikethrough", strikethrough_id)
		treeview.append_column(column)
		return column

	def position_window(self):

		# This function was converted from a C function in clock.c in the GNOME
		# panel applet 'clock', which is GPL 2+ (like this code). Thanks to the
		# GNOME authors,
		#       Miguel de Icaza
 		#       Frederico Mena
		#       Stuart Parmenter
		#       Alexander Larsson
		#       George Lebl
		#       Gediminas Paulauskas
		#       Mark McLoughlin

		x,y = self.panel_image.window.get_origin()

		w,h = self.window.get_size()

		button_w = self.panel_image.get_allocation().width
		button_h = self.panel_image.get_allocation().height

		screen = self.window.get_screen()

		n = screen.get_n_monitors()
		found_monitor = False
		monitor = None
		for i in range(n):
			monitor = screen.get_monitor_geometry(i)
			if (x >= monitor.x and x <= monitor.x + monitor.width and
			    y >= monitor.y and y <= monitor.y + monitor.height):
				found_monitor = True
				break

		if not found_monitor:
			monitor = gtk.gdk.Rectangle()
			monitor.x = 0
			monitor.y = 0
			monitor.width  = screen.get_width()
			monitor.height = screen.get_height()

		if x > monitor.width/2:
			x -= w
			x += button_w

		if y < monitor.height/2:
			y += button_h
		else:
			y -= h

		self.window.move(x, y)


	# Threads

	def check_file(self, iter):
		filename = to_filename(self.files_store.get(iter, 0)[0])
		if os.path.exists(filename):
			self.files_store.set(iter, FILE_STORE_COLOR,         NORMAL_FG_COLOR)
			self.files_store.set(iter, FILE_STORE_STRIKETHROUGH, False)
		else:
			self.files_store.set(iter, FILE_STORE_COLOR,         DISABLE_FG_COLOR)
			self.files_store.set(iter, FILE_STORE_STRIKETHROUGH, True)

	def check_files_thread(self):
		if self.pressed:
			iter = self.files_store.get_iter_first()
			while iter is not None:
				self.check_file(iter)
				iter = self.files_store.iter_next(iter)
			return True
		else:
			return False

	# Utilities

	def get_selected(self):
		return self.treeview.get_selection().get_selected()

	def get_n_files(self):
		ret = self.files_store.iter_n_children(None)
		return ret

	def swap_file_position(self, model, iter, old_val, new_val):
		def swap_file_internal(m, p, i, vs):
			if m.get(i, 1)[0] == vs[1]:
				m.set(i, 1, vs[0])
			              
		if old_val >= 0 and new_val >= 0 and old_val < self.get_n_files() and new_val < self.get_n_files():
			model.foreach(swap_file_internal, (old_val, new_val))
			model.set(iter, 1, new_val)

		self.pickle()

	def remove_file_position(self, model, old_position):
		def remove_file_internal(m, p, i, op):
			if m.get(i, 1)[0] > op:
				m.set(i, 1, m.get(i, 1)[0]-1)

		model.foreach(remove_file_internal, old_position)


### Factory

def factory(applet, iid, standalone_window=None):
	ts = TopShelf(applet, standalone_window)

	return True


### Main

if len(sys.argv) == 1:
	# Run standalone
	main_window = gtk.Window(gtk.WINDOW_TOPLEVEL)
	main_window.set_title("TopShelf")
	main_window.connect("destroy", gtk.main_quit)

	applet = gnomeapplet.Applet()
	factory(applet, None, standalone_window=main_window)
	applet.reparent(main_window)
	main_window.show_all()
	gtk.main()

	sys.exit()

if __name__ == '__main__':
	gnomeapplet.bonobo_factory("OAFIID:Gnome_Panel_TopShelf_Factory", gnomeapplet.Applet.__gtype__, "A current files manager", "1.0", factory)

