/***************************************************************************
                          date.c  -  description
                             -------------------
    begin                : Sun Jun 3 2001
    copyright            : (C) 2001 by Pierre-Michel THEVENY
    email                : pmth@free.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#include "date.h"
#include "utils.h"


/***************************************************************************
* Get current GMT date and time
 ***************************************************************************/

G_DATE *
get_current_time (void)
{
    time_t curtime;
    struct tm *gmt;

    G_DATE *gDate = g_new (G_DATE, 1);

    /* Get the current time. */
    curtime = time (NULL);

    /* Convert it to gmt time representation. */
    gmt = gmtime (&curtime);

    gDate->day = gmt->tm_mday;
    gDate->month = gmt->tm_mon + 1;
    gDate->year = gmt->tm_year + 1900;
    gDate->hour = gmt->tm_hour;
    gDate->min = gmt->tm_min;
    gDate->sec = gmt->tm_sec;

    return (gDate);
}


/***************************************************************************
* Is a gregorian year is leap ?
 ***************************************************************************/

gboolean
is_leap (const gint year)
{
    if (year % 400 == 0)
    {
        return (TRUE);
    }
    else
    {
        if (year % 100 == 0)
        {
            return (FALSE);
        }
        else
        {
            if (year % 4 == 0)
            {
                return (TRUE);
            }
            else
            {
                return (FALSE);
            }
        }
    }
}


/*********************************************************************************
Conversion from Gregorian date to mofified Julian day and vice-versa are
derived from the Communications of the ACM (CACM, volume 11, number
10, October 1968, p.657) Henry F. Fliegel and Thomas C. Van Flandern
*********************************************************************************/

/***************************************************************************
* Convertion G_Date to Mjd
 ***************************************************************************/

gdouble
gregorian_to_mjd (G_DATE * gDate)
{
    gdouble mjd;

    if (gDate == NULL)
    {
        g_error (_
                 ("%s:%i Pointer to Gregorian date is a NULL pointer."),
                 __FILE__, __LINE__);
    }

    mjd = 1461 * (gDate->year + 4800 + (gDate->month - 14) / 12) / 4
          + 367 * (gDate->month - 2 - 12 * ((gDate->month - 14) / 12))
          / 12
          - 3 * ((gDate->year + 4900 + (gDate->month - 14) / 12) / 100)
          / 4
          + gDate->day - 2432076
          + (gDate->hour * 3600 + gDate->min * 60 + gDate->sec) / 86400.0;

    return (mjd);
}


/***************************************************************************
* Convertion Mjd to G_Date
 ***************************************************************************/

G_DATE
mjd_to_gregorian (gdouble mjd)
{
    G_DATE gDate;

    gint jd = (gint) (floor (mjd));	/* Just the integer part */
    gdouble srd = 86400.0 * (mjd - floor (mjd));	/* Seconds left */

    gint i, j;

    gint l = jd + 2468570;
    gint n = 4 * l / 146097;

    l = l - (146097 * n + 3) / 4;

    i = 4000 * (l + 1) / 1461001;

    l = l - 1461 * i / 4 + 31;

    j = 80 * l / 2447;

    gDate.day = l - 2447 * j / 80;
    l = j / 11;
    gDate.month = j + 2 - 12 * l;
    gDate.year = 100 * (n - 49) + i + l;

    gDate.hour = (gint) (srd / 3600.0);
    gDate.min = (gint) (fmod (srd, 3600.0) / 60.0);
    gDate.sec = fmod (fmod (srd, 3600.0), 60.0);

    return (gDate);
}
