/***************************************************************************
                          main_window.c  -  description
                             -------------------
    begin                : Mon Jun 4 2001
    copyright            : (C) 2001 by Pierre-Michel THEVENY
    email                : pmth@free.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <stdlib.h>
#include <math.h>

#include "main_window.h"
#include "date.h"
#include "help.h"
#include "msg.h"
#include "msg_error.h"
#include "utils.h"


/***************************************************************************
* Display the manual: not fully implemented yet
 ***************************************************************************/

static void
on_manual_activate (GtkMenuItem * menuitem, gpointer user_data)
{
    display_help ();
}


/***************************************************************************
* Display About window
 ***************************************************************************/

static void
on_about_activate (GtkMenuItem * menuitem, gpointer user_data)
{
    display_about ();
}


/***************************************************************************
* Get gregorian date from spinbutton
 ***************************************************************************/

static G_DATE *
get_gregorian (GtkButton * button, G_DATE * gDate)
{
    GtkWidget *spin;

    g_return_val_if_fail (button != NULL, NULL);
    g_return_val_if_fail (gDate != NULL, NULL);

    /* Day */
    spin = lookup_widget (GTK_WIDGET (button), "day_spinbutton");
    gDate->day = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin));

    /* Month */
    spin = lookup_widget (GTK_WIDGET (button), "month_spinbutton");
    gDate->month = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin));

    /* Year */
    spin = lookup_widget (GTK_WIDGET (button), "year_spinbutton");
    gDate->year = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin));

    /* Hour */
    spin = lookup_widget (GTK_WIDGET (button), "hour_spinbutton");
    gDate->hour = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin));

    /* Minute */
    spin = lookup_widget (GTK_WIDGET (button), "minute_spinbutton");
    gDate->min = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin));

    /* Second */
    spin = lookup_widget (GTK_WIDGET (button), "second_spinbutton");
    gDate->sec = gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (spin));

    /* Sanity check: other checks are performed by spinbutton adjustement */
    if (gDate->month == 4 && gDate->day > 30)
    {
        msg_error (_("April has only 30 days"));
        return (NULL);
    }
    else if (gDate->month == 6 && gDate->day > 30)
    {
        msg_error (_("June has only 30 days"));
        return (NULL);
    }
    else if (gDate->month == 9 && gDate->day > 30)
    {
        msg_error (_("September has only 30 days"));
        return (NULL);
    }
    else if (gDate->month == 11 && gDate->day > 30)
    {
        msg_error (_("November has only 30 days"));
        return (NULL);
    }
    else if (gDate->month == 2 && is_leap (gDate->year) && gDate->day > 29)
    {
        msg_error (_("Frebruary has only 29 days this year"));
        return (NULL);
    }
    else if (gDate->month == 2 && (!is_leap (gDate->year)) && gDate->day > 28)
    {
        msg_error (_("Frebruary has only 28 days this year"));
        return (NULL);
    }
    else if (gDate->year == (-4713) &&
             (gDate->month < 11 ||
              (gDate->month == 11 &&
               (gDate->day < 24 || (gDate->day == 24 && (gDate->hour < 12))))))
    {

        msg_error (_("Gregorian date is out of limits"));
        return (NULL);
    }
    else
    {
        g_print ("Get gregorian %i/%i/%i %i:%i:%f\n", gDate->year, gDate->month,
                 gDate->day, gDate->hour, gDate->min, gDate->sec);
        return (gDate);
    }
}


/***************************************************************************
* Get Julian day from text entry
 ***************************************************************************/

static gdouble *
get_jd (GtkButton * button, gdouble * jd)
{
    GtkWidget *entry;
    gchar *buffer = NULL;

    g_return_val_if_fail (button != NULL, NULL);
    g_return_val_if_fail (jd != NULL, NULL);

    entry = lookup_widget (GTK_WIDGET (button), "jd_entry");
    buffer = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
    (*jd) = g_strtod (buffer, NULL);
    g_free (buffer);

    /* Sanity check */
    if ((*jd) < (MIN_JD) || (*jd) >= (MAX_JD))
    {
        msg_error (_("Julian day is out of limits"));
        return (NULL);
    }
    else
    {
        g_print ("Get jd %f\n", (*jd));
        return (jd);
    }
}


/***************************************************************************
* Get modified Julian day from text entry
 ***************************************************************************/

static gdouble *
get_mjd (GtkButton * button, gdouble * jd)
{
    GtkWidget *entry;
    gchar *buffer = NULL;

    g_return_val_if_fail (button != NULL, NULL);
    g_return_val_if_fail (jd != NULL, NULL);

    entry = lookup_widget (GTK_WIDGET (button), "mjd_entry");
    buffer = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
    (*jd) = g_strtod (buffer, NULL);
    g_free (buffer);

    /* Sanity check */
    if ((*jd) < (MIN_MJD) || (*jd) >= (MAX_MJD))
    {
        msg_error (_("Modified Julian day is out of limits"));
        return (NULL);
    }
    else
    {
        g_print ("Get mjd %f\n", (*jd));
        return (jd);
    }
}


/***************************************************************************
* Get LOP Julian day from text entry
 ***************************************************************************/

static gdouble *
get_lop (GtkButton * button, gdouble * jd)
{
    GtkWidget *entry;
    gchar *buffer;

    g_return_val_if_fail (button != NULL, NULL);
    g_return_val_if_fail (jd != NULL, NULL);

    entry = lookup_widget (GTK_WIDGET (button), "lop_entry");
    buffer = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
    (*jd) = g_strtod (buffer, NULL);
    g_free (buffer);

    /* Sanity check */
    if ((*jd) < (MIN_LOP_JD) || (*jd) >= (MAX_LOP_JD))
    {
        msg_error (_("LOP Julian day is out of limits"));
        return (NULL);
    }
    else
    {
        g_print ("Get lop_jd %f\n", (*jd));
        return (jd);
    }
}


/***************************************************************************
* Get CNES Julian day from text entry
 ***************************************************************************/

static gdouble *
get_cnes (GtkButton * button, gdouble * jd)
{
    GtkWidget *entry;
    gchar *buffer;

    g_return_val_if_fail (button != NULL, NULL);
    g_return_val_if_fail (jd != NULL, NULL);

    entry = lookup_widget (GTK_WIDGET (button), "cnes_entry");
    buffer = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
    (*jd) = g_strtod (buffer, NULL);
    g_free (buffer);

    /* Sanity check */
    if ((*jd) < (MIN_CNES_JD) || (*jd) >= (MAX_CNES_JD))
    {
        msg_error (_("CNES Julian day is out of limits"));
        return (NULL);
    }
    else
    {
        g_print ("Get cnes_jd %f\n", (*jd));
        return (jd);
    }
}


/***************************************************************************
* Get NASA Julian day from text entry
 ***************************************************************************/

static gdouble *
get_nasa (GtkButton * button, gdouble * jd)
{
    GtkWidget *entry;
    gchar *buffer;

    g_return_val_if_fail (button != NULL, NULL);
    g_return_val_if_fail (jd != NULL, NULL);

    entry = lookup_widget (GTK_WIDGET (button), "nasa_entry");
    buffer = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
    (*jd) = g_strtod (buffer, NULL);
    g_free (buffer);

    /* Sanity check */
    if ((*jd) < (MIN_NASA_JD) || (*jd) >= (MAX_NASA_JD))
    {
        msg_error (_("NASA Julian day is out of limits"));
        return (NULL);
    }
    else
    {
        g_print ("Get nasa_jd %f\n", (*jd));
        return (jd);
    }
}


/***************************************************************************
* Get CCSDS Julian day from text entry
 ***************************************************************************/

static gdouble *
get_ccsds (GtkButton * button, gdouble * jd)
{
    GtkWidget *entry;
    gchar *buffer;

    g_return_val_if_fail (button != NULL, NULL);
    g_return_val_if_fail (jd != NULL, NULL);

    entry = lookup_widget (GTK_WIDGET (button), "ccsds_entry");
    buffer = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
    (*jd) = g_strtod (buffer, NULL);
    g_free (buffer);

    /* Sanity check */
    if ((*jd) < (MIN_CCSDS_JD) || (*jd) >= (MAX_CCSDS_JD))
    {
        msg_error (_("CCSDS Julian day is out of limits"));
        return (NULL);
    }
    else
    {
        g_print ("Get ccsds_jd %f\n", (*jd));
        return (jd);
    }
}


/***************************************************************************
* Get TOPEX/POSEIDON cycle from spinbutton
 ***************************************************************************/

static gint
get_cycle (GtkButton * button)
{
    GtkWidget *spin;
    gint cycle;

    g_return_val_if_fail (button != NULL, 0);

    spin = lookup_widget (GTK_WIDGET (button), "cycle_spinbutton");
    cycle = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin));

    /* Sanity check: other checks are performed by spinbutton adjustement */
    if (cycle == 0)
    {
        msg_error (_("Cycle is out of limits"));
        return (0);
    }
    else
    {
        g_print ("Get topex cycle %i\n", cycle);
        return cycle;
    }
}


/***************************************************************************
* Get TOPEX/POSEIDON pass from spinbutton
 ***************************************************************************/

static gint
get_pass (GtkButton * button)
{
    GtkWidget *spin;
    gint pass;

    g_return_val_if_fail (button != NULL, 0);

    /* Checks are performed by spinbutton adjustement */
    spin = lookup_widget (GTK_WIDGET (button), "pass_spinbutton");
    pass = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin));

    /* Sanity check: other checks are performed by spinbutton adjustement */
    if (pass == 0)
    {
        msg_error (_("Pass is out of limits"));
        return (0);
    }
    else
    {
        g_print ("Get topex pass %i\n", pass);
        return pass;
    }
}


/***************************************************************************
* Set gregorian date to GREGORIAN spinbuttons
 ***************************************************************************/

static void
set_gregorian (GtkButton * button, G_DATE * gDate)
{
    GtkWidget *spin;

    g_return_if_fail (button != NULL);
    g_return_if_fail (gDate != NULL);

    g_print ("Set gregorian to %i/%i/%i %i:%i:%f\n", gDate->year, gDate->month,
             gDate->day, gDate->hour, gDate->min, gDate->sec);

    spin = lookup_widget (GTK_WIDGET (button), "day_spinbutton");
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), gDate->day);

    spin = lookup_widget (GTK_WIDGET (button), "month_spinbutton");
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), gDate->month);

    spin = lookup_widget (GTK_WIDGET (button), "year_spinbutton");
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), gDate->year);

    spin = lookup_widget (GTK_WIDGET (button), "hour_spinbutton");
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), gDate->hour);

    spin = lookup_widget (GTK_WIDGET (button), "minute_spinbutton");
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), gDate->min);

    spin = lookup_widget (GTK_WIDGET (button), "second_spinbutton");
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), gDate->sec);
}


/***************************************************************************
* Set Julian day to JD text entry
 ***************************************************************************/

static void
set_jd (GtkButton * button, gdouble * mjd)
{
    GtkWidget *entry;
    gdouble jd;
    gchar *buffer;

    g_return_if_fail (button != NULL);
    g_return_if_fail (mjd != NULL);

    jd = (*mjd) - (OFFSET_JD);

    g_print ("Set jd to %f\n", jd);

    buffer = g_strdup_printf ("%.6f", jd);

    entry = lookup_widget (GTK_WIDGET (button), "jd_entry");
    gtk_entry_set_text (GTK_ENTRY (entry), buffer);
    g_free (buffer);
}


/***************************************************************************
* Set modified Julian day to MJD text entry
 ***************************************************************************/

static void
set_mjd (GtkButton * button, gdouble * mjd)
{
    GtkWidget *entry;
    gchar *buffer;

    g_return_if_fail (button != NULL);
    g_return_if_fail (mjd != NULL);

    g_print ("Set mjd to %f\n", (*mjd));

    buffer = g_strdup_printf ("%.6f", (*mjd));

    entry = lookup_widget (GTK_WIDGET (button), "mjd_entry");
    gtk_entry_set_text (GTK_ENTRY (entry), buffer);
    g_free (buffer);
}


/***************************************************************************
* Set modified Julian day to LOP text entry
 ***************************************************************************/

static void
set_lop (GtkButton * button, gdouble * mjd)
{
    GtkWidget *entry;
    gdouble jd;
    gchar *buffer;

    g_return_if_fail (button != NULL);
    g_return_if_fail (mjd != NULL);

    jd = (*mjd) - (OFFSET_LOP);

    g_print ("Set lop_jd to %f\n", jd);

    buffer = g_strdup_printf ("%.6f", jd);

    entry = lookup_widget (GTK_WIDGET (button), "lop_entry");
    gtk_entry_set_text (GTK_ENTRY (entry), buffer);
    g_free (buffer);
}


/***************************************************************************
* Set modified Julian day to CNES text entry
 ***************************************************************************/

static void
set_cnes (GtkButton * button, gdouble * mjd)
{
    GtkWidget *entry;
    gdouble jd;
    gchar *buffer;

    g_return_if_fail (button != NULL);
    g_return_if_fail (mjd != NULL);

    jd = (*mjd) - (OFFSET_CNES);

    g_print ("Set cnes_jd to %f\n", jd);

    buffer = g_strdup_printf ("%.6f", jd);

    entry = lookup_widget (GTK_WIDGET (button), "cnes_entry");
    gtk_entry_set_text (GTK_ENTRY (entry), buffer);
    g_free (buffer);
}


/***************************************************************************
* Set modified Julian day to NASA text entry
 ***************************************************************************/

static void
set_nasa (GtkButton * button, gdouble * mjd)
{
    GtkWidget *entry;
    gdouble jd;
    gchar *buffer;

    g_return_if_fail (button != NULL);
    g_return_if_fail (mjd != NULL);

    jd = (*mjd) - (OFFSET_NASA);

    g_print ("Set nasa_jd to %f\n", jd);

    buffer = g_strdup_printf ("%.6f", jd);

    entry = lookup_widget (GTK_WIDGET (button), "nasa_entry");
    gtk_entry_set_text (GTK_ENTRY (entry), buffer);
    g_free (buffer);
}


/***************************************************************************
* Set modified Julian day to CCSDS text entry
 ***************************************************************************/

static void
set_ccsds (GtkButton * button, gdouble * mjd)
{
    GtkWidget *entry;
    gdouble jd;
    gchar *buffer;

    g_return_if_fail (button != NULL);
    g_return_if_fail (mjd != NULL);

    jd = (*mjd) - (OFFSET_CCSDS);

    g_print ("Set ccsds_jd to %f\n", jd);

    buffer = g_strdup_printf ("%.6f", jd);

    entry = lookup_widget (GTK_WIDGET (button), "ccsds_entry");
    gtk_entry_set_text (GTK_ENTRY (entry), buffer);
    g_free (buffer);
}


/***************************************************************************
* Set TOPEX/POSEIDON cycle and pass to T/P spinbuttons
 ***************************************************************************/

static void
set_cycle_and_pass (GtkButton * button, gdouble * mjd)
{
    GtkWidget *spin_cycle;
    GtkWidget *spin_pass;
    gdouble dcycle;
    gint cycle;
    gint pass;

    g_return_if_fail (button != NULL);
    g_return_if_fail (mjd != NULL);

    dcycle = ((*mjd) - (FIRST_CYCLE_MJD)) / (254.0 * PASS_DURATION_S / 86400.0);

    cycle = (gint) (dcycle) + 1;

    pass = (gint) ((dcycle - floor (dcycle)) * 254.0 + 1e-6) + 1;	/* 1e-6 is a hack to correct precicion error */

    spin_cycle = lookup_widget (GTK_WIDGET (button), "cycle_spinbutton");
    spin_pass = lookup_widget (GTK_WIDGET (button), "pass_spinbutton");

    if (cycle > 0 && cycle <= MAX_CYCLE)
    {
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_cycle), cycle);
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_pass), pass);

        g_print ("Set topex_cycle to %i\n", cycle);
        g_print ("Set topex_pass to %i\n", pass);
    }
    else
    {
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_cycle), 0);
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_pass), 0);
        g_print ("Set topex_cycle to 0\n");
        g_print ("Set topex_pass to 1\n");
    }
}


/***************************************************************************
* Set Gregorian date in calendar
 ***************************************************************************/

static void
set_calendar (GtkButton * button, G_DATE * gDate)
{
    GtkWidget *calendar;
    GtkWidget *leap_label;
    gchar *buffer;

    g_return_if_fail (button != NULL);
    g_return_if_fail (gDate != NULL);

    /* Set calendar date */
    calendar = lookup_widget (GTK_WIDGET (button), "calendar");
    gtk_calendar_select_month (GTK_CALENDAR (calendar),
                               gDate->month - 1, gDate->year);
    gtk_calendar_select_day (GTK_CALENDAR (calendar), gDate->day);

    /* Indicate leap year */
    leap_label = lookup_widget (GTK_WIDGET (button), "leap_label");

    if (is_leap (gDate->year))
    {
        buffer = g_strdup_printf (_("%i is a leap year"), gDate->year);
    }
    else
    {
        buffer = g_strdup_printf (_("%i is NOT a leap year"), gDate->year);
    }
    gtk_label_set_text (GTK_LABEL (leap_label), buffer);
    g_free (buffer);
}


/***************************************************************************
* Set all the date formats in one call
 ***************************************************************************/

static void
set_all_dates (GtkButton * button, G_DATE * gDate, gdouble * mjd)
{
    gchar *buffer = NULL;
    GtkWidget *leap_setting;

    g_return_if_fail (button != NULL);
    g_return_if_fail (gDate != NULL);
    g_return_if_fail (mjd != NULL);

    set_gregorian (button, gDate);
    set_jd (button, mjd);
    set_mjd (button, mjd);
    set_lop (button, mjd);
    set_cnes (button, mjd);
    set_nasa (button, mjd);
    set_ccsds (button, mjd);
    set_cycle_and_pass (button, mjd);
    set_calendar (button, gDate);


    /* Msg the user for leap year if asked in settings menu */
    leap_setting = lookup_widget (GTK_WIDGET (button), "leap_setting");
    if (GTK_CHECK_MENU_ITEM (leap_setting)->active)
    {
        if (is_leap (gDate->year))
        {
            buffer = g_strdup_printf (_("%i is a leap year"), gDate->year);
            display_msg (buffer);
            g_free (buffer);
        }
    }
}


/***************************************************************************
* Conversion from Gregorian day
 ***************************************************************************/

static void
on_convert_gregorian_clicked (GtkButton * button, gpointer user_data)
{
    G_DATE gDate;
    gdouble mjd;

    g_return_if_fail (button != NULL);

    g_print ("\n");

    if (get_gregorian (button, &gDate) != NULL)
    {
        mjd = gregorian_to_mjd (&gDate);

        set_all_dates (button, &gDate, &mjd);
    }
}


/***************************************************************************
* Conversion from Julian day
 ***************************************************************************/

static void
on_convert_jd_clicked (GtkButton * button, gpointer user_data)
{
    G_DATE gDate;
    gdouble mjd;

    g_return_if_fail (button != NULL);

    g_print ("\n");

    if (get_jd (button, &mjd) != NULL)
    {
        mjd += (OFFSET_JD);
        gDate = mjd_to_gregorian (mjd);

        set_all_dates (button, &gDate, &mjd);
    }
}


/***************************************************************************
* Conversion from modified Julian day
 ***************************************************************************/

static void
on_convert_mjd_clicked (GtkButton * button, gpointer user_data)
{
    G_DATE gDate;
    gdouble mjd;

    g_return_if_fail (button != NULL);

    g_print ("\n");

    if (get_mjd (button, &mjd) != NULL)
    {
        gDate = mjd_to_gregorian (mjd);

        set_all_dates (button, &gDate, &mjd);
    }
}


/***************************************************************************
* Conversion from LOP Julian day
 ***************************************************************************/

static void
on_convert_lop_clicked (GtkButton * button, gpointer user_data)
{
    G_DATE gDate;
    gdouble mjd;

    g_return_if_fail (button != NULL);

    g_print ("\n");

    if (get_lop (button, &mjd) != NULL)
    {
        mjd += (OFFSET_LOP);
        gDate = mjd_to_gregorian (mjd);

        set_all_dates (button, &gDate, &mjd);
    }
}


/***************************************************************************
* Conversion from CNES Julian day
 ***************************************************************************/

static void
on_convert_cnes_clicked (GtkButton * button, gpointer user_data)
{
    G_DATE gDate;
    gdouble mjd;

    g_return_if_fail (button != NULL);

    g_print ("\n");

    if (get_cnes (button, &mjd) != NULL)
    {
        mjd += (OFFSET_CNES);
        gDate = mjd_to_gregorian (mjd);

        set_all_dates (button, &gDate, &mjd);
    }
}


/***************************************************************************
* Conversion from NASA Julian day
 ***************************************************************************/

static void
on_convert_nasa_clicked (GtkButton * button, gpointer user_data)
{
    G_DATE gDate;
    gdouble mjd;

    g_return_if_fail (button != NULL);

    g_print ("\n");

    if (get_nasa (button, &mjd) != NULL)
    {
        mjd += (OFFSET_NASA);
        gDate = mjd_to_gregorian (mjd);

        set_all_dates (button, &gDate, &mjd);
    }
}


/***************************************************************************
* Conversion from CCSDS Julian day
 ***************************************************************************/

static void
on_convert_ccsds_clicked (GtkButton * button, gpointer user_data)
{
    G_DATE gDate;
    gdouble mjd;

    g_return_if_fail (button != NULL);

    g_print ("\n");

    if (get_ccsds (button, &mjd) != NULL)
    {
        mjd += (OFFSET_CCSDS);
        gDate = mjd_to_gregorian (mjd);

        set_all_dates (button, &gDate, &mjd);
    }
}


/***************************************************************************
* Conversion from TOPEX/POSEIDON cycle and pass
 ***************************************************************************/

static void
on_convert_topex_clicked (GtkButton * button, gpointer user_data)
{
    G_DATE gDate;
    gdouble mjd;
    gint cycle;
    gint pass;

    g_return_if_fail (button != NULL);

    g_print ("\n");

    cycle = get_cycle (button);
    pass = get_pass (button);

    /* SpinButtons adj limit cycle to 0-MAX_CYCLE and pass to 0-254 */
    if (cycle != 0 && pass != 0)
    {
        mjd =
            (FIRST_CYCLE_MJD) + PASS_DURATION_S * ((cycle - 1) * 254.0 +
                                                   (pass - 1)) / 86400.0;

        gDate = mjd_to_gregorian (mjd);

        set_all_dates (button, &gDate, &mjd);
    }
}


/***************************************************************************
* Create the main window
 ***************************************************************************/

GtkWidget *
create_main_window (void)
{
    GtkWidget *main_window;
    GtkWidget *vbox;
    GtkWidget *handlebox;
    GtkWidget *menubar;
    guint tmp_key;

    GtkWidget *file;
    GtkWidget *file_menu;
    GtkAccelGroup *file_menu_accels;
    GtkWidget *quit;

    GtkWidget *settings;
    GtkWidget *settings_menu;
    GtkAccelGroup *settings_menu_accels;
    GtkWidget *leap_setting;

    GtkWidget *help;
    GtkWidget *help_menu;
    GtkAccelGroup *help_menu_accels;
    GtkWidget *manual;
    GtkWidget *sep;
    GtkWidget *about;

    GtkWidget *convert_table;

    GtkWidget *gregorian_button;
    GtkWidget *jd_button;
    GtkWidget *mjd_button;
    GtkWidget *lop_button;
    GtkWidget *cnes_button;
    GtkWidget *nasa_button;
    GtkWidget *ccsds_button;
    GtkWidget *topex_button;

    GtkWidget *gregorian_table;
    GtkWidget *day_label;
    GtkWidget *month_label;
    GtkWidget *year_label;
    GtkWidget *hour_label;
    GtkWidget *minute_label;
    GtkWidget *second_label;
    GtkObject *day_spinbutton_adj;
    GtkWidget *day_spinbutton;
    GtkObject *month_spinbutton_adj;
    GtkWidget *month_spinbutton;
    GtkObject *year_spinbutton_adj;
    GtkWidget *year_spinbutton;
    GtkObject *hour_spinbutton_adj;
    GtkWidget *hour_spinbutton;
    GtkObject *minute_spinbutton_adj;
    GtkWidget *minute_spinbutton;
    GtkObject *second_spinbutton_adj;
    GtkWidget *second_spinbutton;

    GtkWidget *hseparator1;

    GtkWidget *jd_label;
    GtkWidget *jd_entry;

    GtkWidget *mjd_label;
    GtkWidget *mjd_entry;

    GtkWidget *lop_label;
    GtkWidget *lop_entry;

    GtkWidget *cnes_label;
    GtkWidget *cnes_entry;

    GtkWidget *nasa_label;
    GtkWidget *nasa_entry;

    GtkWidget *ccsds_label;
    GtkWidget *ccsds_entry;

    GtkWidget *hseparator2;

    GtkWidget *cycle_pass_label;
    GtkWidget *cycle_table;
    GtkWidget *cycle_label;
    GtkObject *cycle_spinbutton_adj;
    GtkWidget *cycle_spinbutton;
    GtkWidget *pass_label;
    GtkObject *pass_spinbutton_adj;
    GtkWidget *pass_spinbutton;

    GtkWidget *gregorian_label;

    GtkAccelGroup *accel_group;

    GtkWidget *notebook;
    GtkWidget *converter_page_label;
    GtkWidget *calendar_page_label;
    GtkWidget *calendar_vbox;
    GtkWidget *calendar;
    GtkWidget *leap_label;

    GtkTooltips *tooltips;

    G_DATE *gDate;

    /*################### MISC ################### */

    /* Main window */
    main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_position (GTK_WINDOW (main_window), GTK_WIN_POS_CENTER);
    gtk_object_set_data (GTK_OBJECT (main_window), "main_window", main_window);
    gtk_window_set_title (GTK_WINDOW (main_window), _("TP time"));

    /* Accel group */
    accel_group = gtk_accel_group_new ();
    gtk_window_add_accel_group (GTK_WINDOW (main_window), accel_group);

    /* Tooltips */
    tooltips = gtk_tooltips_new ();
    gtk_object_set_data (GTK_OBJECT (main_window), "tooltips", tooltips);

    /* Vertical box */
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_ref (vbox);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "vbox", vbox,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_add (GTK_CONTAINER (main_window), vbox);
    gtk_widget_show (vbox);

    /* Handle box for menu bar */
    handlebox = gtk_handle_box_new ();
    gtk_widget_ref (handlebox);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "handlebox",
                              handlebox, (GtkDestroyNotify) gtk_widget_unref);
    gtk_box_pack_start (GTK_BOX (vbox), handlebox, FALSE, FALSE, 0);
    gtk_widget_show (handlebox);

    /* Initial values */
    gDate = get_current_time ();

    /*################### MENUS ################### */

    /* Menu bar */
    menubar = gtk_menu_bar_new ();
    gtk_widget_ref (menubar);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "menubar", menubar,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_add (GTK_CONTAINER (handlebox), menubar);
    gtk_widget_show (menubar);

    /* File sub-menu */
    file = gtk_menu_item_new_with_label ("");
    tmp_key = gtk_label_parse_uline (GTK_LABEL (GTK_BIN (file)->child),
                                     _("_File"));
    gtk_widget_add_accelerator (file, "activate_item", accel_group,
                                tmp_key, GDK_MOD1_MASK, (GtkAccelFlags) 0);
    gtk_widget_ref (file);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "file", file,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_add (GTK_CONTAINER (menubar), file);
    gtk_widget_show (file);

    file_menu = gtk_menu_new ();
    gtk_widget_ref (file_menu);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "file_menu", file_menu,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (file), file_menu);
    file_menu_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (file_menu));

    quit = gtk_menu_item_new_with_label ("");
    tmp_key = gtk_label_parse_uline (GTK_LABEL (GTK_BIN (quit)->child),
                                     _("Quit"));
    gtk_widget_add_accelerator (quit, "activate_item", file_menu_accels,
                                tmp_key, 0, 0);
    gtk_widget_ref (quit);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "quit", quit,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_add (GTK_CONTAINER (file_menu), quit);
    gtk_widget_add_accelerator (quit, "activate", accel_group,
                                GDK_q, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
    gtk_widget_show (quit);


    /* Settings sub-menu */
    settings = gtk_menu_item_new_with_label ("");
    tmp_key = gtk_label_parse_uline (GTK_LABEL (GTK_BIN (settings)->child),
                                     _("_Settings"));
    gtk_widget_add_accelerator (settings, "activate_item", accel_group,
                                tmp_key, GDK_MOD1_MASK, (GtkAccelFlags) 0);
    gtk_widget_ref (settings);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "settings", settings,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_add (GTK_CONTAINER (menubar), settings);
    gtk_widget_show (settings);

    settings_menu = gtk_menu_new ();
    gtk_widget_ref (settings_menu);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "settings_menu",
                              settings_menu,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (settings), settings_menu);
    settings_menu_accels =
        gtk_menu_ensure_uline_accel_group (GTK_MENU (settings_menu));

    leap_setting = gtk_check_menu_item_new_with_label ("");
    tmp_key = gtk_label_parse_uline (GTK_LABEL (GTK_BIN (leap_setting)->child),
                                     _("Warn on leap years"));
    gtk_widget_add_accelerator (leap_setting, "activate_item",
                                settings_menu_accels, tmp_key, 0, 0);
    gtk_widget_ref (leap_setting);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "leap_setting",
                              leap_setting,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_add (GTK_CONTAINER (settings_menu), leap_setting);
    gtk_check_menu_item_set_show_toggle (GTK_CHECK_MENU_ITEM (leap_setting),
                                         TRUE);
    gtk_widget_show (leap_setting);

    /* Help sub-menu */
    help = gtk_menu_item_new_with_label ("");
    tmp_key = gtk_label_parse_uline (GTK_LABEL (GTK_BIN (help)->child),
                                     _("_Help"));
    gtk_widget_add_accelerator (help, "activate_item", accel_group,
                                tmp_key, GDK_MOD1_MASK, (GtkAccelFlags) 0);
    gtk_widget_ref (help);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "help", help,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_add (GTK_CONTAINER (menubar), help);
    gtk_widget_show (help);

    help_menu = gtk_menu_new ();
    gtk_widget_ref (help_menu);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "help_menu", help_menu,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (help), help_menu);
    help_menu_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (help_menu));

    manual = gtk_menu_item_new_with_label ("");
    tmp_key = gtk_label_parse_uline (GTK_LABEL (GTK_BIN (manual)->child),
                                     _("Manual"));
    gtk_widget_add_accelerator (manual, "activate_item", help_menu_accels,
                                tmp_key, 0, 0);
    gtk_widget_ref (manual);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "manual", manual,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_add (GTK_CONTAINER (help_menu), manual);
    gtk_widget_add_accelerator (manual, "activate", accel_group,
                                GDK_F1, 0, GTK_ACCEL_VISIBLE);
    gtk_widget_show (manual);

    sep = gtk_menu_item_new ();
    gtk_widget_ref (sep);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "sep", sep,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_add (GTK_CONTAINER (help_menu), sep);
    gtk_widget_set_sensitive (sep, FALSE);
    gtk_widget_show (sep);

    about = gtk_menu_item_new_with_label ("");
    tmp_key = gtk_label_parse_uline (GTK_LABEL (GTK_BIN (about)->child),
                                     _("About ..."));
    gtk_widget_add_accelerator (about, "activate_item", help_menu_accels,
                                tmp_key, 0, 0);
    gtk_widget_ref (about);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "about", about,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_add (GTK_CONTAINER (help_menu), about);
    gtk_widget_show (about);

    /*################### NOTEBOOK ################### */

    /* General setup */
    notebook = gtk_notebook_new ();
    gtk_notebook_set_homogeneous_tabs (GTK_NOTEBOOK (notebook), TRUE);
    gtk_notebook_set_tab_border (GTK_NOTEBOOK (notebook), 5);
    gtk_box_pack_start (GTK_BOX (vbox), notebook, TRUE, TRUE, 0);
    gtk_widget_show (notebook);

    /*################### CONVERTER PAGE ################### */

    /* Converter page label */
    converter_page_label = gtk_label_new (_("Date converter"));
    gtk_widget_ref (converter_page_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "converter_page_label",
                              converter_page_label,
                              (GtkDestroyNotify) gtk_widget_unref);

    /* Table for conversion */
    convert_table = gtk_table_new (10, 3, FALSE);
    gtk_widget_ref (convert_table);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "convert_table",
                              convert_table,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_set_border_width (GTK_CONTAINER (convert_table), 10);
    gtk_table_set_col_spacings (GTK_TABLE (convert_table), 10);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), convert_table,
                              converter_page_label);
    gtk_notebook_set_tab_label_packing (GTK_NOTEBOOK (notebook), convert_table,
                                        TRUE, TRUE, GTK_PACK_START);
    gtk_widget_show (convert_table);

    /*################### GREGORIAN ################### */

    /* Label: Gregorian date */
    gregorian_label = gtk_label_new (_("Gregorian date"));
    gtk_widget_ref (gregorian_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "gregorian_label",
                              gregorian_label,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), gregorian_label, 0, 1,
                               0, 1);
    gtk_misc_set_alignment (GTK_MISC (gregorian_label), 0., 0.5);
    gtk_widget_show (gregorian_label);

    /* Table: Gregorian date */
    gregorian_table = gtk_table_new (4, 4, TRUE);
    gtk_widget_ref (gregorian_table);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "gregorian_table",
                              gregorian_table,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), gregorian_table, 1, 2,
                               0, 1);
    gtk_container_set_border_width (GTK_CONTAINER (gregorian_table), 5);
    gtk_table_set_col_spacings (GTK_TABLE (gregorian_table), 5);
    gtk_widget_show (gregorian_table);

    /* Label: year */
    year_label = gtk_label_new (_("Year"));
    gtk_widget_ref (year_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "year_label",
                              year_label, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (gregorian_table), year_label, 0, 2, 0,
                               1);
    gtk_misc_set_alignment (GTK_MISC (year_label), 0., 0.5);
    gtk_widget_show (year_label);

    /* Label: month */
    month_label = gtk_label_new (_("Month"));
    gtk_widget_ref (month_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "month_label",
                              month_label, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (gregorian_table), month_label, 2, 3,
                               0, 1);
    gtk_misc_set_alignment (GTK_MISC (month_label), 0., 0.5);
    gtk_widget_show (month_label);

    /* Label: day */
    day_label = gtk_label_new (_("Day"));
    gtk_widget_ref (day_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "day_label", day_label,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (gregorian_table), day_label, 3, 4, 0,
                               1);
    gtk_misc_set_alignment (GTK_MISC (day_label), 0., 0.5);
    gtk_widget_show (day_label);

    /* Label: hour */
    hour_label = gtk_label_new (_("Hour"));
    gtk_widget_ref (hour_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "hour_label",
                              hour_label, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (gregorian_table), hour_label, 0, 1, 2,
                               3);
    gtk_misc_set_alignment (GTK_MISC (hour_label), 0., 0.5);
    gtk_widget_show (hour_label);

    /* Label: minute */
    minute_label = gtk_label_new (_("Minute"));
    gtk_widget_ref (minute_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "minute_label",
                              minute_label,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (gregorian_table), minute_label, 1, 2,
                               2, 3);
    gtk_misc_set_alignment (GTK_MISC (minute_label), 0., 0.5);
    gtk_widget_show (minute_label);

    /* Label: second */
    second_label = gtk_label_new (_("Second"));
    gtk_widget_ref (second_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "second_label",
                              second_label,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (gregorian_table), second_label, 2, 4,
                               2, 3);
    gtk_misc_set_alignment (GTK_MISC (second_label), 0., 0.5);
    gtk_widget_show (second_label);

    /* SpinButton: Years limited from -4713 to 8000 */
    year_spinbutton_adj =
        gtk_adjustment_new (gDate->year, -4713., 8000., 1., 10., 10.);
    year_spinbutton =
        gtk_spin_button_new (GTK_ADJUSTMENT (year_spinbutton_adj), 1., 0);
    gtk_widget_ref (year_spinbutton);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "year_spinbutton",
                              year_spinbutton,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (gregorian_table), year_spinbutton, 0,
                               2, 1, 2);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (year_spinbutton), TRUE);
    gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (year_spinbutton),
                                       GTK_UPDATE_IF_VALID);
    gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (year_spinbutton), TRUE);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (year_spinbutton), TRUE);
    gtk_tooltips_set_tip (tooltips, year_spinbutton,
                          _
                          ("Enter here the year part of your Gregorian date (UT)"),
                          NULL);
    gtk_widget_show (year_spinbutton);

    /* SpinButton: Monthes limited from 1 to 12 */
    month_spinbutton_adj =
        gtk_adjustment_new (gDate->month, 1., 12., 1., 10., 10.);
    month_spinbutton =
        gtk_spin_button_new (GTK_ADJUSTMENT (month_spinbutton_adj), 1., 0);
    gtk_widget_ref (month_spinbutton);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "month_spinbutton",
                              month_spinbutton,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (gregorian_table), month_spinbutton, 2,
                               3, 1, 2);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (month_spinbutton), TRUE);
    gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (month_spinbutton),
                                       GTK_UPDATE_IF_VALID);
    gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (month_spinbutton),
                                       TRUE);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (month_spinbutton), TRUE);
    gtk_tooltips_set_tip (tooltips, month_spinbutton,
                          _
                          ("Enter here the month part of your Gregorian date (UT)"),
                          NULL);
    gtk_widget_show (month_spinbutton);

    /* SpinButton: Days limited from 1 to 31 */
    day_spinbutton_adj = gtk_adjustment_new (gDate->day, 1., 31., 1., 10., 10.);
    day_spinbutton =
        gtk_spin_button_new (GTK_ADJUSTMENT (day_spinbutton_adj), 1., 0);
    gtk_widget_ref (day_spinbutton);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "day_spinbutton",
                              day_spinbutton,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (gregorian_table), day_spinbutton, 3,
                               4, 1, 2);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (day_spinbutton), TRUE);
    gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (day_spinbutton),
                                       GTK_UPDATE_IF_VALID);
    gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (day_spinbutton), TRUE);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (day_spinbutton), TRUE);
    gtk_tooltips_set_tip (tooltips, day_spinbutton,
                          _
                          ("Enter here the day part of your Gregorian date (UT)"),
                          NULL);
    gtk_widget_show (day_spinbutton);

    /* SpinButton: Hours limited from 0 to 23 */
    hour_spinbutton_adj =
        gtk_adjustment_new (gDate->hour, 0., 23., 1., 10., 10.);
    hour_spinbutton =
        gtk_spin_button_new (GTK_ADJUSTMENT (hour_spinbutton_adj), 1., 0);
    gtk_widget_ref (hour_spinbutton);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "hour_spinbutton",
                              hour_spinbutton,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (gregorian_table), hour_spinbutton, 0,
                               1, 3, 4);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (hour_spinbutton), TRUE);
    gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (hour_spinbutton),
                                       GTK_UPDATE_IF_VALID);
    gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (hour_spinbutton), TRUE);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (hour_spinbutton), TRUE);
    gtk_tooltips_set_tip (tooltips, hour_spinbutton,
                          _
                          ("Enter here the hour part of your Gregorian date (UT)"),
                          NULL);
    gtk_widget_show (hour_spinbutton);

    /* SpinButton: Minutes limited from 0 to 59 */
    minute_spinbutton_adj =
        gtk_adjustment_new (gDate->min, 0., 59., 1., 10., 10.);
    minute_spinbutton =
        gtk_spin_button_new (GTK_ADJUSTMENT (minute_spinbutton_adj), 1., 0);
    gtk_widget_ref (minute_spinbutton);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "minute_spinbutton",
                              minute_spinbutton,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (gregorian_table), minute_spinbutton,
                               1, 2, 3, 4);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (minute_spinbutton), TRUE);
    gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (minute_spinbutton),
                                       GTK_UPDATE_IF_VALID);
    gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (minute_spinbutton),
                                       TRUE);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (minute_spinbutton), TRUE);
    gtk_tooltips_set_tip (tooltips, minute_spinbutton,
                          _
                          ("Enter here the minute part of your Gregorian date (UT)"),
                          NULL);
    gtk_widget_show (minute_spinbutton);

    /* SpinButton: Seconds limited from 0 to 60 */
    second_spinbutton_adj =
        gtk_adjustment_new (gDate->sec, 0., 60., 1., 10., 10.);
    second_spinbutton =
        gtk_spin_button_new (GTK_ADJUSTMENT (second_spinbutton_adj), 1., 5);
    gtk_widget_ref (second_spinbutton);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "second_spinbutton",
                              second_spinbutton,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (gregorian_table), second_spinbutton,
                               2, 4, 3, 4);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (second_spinbutton), TRUE);
    gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (second_spinbutton),
                                       GTK_UPDATE_IF_VALID);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (second_spinbutton), TRUE);
    gtk_tooltips_set_tip (tooltips, second_spinbutton,
                          _
                          ("Enter here the second part of your Gregorian date (UT)"),
                          NULL);
    gtk_widget_show (second_spinbutton);

    /* Button: Conversion from Gregorian day */
    gregorian_button = gtk_button_new_with_label (_(" Convert "));
    gtk_widget_ref (gregorian_button);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "gregorian_button",
                              gregorian_button,
                              (GtkDestroyNotify) gtk_widget_unref);
    GTK_WIDGET_SET_FLAGS (gregorian_button, GTK_CAN_DEFAULT);
    gtk_table_attach (GTK_TABLE (convert_table), gregorian_button, 2, 3, 0, 1,
                      (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
    gtk_tooltips_set_tip (tooltips, gregorian_button,
                          _("Hit me to convert from a Gregorian date"), NULL);
    gtk_widget_show (gregorian_button);

    /* Horizontal separator */
    hseparator1 = gtk_hseparator_new ();
    gtk_widget_ref (hseparator1);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "hseparator1",
                              hseparator1, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), hseparator1, 0, 3, 1,
                               2);
    gtk_widget_show (hseparator1);

    /*################### JULIAN DAY ################### */

    /* Label: Julian day */
    jd_label = gtk_label_new (_("Julian  day"));
    gtk_widget_ref (jd_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "jd_label", jd_label,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), jd_label, 0, 1, 2, 3);
    gtk_misc_set_alignment (GTK_MISC (jd_label), 0., 0.5);
    gtk_widget_show (jd_label);

    /* TextEntry: Julian day */
    jd_entry = gtk_entry_new_with_max_length (32);
    gtk_widget_ref (jd_entry);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "jd_entry", jd_entry,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), jd_entry, 1, 2, 2, 3);
    gtk_tooltips_set_tip (tooltips, jd_entry,
                          _("Enter here the Julian day to convert"), NULL);
    gtk_widget_show (jd_entry);

    /* Button: Conversion from Julian day */
    jd_button = gtk_button_new_with_label (_(" Convert "));
    gtk_widget_ref (jd_button);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "jd_button",
                              jd_button, (GtkDestroyNotify) gtk_widget_unref);
    GTK_WIDGET_SET_FLAGS (jd_button, GTK_CAN_DEFAULT);
    gtk_table_attach (GTK_TABLE (convert_table), jd_button, 2, 3, 2, 3,
                      (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
    gtk_tooltips_set_tip (tooltips, jd_button,
                          _("Hit me to convert from a Julian day"), NULL);
    gtk_widget_show (jd_button);

    /*################### MODIFIED JULIAN DAY ################### */

    /* Label: Modified Julian day */
    mjd_label = gtk_label_new (_("Modified Julian day"));
    gtk_widget_ref (mjd_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "mjd_label", mjd_label,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), mjd_label, 0, 1, 3,
                               4);
    gtk_misc_set_alignment (GTK_MISC (mjd_label), 0., 0.5);
    gtk_widget_show (mjd_label);

    /* TextEntry: Modified Julian day */
    mjd_entry = gtk_entry_new ();
    gtk_widget_ref (mjd_entry);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "mjd_entry", mjd_entry,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), mjd_entry, 1, 2, 3,
                               4);
    gtk_tooltips_set_tip (tooltips, mjd_entry,
                          _
                          ("Enter here the modified Julian day to convert (origin 1858/11/17)"),
                          NULL);
    gtk_widget_show (mjd_entry);

    /* Button: Conversion from modified Julian day */
    mjd_button = gtk_button_new_with_label (_(" Convert "));
    gtk_widget_ref (mjd_button);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "mjd_button",
                              mjd_button, (GtkDestroyNotify) gtk_widget_unref);
    GTK_WIDGET_SET_FLAGS (mjd_button, GTK_CAN_DEFAULT);
    gtk_table_attach (GTK_TABLE (convert_table), mjd_button, 2, 3, 3, 4,
                      (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
    gtk_tooltips_set_tip (tooltips, mjd_button,
                          _
                          ("Hit me to convert from a modified Julian day"),
                          NULL);
    gtk_widget_show (mjd_button);

    /*################### LOP JULIAN DAY ################### */

    /* Label: LOP Julian day */
    lop_label = gtk_label_new (_("LOP Julian day"));
    gtk_widget_ref (lop_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "lop_label", lop_label,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), lop_label, 0, 1, 4,
                               5);
    gtk_misc_set_alignment (GTK_MISC (lop_label), 0., 0.5);
    gtk_widget_show (lop_label);

    /* TextEntry: LOP Julian day */
    lop_entry = gtk_entry_new ();
    gtk_widget_ref (lop_entry);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "lop_entry", lop_entry,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), lop_entry, 1, 2, 4,
                               5);
    gtk_tooltips_set_tip (tooltips, lop_entry,
                          _
                          ("Enter here the LOP Julian day to convert (origin 1992/01/01)"),
                          NULL);
    gtk_widget_show (lop_entry);

    /* Button: Conversion from LOP Julian day */
    lop_button = gtk_button_new_with_label (_(" Convert "));
    gtk_widget_ref (lop_button);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "lop_button",
                              lop_button, (GtkDestroyNotify) gtk_widget_unref);
    GTK_WIDGET_SET_FLAGS (lop_button, GTK_CAN_DEFAULT);
    gtk_table_attach (GTK_TABLE (convert_table), lop_button, 2, 3, 4, 5,
                      (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
    gtk_tooltips_set_tip (tooltips, lop_button,
                          _("Hit me to convert from a LOP Julian day"), NULL);
    gtk_widget_show (lop_button);

    /*################### CNES JULIAN DAY ################### */

    /* Label: CNES Julian day */
    cnes_label = gtk_label_new (_("CNES Julian day"));
    gtk_widget_ref (cnes_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "cnes_label",
                              cnes_label, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), cnes_label, 0, 1, 5,
                               6);
    gtk_misc_set_alignment (GTK_MISC (cnes_label), 0., 0.5);
    gtk_widget_show (cnes_label);

    /* TextEntry: CNES Julian day */
    cnes_entry = gtk_entry_new ();
    gtk_widget_ref (cnes_entry);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "cnes_entry",
                              cnes_entry, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), cnes_entry, 1, 2, 5,
                               6);
    gtk_tooltips_set_tip (tooltips, cnes_entry,
                          _
                          ("Enter here the CNES Julian day to convert (origin 1950/01/01)"),
                          NULL);
    gtk_widget_show (cnes_entry);

    /* Button: Conversion from CNES Julian day */
    cnes_button = gtk_button_new_with_label (_(" Convert "));
    gtk_widget_ref (cnes_button);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "cnes_button",
                              cnes_button, (GtkDestroyNotify) gtk_widget_unref);
    GTK_WIDGET_SET_FLAGS (cnes_button, GTK_CAN_DEFAULT);
    gtk_table_attach (GTK_TABLE (convert_table), cnes_button, 2, 3, 5, 6,
                      (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
    gtk_tooltips_set_tip (tooltips, cnes_button,
                          _("Hit me to convert from a CNES Julian day"), NULL);
    gtk_widget_show (cnes_button);

    /*################### NASA JULIAN DAY ################### */

    /* Label: NASA Julian day */
    nasa_label = gtk_label_new (_("NASA Julian day"));
    gtk_widget_ref (nasa_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "nasa_label",
                              nasa_label, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), nasa_label, 0, 1, 6,
                               7);
    gtk_misc_set_alignment (GTK_MISC (nasa_label), 0., 0.5);
    gtk_widget_show (nasa_label);

    /* TextEntry: NASA Julian day */
    nasa_entry = gtk_entry_new ();
    gtk_widget_ref (nasa_entry);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "nasa_entry",
                              nasa_entry, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), nasa_entry, 1, 2, 6,
                               7);
    gtk_tooltips_set_tip (tooltips, nasa_entry,
                          _
                          ("Enter here the NASA Julian day to convert (origin 1968/05/24)"),
                          NULL);
    gtk_widget_show (nasa_entry);

    /* Button: Conversion from NASA Julian day */
    nasa_button = gtk_button_new_with_label (_(" Convert "));
    gtk_widget_ref (nasa_button);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "nasa_button",
                              nasa_button, (GtkDestroyNotify) gtk_widget_unref);
    GTK_WIDGET_SET_FLAGS (nasa_button, GTK_CAN_DEFAULT);
    gtk_table_attach (GTK_TABLE (convert_table), nasa_button, 2, 3, 6, 7,
                      (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
    gtk_tooltips_set_tip (tooltips, nasa_button,
                          _("Hit me to convert from a NASA Julian day"), NULL);
    gtk_widget_show (nasa_button);

    /*################### CCSDS JULIAN DAY ################### */

    /* Label: CCSDS Julian day */
    ccsds_label = gtk_label_new (_("CCSDS Julian day"));
    gtk_widget_ref (ccsds_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "ccsds_label",
                              ccsds_label, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), ccsds_label, 0, 1, 7,
                               8);
    gtk_misc_set_alignment (GTK_MISC (ccsds_label), 0., 0.5);
    gtk_widget_show (ccsds_label);

    /* TextEntry: CCSDSJulian day */
    ccsds_entry = gtk_entry_new ();
    gtk_widget_ref (ccsds_entry);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "ccsds_entry",
                              ccsds_entry, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), ccsds_entry, 1, 2, 7,
                               8);
    gtk_tooltips_set_tip (tooltips, ccsds_entry,
                          _
                          ("Enter here the CCSDS Julian day to convert (origin 1958/01/01)"),
                          NULL);
    gtk_widget_show (ccsds_entry);

    /* Button: Conversion from CCSDS Julian day */
    ccsds_button = gtk_button_new_with_label (_(" Convert "));
    gtk_widget_ref (ccsds_button);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "ccsds_button",
                              ccsds_button,
                              (GtkDestroyNotify) gtk_widget_unref);
    GTK_WIDGET_SET_FLAGS (ccsds_button, GTK_CAN_DEFAULT);
    gtk_table_attach (GTK_TABLE (convert_table), ccsds_button, 2, 3, 7, 8,
                      (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
    gtk_tooltips_set_tip (tooltips, ccsds_button,
                          _("Hit me to convert from a CCSDS Julian day"), NULL);
    gtk_widget_show (ccsds_button);

    /* Horizontal separator */
    hseparator2 = gtk_hseparator_new ();
    gtk_widget_ref (hseparator2);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "hseparator2",
                              hseparator2, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), hseparator2, 0, 3, 8,
                               9);
    gtk_widget_show (hseparator2);

    /*################### TOPEX/POSEIDON ################### */

    /* Label: Topex cycle and pass */
    cycle_pass_label = gtk_label_new (_("TOPEX Cycle & Pass"));
    gtk_widget_ref (cycle_pass_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "cycle_pass_label",
                              cycle_pass_label,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), cycle_pass_label, 0,
                               1, 9, 10);
    gtk_misc_set_alignment (GTK_MISC (cycle_pass_label), 0., 0.5);
    gtk_widget_show (cycle_pass_label);

    /* Table: Topex cycle and pass */
    cycle_table = gtk_table_new (2, 2, FALSE);
    gtk_widget_ref (cycle_table);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "cycle_table",
                              cycle_table, (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_set_border_width (GTK_CONTAINER (cycle_table), 5);
    gtk_table_attach_defaults (GTK_TABLE (convert_table), cycle_table, 1, 2, 9,
                               10);
    gtk_widget_show (cycle_table);

    /* Label: Topex cycle */
    cycle_label = gtk_label_new (_("Cycle"));
    gtk_widget_ref (cycle_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "cycle_label",
                              cycle_label, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (cycle_table), cycle_label, 0, 1, 0,
                               1);
    gtk_misc_set_alignment (GTK_MISC (cycle_label), 0., 0.5);
    gtk_widget_show (cycle_label);

    /* Label: Topex pass */
    pass_label = gtk_label_new (_("Pass"));
    gtk_widget_ref (pass_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "pass_label",
                              pass_label, (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (cycle_table), pass_label, 1, 2, 0, 1);
    gtk_misc_set_alignment (GTK_MISC (pass_label), 0., 0.5);
    gtk_widget_show (pass_label);

    /* SpinButton: Topex cycle */
    cycle_spinbutton_adj = gtk_adjustment_new (0., 0., MAX_CYCLE, 1., 10., 10.);
    cycle_spinbutton =
        gtk_spin_button_new (GTK_ADJUSTMENT (cycle_spinbutton_adj), 1., 0);
    gtk_widget_ref (cycle_spinbutton);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "cycle_spinbutton",
                              cycle_spinbutton,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (cycle_table), cycle_spinbutton, 0, 1,
                               1, 2);
    gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (cycle_spinbutton),
                                       GTK_UPDATE_IF_VALID);
    gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (cycle_spinbutton),
                                       TRUE);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (cycle_spinbutton), TRUE);
    gtk_tooltips_set_tip (tooltips, cycle_spinbutton,
                          _("Enter here your TOPEX/POSEIDON cycle number"),
                          NULL);
    gtk_widget_show (cycle_spinbutton);

    /* SpinButton: Topex pass */
    pass_spinbutton_adj = gtk_adjustment_new (0., 0., 254., 1., 10., 10.);
    pass_spinbutton =
        gtk_spin_button_new (GTK_ADJUSTMENT (pass_spinbutton_adj), 1., 0);
    gtk_widget_ref (pass_spinbutton);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "pass_spinbutton",
                              pass_spinbutton,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_table_attach_defaults (GTK_TABLE (cycle_table), pass_spinbutton, 1, 2,
                               1, 2);
    gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (pass_spinbutton),
                                       GTK_UPDATE_IF_VALID);
    gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (pass_spinbutton), TRUE);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (pass_spinbutton), TRUE);
    gtk_tooltips_set_tip (tooltips, pass_spinbutton,
                          _("Enter here your TOPEX/POSEIDON pass number"),
                          NULL);
    gtk_widget_show (pass_spinbutton);

    /* Button: Conversion from TOPEX cycle and pass */
    topex_button = gtk_button_new_with_label (_(" Convert "));
    gtk_widget_ref (topex_button);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "topex_button",
                              topex_button,
                              (GtkDestroyNotify) gtk_widget_unref);
    GTK_WIDGET_SET_FLAGS (topex_button, GTK_CAN_DEFAULT);
    gtk_table_attach (GTK_TABLE (convert_table), topex_button, 2, 3, 9, 10,
                      (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
    gtk_tooltips_set_tip (tooltips, topex_button,
                          _
                          ("Hit me to convert from TOPEX/POSEIDON cycle and pass numbers"),
                          NULL);
    gtk_widget_show (topex_button);


    /*################### CALENDAR PAGE ################### */

    /* Calendar page label */
    calendar_page_label = gtk_label_new (_("Calendar"));
    gtk_widget_ref (calendar_page_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "calendar_page_label",
                              calendar_page_label,
                              (GtkDestroyNotify) gtk_widget_unref);

    /* Calendar vertical box */
    calendar_vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_ref (calendar_vbox);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "calendar_vbox",
                              calendar_vbox,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), calendar_vbox,
                              calendar_page_label);
    gtk_notebook_set_tab_label_packing (GTK_NOTEBOOK (notebook), calendar_vbox,
                                        TRUE, TRUE, GTK_PACK_START);
    gtk_widget_show (calendar_vbox);

    /* Calendar */
    calendar = gtk_calendar_new ();
    gtk_widget_ref (calendar);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "calendar", calendar,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_calendar_display_options (GTK_CALENDAR (calendar),
                                  GTK_CALENDAR_SHOW_HEADING
                                  | GTK_CALENDAR_SHOW_DAY_NAMES |
                                  GTK_CALENDAR_SHOW_WEEK_NUMBERS |
                                  GTK_CALENDAR_WEEK_START_MONDAY);
    /* initial values */
    gtk_calendar_select_month (GTK_CALENDAR (calendar), gDate->month - 1,
                               gDate->year);
    gtk_calendar_select_day (GTK_CALENDAR (calendar), gDate->day);
    gtk_box_pack_start (GTK_BOX (calendar_vbox), calendar, FALSE, TRUE, 10);

    gtk_widget_show (calendar);

    /* Leap year label */
    leap_label = gtk_label_new ("");
    gtk_widget_ref (leap_label);
    gtk_object_set_data_full (GTK_OBJECT (main_window), "leap_label",
                              leap_label, (GtkDestroyNotify) gtk_widget_unref);
    gtk_misc_set_alignment (GTK_MISC (leap_label), 0.5, 0.5);
    gtk_box_pack_start (GTK_BOX (calendar_vbox), leap_label, TRUE, TRUE, 10);
    gtk_widget_show (leap_label);

    /* misc */
    g_free (gDate);

    /*################### SIGNAL HANDLING ################### */

    /* Misc */
    gtk_signal_connect (GTK_OBJECT (main_window), "destroy",
                        GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

    gtk_signal_connect (GTK_OBJECT (main_window), "delete_event",
                        GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

    /* Menus */
    gtk_signal_connect (GTK_OBJECT (quit), "activate",
                        GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

    gtk_signal_connect (GTK_OBJECT (manual), "activate",
                        GTK_SIGNAL_FUNC (on_manual_activate), NULL);

    gtk_signal_connect (GTK_OBJECT (about), "activate",
                        GTK_SIGNAL_FUNC (on_about_activate), NULL);

    /* Convert buttons */
    gtk_signal_connect (GTK_OBJECT (gregorian_button), "clicked",
                        GTK_SIGNAL_FUNC (on_convert_gregorian_clicked), NULL);

    gtk_signal_connect (GTK_OBJECT (jd_button), "clicked",
                        GTK_SIGNAL_FUNC (on_convert_jd_clicked), NULL);

    gtk_signal_connect (GTK_OBJECT (mjd_button), "clicked",
                        GTK_SIGNAL_FUNC (on_convert_mjd_clicked), NULL);

    gtk_signal_connect (GTK_OBJECT (lop_button), "clicked",
                        GTK_SIGNAL_FUNC (on_convert_lop_clicked), NULL);

    gtk_signal_connect (GTK_OBJECT (cnes_button), "clicked",
                        GTK_SIGNAL_FUNC (on_convert_cnes_clicked), NULL);

    gtk_signal_connect (GTK_OBJECT (nasa_button), "clicked",
                        GTK_SIGNAL_FUNC (on_convert_nasa_clicked), NULL);

    gtk_signal_connect (GTK_OBJECT (ccsds_button), "clicked",
                        GTK_SIGNAL_FUNC (on_convert_ccsds_clicked), NULL);

    gtk_signal_connect (GTK_OBJECT (topex_button), "clicked",
                        GTK_SIGNAL_FUNC (on_convert_topex_clicked), NULL);

    return main_window;
}
