/*
 * This file is part of TraceProto.
 * Copyright 2004 Eric Hope
 * skippysaurus@skippylair.net
 *
 * TraceProto is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * TraceProto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TraceProto; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <libnet.h>
#include <pcap.h>
#include <unistd.h>
#include <errno.h>
#include <sys/time.h>
#include <stdio.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <ctype.h>

#include "traceproto.h"
#include "config.h"


/*
 * The report_* functions are the various possible output
 * formats that can be specified by the -o flag.  The first
 * parameter is one of the output_items enumerated in traceproto.h
 * These must match the *_packet_id array provided by each of the
 * report functions.  Thus it should be easy to add new report
 * formats.  The third parameter similarly should match one of
 * the packet_numbers from traceproto.h.
 */

/* spit out the header and nothing else (accept whatever accounting) */
void report_none ( int output_item, struct in_addr * from, int packet_type )
{
	if ( output_item == TP_OUT_HEADER )
	{
		printf ( "%s to %s (%s), %d hops max\n",
			state.prog,
			behavior.target,
			behavior.target_reverse,
			behavior.max_ttl );
	}

	fflush ( stdout );

	return;
}

/* spit out a single char, all responses for each hop on the same line */
void report_minimum ( int output_item, struct in_addr * from, int packet_type )
{
	char * minimal_packet_id[] = {
		" ",
		"*",
		"S",
		"A",
		"S",
		"R",
		"R",
		"F",
		"U",
		"X",
		"P",
		"H",
		"N",
		"I",
		"?"
	};

	switch ( output_item )
	{
	case TP_OUT_HEADER:
		break;
	case TP_OUT_HOP_NUMBER:
		printf ( "%d  ", state.current_hop );
		break;
	case TP_OUT_HOP_INFO:
		printf ( "%s ", minimal_packet_id [ packet_type ] );
		if ( ( state.packets_this_hop + 1 ) == behavior.packets_per_hop )
			printf ( "\n" );
	}

	fflush ( stdout );

	return;
}

/* minimal output designed for scripting, not nessasarily
 * human readability */
void report_scriptable ( int output_item, struct in_addr * from, int packet_type )
{
	char * script_packet_id[] = {
		"NULL",
		"NR",
		"TS",
		"TA",
		"TSA",
		"TR",
		"TAR",
		"TF",
		"U",
		"ITX",
		"IPU",
		"IHU",
		"INU",
		"I",
		"O",
	};
        switch ( output_item )
        {
        case TP_OUT_HEADER:
                break;
        case TP_OUT_HOP_NUMBER:
                break;
	case TP_OUT_HOP_INFO:
		if ( packet_type == TP_TYPE_NR )
		{
			printf ( "%d 0.0.0.0 %s 0\n",
				state.current_hop,
				script_packet_id [ packet_type ] );
		} else {
			printf ( "%d %s %s %3g\n",
				state.current_hop,
				inet_ntoa ( * from ),
				script_packet_id [ packet_type ],
				state.trip_time );
		}
	}
	fflush ( stdout );
			
	return;
}

/* as close as possible to the original traceroute
 * needs a bit of work to make it match exactly */
void report_classic ( int output_item, struct in_addr * from, int packet_type )
{
	static struct in_addr previous_in_addr;

	char * classic_packet_id[] = {
		" ",
		"*",
		" ",
		" ",
		" ",
		"!R",
		"!R",
		" ",
		" ",
		" ",
		"!P",
		"!H",
		"!N",
		" ",
		"!?"
	};

	switch ( output_item  )
	{
	case TP_OUT_HEADER:
		printf ( "%s to %s (%s), %d hops max\n",
			state.prog,
			behavior.target,
			behavior.target_reverse,
			behavior.max_ttl );
		break;
	case TP_OUT_HOP_NUMBER:
		printf ( "%d  ", state.current_hop );
		previous_in_addr.s_addr = ( unsigned long ) NULL;
		break;
	case TP_OUT_HOP_INFO:
		if ( from != NULL && previous_in_addr.s_addr != from->s_addr )
		{
			if ( previous_in_addr.s_addr != ( unsigned long ) NULL )
				printf ( "\n\t" );
			printf ( "%s (%s)  ",
				libnet_addr2name4 ( from->s_addr, behavior.libnet_resolve_choice ),
				inet_ntoa ( * from ) );
		}
		if ( packet_type == TP_TYPE_NR )
			printf ( "*  " );
		else
			printf ( "%3g ms %s ",
				state.trip_time,
				classic_packet_id [ packet_type ] );

		if ( ( state.packets_this_hop + 1 ) >= behavior.packets_per_hop )
			printf ( "\n" );

		break;
	}

	fflush ( stdout );
	if ( from != 0 )
		previous_in_addr = * from;

	return;
}

/*
 * traceproto standard format, somewhat more readable than the original
 * but still fairly space efficient
 */
void report_std ( int output_item, struct in_addr * from, int packet_type )
{
	static struct in_addr previous_in_addr;

	char * std_packet_id[] = {
		"null",
		"no response",
		"TCP Syn",
		"TCP Ack",
		"TCP Syn Ack",
		"TCP Reset",
		"TCP Ack Reset",
		"TCP Fin",
		"UDP",
		"ICMP Time Exceeded",
		"Port Unreachable",
		"Host Unreachable",
		"Network Unreachable",
		"ICMP",
		"Unknown"
	};

	switch ( output_item )
	{
	case TP_OUT_HEADER:
		printf ( "%s: trace to %s (%s), port %d\n",
			state.prog,
			behavior.target,
			behavior.target_reverse,
			packet.dst_port );
		break;
	case TP_OUT_HOP_NUMBER:
		printf ( "ttl  %d:", state.current_hop );
		previous_in_addr.s_addr = ( unsigned long ) NULL;
		break;
	case TP_OUT_HOP_INFO:
		if ( from != NULL && previous_in_addr.s_addr != from->s_addr )
		{
			if ( previous_in_addr.s_addr != ( unsigned long ) NULL )
				printf ( "\n\t" );
			printf ( "  %s from %s (%s)\n",
				std_packet_id [ packet_type ],
				libnet_addr2name4 ( from->s_addr, behavior.libnet_resolve_choice ),
				inet_ntoa ( * from ) );
		}
		if ( packet_type == TP_TYPE_NR )
	//		printf ( "\t%s", std_packet_id [ packet_type ] );
			printf ( "%s     ", std_packet_id [ packet_type ] );
		else
			printf ( "\t%#.5g ms", state.trip_time );

		if ( ( state.packets_this_hop + 1 ) == behavior.packets_per_hop )
			printf ( "\n" );
		break;
	}

	fflush ( stdout );
	if ( from != 0 )
		previous_in_addr = * from;

	return;
}

/*
 * the exec summary version, each response on a separate line with an 
 * ascii line showing the time
 */
void report_graphic ( int output_item, struct in_addr * from, int packet_type )
{
	int i;
	static struct in_addr previous_in_addr;

	char * graphic_packet_id[] = {
		"null",
		"no response",
		"TCP Syn",
		"TCP Ack",
		"TCP Syn Ack",
		"TCP Reset",
		"TCP Ack Reset",
		"TCP Fin",
		"UDP",
		"ICMP Time Exceeded",
		"Port Unreachable",
		"Host Unreachable",
		"Network Unreachable",
		"ICMP",
		"Unknown"
	};

	switch ( output_item )
	{
	case TP_OUT_HEADER:
		printf ( "%s: trace to %s, port %d\n",
			state.prog,
			behavior.target,
			packet.dst_port );
		break;
	case TP_OUT_HOP_NUMBER:
		printf ( "ttl  %d:  ", state.current_hop );
		break;
	case TP_OUT_HOP_INFO:
		if ( from != NULL && previous_in_addr.s_addr != from->s_addr )
		{
			printf ( "%s from %s (%s)\n",
				graphic_packet_id [ packet_type ],
				libnet_addr2name4 ( from->s_addr, behavior.libnet_resolve_choice ),
				inet_ntoa ( * from ) );
		}
		if ( packet_type == TP_TYPE_NR )
			state.trip_time = 0;

		for ( i = 0; i < ( int ) state.trip_time; i++ )
			printf ( "#" );

		printf ( "  %#.5g ms\n", state.trip_time );

		if ( ( state.packets_this_hop + 1 ) == behavior.packets_per_hop )
			printf ( "\n" );
		break;
	}

	fflush ( stdout );
	if ( from != 0 )
		previous_in_addr = * from;

	return;
}


/*
 * the "how to do what" section
 */
void usage( char * prog )
{
	int i;
char * usage_str[] = {
	"usage:",
	"\ttraceproto [options] destination",
	"\t-p protocol (default tcp)",
	"\t\tcurrently available: tcp, udp, and icmp",
	"\t-d minimum (or only) destination port (default 80)",
	"\t-D maximum destination port",
	"\t-s minimum source port (default 10240)",
	"\t-S maximum source port",
	"\t-m minimum ttl (default 1)",
	"\t-M maximum ttl (default 30)",
	"\t-w wait timeout (default 5 sec)",
	"\t-W wait timeout before sending a new packet (default 0 ms)",
	"\t-a hop accounting (default 2)",
	"\t\t 0 == no accounting",
	"\t\t 1 == total only",
	"\t\t 2 == full accounting",
	"\t-P payload byte count (default 12)",
	"\t-k comma separated list of hops to skip (default none)",
	"\t-c trace continuously without accounting",
	"\t-C continuous trace with accounting",
	"\t-I number of reoccuring traces",
	"\t-H packets per hop (default 3)",
	"\t-f set the don't fragment bit",
	"\t-i incr/decr source/destination ports",
	"\t\t s == decrement source port",
	"\t\t S == increment source port (default)",
	"\t\t d == decrement destination port",
	"\t\t D == increment destination port",
	"\t\t n == static source port",
	"\t\t N == static destination port (default)",
	"\t\t Note: nN may result in inaccurate responses",
	"\t-o output style",
	"\t\t s == standard output (default)",
	"\t\t g == graphical output",
	"\t\t c == classic output",
	"\t\t m == minimal output",
	"\t\t p == scriptable output",
	"\t\t n == no individual hop output, accounting only",
	"\t-t tcp flags to use",
	"\t\t S == SYN (default)",
	"\t\t A == ACK",
	"\t\t R == RST",
	"\t\t U == URG",
	"\t\t P == PUSH",
	"\t\t F == FIN",
	"\t-R start at max ttl and decrement",
	"\t-h this help message",
	"\t-v version info",
	"",
	"Note that you can set generally set illogical or contradictory",
	"options if you wish, combining -d 80 and -p icmp is permissible",
	"but silly.  The option that doesn't make sense is usually ignored.",
	"In the case of contradictory options, the last one seen is",
	"generally authoritative",
	"" };

	for ( i = 0; i < sizeof (  usage_str ) / sizeof ( char * ); i++ )
		printf ( "%s\n", usage_str [ i ] );
}

/*
 * take a comma separated list (sans spaces) and give them a miss
 * (ie dont send probes with the ttl set to these numbers)
 */
int parse_skips ( char * skip_ptr )
{
	int c;
	char tmp[ 4 ];
	int tmp_c = 0;

	behavior.skips = ( int * ) calloc ( behavior.max_ttl + 1, sizeof ( int ) );
	if ( behavior.skips == NULL )
	{
		printf ( "memory allocation error" );
		tixe ( tixe_cleanup, 1 );
	}

	tixe_cleanup.skips_free = YES;

	/*
	 * loop through the str from the cmd line
	 * if its a digit remember it
	 * if its a comma, add it to the list of skipped hops
	 */
	memset ( & tmp, '\0', sizeof ( tmp ) );
	for ( c = 0; c < strlen ( skip_ptr ); c++ )
	{
		if ( isdigit ( skip_ptr[ c ] ) && tmp_c < 4 )
		{
			tmp[ tmp_c++ ] = skip_ptr[ c ];
		}
		else if ( isdigit ( skip_ptr[ c ] ) && tmp_c >= 4 )
		{
			return 1;
		}
		else if ( skip_ptr[ c ] == ',' )
		{
			tmp_c = atoi ( tmp );
			if ( tmp_c > 0 && tmp_c <= behavior.max_ttl )
			{
				behavior.skips[ tmp_c ] = 1;
			}
			else
			{
				return 1;
			}
			tmp_c = 0;
			memset ( & tmp, '\0', sizeof ( tmp ) );
		}
		else
		{
			return 1;
		}
	} // end of for loop

	// one more time so it doesn't require a trailing comma
	if ( tmp_c != 0 )
	{
		tmp_c = atoi ( tmp );
		if ( tmp_c > 0 && tmp_c <= behavior.max_ttl )
		{
			behavior.skips[ tmp_c ] = 1;
		}
		else
		{
			return 1;
		}
		tmp_c = 0;
		memset ( & tmp, '\0', sizeof ( tmp ) );
	}

	return 0;
}

/*
 * remember if the user hits it
 */
void ctrl_c ()
{
	behavior.do_audit_exit = YES;
}
void ctrl_z ()
{
	behavior.do_audit = YES;
}

/*
 * one timeval - another timeval
 */
double diff_time ( struct timeval * start, struct timeval * end )
{
	register double diff;

	diff = ( double )( end->tv_sec  - start->tv_sec  ) * 1000.0
	     + ( double )( end->tv_usec - start->tv_usec ) / 1000.0;

	return diff;
}

/*
 * create and apply the Berkley Packet Filter filter
 */
int do_filter()
{
	char filter [ FILTERSIZE ];
	struct bpf_program compiled_filter;
	bpf_u_int32 netmask = 0;

	sprintf ( filter,		/* the buffer fed to pcap */
		behavior.filter_text,	/* the static text set on traceproto.c */
		behavior.target,
		packet.src_port,
		packet.dst_port,
		packet.ip_id );		/* the randomly generated id */

	if ( pcap_compile(
		state.psocket,		 /* pcap_t */
		& compiled_filter,
		filter,
		0,			/* don't optimise */
		netmask ) == -1 )
	{
		printf("pcap error: %s\n", pcap_geterr( state.psocket ) );
		return 1;
	}

	if ( pcap_setfilter(
		state.psocket,
		& compiled_filter ) != 0 )
	{
		printf("pcap error: %s\n", pcap_geterr( state.psocket ) );
		return 1;
	}
	return 0;
}

/*
 * shove the details into memory
 */
int account_packet ( double return_time )
{

/*
 * record total for all hops
 */
	if ( return_time == ( double ) 0 )
		state.hop_record[ 0 ].lost_packets++;
	else
		state.hop_record[ 0 ].num_packets++;

	if ( state.hop_record[ 0 ].min_time == ( double ) 0
			|| state.hop_record[ 0 ].min_time > return_time )
		state.hop_record[ 0 ].min_time = return_time;

	if ( state.hop_record[ 0 ].max_time < return_time )
		state.hop_record[ 0 ].max_time = return_time;

	state.hop_record[ 0 ].ave_time =
		      ( state.hop_record[ 0 ].ave_time
		*	state.hop_record[ 0 ].num_packets
		+	return_time )
		/     ( state.hop_record[ 0 ].num_packets
		+	1 );

/*
 * record the individual hop stats
 */
	state.hop_record[ state.current_hop ].distance = state.current_hop;

	/*
	 * if there was no response, record the lost packet, but not the time
	 */
	if ( return_time == ( double ) 0 )
	{
		state.hop_record[ state.current_hop ].lost_packets++;
		return 0;
	}
	else
		state.hop_record[ state.current_hop ].num_packets++;

	if ( state.hop_record[ state.current_hop ].min_time == (double)0
		|| state.hop_record[ state.current_hop ].min_time > return_time )
	{
		state.hop_record[ state.current_hop ].min_time = return_time;
	}

	if ( state.hop_record[ state.current_hop ].max_time < return_time )
		state.hop_record[ state.current_hop ].max_time = return_time;

	/*
	 * math: ave = ( ave * (#pkts - 1) + response ) / #pkts
	 */
	state.hop_record[ state.current_hop ].ave_time =
			( state.hop_record[ state.current_hop ].ave_time
		*	( state.hop_record[ state.current_hop ].num_packets
		-	  1 )
		+	  return_time )
		/	( state.hop_record[ state.current_hop ].num_packets );

	return 1;
}

/*
 * formal info on all packets seen so far
 * ( may or may not indicate the end of the run )
 */
void hop_audit()
{
	int hop;

	if ( behavior.account_level > TP_ACCOUNT_NONE )
	{
		printf ( "\nhop :  min   /  ave   /  max   :  # packets  :  # lost\n" );
	}

	if ( behavior.account_level == TP_ACCOUNT_FULL )
	{
		printf (     "-------------------------------------------------------\n" );
		for ( hop = behavior.min_ttl; hop <= behavior.max_ttl; hop++ )
		{
			if ( state.hop_record[ hop ].num_packets != 0
				|| state.hop_record[ hop ].lost_packets != 0 )
			{
				printf ( "%3d : %#.5g / %#.5g / %#.5g : %3d packets : %3d lost\n",
					hop,
					state.hop_record[ hop ].min_time,
					state.hop_record[ hop ].ave_time,
					state.hop_record[ hop ].max_time,
					state.hop_record[ hop ].num_packets,
					state.hop_record[ hop ].lost_packets );
			}

		} // end for hop loop
	} // end if full accounting

	if ( behavior.account_level > TP_ACCOUNT_NONE )
	{
		printf (     "------------------------Total--------------------------\n" );
		if ( state.hop_record[ 0 ].num_packets != 0
			|| state.hop_record[ 0 ].lost_packets != 0 )
		{
			printf ( "total %#.5g / %#.5g / %#.5g : %3d packets : %3d lost\n",
				state.hop_record[ 0 ].min_time,
				state.hop_record[ 0 ].ave_time,
				state.hop_record[ 0 ].max_time,
				state.hop_record[ 0 ].num_packets,
				state.hop_record[ 0 ].lost_packets );
		}
		printf ( "\n" );
	} // end if ( account level > TP_ACCOUNT_NONE )

	if ( behavior.do_audit_exit == YES )
	{
		printf ( "\n" );
		tixe ( tixe_cleanup, 0 );
	}

	behavior.do_audit = NO;
}

/*
 * take the name used to call and make it canonical
 */
int reg_name ( char * arg0 )
{
	int c;
	state.prog = & arg0 [ 0 ];
	for ( c = 0; c < strlen ( arg0 ); c++ )
	{
		if ( arg0[ c ] == '/' )
		{
			state.prog = & arg0[ c+1 ];
		}
	}

	return strlen ( state.prog );
}

/*
 * cleanup our mess and vamoos
 */
void tixe ( struct cleanup clean_list, int exit_status )
{
	if ( clean_list.payload_free      == YES ) { free ( packet.payload     ); }
	if ( clean_list.skips_free        == YES ) { free ( behavior.skips     ); }
	if ( clean_list.hop_record_free   == YES ) { free ( state.hop_record   ); }
	if ( clean_list.fake_psocket_free == YES ) { free ( state.fake_psocket ); }

	if ( clean_list.libnet_cleanup  == YES ) { libnet_destroy ( state.packet ); }
	if ( clean_list.pcap_cleanup    == YES ) { pcap_close ( state.psocket );    }

	exit ( exit_status );
}

/*
 * who are we
 */
void version( char * prog )
{
//        char * version = "0.9.0beta0";
//        char * release_date = "July 19, 2004";
//
//        printf("%s version %s, released %s\n", prog, version, release_date );
        printf("%s version %s, released %s\n", PACKAGE, VERSION, RELEASE_DATE );
}

/*
 * sort out which tcp flags to use
 * called from the cmd line flags parsing
 */
u_char parse_flags ( char * user_flags )
{
	int i, errors = 0;
	u_char packet_flags = '\0';

	for ( i = strlen ( user_flags ); i > 0; i-- )
	{
		switch ( user_flags [ 0 ] )
		{
		case 'S':
		case 's':
			packet_flags |= TH_SYN;
			break;
		case 'A':
		case 'a':
			packet_flags |= TH_ACK;
			break;
		case 'R':
		case 'r':
			packet_flags |= TH_RST;
			break;
		case 'U':
		case 'u':
			packet_flags |= TH_URG;
			break;
		case 'p':
		case 'P':
			packet_flags |= TH_PUSH;
			break;
		case 'F':
		case 'f':
			packet_flags |= TH_FIN;
			break;
		default:
			errors++;
			break;
		}
		user_flags++;
	}

	return packet_flags;
}
