/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    Copyright (C) 2009-2012 Martin Brehm

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "microhet.h"
#include "travis.h"
#include "maintools.h"


CMicroHet::CMicroHet()
{
}


CMicroHet::~CMicroHet()
{
}


CMicroHetObservation::CMicroHetObservation()
{
}


CMicroHetObservation::~CMicroHetObservation()
{
}


void CMicroHet::Parse()
{
	CMicroHetObservation *o;
	int z, i1, i2, i3;

	mprintf(WHITE,">>> Microheterogeneity Analysis >>>\n\n");

	m_bSphere = AskYesNo("    Use cuboid (n) or sphere (y)? [no] ",false);

	m_iRes = AskUnsignedInteger("    Please enter the spatial resolution per dimension: [100] ",100);
	m_iResSqr = m_iRes * m_iRes;
	m_iResTri = m_iResSqr * m_iRes;
	m_fBinVolume = g_fBoxX * g_fBoxY * g_fBoxZ / 1e9 / m_iRes / m_iRes / m_iRes;

	mprintf("\n    One bin has the spatial extent of %.3f x %.3f x %.3f pm.\n",g_fBoxX/m_iRes,g_fBoxY/m_iRes,g_fBoxZ/m_iRes);
	mprintf("    This equals a volume of %.3f pm^3.\n",m_fBinVolume*1E9);

	mprintf("\n    This will require %s of RAM.\n\n",FormatBytes((double)m_iResTri*sizeof(MH_BIN_TYPE)));

	if (AskYesNo("    Use multiple analysis boxes with different sizes (y/n)? [no] ",false))
	{
		if (AskYesNo("    Use equidistant box sizes (y) or enter sizes manually (n)? [yes] ",true))
		{
			i1 = AskUnsignedInteger("    Enter smallest box size (in bins): [1] ",1);
			i2 = AskUnsignedInteger("    Enter box size increment (in bins): [1] ",1);
			i3 = AskUnsignedInteger("    Enter increment count: [%d] ",(m_iRes-i1)/i2+1,(m_iRes-i1)/i2+1);
			for (z=0;z<i3;z++)
				m_iaCuboidLength.Add(i1+z*i2);
		} else
		{
			do {
				i1 = AskUnsignedInteger("    Enter box size (in bins): [finished] ",0);
				if (i1 != 0)
					m_iaCuboidLength.Add(i1);
			} while (i1 != 0);
		}
	} else
	{
		i1 = AskUnsignedInteger("    Please enter the edge length of the analysis box (in bins): [%d] ",m_iRes/2,m_iRes/2);
		m_iaCuboidLength.Add(i1);
	}

	mprintf("\n    Using %d different analysis boxes:\n\n",m_iaCuboidLength.GetSize());
	for (z=0;z<m_iaCuboidLength.GetSize();z++)
	{
		mprintf("    - %3d bins edge length (%8.3f x %8.3f x %8.3f pm)\n",m_iaCuboidLength[z],g_fBoxX/m_iRes*m_iaCuboidLength[z],g_fBoxY/m_iRes*m_iaCuboidLength[z],g_fBoxZ/m_iRes*m_iaCuboidLength[z]);
	}

	do {
		mprintf(WHITE,"\n*** Analysis %d ***\n\n",m_oaObservations.GetSize()+1);
		o = new CMicroHetObservation();
		o->m_pParent = this;
		o->Parse();
		m_oaObservations.Add(o);
		mprintf("\n");
	} while (AskYesNo("    Add another microheterogeneity analysis (y/n)? [no] ",false));

	mprintf(WHITE,"\n<<< End of Microheterogeneity Analysis <<<\n\n");
}


void CMicroHetObservation::Parse()
{
	int z, z2, z3, c;
	CAtomGroup *ag;
	char buf[256];

	m_bMass = AskYesNo("    Use mass (y) or atom count (n)? [yes] ",true);
	m_bDensity = AskYesNo("    Use density (y) or absolute value (n)? [yes] ",true);

	if (m_bDensity)
		m_bUniform = AskYesNo("    Take density relative to uniform density (y) or absolute values (n)? [yes] ",true);

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		mprintf("\n");
		if (AskYesNo("    Use atoms from molecule %s (y/n)? [yes] ",true,((CMolecule*)g_oaMolecules[z])->m_sName))
		{
			ag = new CAtomGroup();
_a:
			AskString("      Which atoms to use from this molecule? [all] ",buf,"*");
			if (!ag->ParseAtoms((CMolecule*)g_oaMolecules[z],buf))
				goto _a;
			m_oaAtomGroups.Add(ag);
		}
	}

	BuildName();

	mprintf("\n    Observing %s.\n",m_sName);

	c = 0;
	for (z2=0;z2<m_oaAtomGroups.GetSize();z2++)
	{
		ag = (CAtomGroup*)m_oaAtomGroups[z2];
		for (z3=0;z3<ag->m_oaAtoms.GetSize();z3++)
		{
			if (m_bMass)
				c += (int)(((CAtom*)g_oaAtoms[ag->m_baRealAtomType[z3]])->m_pElement->m_fMass+0.5) * ((CxIntArray*)ag->m_oaAtoms[z3])->GetSize() * ag->m_pMolecule->m_laSingleMolIndex.GetSize();
					else c += ((CxIntArray*)ag->m_oaAtoms[z3])->GetSize() * ag->m_pMolecule->m_laSingleMolIndex.GetSize();
		}
	}
	mprintf("\n    Largest expected bin value is %d.\n",c);
	mprintf("    Maximum allowed bin value is %.0f.\n\n",pow(256.0,sizeof(MH_BIN_TYPE)));
	m_iHistoRes = c+1;
	m_fMaxVal = c;

	if (c >= pow(256.0,sizeof(MH_BIN_TYPE)))
	{
		AskYesNo("    WARNING: Bin overflow may occur (adjust MH_BIN_TYPE in microhet.h)! [Ok] ",true);
		mprintf("\n");
	}

	if (m_pParent->m_iaCuboidLength.GetSize() > 1)
		m_bCDF = AskYesNo("    Create two-dimensional distribution function (y/n)? [yes] ",true);
			else m_bCDF = false;

	if (m_bCDF)
	{
		mprintf("\n");
		m_bEdgeUniform = AskYesNo("    Take cube edge length relative to simulation box (y) or absolute (n)? [yes] ",true);
		m_iCDFRes = AskUnsignedInteger("    Please enter vertical CDF resolution: [150] ",150);
		if (m_bDensity)
			m_fRangeFac = AskFloat("    Please enter maximum vertical CDF range (in units of uniform density): [5.0] ",5.0);
	} 
}


void CMicroHet::Create()
{
	float f;
	int z, ch, zx, zy, zz, zyq, zzq, chq;

	try { m_pIBin = new MH_BIN_TYPE[m_iResTri]; } catch(...) {m_pIBin = NULL; }
	if (m_pIBin == NULL) NewException((double)m_iResTri*sizeof(MH_BIN_TYPE),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<m_iaCuboidLength.GetSize();z++)
	{
		f = 0;
		if (m_bSphere)
		{
			ch = m_iaCuboidLength[z]/2;
			chq = ch*ch;
			for (zz=-ch;zz<ch;zz++)
			{
				zzq = zz*zz;
				for (zy=-ch;zy<ch;zy++)
				{
					zyq = zy*zy;
					for (zx=-ch;zx<ch;zx++)
					{
						if (zzq + zyq + zx*zx <= chq)
							f += m_fBinVolume;
					}
				}
			}
		} else
		{
			f = m_iaCuboidLength[z] * m_iaCuboidLength[z] * m_iaCuboidLength[z] * m_fBinVolume;
		}
		m_faCuboidVolume.Add(f);
	}

	for (z=0;z<m_oaObservations.GetSize();z++)
	{
		((CMicroHetObservation*)m_oaObservations[z])->Create();
	}
}


void CMicroHet::Zero()
{
	int z;

	for (z=0;z<m_iResTri;z++)
		m_pIBin[z] = 0;
}


void CMicroHetObservation::Create()
{
	int z;

	try { m_pHistogram = new CDF*[m_pParent->m_iaCuboidLength.GetSize()]; } catch(...) { m_pHistogram = NULL; }
	if (m_pHistogram == NULL) NewException((double)m_pParent->m_iaCuboidLength.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<m_pParent->m_iaCuboidLength.GetSize();z++)
	{
		try { m_pHistogram[z] = new CDF(); } catch(...) { m_pHistogram[z] = NULL; }
		if (m_pHistogram[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		m_pHistogram[z]->m_iResolution = m_iHistoRes;
		m_pHistogram[z]->m_fMinVal = 0;

		m_pHistogram[z]->SetLabelY("Occurrence");

		if (m_bDensity)
		{
			if (m_bUniform)
			{
				m_pHistogram[z]->m_fMaxVal = g_fBoxX*g_fBoxY*g_fBoxZ/1e9 / m_pParent->m_faCuboidVolume[z];
				if (m_bMass)
				{
					m_pHistogram[z]->SetLabelX("Relative mass density");
				} else
				{
					m_pHistogram[z]->SetLabelX("Relative particle density");
				}
			} else
			{
				if (m_bMass)
				{
					m_pHistogram[z]->SetLabelX("Mass density [g/cm^3]");
					m_pHistogram[z]->m_fMaxVal = m_fMaxVal / m_pParent->m_faCuboidVolume[z] * 0.0016605;
				} else
				{
					m_pHistogram[z]->SetLabelX("Particle density [1/nm^3]");
					m_pHistogram[z]->m_fMaxVal = m_fMaxVal / m_pParent->m_faCuboidVolume[z];
				}
			}
		} else
		{
			m_pHistogram[z]->m_fMaxVal = m_fMaxVal;
			if (m_bMass)
				m_pHistogram[z]->SetLabelX("Mass [g/mol]");
					else m_pHistogram[z]->SetLabelX("Particle count");
		}

		m_pHistogram[z]->m_bLeft = true;

		m_pHistogram[z]->Create();
	}

	if (m_bCDF)
	{
		m_pCDF = new C2DF();

		m_pCDF->m_fMinVal[0] = 0;

		if (m_bEdgeUniform)
			m_pCDF->m_fMaxVal[0] = 1.0;
				else m_pCDF->m_fMaxVal[0] = m_pParent->m_iaCuboidLength[m_pParent->m_iaCuboidLength.GetSize()-1]*g_fBoxX/m_pParent->m_iRes;

		m_pCDF->m_iRes[0] = m_pParent->m_iaCuboidLength[m_pParent->m_iaCuboidLength.GetSize()-1] / (m_pParent->m_iaCuboidLength[1] - m_pParent->m_iaCuboidLength[0]) + 1;
		m_iCDFOffset = m_pParent->m_iaCuboidLength[0] / (m_pParent->m_iaCuboidLength[1] - m_pParent->m_iaCuboidLength[0]);

		m_pCDF->m_fMinVal[1] = 0;
		if (m_bDensity)
		{
			if (m_bUniform)
			{
				m_pCDF->m_fMaxVal[1] = m_fRangeFac;
				if (m_bMass)
				{
					m_pCDF->SetLabelY("Relative mass density");
				} else
				{
					m_pCDF->SetLabelY("Relative particle density");
				}
			} else
			{
				if (m_bMass)
				{
					m_pCDF->m_fMaxVal[1] = m_fMaxVal / g_fBoxX / g_fBoxY / g_fBoxZ * 1e9 * m_fRangeFac * 0.0016605;
					m_pCDF->SetLabelY("Mass density [g/cm^3]");
				} else
				{
					m_pCDF->m_fMaxVal[1] = m_fMaxVal / g_fBoxX / g_fBoxY / g_fBoxZ * 1e9 * m_fRangeFac;
					m_pCDF->SetLabelY("Particle density [1/nm^3]");
				}
			}
		} else
		{
			m_pCDF->m_fMaxVal[1] = m_fMaxVal;
			if (m_bMass)
				m_pCDF->SetLabelY("Mass [g/mol]");
					else m_pCDF->SetLabelY("Particle count");
		}
		m_pCDF->m_iRes[1] = m_iCDFRes;

		if (m_bEdgeUniform)
			m_pCDF->SetLabelX("Relative cube edge length");
				else m_pCDF->SetLabelX("Cube edge length [pm]");

		m_pCDF->SetLabelZ("Occurrence");

		m_pCDF->m_iPlotType = 2;
		m_pCDF->m_iSmoothGrade = 0;
		m_pCDF->m_fPlotExp = 0.2;
		m_pCDF->m_iInterpolationOrder = 1;

		m_pCDF->Create();
	}
}


void CMicroHet::Bin(CMicroHetObservation *o, int zi)
{
	int z;

	for (z=0;z<m_iResTri;z++)
		o->m_pHistogram[zi]->AddToBinInt_Fast(m_pIBin[z]);
}


void CMicroHet::Process(CTimeStep *ts)
{
	int z, z2, z3, z4, z5, cl, zi;
	int px, py, pz;
	int ival;
	CMicroHetObservation *o;
	CAtomGroup *ag;
	CSingleMolecule *sm;
	CxVector3 *v;

	for (zi=0;zi<m_iaCuboidLength.GetSize();zi++)
	{
		cl = m_iaCuboidLength[zi];

		for (z=0;z<m_oaObservations.GetSize();z++)
		{
			Zero();

			o = (CMicroHetObservation*)m_oaObservations[z];

			for (z2=0;z2<o->m_oaAtomGroups.GetSize();z2++)
			{
				ag = (CAtomGroup*)o->m_oaAtomGroups[z2];

				for (z3=0;z3<ag->m_oaAtoms.GetSize();z3++)
				{
					if (o->m_bMass)
						ival = (int)(((CAtom*)g_oaAtoms[ag->m_baRealAtomType[z3]])->m_pElement->m_fMass+0.5);

					for (z4=0;z4<((CxIntArray*)ag->m_oaAtoms[z3])->GetSize();z4++)
					{
						for (z5=0;z5<ag->m_pMolecule->m_laSingleMolIndex.GetSize();z5++)
						{
							sm = (CSingleMolecule*)g_oaSingleMolecules[ag->m_pMolecule->m_laSingleMolIndex[z5]];
							v = &ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[ag->m_baAtomType[z3]])->GetAt(z4)];

							px = (int)(v->GetAt(0) / g_fBoxX * m_iRes);
							py = (int)(v->GetAt(1) / g_fBoxY * m_iRes);
							pz = (int)(v->GetAt(2) / g_fBoxZ * m_iRes);

							if (o->m_bMass)
								ProcessAtom(px,py,pz,ival,cl);
									else ProcessAtom(px,py,pz,0,cl);

						} // END FOR Z5
					} // END FOR Z4
				} // END FOR Z3
			} // END FOR Z2

			Bin(o,zi);
		} // END FOR Z
	} // END FOR ZI
}


void CMicroHet::ProcessAtom(int px, int py, int pz, int mass, int cl)
{
	int ch, chq, ix, iy, iz, zzq, zyq, zx, zy, zz, ty, tz;
	MH_BIN_TYPE *ib2;

	ch = cl/2;
	chq = ch*ch;

	if (mass == 0)
	{
		if (m_bSphere)
		{
			iz = pz;
			for (zz=-ch;zz<ch;zz++)
			{
				zzq = zz*zz;
				iz++;
				if (iz >= m_iRes)
					iz = 0;
				tz = iz*m_iResSqr;
				iy = py;
				for (zy=-ch;zy<ch;zy++)
				{
					zyq = zy*zy;
					iy++;
					if (iy >= m_iRes)
						iy = 0;
					ty = iy*m_iRes;
					ix = px;
					for (zx=-ch;zx<ch;zx++)
					{
						ix++;
						if (ix >= m_iRes)
							ix = 0;
						if (zzq + zyq + zx*zx <= chq)
							m_pIBin[tz+ty+ix]++;
					}
				}
			}
		} else // ELSE IF NOT SPHERE
		{
			iz = pz;
			for (zz=0;zz<cl;zz++)
			{
				iz++;
				if (iz >= m_iRes)
					iz = 0;
				tz = iz*m_iResSqr;
				iy = py;
				for (zy=0;zy<cl;zy++)
				{
					iy++;
					if (iy >= m_iRes)
						iy = 0;
					ty = iy*m_iRes;

/*					ix = px;
					for (zx=0;zx<cl;zx++)
					{
						ix++;
						if (ix >= m_iRes)
							ix = 0;
						m_pIBin[tz+ty+ix]++;
					}*/

					ib2 = &m_pIBin[tz+ty+px];
					if (px+cl >= m_iRes)
					{
						for (zx=0;zx<m_iRes-px;zx++)
						{
							(*ib2)++;
							ib2++;
						}
						ib2 -= m_iRes;
						for (;zx<cl;zx++)
						{
							(*ib2)++;
							ib2++;
						}
					} else
					{
						for (zx=0;zx<cl;zx++)
						{
							(*ib2)++;
							ib2++;
						}

/*						ib3 = ib2+cl;
						for (;ib2<ib3;ib2++)
							(*ib2)++;*/
					}

				}
			}
		} // END IF NOT SPHERE
	} else // ELSE IF MASS
	{
		if (m_bSphere)
		{
			iz = pz;
			for (zz=-ch;zz<ch;zz++)
			{
				zzq = zz*zz;
				iz++;
				if (iz >= m_iRes)
					iz = 0;
				tz = iz*m_iResSqr;
				iy = py;
				for (zy=-ch;zy<ch;zy++)
				{
					zyq = zy*zy;
					iy++;
					if (iy >= m_iRes)
						iy = 0;
					ty = iy*m_iRes;

					ix = px;
					for (zx=-ch;zx<ch;zx++)
					{
						ix++;
						if (ix >= m_iRes)
							ix = 0;
						if (zzq + zyq + zx*zx <= chq)
							m_pIBin[tz+ty+ix] += mass;
					}

				}
			}
		} else // ELSE IF NOT SPHERE
		{
			iz = pz;
			for (zz=0;zz<cl;zz++)
			{
				iz++;
				if (iz >= m_iRes)
					iz = 0;
				tz = iz*m_iResSqr;
				iy = py;
				for (zy=0;zy<cl;zy++)
				{
					iy++;
					if (iy >= m_iRes)
						iy = 0;
					ty = iy*m_iRes;

				/*	ix = px;
					for (zx=0;zx<cl;zx++)
					{
						ix++;
						if (ix >= m_iRes)
							ix = 0;
						ib[tz+ty+ix] += mass;
					}*/
					
					ib2 = &m_pIBin[tz+ty+px];
					if (px+cl >= m_iRes)
					{
						for (zx=0;zx<m_iRes-px;zx++)
						{
							(*ib2) += mass;
							ib2++;
						}
						ib2 -= m_iRes;
						for (;zx<cl;zx++)
						{
							(*ib2) += mass;
							ib2++;
						}
					} else
					{
						for (zx=0;zx<cl;zx++)
						{
							(*ib2) += mass;
							ib2++;
						}
					}

			/*		ib2 = &ib[tz+ty+px];
					for (zx=0;zx<cl;zx++)
					{
						*ib2 += mass;
						ib2++;
					}*/
				}
			}
		} // END IF NOT SPHERE
	} // END IF MASS
}


void CMicroHetObservation::BuildCDF()
{
//	double d;
	int zi, z;

	for (zi=0;zi<m_pParent->m_iaCuboidLength.GetSize();zi++)
	{
//		d = m_pParent->m_faCuboidVolume[zi];

		for (z=0;z<m_pHistogram[zi]->m_iResolution;z++)
			m_pCDF->AddToBin_IntX(zi+m_iCDFOffset,z*m_pHistogram[zi]->m_fMaxVal/m_pHistogram[zi]->m_iResolution,m_pHistogram[zi]->m_pBin[z]);
	}
	m_pCDF->NormalizeBinIntegral(1000000.0);
}


void CMicroHetObservation::BuildName()
{
	char buf[1024];
	int z;

	buf[0] = 0;

	for (z=0;z<m_oaAtomGroups.GetSize();z++)
	{
		strcat(buf,((CAtomGroup*)m_oaAtomGroups[z])->m_pMolecule->m_sName);
		strcat(buf,": ");
		strcat(buf,((CAtomGroup*)m_oaAtomGroups[z])->m_sName);
		if (z < m_oaAtomGroups.GetSize()-1)
			strcat(buf,"; ");
	}

	try { m_sName = new char[strlen(buf)+1]; } catch(...) { m_sName = NULL; }
	if (m_sName == NULL) NewException((double)sizeof(char)*(strlen(buf)+1),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	strcpy(m_sName,buf);
}


void CMicroHetObservation::BuildZeroPlot(const char *s, bool absolute)
{
	int zi, z, z2;
	double d;
	CDF *df;
	char buf[256];

	df = new CDF();

	if (absolute)
	{
		df->m_fMinVal = (double)m_pParent->m_iaCuboidLength[0] / (double)m_pParent->m_iRes * g_fBoxX;
		df->m_fMaxVal = ((double)(m_pParent->m_iaCuboidLength.GetSize()+1.0) / m_pParent->m_iaCuboidLength.GetSize()) * (double)m_pParent->m_iaCuboidLength[m_pParent->m_iaCuboidLength.GetSize()-1] / (double)m_pParent->m_iRes * g_fBoxX;
	} else
	{
		df->m_fMinVal = (double)m_pParent->m_iaCuboidLength[0] / (double)m_pParent->m_iRes;
		df->m_fMaxVal = (double)m_pParent->m_iaCuboidLength[m_pParent->m_iaCuboidLength.GetSize()-1] / (double)m_pParent->m_iRes;
	}

	df->m_iResolution = m_pParent->m_iaCuboidLength.GetSize();
	df->m_bLeft = true;

	df->CreateMulti(m_iHistoRes);

	if (absolute)
		df->SetLabelX("Cube edge length [pm]");
			else df->SetLabelX("Relative cube edge length");

	df->SetLabelY("Total percentage");

	for (z2=0;z2<m_iHistoRes;z2++)
	{
		if (m_bMass)
			sprintf(buf,"Mass %d percentage",z2);
				else sprintf(buf,"Count %d percentage",z2);
		df->SetLabelMulti(z2,buf);
		for (zi=0;zi<m_pParent->m_iaCuboidLength.GetSize();zi++)
		{
			d = 0;
			for (z=0;z<m_iHistoRes;z++)
				d += m_pHistogram[zi]->m_pBin[z];
			df->AddToBin_Multi_Int(z2,zi,m_pHistogram[zi]->m_pBin[z2] / d * 100.0);
		}
	}

	mprintf("      Saving entry plot as %s.csv ...\n",s);
	df->WriteMulti("",s,".csv");
	mprintf("      Saving entry plot AGR file as %s.agr ...\n",s);
	df->WriteMultiAgr("",s,".agr","Bin Value Percentage",false);

	delete df;
}


void CMicroHetObservation::BuildSlices(char *s)
{
	int zi, z2;
	CDF *df;
	double d;
	char buf[256];

	df = new CDF();

	df->m_fMinVal = 0;
	df->m_fMaxVal = /*m_fMaxVal*/m_iHistoRes;
	df->m_iResolution = m_iHistoRes;
	df->m_bLeft = true;

	df->CreateMulti(m_pParent->m_iaCuboidLength.GetSize());

	df->SetLabelY("Total percentage");

	if (m_bMass)
		df->SetLabelX("Mass [g/mol]");
			else df->SetLabelX("Particle count");

	d = 0;
	for (zi=0;zi<m_pHistogram[0]->m_iResolution;zi++)
		d += m_pHistogram[0]->m_pBin[zi];

	for (z2=0;z2<m_pParent->m_iaCuboidLength.GetSize();z2++)
	{
		sprintf(buf,"%.1f pm",(double)m_pParent->m_iaCuboidLength[z2]/m_pParent->m_iRes*g_fBoxX);
		df->SetLabelMulti(z2,buf);
		for (zi=0;zi<m_pHistogram[z2]->m_iResolution;zi++)
			df->AddToBin_Multi_Int(z2,zi,m_pHistogram[z2]->m_pBin[zi]/d*100.0);
	}

	for (zi=0;zi<df->m_iResolution;zi++)
		df->m_pBin[zi] /= m_pParent->m_iaCuboidLength.GetSize();

	mprintf("      Saving Microheterogeneity plot as %s.csv ...\n",s);
	df->WriteMulti("",s,".csv");
	mprintf("      Saving Microheterogeneity plot AGR file as %s.agr ...\n",s);
	df->WriteMultiAgr("",s,".agr","Microheterogeneity plot",false);

	delete df;
}
