//-*-c++-*-
/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 01/05/2002
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cmath>

#include "tulip/FanError.h"


static const double minLengthThreshold = 1.0;

FanError::FanError(const Coord &center,const double angleThreshold,const double lengthThreshold):
  fanCenter(center),
  angleThreshold(angleThreshold),
  lengthThreshold(lengthThreshold) 
{
  lengthValues.clear();
  lengthSum=0;
  currentAngle=0;
  numberOfVertex=0;
}


unsigned int FanError::addVertex(Coord newVertex) {
  //cerr << "add vertex" << endl;
  if (numberOfVertex==0) {
    Coord translatedNewVertex=newVertex-fanCenter;
    lastVertex=newVertex;
    double length=translatedNewVertex.norm();
    numberOfVertex++;
    lengthSum+=length;
    lengthValues.push_back(length);
    return 0;
  }

  //Compute the new angle "alpha" and new length "length"
  //cerr << "compute angle" << endl;
  Coord translatedNewVertex=newVertex-fanCenter;
  Coord translatedlastVertex=lastVertex-fanCenter;
  double cosAlpha=translatedlastVertex*translatedNewVertex/(translatedNewVertex.norm()*translatedlastVertex.norm());
  double alpha=acos(cosAlpha);
  //cerr << "alpha =" << alpha*360/6.28 << endl;
  double length=translatedNewVertex.norm();
  
  //Update the values needed for statistic
  numberOfVertex++;
  lengthSum+=length;
  lengthValues.push_back(length);

  //Compute the new standart deviation
  //cerr << "compute deviation" << endl;
  double sigma=0;
  list<double>::iterator it=lengthValues.begin();
  double mean=lengthSum/numberOfVertex;
  for (;it!=lengthValues.end();++it) sigma+=(*it-mean)*(*it-mean);
  sigma/=numberOfVertex;


  bool changeBlade=false;
  bool changeFan=false;

  //Determine wether or not a vertex should be inserted
  //- 1 test angle
  if (currentAngle+alpha>angleThreshold) changeBlade=true;
  if (alpha>angleThreshold) changeFan=true;
  
  //- 2 test length
  if (length>(1+1/numberOfVertex)*mean+sigma+lengthThreshold) changeFan=true;
  if (length<(1+1/numberOfVertex)*mean-lengthThreshold-sigma) changeFan=true;
  
  lastVertex=newVertex;
  currentAngle+=alpha;
  lengthSum+=length;
  if (changeFan) return 2;
  if (changeBlade) return 1;
  return 0;
}

void FanError::reset() {
  //cerr << "reset" << endl;  
  lengthValues.clear();
  lengthSum=0;
  currentAngle=0;
  numberOfVertex=0;
}
