#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <tulip/SelectionProxy.h>
#include <tulip/StringProxy.h>
#include <tulip/MetricProxy.h>
#include <tulip/MetaGraphProxy.h>
#include <tulip/IntProxy.h>
#include <tulip/LayoutProxy.h>
#include <tulip/SizesProxy.h>
#include <tulip/ColorsProxy.h>
#include "tulip/GlFonts.h"
#include "tulip/GlLines.h"
#include "tulip/GlGraph.h"
#include "tulip/FanError.h"
#include "tulip/Glyph.h"

static const unsigned int TLPMAXDEPTH=2;
const Color colorSelect= Color(255, 102, 255, 255);


//====================================================
void SetColor(const Color &c) {
  glColor4ub(c[0],c[1],c[2],c[3]);
}
//====================================================
void SetMaterial(const Color &C) {
  float color[4];
  color[0]=((float)C.getR())/255.0;
  color[1]=((float)C.getG())/255.0;
  color[2]=((float)C.getB())/255.0;
  color[3]=1.0;
  glMaterialfv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color );
}
//====================================================
void GlGraph::initProxies() {
  if (_superGraph != NULL) {
    elementSelected=getProxy<SelectionProxy>(_superGraph,"viewSelection");
    elementLabel=getProxy<StringProxy>(_superGraph,"viewLabel");
    elementColor=getProxy<ColorsProxy>(_superGraph,"viewColors");
    elementShape=getProxy<IntProxy>(_superGraph,"viewShape");
    elementSize=getProxy<SizesProxy>(_superGraph,"viewSize");
    elementLayout=getProxy<LayoutProxy>(_superGraph,"viewLayout");
    elementMetaGraph=getProxy<MetaGraphProxy>(_superGraph,"viewMetaGraph");
    elementTexture=getProxy<StringProxy>(_superGraph, "viewTexture");
  }
}

//====================================================
void GlGraph::drawMetaNode(node n,unsigned int depth) {
  SuperGraph *sp=_superGraph;
  _superGraph=elementMetaGraph->getNodeValue(n);
  initProxies();
  Coord maxL=elementLayout->getMax(_superGraph);
  Coord minL=elementLayout->getMin(_superGraph);
  Size  maxS=elementSize->getMax(_superGraph);

  Coord translate= (elementLayout->getMax(_superGraph)+elementLayout->getMin(_superGraph))/-2.0;
  double max=maxS[0]+maxL[0]-minL[0];
  max=max>?maxS[1]+maxL[1]-minL[1];
  max=max>?maxS[2]+maxL[2]-minL[2];
  
  if (max<0.0001) max=1;
  max=1/max;
  glPushMatrix();
  glScalef(max,max,max);
  glTranslatef(translate[0],translate[1],translate[2]);
  Iterator<node> *itN=_superGraph->getNodes();
  for (;itN->hasNext();) {
    node itn=itN->next();
    drawNode(itn,depth+1);
  }delete itN;

  Iterator<edge> *itE=_superGraph->getEdges();
  for (;itE->hasNext();) {
    edge ite=itE->next();
    drawEdge(ite);
  }delete itE;
  
  glPopMatrix();
  _superGraph=sp;
  initProxies();
}


//====================================================
void GlGraph::drawNode(node n,unsigned int depth) {
  if (depth>TLPMAXDEPTH) return;

  Coord nodeCoord;
  Color nodeColor;
  this->desactivateTexture();
  nodeCoord=elementLayout->getNodeValue(n);
  glPushMatrix();
  glTranslatef( nodeCoord.getX() , nodeCoord.getY() , nodeCoord.getZ() );
  glPushMatrix();
  Size nodeSize=elementSize->getNodeValue(n);
  glScalef(nodeSize.getW(),nodeSize.getH(),nodeSize.getD());
  //marker used for getting triangles infos in feedback mode
  glPassThrough(n.id);
  if (elementMetaGraph->getNodeValue(n) ==0) {
    int index = elementShape->getNodeValue(n);
    if (glyphTable.find(index) == glyphTable.end()) {
      index=0;
    }
    glyphTable[index]->draw(n);
  }
  else {
    this->desactivateTexture();
    SetMaterial(colorSelect);
    glCallList(metaGraphDL);
    drawMetaNode(n,depth);
  }
  glPopMatrix();
  this->desactivateTexture();
  if (elementSelected->getNodeValue(n)) {

    //glScalef(nodeSize.getW()*1.5,nodeSize.getH()*1.5,nodeSize.getD()*1.5);
    glScalef(nodeSize.getW(),nodeSize.getH(),nodeSize.getD());
    glCallList(selectionDL);
  }
  glPopMatrix();
  
  if (_viewLabel) {
    int tmpICol=nodeColor.getR()+nodeColor.getG()+nodeColor.getB();
    Color fontColor;
    if (tmpICol>384)
      fontColor.set(0,0,0,255);
    else
      fontColor.set(255,255,255,255);
    fontColor.set(0,0,0,255);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_LIGHTING);
    glColor4ub(fontColor.getR(),fontColor.getG(),fontColor.getB(),fontColor.getA());
    string tmp(elementLabel->getNodeValue(n));
    int len=tmp.length();
    double wi,he;
    int rastPos[4];
    if (len>0) {
      switch(_FontsType){
      case 0:
	glLineWidth(1.0);
	wi=GlFonts::getStrokeLength(tmp.c_str());
	glPushMatrix();
	glTranslatef(nodeCoord.getX()+nodeSize.getW()/2.0, nodeCoord.getY()+nodeSize.getH()/4.0, nodeCoord.getZ());
	glScalef(-nodeSize.getW()/wi, -nodeSize.getH()/240.0, 1);
	GlFonts::drawStrokeString(tmp.c_str());
	glPopMatrix();
	break;
      case 1:
	glRasterPos3f(nodeCoord.getX(), nodeCoord.getY()+nodeSize.getH()/2 , nodeCoord.getZ());
	glBitmap(0,0,0,0,-len/2*9-1,-8,0);
	glGetIntegerv(GL_CURRENT_RASTER_POSITION,rastPos);
	if (!occlusionTest.addRectangle(RectangleInt2D(rastPos[0],rastPos[1],rastPos[0]+len*9+1,rastPos[1]+12))) {
	  glEnable(GL_BLEND);
	  glBlendFunc(GL_SRC_ALPHA,GL_ONE);
	  glDrawPixels(len*9+1,12,GL_RGB,GL_BYTE,(void*)&bitmapMask);
	  glDisable(GL_BLEND);
	  glRasterPos3f(nodeCoord.getX(), nodeCoord.getY()+nodeSize.getH()/2 , nodeCoord.getZ());
	  GlFonts::drawBitmapString(tmp.c_str());
	}
	break;
      case 2:
	unsigned int w,h;
	GlFonts::getTextureStringSize(tmp.c_str(),w,h);
	wi=w+2;
	he=h+2;
	glPushMatrix();
	glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
	glScalef(-nodeSize.getW()/wi, -nodeSize.getH()/he, 1);
	glEnable(GL_BLEND);
	glBlendFunc(GL_ONE_MINUS_DST_COLOR,GL_ONE_MINUS_SRC_COLOR);
	GlFonts::drawTextureString(tmp.c_str());
	glDisable(GL_BLEND);
	glPopMatrix();
	break;
      default:
	cerr << "GlGraph::DrawNodes unknown fonts" << endl;
	break;
      }
    }
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_LIGHTING);
  }
}

//====================================================
unsigned int GlGraph::drawNodes(unsigned int number, Iterator<node> *itN) {
  unsigned int tmp=number;
  //  glEnable(GL_COLOR_MATERIAL);
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glMatrixMode(GL_MODELVIEW); 
  glEnable(GL_LIGHTING);
  glDisable(GL_COLOR_MATERIAL);
  glDisable(GL_BLEND);
  initProxies();
  glPassThrough(-1.0);	  //mark beginning of nodes drawing for feedback mode
  while ((itN->hasNext()) && (number >0)) {
    number--;
    node itv=itN->next();
    drawNode(itv,0);
  }
  glPassThrough(-1.0);  //mark end of nodes drawing
  glPopAttrib();
  if (!itN->hasNext()) occlusionTest.reset();
  //  glDisable(GL_COLOR_MATERIAL);
  return tmp-number;
}

//=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*
#define L3D_BIT (1<<9)
void GlGraph::drawEdge(const Coord &startNode, const Coord &finalNode,
                       const Coord &startPoint,const LineType::RealType &bends, const Coord &endPoint,
                       const Color &startColor, const Color &endColor, const Size &size, int shape, const bool selected) {
  const double width = selected ? 3.0 : 1.0;
  
  if (_edge3D) shape |= L3D_BIT;
  
  if (shape & L3D_BIT) {
    if (shape < (L3D_BIT+16)) {
      glMatrixMode(GL_MODELVIEW);
      glEnable(GL_LIGHTING);
      glEnable(GL_COLOR_MATERIAL);
      glPushMatrix();
      if (selected) glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    }
    else shape  = shape & ~L3D_BIT; //no 3D model defined, falling back to Line model
  }
  
  switch (shape)
    {
    case 0:GlLines::glDrawCurve(startPoint,bends,endPoint,width,0,startColor,endColor);break;
    case 1:GlLines::glDrawCurve(startPoint,bends,endPoint,width,1,startColor,endColor);break;
    case 2:GlLines::glDrawCurve(startPoint,bends,endPoint,width,2,startColor,endColor);break;
    case 3:GlLines::glDrawCurve(startPoint,bends,endPoint,width,3,startColor,endColor);break;
    case 4:GlLines::glDrawBezierCurve(startPoint,bends,endPoint,10,width,0,startColor,endColor);break;
    case 5:GlLines::glDrawBezierCurve(startPoint,bends,endPoint,10,width,1,startColor,endColor);break;
    case 6:GlLines::glDrawBezierCurve(startPoint,bends,endPoint,10,width,2,startColor,endColor);break;
    case 7:GlLines::glDrawBezierCurve(startPoint,bends,endPoint,10,width,3,startColor,endColor);break;
    case 8:GlLines::glDrawSplineCurve(startPoint,bends,endPoint,10,width,0,startColor,endColor);break;
    case 9:GlLines::glDrawSplineCurve(startPoint,bends,endPoint,10,width,1,startColor,endColor);break;
    case 10:GlLines::glDrawSplineCurve(startPoint,bends,endPoint,10,width,2,startColor,endColor);break;
    case 11:GlLines::glDrawSplineCurve(startPoint,bends,endPoint,10,width,3,startColor,endColor);break;
    case 12:GlLines::glDrawSpline2Curve(startPoint,bends,endPoint,10,width,0,startColor,endColor);break;
    case 13:GlLines::glDrawSpline2Curve(startPoint,bends,endPoint,10,width,1,startColor,endColor);break;
    case 14:GlLines::glDrawSpline2Curve(startPoint,bends,endPoint,10,width,2,startColor,endColor);break;
    case 15:GlLines::glDrawSpline2Curve(startPoint,bends,endPoint,10,width,3,startColor,endColor);break;
      //3D lines
    case L3D_BIT: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::PLAIN,
                                           GlLines::LINEAR, startColor, endColor); break;
    case L3D_BIT+1: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DOT,
                                           GlLines::LINEAR, startColor, endColor); break;
    case L3D_BIT+2: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DASHED,
                                           GlLines::LINEAR, startColor, endColor); break;
    case L3D_BIT+3: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::ALTERNATE,
                                           GlLines::LINEAR, startColor, endColor); break;

    case L3D_BIT+4: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::PLAIN,
                                           GlLines::BEZIER, startColor, endColor); break;
    case L3D_BIT+5: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DOT,
                                           GlLines::BEZIER, startColor, endColor); break;
    case L3D_BIT+6: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DASHED,
                                           GlLines::BEZIER, startColor, endColor); break;
    case L3D_BIT+7: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::ALTERNATE,
                                           GlLines::BEZIER, startColor, endColor); break;

    case L3D_BIT+8: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::PLAIN,
                                           GlLines::SPLINE3, startColor, endColor); break;
    case L3D_BIT+9: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DOT,
                                           GlLines::SPLINE3, startColor, endColor); break;
    case L3D_BIT+10: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DASHED,
                                           GlLines::SPLINE3, startColor, endColor); break;
    case L3D_BIT+11: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::ALTERNATE,
                                           GlLines::SPLINE3, startColor, endColor); break;
    
    case L3D_BIT+12: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::PLAIN,
                                           GlLines::SPLINE4, startColor, endColor); break;
    case L3D_BIT+13: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DOT,
                                           GlLines::SPLINE4, startColor, endColor); break;
    case L3D_BIT+14: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DASHED,
                                           GlLines::SPLINE4, startColor, endColor); break;
    case L3D_BIT+15: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::ALTERNATE,
                                           GlLines::SPLINE4, startColor, endColor); break;

    default:
      glDisable(GL_LIGHTING);
      GlLines::glDrawCurve(startPoint,bends,endPoint,width,0,startColor,endColor);break;
    }
  
  if ((shape & L3D_BIT) && (shape < (L3D_BIT+16))) {
    if (selected) glPolygonMode(GL_FRONT, GL_FILL);
    glDisable(GL_LIGHTING);
    glDisable(GL_COLOR_MATERIAL);
    glPopMatrix();
  }
}

void GlGraph::drawEdge(edge ite) {
  Coord srcCoord = elementLayout->getNodeValue(_superGraph->source(ite));
  Coord tgtCoord = elementLayout->getNodeValue(_superGraph->target(ite));
  Coord srcAnchor, tgtAnchor, endLineAnchor, tmpAnchor;
  Size srcSize = elementSize->getNodeValue(_superGraph->source(ite));
  Size tgtSize = elementSize->getNodeValue(_superGraph->target(ite));
  Size edgeSize = elementSize->getEdgeValue(ite);
  Color C1,C2;
  GLfloat *matrix = new GLfloat[16];
  LineType::RealType &lCoord = elementLayout->getEdgeValue(ite);

  // set color
  if (isEdgeColorInterpolate()) {
    C1=elementColor->getNodeValue(_superGraph->source(ite));
    C2=elementColor->getNodeValue(_superGraph->target(ite));
  }
  else {
    C1=C2=elementColor->getEdgeValue(ite);
  }
  
  // set srcAnchor, tgtAnchor. tmpAnchor will be on the point just before tgtAnchor
  int index = elementShape->getNodeValue(_superGraph->source(ite));
  if (glyphTable.find(index) == glyphTable.end()) {
    index=0;
  }

  tmpAnchor = (lCoord.size() > 0) ? lCoord.front() : tgtCoord;
  srcAnchor = glyphTable.find(index)->second->getAnchor(srcCoord, tmpAnchor, srcSize);

  index = elementShape->getNodeValue(_superGraph->target(ite));
  if (glyphTable.find(index) == glyphTable.end()) {
    index=0;
  }

  //this time we don't take srcCoord but srcAnchor to be oriented to where the line comes from
  tmpAnchor = (lCoord.size() > 0) ? lCoord.back() : srcAnchor;
  tgtAnchor = glyphTable.find(index)->second->getAnchor(tgtCoord, tmpAnchor, tgtSize);

  //draw Arrow
  if (isViewArrow()) {
    float sizeT = edgeSize.getD();
    
    makeArrowMatrix(matrix, tmpAnchor, tgtAnchor, 1.0, 1.0, 1.0);
    glMatrixMode(GL_MODELVIEW);
    glEnable(GL_LIGHTING);
    glPushMatrix();
    glMultMatrixf(matrix);
    glScalef(sizeT, sizeT, sizeT);
    SetMaterial(C2);
    glCallList(objectCone);
    glPopMatrix();

    if (elementSelected->getEdgeValue(ite)==true) {
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      glPushMatrix();
      glMultMatrixf(matrix);
      glScalef(sizeT+0.1, sizeT+0.1, sizeT+0.1);
      SetMaterial(colorSelect);
      glCallList(objectCone);
      glPopMatrix();
      glPolygonMode(GL_FRONT, GL_FILL);
    }

    endLineAnchor = tmpAnchor - tgtAnchor;
    endLineAnchor *= sizeT / endLineAnchor.norm();
    endLineAnchor += tgtAnchor;

    tgtCoord = tgtAnchor; //this defines in drawEdge the arrow head as being the final node
  }
  else {
    endLineAnchor = tgtAnchor;
  }
  
  glDisable(GL_LIGHTING);
  
  //draw Edges
  if (elementSelected->getEdgeValue(ite)==true) {
    float w,h,d;
    edgeSize.get(w, h, d);
    Size selSize(w+0.05, h+0.05, d);
    drawEdge(srcCoord, tgtCoord, srcAnchor, lCoord, endLineAnchor, colorSelect, colorSelect,
	     selSize, elementShape->getEdgeValue(ite), true);
  }
  
  drawEdge(srcCoord, tgtCoord, srcAnchor, lCoord, endLineAnchor, C1, C2, edgeSize, elementShape->getEdgeValue(ite));

  delete [] matrix;
  glEnable( GL_LIGHTING );
}


unsigned int GlGraph::drawEdges(unsigned int number,Iterator<edge> *itE) {
  unsigned int tmp=number;
  glPushAttrib(GL_ALL_ATTRIB_BITS); 
  initProxies();
  while ((itE->hasNext()) && (number>0)) {
    number--;
    edge ite=itE->next();
    drawEdge(ite);
  }
  glPopAttrib();
  return tmp-number;
}



//Fan rendering methods for nodes David Auber, David Duke
void GlGraph::drawFanNode(node n) {
  if (elementSelected->getNodeValue(n)) {
    drawNode(n,0);
    Iterator<edge> *itE=_superGraph->getOutEdges(n);
    for (;itE->hasNext();) {
      edge ite=itE->next();
      drawEdge(ite);
    }delete itE;
  } 
  else {
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA,GL_ONE);
    Coord centerCoord,lastValid,lastTested;
    Color centerColor,lastTestedColor,lastValidColor;
    int numberOfChildren;
    if ((numberOfChildren=_superGraph->outdeg(n))==0)  {
      if (_superGraph->indeg(n)==0)  drawNode(n,0);
      return;
    }
    centerCoord=elementLayout->getNodeValue(n);
    centerColor=elementColor->getNodeValue(n);
    //Initiallize the FanError Object
    int k;
    if (numberOfChildren<5) k=2; else k=0;
    FanError fanError(centerCoord,6.28/(0.7*numberOfChildren+k),3);

    //We strore coordinates in order to compute normals
    vector<Coord> neighboors(numberOfChildren);
    vector<Color> neighboorsColors(numberOfChildren);
    Iterator<node> *itChild=_superGraph->getOutNodes(n);
    for (int i=0;itChild->hasNext();++i) {
      node itchild=itChild->next();
      neighboors[i]=elementLayout->getNodeValue(itchild);
      neighboorsColors[i]=elementColor->getNodeValue(itchild);
    }delete itChild;
    glFrontFace(GL_CW);

    //compute normal of the center average of the normal of all faces.
    Coord centerNormal(0,0,0);
    for (int j=0;j<numberOfChildren-1;++j) {
      centerNormal+=(neighboors[j]-centerCoord)^(neighboors[j+1]-centerCoord);
    }
    centerNormal/=(numberOfChildren-1);

    //Compute and draw the FRONT fans
    unsigned int result;
    unsigned int nbElements;
    node itchild;
    itChild=_superGraph->getOutNodes(n);
    itchild=itChild->next();
    lastTested=lastValid=elementLayout->getNodeValue(itchild);
    lastTestedColor=lastValidColor=elementColor->getNodeValue(itchild);
    result=2;
    nbElements=0;
    for (int i=0;itChild->hasNext();) {
      fanError.reset();
      fanError.addVertex(lastTested);
      if (result==2) {
	nbElements=0;
	glBegin(GL_TRIANGLE_FAN);
	//	SetMaterial(centerColor);
	SetColor(centerColor);
	glNormal3f(centerNormal.getX(),centerNormal.getY(),centerNormal.getZ());
	glVertex3f(centerCoord.getX() , centerCoord.getY() , centerCoord.getZ());
	//	SetMaterial(lastTestedColor);
	SetColor(lastTestedColor);
	if (i+1<numberOfChildren) {
	  Coord resultCoord=(neighboors[i+1]-neighboors[i])^(centerCoord-neighboors[i]);
	  glNormal3f(resultCoord.getX(),resultCoord.getY(),resultCoord.getZ());
	}
	glVertex3f(lastTested.getX() , lastTested.getY() , lastTested.getZ());	
      }
      lastValid=lastTested;
      for (;itChild->hasNext();) {
	node itchild=itChild->next();
	i++;
	lastTested=elementLayout->getNodeValue(itchild);
	lastTestedColor=elementColor->getNodeValue(itchild);
	if ((result=fanError.addVertex(lastTested))!=0) break;
	lastValid=lastTested;
	lastValidColor=lastTestedColor;
	nbElements++;
      }
      //      SetMaterial(lastValidColor);
      SetColor(lastValidColor);

      if (i-1>0) {
	Coord resultCoord=(centerCoord-neighboors[i])^(neighboors[i-1]-neighboors[i]);
	glNormal3f(resultCoord.getX(),resultCoord.getY(),resultCoord.getZ());
      }
      glVertex3f(lastValid.getX() , lastValid.getY() , lastValid.getZ());
      if (result==2) glEnd();
      if ((nbElements==0) && (result==2)) {
	glDisable(GL_BLEND);
	glDisable(GL_LIGHTING);
	GlLines::glDrawLine(centerCoord,lastValid,1.0f,0,centerColor,lastValidColor);
	glEnable(GL_LIGHTING);
	glEnable(GL_BLEND);
      }
    }
    if (result!=2) glEnd();
    else {
      glDisable(GL_BLEND);
      glDisable(GL_LIGHTING);
      GlLines::glDrawLine(centerCoord,lastTested,1.0f,0,centerColor,lastTestedColor);
      glEnable(GL_LIGHTING);
      glEnable(GL_BLEND);
    }
    delete itChild;


    //We make the BACK drawing now
    glFrontFace(GL_CCW);
    centerNormal*=-1;
    itChild=_superGraph->getOutNodes(n);
    itchild=itChild->next();
    lastTested=lastValid=elementLayout->getNodeValue(itchild);
    lastTestedColor=lastValidColor=elementColor->getNodeValue(itchild);
    result=2;
    nbElements=0;
    for (int i=0;itChild->hasNext();) {
      fanError.reset();
      fanError.addVertex(lastTested);
      if (result==2) {
	nbElements=0;
	glBegin(GL_TRIANGLE_FAN);
	//SetMaterial(centerColor);
	SetColor(centerColor);
	glNormal3f(centerNormal.getX(),centerNormal.getY(),centerNormal.getZ());
	glVertex3f(centerCoord.getX() , centerCoord.getY() , centerCoord.getZ());
	//	SetMaterial(lastTestedColor);
	SetColor(lastTestedColor);
	if (i+1<numberOfChildren) {
	  Coord resultCoord=(centerCoord-neighboors[i])^(neighboors[i+1]-neighboors[i]);
	  glNormal3f(resultCoord.getX(),resultCoord.getY(),resultCoord.getZ());
	}
	glVertex3f(lastTested.getX() , lastTested.getY() , lastTested.getZ());	
      }
      lastValid=lastTested;
      for (;itChild->hasNext();) {
	node itchild=itChild->next();
	i++;
	lastTested=elementLayout->getNodeValue(itchild);
	lastTestedColor=elementColor->getNodeValue(itchild);
	if ((result=fanError.addVertex(lastTested))!=0) break;
	lastValid=lastTested;
	lastValidColor=lastTestedColor;
	nbElements++;
      }
      //      SetMaterial(lastValidColor);
      SetColor(lastValidColor);

      if (i-1>0) {
	Coord resultCoord=(neighboors[i-1]-neighboors[i])^(centerCoord-neighboors[i]);
	glNormal3f(resultCoord.getX(),resultCoord.getY(),resultCoord.getZ());
      }

      glVertex3f(lastValid.getX() , lastValid.getY() , lastValid.getZ());
      if (result==2) glEnd();
    }
    if (result!=2) glEnd();
    delete itChild;
  }

  glFrontFace(GL_CCW);
  glDisable(GL_BLEND);
}
//====================================================
unsigned int GlGraph::drawFanNodes(unsigned int number, Iterator<node> *itN) {
  unsigned int tmp=number;
  glPushAttrib(GL_ALL_ATTRIB_BITS); 
  initProxies();
  while ((itN->hasNext()) && (number >0)) {
    number--;
    node itv=itN->next();
    drawFanNode(itv);
  }
  glPopAttrib();
  return tmp-number;
}


//==============================================================================
//Deprecated Function
//==============================================================================
///Deprecated function
unsigned int GlGraph::drawEdges(unsigned int number,list<edge>::iterator &itE){
  stlListIterator<edge> itEE(itE,orderedEdge.end());
  return drawEdges(number,&itEE);
}
///deprecated function
unsigned  int GlGraph::drawNodes(unsigned int number, list<node>::iterator &itN){ 
  stlListIterator<node> itNN(itN,orderedNode.end());
  return drawNodes(number,&itNN);
}


