#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <qlistview.h>
#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qpopupmenu.h>

#include <tulip/SuperGraph.h>
#include <tulip/SubGraph.h>
#include <tulip/Cluster.h>
#include <tulip/PropertyProxyContainer.h>
#include <tulip/SelectionProxy.h>

#include "../include/tulip/ClusterTree.h"

using namespace std;

struct ClusterListViewItem: public QListViewItem {
  SubGraph *_subGraph;
  ClusterListViewItem(SubGraph * subgraph, QListViewItem *item) : QListViewItem(item), _subGraph(subgraph) {}
  ClusterListViewItem(SubGraph * subgraph, QListView *item) : QListViewItem(item), _subGraph(subgraph) {}
  SubGraph * getSubGraph() const {return _subGraph;}
};

/* 
 *  Constructs a ClusterTree which is a child of 'parent', with the 
 *  name 'name' and widget flags set to 'f' 
 */
ClusterTree::ClusterTree(SuperGraph *rootGraph , QWidget* parent,  const char* name, WFlags fl)
  : ClusterTreeData(parent, name, fl),
    _currentSuperGraph(rootGraph),
    _rootSuperGraph(rootGraph),
    _clusterTree(_rootSuperGraph->getClusterTree()),
    _currentSubGraph(_clusterTree->getRootSubGraph()),
    removeOperation(false),
    contextMenu(NULL)
{
  setCaption(trUtf8("Cluster Tree"));
  connect(treeView, SIGNAL(currentChanged(QListViewItem*)), SLOT(showHidePartition(QListViewItem*)));
  connect(treeView, SIGNAL(contextMenuRequested ( QListViewItem *, const QPoint &, int )),
	  SLOT(rightButtonClusterTree( QListViewItem *, const QPoint &, int )));
  update();
}

ClusterTree::ClusterTree(QWidget* parent, const char* name, WFlags fl) :
  ClusterTreeData(parent, name,fl),
  _currentSuperGraph(NULL),
  _rootSuperGraph(NULL),
  _clusterTree(NULL),
  _currentSubGraph(NULL),
  removeOperation(false),
  contextMenu(NULL)
{
  setCaption(trUtf8("Cluster Tree"));
  connect(treeView, SIGNAL(currentChanged(QListViewItem*)), SLOT(showHidePartition(QListViewItem*)));
  connect(treeView, SIGNAL(contextMenuRequested ( QListViewItem *, const QPoint &, int )),
	  SLOT(rightButtonClusterTree( QListViewItem *, const QPoint &, int )));
}

/*  
 *  Destroys the object and frees any allocated resources
 */
ClusterTree::~ClusterTree(){ if (contextMenu != NULL) delete contextMenu;}

void ClusterTree::setRootSuperGraph(SuperGraph *sg) {
  treeView->clear();
  _rootSuperGraph = _currentSuperGraph = sg;
  if (sg != NULL) {
    _clusterTree = _rootSuperGraph->getClusterTree();
    _currentSubGraph = _clusterTree->getRootSubGraph();
    update();
  }
  else {
    _clusterTree = NULL;
    _currentSubGraph = NULL;
  }
}

SuperGraph* ClusterTree::getSuperGraph() const {return _currentSuperGraph;}

static QListViewItem *findItemBySuperGraph(QListViewItem *item, const int id) {
  QListViewItem *i = item;
  if (((ClusterListViewItem *) i)->getSubGraph()->getAssociatedSuperGraph()->getId() == id)
    return i;
  for (i = i->firstChild(); i != NULL; i = i->nextSibling()) {
    QListViewItem *tmp = findItemBySuperGraph(i, id);
    if (tmp != NULL) return tmp;
  }
  if (item->nextSibling() != NULL)
    return findItemBySuperGraph(item->nextSibling(), id);
  return NULL;
}

void ClusterTree::setSuperGraph(const SuperGraph *graph) {
  currentSuperGraphChanged(graph);
  emit supergraphChanged(_currentSuperGraph);
}

void ClusterTree::currentSuperGraphChanged(const SuperGraph *graph) {
#ifdef QT_DEBUG
  cerr << __PRETTY_FUNCTION__ << endl;
  Q_CHECK_PTR(graph);
#endif
  QListViewItem *item = findItemBySuperGraph(treeView->firstChild(), graph->getId());
  if (item != NULL) {
    disconnect(treeView, SIGNAL(currentChanged(QListViewItem*)), this, SLOT(showHidePartition(QListViewItem*)));
    treeView->setCurrentItem(item);
    connect(treeView, SIGNAL(currentChanged(QListViewItem*)), this, SLOT(showHidePartition(QListViewItem*)));
    
    _currentSubGraph=((ClusterListViewItem *)item)->getSubGraph();
    _currentSuperGraph=_currentSubGraph->getAssociatedSuperGraph();
  }
}

void ClusterTree::update() {
#ifndef NDEBUG
  cerr << "ClusterTree::update()" << endl;
#endif
  treeView->clear();
  if (_clusterTree != NULL) {
    _currentSubGraph = _clusterTree->getRootSubGraph();
    _currentSuperGraph = _currentSubGraph->getAssociatedSuperGraph();
  }
  if (_currentSubGraph != NULL) {
    buildTreeView(treeView, _currentSubGraph);
    if (!removeOperation)
      emit supergraphChanged(_currentSuperGraph);
  }
}
//=======================================================================================
//Building of the cluster tree view
void ClusterTree::buildTreeView(QListView *item, SubGraph *p) {
  QListViewItem *tmpItem = new ClusterListViewItem(p, item);
  tmpItem->setText(0,QString(p->getName().c_str()));
  tmpItem->setExpandable(true);
  item->setOpen(tmpItem,true);
  list<SubGraph *>::iterator itLP;
  for (itLP = p->getSubGraphChildren()->begin(); itLP != p->getSubGraphChildren()->end(); ++itLP)
    buildTreeView(tmpItem,(*itLP));
}

void ClusterTree::buildTreeView(QListViewItem *item, SubGraph *p) {
  QListViewItem *tmpItem=new ClusterListViewItem(p,item);
  tmpItem->setText(0, QString(p->getName().c_str()));
  tmpItem->setExpandable(true);
  treeView->setOpen(tmpItem,true);
  list<SubGraph *>::iterator itLP;
  for (itLP = p->getSubGraphChildren()->begin(); itLP != p->getSubGraphChildren()->end(); ++itLP)
    buildTreeView(tmpItem,(*itLP));
}

//=======================================================================================
//Cluster Tree Structure modification
void ClusterTree::contextRemoveCluster() {
  if (_currentSuperGraph == _rootSuperGraph) {
    QMessageBox::critical( 0, "Tulip Cluster Tree Editor Remove Failed", QString("You cannot remove the root cluster"));
    return;
  }
  
  Observable::holdObservers();
  removeOperation = true;
  emit aboutToRemoveView(_currentSuperGraph);
  _currentSuperGraph=_rootSuperGraph;
  _rootSuperGraph->delView(_currentSubGraph);
  _currentSubGraph=_clusterTree->getRootSubGraph();
  update();
  removeOperation = false;
  Observable::unholdObservers();
}

void ClusterTree::contextRemoveAllCluster() {
  if (_currentSuperGraph == _rootSuperGraph) {
    QMessageBox::critical( 0, "Tulip Cluster Tree Editor Remove Failed",QString("You cannot remove the root cluster"));
    return;
  }
  Observable::holdObservers();
  removeOperation = true;
  emit aboutToRemoveAllView(_currentSuperGraph);
  _currentSuperGraph=_rootSuperGraph;
  _rootSuperGraph->delAllView(_currentSubGraph);
  _currentSubGraph=_clusterTree->getRootSubGraph();
  update();
  removeOperation = false;
  Observable::unholdObservers();
}

void ClusterTree::contextCloneCluster() {  
  if (_currentSuperGraph == _rootSuperGraph) {
    QMessageBox::critical( 0, "Tulip Cluster Tree Editor Clone Failed",QString("You cannot clone the root cluster"));
    return;
  }
  bool ok;
  QString text = QInputDialog::getText( "Cluster name" ,  "Please enter the cluster name" , QLineEdit::Normal,QString::null, &ok, this );
  if (ok) {
    //    SuperGraph *fatherGraph=_currentSuperGraph->getFather();
    SelectionProxy *sel1 = getLocalProxy<SelectionProxy>(_currentSuperGraph,"tmpselect");
    sel1->setAllNodeValue(true);
    sel1->setAllEdgeValue(true);
    //    SubGraph * tmpSubGraph = fatherGraph->addView(text.ascii(),sel1);
    _currentSuperGraph->getPropertyProxyContainer()->delLocalProxy("tmpselect");
    update();
  }
}

void ClusterTree::contextCloneSubgraphCluster() {
  bool ok;
  QString text = QInputDialog::getText( "Cluster name" , "Please enter the cluster name" ,
                                        QLineEdit::Normal,QString::null, &ok, this);
  if (ok) {
    SelectionProxy *sel1 = getLocalProxy<SelectionProxy>(_currentSuperGraph,"tmpselect");
    sel1->setAllNodeValue(true);
    sel1->setAllEdgeValue(true);
    _currentSuperGraph->addView(text.ascii(),sel1);
    _currentSuperGraph->getPropertyProxyContainer()->delLocalProxy("tmpselect");
    update();
  }
}

void ClusterTree::contextMoveUpCluster() {
  _clusterTree->moveUp(_currentSubGraph);
  update();
}
//=======================================================================================
void ClusterTree::contextRenameCluster() {
  bool ok;
  QString text = QInputDialog::getText( trUtf8("Cluster Name") ,  trUtf8("Please enter the cluster name"),
                                        QLineEdit::Normal, _currentSubGraph->getName().c_str(), &ok, this);
  if (ok) {
    _currentSubGraph->setName(text.latin1());
  }
  update();
}

void ClusterTree::setContextMenu(QPopupMenu *menu) {
  if (contextMenu != NULL) delete contextMenu;
  contextMenu = menu;
}

QPopupMenu *ClusterTree::getContextMenu() const {return contextMenu;}

void ClusterTree::rightButtonClusterTree(QListViewItem *item, const QPoint &p, int c) {
  if (item == NULL) return;
  
  if (contextMenu == NULL) {
    contextMenu = new QPopupMenu(this, "cluster_tree_context_menu");
    contextMenu->insertItem(trUtf8("Remove"), this, SLOT(contextRemoveCluster()));
    contextMenu->insertItem(trUtf8("Remove all"), this, SLOT(contextRemoveAllCluster()));
    contextMenu->insertItem(trUtf8("Move up"), this, SLOT(contextMoveUpCluster()));
    contextMenu->insertItem(trUtf8("Clone"), this, SLOT(contextCloneCluster()));
    contextMenu->insertItem(trUtf8("SubGraph Clone"), this, SLOT(contextCloneSubgraphCluster()));
    contextMenu->insertItem(trUtf8("Rename"), this, SLOT(contextRenameCluster()));
  }
  contextMenu->exec(p);
}

void ClusterTree::showHidePartition(QListViewItem *item,const QPoint &p, int i) {
  showHidePartition(item);
}
//**********************************************************************
///Make the current visible graph equal to the subgraph induce by the 
///Selected partition
void ClusterTree::showHidePartition(QListViewItem *item) {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << endl;
#endif
  _currentSubGraph=((ClusterListViewItem *)item)->getSubGraph();
  _currentSuperGraph=_currentSubGraph->getAssociatedSuperGraph();
  emit supergraphChanged(_currentSuperGraph);
}
