#include <qlabel.h>
#include <qmessagebox.h>

#include <tulip/PropertyProxyContainer.h>
#include <tulip/TypeConverter.h>

#include "tulip/PropertyWidgets.h"
#include "tulip/TulipElementProperties.h"

using namespace Tulip;

TulipElementProperties::TulipElementProperties(SuperGraph *supergraph,
                                               const QStringList &nodeListedProperties,
                                               const QStringList &edgeListedProperties,
                                               QWidget *parent, const char *name, WFlags fl) :
  TulipElementPropertiesData(parent, name, fl), displayMode(NODE), nodeSet(false), edgeSet(false)
{
  setSuperGraph(supergraph);
  setNodeListedProperties(nodeListedProperties);
  setEdgeListedProperties(edgeListedProperties);
  propertyTable->horizontalHeader()->setLabel(0, tr("Property"));
  propertyTable->horizontalHeader()->setLabel(1, tr("Value"));
}

TulipElementProperties::TulipElementProperties(QWidget *parent, const char *name, WFlags fl) :
  TulipElementPropertiesData(parent, name, fl), displayMode(NODE), nodeSet(false), edgeSet(false)
{
  setSuperGraph(NULL);
  propertyTable->horizontalHeader()->setLabel(0, tr("Property"));
  propertyTable->horizontalHeader()->setLabel(1, tr("Value"));
}

TulipElementProperties::~TulipElementProperties() {}

QStringList TulipElementProperties::getCurrentListedProperties() const
{
  switch(displayMode)
    {
    case NODE: return nodeListedProperties;
    case EDGE: return edgeListedProperties;
    default:
      qWarning(QString(__PRETTY_FUNCTION__) + ": current displayMode is invalid!");
      return QStringList();
    }
}

QStringList TulipElementProperties::getNodeListedProperties() const {return nodeListedProperties;}
QStringList TulipElementProperties::getEdgeListedProperties() const {return edgeListedProperties;}

SuperGraph *TulipElementProperties::getSuperGraph() const {return supergraph;}
node TulipElementProperties::getCurrentNode() const {return currentNode;}
edge TulipElementProperties::getCurrentEdge() const {return currentEdge;}
Tulip::AtomType TulipElementProperties::getDisplayMode() const {return displayMode;}

bool TulipElementProperties::isElementDisplayed() const {
  switch(displayMode) {
  case NODE: return nodeSet; break;
  case EDGE: return edgeSet; break;
  }
  return true; //never reached, but suppresses a compiler warning
}

void TulipElementProperties::setDisplayMode(const Tulip::AtomType mode) {
  switch(mode)
    {
    case NODE:
    case EDGE:
      nodeSet = false;
      edgeSet = false;
      displayMode = mode;
      updateTable();
      break;
    }
}

void TulipElementProperties::setCurrentNode(const node &n)
{
  displayMode = NODE;
  nodeSet = true;
  currentNode = n;
  this->setCaption(QString("Node %1").arg(n.id));
  label->setText(QString("Node %1").arg(n.id));
  updateTable();
}

void TulipElementProperties::setCurrentEdge(const edge &e)
{
  displayMode = EDGE;
  edgeSet = true;
  currentEdge = e;
  this->setCaption(QString("Edge %1").arg(e.id));
  label->setText(QString("Edge %1").arg(e.id));
  updateTable();
}

void TulipElementProperties::setSuperGraph(SuperGraph *s)
{
  supergraph = s;
  propertyTable->setSuperGraph(s);
  nodeSet = false;
  edgeSet = false;
  this->setCaption("");
  label->setText("No element selected");
}

void TulipElementProperties::setCurrentListedProperties(const QStringList &l)
{
  switch(displayMode)
    {
    case NODE: return setNodeListedProperties(l);
    case EDGE: return setEdgeListedProperties(l);
    }
}

void TulipElementProperties::setNodeListedProperties(const QStringList &l)
{
  nodeListedProperties = l;
  propertyTable->setNumRows(nodeListedProperties.size());
  updateTable();
}

void TulipElementProperties::setEdgeListedProperties(const QStringList &l)
{
  edgeListedProperties = l;
  propertyTable->setNumRows(edgeListedProperties.size());
  updateTable();
}

void TulipElementProperties::updateTable()
{
  if (supergraph == NULL) return;
  int i=0;
  PropertyProxyContainer *ppc = supergraph->getPropertyProxyContainer();
  QStringList *listedProperties = NULL;
  switch(displayMode)
    {
    case NODE: listedProperties = &nodeListedProperties;
    case EDGE: listedProperties = &edgeListedProperties;
    }
  
  for (QStringList::const_iterator it=listedProperties->begin();
       it != listedProperties->end();
       ++it, ++i)
  {
    propertyTable->setText(i, 0, *it);
    if (ppc->existProxy((*it).latin1())) { //segfault may occur in getProxy (assertion fail) if we don't verify this
      switch(displayMode)
        {
        case NODE:
          if (nodeSet) {
            PProxy *editedProperty = ppc->getProxy((*it).latin1());
            propertyTable->setTulipNodeItem(editedProperty, (*it).latin1(), currentNode, i, 1);
          }
          break;
        case EDGE:
          if (edgeSet) {
            PProxy *editedProperty = ppc->getProxy((*it).latin1());
            propertyTable->setTulipEdgeItem(editedProperty, (*it).latin1(), currentEdge, i, 1);
          }
        }
    }
  }
  propertyTable->adjustColumn(0);
  propertyTable->adjustColumn(1);
}

void TulipElementProperties::propertyTableValueChanged(int row, int col)
{
  bool *elementSet;
  QStringList *listedProperties;
  switch(displayMode)
    {
    case NODE: elementSet = &nodeSet; listedProperties = &nodeListedProperties; break;
    case EDGE: elementSet = &edgeSet; listedProperties = &edgeListedProperties; break;
    }
  
  if (*elementSet &&
      (col == 1) && (row < (int)listedProperties->count()))
    {
      QString property = propertyTable->text(row, 0);
      QString value = propertyTable->text(row, col);
      PProxy *editedProperty = supergraph->getPropertyProxyContainer()->getProxy(property.ascii());

      std::string tmp = value.latin1();
      bool result;
      switch(displayMode)
        {
        case NODE:
          if (result=stringToNodeProperty(editedProperty, currentNode, tmp)) {
            emit tulipNodePropertyChanged(supergraph, currentNode, property, value);
          }
          break;
        case EDGE:
          if (result=stringToEdgeProperty(editedProperty, currentEdge, tmp)) {
            emit tulipEdgePropertyChanged(supergraph, currentEdge, property, value);
          }
          break;
        }
      
      if (!result) {
        QMessageBox::critical( 0, "Tulip Property Editor Change Failed",
                               QString("The value entered for this property is not correct,\n"
                                       "The change won't be applied\n"
                                       "Modify the entered value to apply the changes.")
                               );
      }
    }
}

void TulipElementProperties::nodePropertyChanged(SuperGraph *s, node const &n, QString const &property, QString const &value)
{
  if ((displayMode == NODE) && (supergraph == s) && (currentNode == n))
    updateTable();
}

void TulipElementProperties::edgePropertyChanged(SuperGraph *s, edge const &e, QString const &property, QString const &value)
{
  if ((displayMode == EDGE) && (supergraph == s) && (currentEdge == e))
    updateTable();
}
