//-*-c++-*-
/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 20/08/2001
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "tulip/SuperGraph.h"
#include "tulip/SuperGraphImpl.h"
#include "tulip/SuperGraphView.h"
#include "tulip/SelectionProxy.h"
#include "tulip/Cluster.h"
#include "tulip/IntProxy.h"

using namespace std;

//----------------------------------------------------------------
SuperGraphImpl::SuperGraphImpl()
{
  _propertiesProxyContainer=new PropertyProxyContainerImpl(this);
  _clusterTree=new Cluster();
  _rootView=_clusterTree->getRootSubGraph();
  _superGraph=this;
  _rootView->setAssociatedSuperGraph(this);
  nbNodes=0;
  nbEdges=0;
  outDegree=getProxy<IntProxy>(this,"outdegree");
  outDegree->setAllNodeValue(0);
}
//----------------------------------------------------------------
SuperGraphImpl::~SuperGraphImpl()
{
#ifndef NDEBUG
  cerr << "SuperGraphImpl::~SuperGraphImpl delete proxyContainer" << endl;
#endif
  delete _propertiesProxyContainer;
#ifndef NDEBUG
  cerr << "SuperGraphImpl::~SuperGraphImpl delete cluster tree" << endl;
#endif
  delete _clusterTree;
#ifndef NDEBUG
  cerr << "SuperGraphImpl::~SuperGraphImpl Ok..." << endl;
#endif
  for (Nodes::iterator i=nodes.begin();i!=nodes.end();++i) {
    i->deallocateAll();
  }
}
//----------------------------------------------------------------
PropertyProxyContainer * SuperGraphImpl::getPropertyProxyContainer()
{return  _propertiesProxyContainer;}
//----------------------------------------------------------------
Cluster * SuperGraphImpl::getClusterTree(){return _clusterTree;}
//----------------------------------------------------------------
SuperGraph *SuperGraphImpl:: getFather() const {return _superGraph;}
//----------------------------------------------------------------
SubGraph* SuperGraphImpl::getSubGraph(){return _rootView;}
//----------------------------------------------------------------
SuperGraph* SuperGraphImpl::getView(SubGraph *subGraph)
{
  if(subGraph->getAssociatedSuperGraph()==0)
    {
      SuperGraph *tmp=new SuperGraphView(this,subGraph);
      subGraph->setAssociatedSuperGraph(tmp);
      return tmp;
    }
  else
    return subGraph->getAssociatedSuperGraph();
}
//----------------------------------------------------------------
SubGraph *SuperGraphImpl::addView(const string & name,SelectionProxy *selectionProxy)
{
  SubGraph *tmpSubGraph;
  PropertyContext context;
  context.superGraph=this;
  SelectionProxy *tmpSel =new SelectionProxy(&context);
  tmpSel->setAllNodeValue(false);
  tmpSel->setAllEdgeValue(false);
  Iterator<node> *iteN=getNodes();
  Iterator<edge> *iteE=getEdges();
  for (;iteN->hasNext();) 
    {
      node n=iteN->next();
      if (selectionProxy->getNodeValue(n)) tmpSel->setNodeValue(n,true);
    }
  delete iteN;
  for (;iteE->hasNext();) 
    {
      edge e=iteE->next();
      if (selectionProxy->getEdgeValue(e)) tmpSel->setEdgeValue(e,true);
    }
  delete iteE;

  //  *sel1=*selectionProxy;
  tmpSubGraph=_clusterTree->createSubGraph(name ,_rootView,tmpSel);
  tmpSubGraph->setAssociatedSuperGraph(getView(tmpSubGraph));
#ifndef NDEBUG
  cerr << "addView" << numberOfEdges() << " n=" << numberOfNodes() << endl;
#endif
  return tmpSubGraph;
}
//----------------------------------------------------------------
void SuperGraphImpl::delView(SubGraph *view)
{_clusterTree->erase(view);}
//----------------------------------------------------------------
void SuperGraphImpl::delAllView(SubGraph *view)
{_clusterTree->recErase(view);}
//----------------------------------------------------------------
bool SuperGraphImpl::isElement(const node n) 
{return !nodeIds.is_free(n.id);}
//----------------------------------------------------------------
bool SuperGraphImpl::isElement(const edge e) {return !edgeIds.is_free(e.id);}
//----------------------------------------------------------------
node SuperGraphImpl::addNode()
{
  node newNode(nodeIds.get());
  outDegree->setNodeValue(newNode, 0);
  while (nodes.size()<=newNode.id){
    nodes.push_back(EdgeContainer());
  }
  assert(nodes[newNode.id].empty());
  assert(nodes[newNode.id].empty());
  nbNodes++;
  return newNode;
}
//----------------------------------------------------------------
void SuperGraphImpl::addNode(const node n)
{
  cerr << "SuperGraphImpl::addNode(const node n), The "
       << "current version of Tulip doesn't manage this"
       << "operation on Graph Implementation (coming soon)." << endl;
}
//----------------------------------------------------------------
edge SuperGraphImpl::addEdge(const node s,const node t)
{
  assert(isElement(s) && isElement(t));
  pair< node , node > tmp(s,t);
  outDegree->setNodeValue(s,1+outDegree->getNodeValue(s));
  edge newEdge(edgeIds.get());
  while (edges.size()<=newEdge.id){
    edges.push_back(tmp);
  }
  edges[newEdge.id]=tmp;
  nodes[s.id].push_back(newEdge);
  nodes[t.id].push_back(newEdge);
  nbEdges++;
  return newEdge;
}
//----------------------------------------------------------------
void SuperGraphImpl::addEdge(const edge e)
{
  cerr << "SuperGraphImpl::addEdge(const edge e), The"
       << " current version of Tulip doesn't manage this"
       << "operation on Graph Implementation (coming soon)." << endl;
}
//----------------------------------------------------------------
void SuperGraphImpl::delNode(const node n)
{
  //Warning, the current implementation doesn't manage the 
  //updating of properties for upper_subgraph in the case of 
  //Super Graph Implementation

  externRemove(n);

  for(EdgeContainer::iterator i=nodes[n.id].begin();i!=nodes[n.id].end();++i) {
    externRemove(*i);
    node s=opposite(*i,n);
    removeEdge(nodes[s.id],*i);
    if (s==edges[(*i).id].first)
      outDegree->setNodeValue(s,outDegree->getNodeValue(s)-1);
  }

  nodes[n.id].clear();

  //  cout << "Nodes ID" << endl << nodeIds << endl;
  //  nodes.erase(n);
}
//----------------------------------------------------------------


void SuperGraphImpl::delEdge(const edge e)
{
  //Warning, the current implementation doesn't manage the updating of 
  //properties for upper_subgraph in the case of Super Graph Implementation

  externRemove(e);
  node s=edges[e.id].first;
  node t=edges[e.id].second;
  outDegree->setNodeValue(s,outDegree->getNodeValue(s)-1);
  removeEdge(nodes[s.id],e);
  removeEdge(nodes[t.id],e);

  //cout << "Edges ID" << endl << edgeIds << endl;
}
//----------------------------------------------------------------
void SuperGraphImpl::delAllNode(const node n){delNode(n);}
//----------------------------------------------------------------
void SuperGraphImpl::delAllEdge(const edge e){delEdge(e);}
//----------------------------------------------------------------
Iterator<node>* SuperGraphImpl::getNodes()const
{return (new xSGraphNodeIterator(this));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphImpl::getInNodes(const node n)const
{return (new xInNodesIterator(this,n));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphImpl::getOutNodes(const node n)const
{return (new xOutNodesIterator(this,n));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphImpl::getInOutNodes(const node n)const
{return (new xInOutNodesIterator(this,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphImpl::getEdges()const
{return (new xSGraphEdgeIterator(this));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphImpl::getInEdges(const node n)const
{return (new xInEdgesIterator(this,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphImpl::getOutEdges(const node n)const
{return (new xOutEdgesIterator(this,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphImpl::getInOutEdges(const node n)const
{return (new xInOutEdgesIterator(this,n));}
//----------------------------------------------------------------
int SuperGraphImpl::deg(const node n) const {return nodes[n.id].size();}
//----------------------------------------------------------------
int SuperGraphImpl::indeg(const node n) const {return nodes[n.id].size()-outDegree->getNodeValue(n);}
//----------------------------------------------------------------
int SuperGraphImpl::outdeg(const node n) const {return outDegree->getNodeValue(n);}
//----------------------------------------------------------------
node SuperGraphImpl::source(const edge e)const{return edges[e.id].first;}
//----------------------------------------------------------------
node SuperGraphImpl::target(const edge e)const{return edges[e.id].second;}
//----------------------------------------------------------------
void SuperGraphImpl::reverse(const edge e)
{
  assert(isElement(e));
  node s=edges[e.id].first;
  edges[e.id].first=edges[e.id].second;
  edges[e.id].second=s;
  outDegree->setNodeValue(s,outDegree->getNodeValue(s)-1);
  outDegree->setNodeValue(edges[e.id].first,outDegree->getNodeValue(edges[e.id].first)+1);
}
//----------------------------------------------------------------
int SuperGraphImpl::numberOfEdges()const{return nbEdges;}
//----------------------------------------------------------------
int SuperGraphImpl::numberOfNodes()const{return nbNodes;}
//----------------------------------------------------------------
void SuperGraphImpl::removeEdge(EdgeContainer &c, const edge e) {
  edge e1=c.back();
  c.pop_back();
  if (e!=e1)
    for (EdgeContainer::iterator i=c.begin();i!=c.end();++i)
      if (*i==e) { *i=e1; break; }
}
//----------------------------------------------------------------
void SuperGraphImpl::externRemove(const edge e) {
  assert(isElement(e));
  list<SubGraph *> *subGraphs=_rootView->getSubGraphChildren();
  for(  list<SubGraph *>::iterator it=subGraphs->begin();it!=subGraphs->end();++it) 
    if ((*it)->getAssociatedSuperGraph()->isElement(e)) (*it)->getAssociatedSuperGraph()->delEdge(e);

  getPropertyProxyContainer()->erase(e);
  edgeIds.free(e.id);
  nbEdges--;
}
//----------------------------------------------------------------
void SuperGraphImpl::externRemove(const node n) {
  assert(isElement(n));
  list<SubGraph *> *subGraphs=_rootView->getSubGraphChildren();
  for(list<SubGraph *>::iterator it=subGraphs->begin();it!=subGraphs->end();++it) 
    if ((*it)->getAssociatedSuperGraph()->isElement(n)) (*it)->getAssociatedSuperGraph()->delNode(n);

  getPropertyProxyContainer()->erase(n);
  nodeIds.free(n.id);
  nbNodes--;
}
//----------------------------------------------------------------
