/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 20/08/2001
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "tulip/SuperGraphView.h"
#include "tulip/PropertyProxyContainer.h"
#include "tulip/SelectionProxy.h"
#include "tulip/Cluster.h"
#include "tulip/SubGraph.h"
#include "tulip/StableIterator.h"
using namespace std;
//----------------------------------------------------------------
SuperGraphView::SuperGraphView(SuperGraph *superGraph,SubGraph *subGraph):
  _superGraph(superGraph),
  _currentView(subGraph),
  _propertiesProxyContainer(new PropertyProxyContainerImpl(this)),
  nNodes(0),
  nEdges(0)
{}
//----------------------------------------------------------------
SuperGraphView::~SuperGraphView() {
  cerr << "SuperGraphView::~SuperGraphView" << endl;
  delete _propertiesProxyContainer;
  cerr << "SuperGraphView::~SuperGraphView Ok......" << endl;
}  
//----------------------------------------------------------------
PropertyProxyContainer *SuperGraphView::getPropertyProxyContainer()
{return  _propertiesProxyContainer;}
//----------------------------------------------------------------
SuperGraph *SuperGraphView::getFather() const
{return _currentView->getFather()->getAssociatedSuperGraph();}
//----------------------------------------------------------------
Cluster *SuperGraphView::getClusterTree()
{return _currentView->getFather()->getAssociatedSuperGraph()->getClusterTree();}
//----------------------------------------------------------------
SubGraph* SuperGraphView::getSubGraph(){return _currentView;}
//----------------------------------------------------------------
SuperGraph *SuperGraphView::getView(SubGraph *subGraph) {
  if(subGraph->getAssociatedSuperGraph()==0) {
    SuperGraph *tmp=new SuperGraphView(this,subGraph);
    subGraph->setAssociatedSuperGraph(tmp);
    return tmp;
  }
  else
    return subGraph->getAssociatedSuperGraph();
}
//----------------------------------------------------------------
SubGraph *SuperGraphView::addView(const string &name,SelectionProxy *selectionProxy)
{
  _superGraph=_currentView->getFather()->getAssociatedSuperGraph();
  PropertyContext context;
  context.superGraph=this;
  SelectionProxy *tmpSel =new SelectionProxy(&context);
  //  *tmpSel=*selectionProxy; 
  tmpSel->setAllNodeValue(false);
  tmpSel->setAllEdgeValue(false);
  Iterator<node> *iteN=getNodes();
  for (;iteN->hasNext();) {
      node n=iteN->next();
      if (selectionProxy->getNodeValue(n)) tmpSel->setNodeValue(n,true);
    }
  delete iteN;
  Iterator<edge> *iteE=getEdges();
  for (;iteE->hasNext();) {
      edge e=iteE->next();
      if (selectionProxy->getEdgeValue(e)) tmpSel->setEdgeValue(e,true);
    }
  delete iteE;

  SubGraph *tmpSubGraph;
  tmpSubGraph=(_superGraph->getClusterTree())->createSubGraph(name , _currentView ,tmpSel);
  tmpSubGraph->setAssociatedSuperGraph(getView(tmpSubGraph));
  return (tmpSubGraph);
}
//----------------------------------------------------------------
void SuperGraphView::delView(SubGraph *view){
  _superGraph=_currentView->getFather()->getAssociatedSuperGraph();
  _superGraph->delView(view);
}
//----------------------------------------------------------------
void SuperGraphView::delAllView(SubGraph *view){
  _superGraph=_currentView->getFather()->getAssociatedSuperGraph();
  _superGraph->delAllView(view);
}
//----------------------------------------------------------------
bool SuperGraphView::isElement(const node n) 
{return _currentView->getSubGraphProxy()->getNodeValue(n);}
//----------------------------------------------------------------
bool SuperGraphView::isElement(const edge e) 
{return _currentView->getSubGraphProxy()->getEdgeValue(e);}
//----------------------------------------------------------------
node SuperGraphView::addNode() {
  _superGraph=_currentView->getFather()->getAssociatedSuperGraph();
  node tmp=_superGraph->addNode();
  _currentView->getSubGraphProxy()->setNodeValue(tmp,true);
  if (nNodes>0) nNodes++;
  return tmp;
}
//----------------------------------------------------------------
void SuperGraphView::addNode(const node n)
{
  assert(!isElement(n));
  if (!isElement(n)) {
      _superGraph=_currentView->getFather()->getAssociatedSuperGraph();
      if (!_superGraph->isElement(n)) _superGraph->addNode(n);
      _currentView->getSubGraphProxy()->setNodeValue(n,true);
        if (nNodes>0) nNodes++;
    }
}
//----------------------------------------------------------------
edge SuperGraphView::addEdge(const node n1,const node n2)
{
  assert(isElement(n1));
  assert(isElement(n2));
  _superGraph=_currentView->getFather()->getAssociatedSuperGraph();
  edge tmp=_superGraph->addEdge(n1,n2);
  _currentView->getSubGraphProxy()->setEdgeValue(tmp,true);
    if (nEdges>0) nEdges++;
  return tmp;
}
//----------------------------------------------------------------
void SuperGraphView::addEdge(const edge e) {
  assert(!isElement(e));
  if (!_currentView->getSubGraphProxy()->getEdgeValue(e)) {
      _superGraph=_currentView->getFather()->getAssociatedSuperGraph();
      if (!_superGraph->isElement(e)) _superGraph->addEdge(e);
      _currentView->getSubGraphProxy()->setEdgeValue(e,true);
        if (nEdges>0) nEdges++;
    }
}
//----------------------------------------------------------------
void SuperGraphView::delNode(const node n) {
  assert(isElement(n));
  if (isElement(n)) {
      list<SubGraph *> *subGraphs=_currentView->getSubGraphChildren();
      list<SubGraph *>::iterator it=subGraphs->begin();
      for(;it!=subGraphs->end();++it) 
	if ((*it)->getAssociatedSuperGraph()->isElement(n)) (*it)->getAssociatedSuperGraph()->delNode(n);
      set<edge> tmpSet;
      Iterator<edge> *itE=getInOutEdges(n);
      for (;itE->hasNext();) tmpSet.insert(itE->next());
      delete itE;
      for (set<edge>::const_iterator ite=tmpSet.begin();ite!=tmpSet.end();++ite)
	delEdge(*ite);
      _currentView->getSubGraphProxy()->setNodeValue(n,false);
      getPropertyProxyContainer()->erase(n);
        if (nNodes>0) nNodes--;
    }
}
//----------------------------------------------------------------
void SuperGraphView::delEdge(const edge e){
  assert(isElement(e));
  if (isElement(e)) {
      list<SubGraph *> *subGraphs=_currentView->getSubGraphChildren();
      list<SubGraph *>::iterator it=subGraphs->begin();
      for(;it!=subGraphs->end();++it) 
	if ((*it)->getAssociatedSuperGraph()->isElement(e)) (*it)->getAssociatedSuperGraph()->delEdge(e);
      _currentView->getSubGraphProxy()->setEdgeValue(e,false);
      getPropertyProxyContainer()->erase(e);
      if (nEdges>0) nEdges--;
    }
}
//----------------------------------------------------------------
void SuperGraphView::delAllNode(const node n){
  _superGraph=_currentView->getFather()->getAssociatedSuperGraph();
  _superGraph->delAllNode(n);
}
//----------------------------------------------------------------
void SuperGraphView::delAllEdge(const edge e){
  _superGraph=_currentView->getFather()->getAssociatedSuperGraph();
  _superGraph->delAllEdge(e);
}
//----------------------------------------------------------------
Iterator<node>* SuperGraphView::getNodes() const 
{return (new SGraphNodeIterator(this,_currentView));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphView::getInNodes(const node n) const 
{return (new InNodesIterator(_currentView->getFather()->getAssociatedSuperGraph(),_currentView,n));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphView::getOutNodes(const node n)const 
{return (new OutNodesIterator(_currentView->getFather()->getAssociatedSuperGraph(),_currentView,n));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphView::getInOutNodes(const node n)const 
{return (new InOutNodesIterator(_currentView->getFather()->getAssociatedSuperGraph(),_currentView,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphView::getEdges()const 
{return (new SGraphEdgeIterator(this,_currentView));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphView::getInEdges(const node n)const 
{return (new InEdgesIterator(_currentView->getFather()->getAssociatedSuperGraph(),_currentView,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphView::getOutEdges(const node n)const 
{return (new OutEdgesIterator(_currentView->getFather()->getAssociatedSuperGraph(),_currentView,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphView::getInOutEdges(const node n)const
{return (new InOutEdgesIterator(_currentView->getFather()->getAssociatedSuperGraph(),_currentView,n));}
//----------------------------------------------------------------
int SuperGraphView::numberOfNodes()const {
  if (nNodes<=0) 
    return (nNodes=this->SuperGraphAbstract::numberOfNodes());
  else
    return nNodes;
}
//----------------------------------------------------------------
int SuperGraphView::numberOfEdges() const {
  if (nEdges<=0) 
    return (nEdges=this->SuperGraphAbstract::numberOfEdges());
  else
    return nEdges;
}
//----------------------------------------------------------------
