#ifndef _TLPMUTABLECONTAINER_
#define _TLPMUTABLECONTAINER_

#include <iostream>
#if (__GNUC__ < 3)
#include <hash_map>
#else
#include <ext/hash_map>
#endif
#include <vector>
#include "tulipconf.h"
#include "ReturnType.h"
#include <exception>
#include "tulip/Iterator.h"

enum State { VECT=0, HASH=1 };
static const unsigned int MAXSET=100;

class ImpossibleOperation : public std::exception {
};

template <typename TYPE> 
class IteratorVector : public Iterator<unsigned int> {
 public:
  IteratorVector(const TYPE value, std::vector<TYPE> *vData, unsigned int max):value(value),pos(0),max(max),vData(vData) {
    while (pos<(*vData).size() && (*vData)[pos]!=value ) 
      ++pos;
  }
  bool hasNext() {
    if (pos>=(*vData).size()) 
      return false;
    else
      return true;
  }
  unsigned int next() {
    unsigned int tmp = pos;
    ++pos;
    while (pos<(*vData).size() && (*vData)[pos]!=value)
      ++pos;
    return tmp;
  }
 private:
  TYPE value;
  unsigned int pos;
  unsigned int max;
  std::vector<TYPE> *vData;
};

template <typename TYPE> 
class IteratorHash : public Iterator<unsigned int> {
 public:
  IteratorHash(const TYPE value, stdext::hash_map<unsigned int,TYPE> *hData, unsigned int max):value(value),pos(0),max(max),hData(hData) {
    it=(*hData).begin();
    while (it!=(*hData).end() && (*it).second!=value)
      ++it;
  }
  bool hasNext() {
    return (it!=(*hData).end());
  }
  unsigned int next() {
    unsigned int tmp = (*it).first;
    ++it;
    while (it!=(*hData).end() && (*it).second!=value)
      ++it;
    return tmp;
  }
 private:
  TYPE value;
  unsigned int pos;
  unsigned int max;
  stdext::hash_map<unsigned int,TYPE> *hData;
  typename stdext::hash_map<unsigned int,TYPE>::const_iterator it;
};

template <typename TYPE> 
class MutableContainer {
public:
  MutableContainer();
  ~MutableContainer();
  void setAll(const TYPE &value);
  void set(const unsigned int i,const TYPE &value);  
  typename ReturnType<TYPE>::Value get(const unsigned int i) const;
  Iterator<unsigned int>* findAll(const TYPE &value) throw (ImpossibleOperation);
private:
  void vecttohash();
  void hashtovect();
  void compress();
private:
  std::vector<TYPE> *vData;
  stdext::hash_map<unsigned int,TYPE> *hData;
  unsigned int minIndex,maxIndex;
  TYPE defaultValue;
  State state;
  unsigned int toCompress;
  unsigned int elementInserted;
  unsigned int phiP,phi;
  double ratio;
};

template<typename TYPE> 
MutableContainer<TYPE>::MutableContainer() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  state=VECT;
  vData=new std::vector<TYPE>();
  hData=0;
  maxIndex=0;
  minIndex=0;
  toCompress=0;
  elementInserted=0;
  phiP=sizeof(void *);
  phi=sizeof(TYPE);
  ratio=double(phi) / (3.0*double(phiP)+double(phi));
}

template <typename TYPE> 
MutableContainer<TYPE>::~MutableContainer() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  switch (state) {
  case VECT: 
    delete vData;
    break;
  case HASH:
    delete hData;
    break;
  default:
    std::cerr << __PRETTY_FUNCTION__ << "unexpected state value (serious bug)" << std::endl;
    break;  
  }
}

template <typename TYPE> 
void MutableContainer<TYPE>::setAll(const TYPE &value) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  switch (state) {
  case VECT: 
    delete vData;
    break;
  case HASH:
    delete hData;
    break;
  default:
    std::cerr << __PRETTY_FUNCTION__ << "unexpected state value (serious bug)" << std::endl;
    break; 
  }
  defaultValue=value;
  state=VECT;
  vData=new std::vector<TYPE>();
  maxIndex=0;
  minIndex=0;
  toCompress=0;
  elementInserted=0;
}

template <typename TYPE> 
Iterator<unsigned int>* MutableContainer<TYPE>::findAll(const TYPE &value) throw (ImpossibleOperation) {
  if (value==defaultValue) 
    throw ImpossibleOperation();
  else {
    switch (state) {
    case VECT: 
      return new IteratorVector<TYPE>(value,vData,maxIndex+1);
      break;
    case HASH:
      return new IteratorHash<TYPE>(value,hData,maxIndex+1);
      break;
    default:
      std::cerr << __PRETTY_FUNCTION__ << "unexpected state value (serious bug)" << std::endl;
      break; 
    }
  }
}


template <typename TYPE> 
void MutableContainer<TYPE>::set(const unsigned int i,const TYPE &value) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  typename stdext::hash_map<unsigned int,TYPE>::iterator it;
  if (value==defaultValue) {
    switch (state) {
    case VECT : 
      if (i<=maxIndex) {
	if ((*vData)[i] != defaultValue) {
	  elementInserted--;
	  (*vData)[i]= defaultValue;
	}
      }
      break;
    case HASH :
      if ((it=hData->find(i))!=hData->end()) {
	hData->erase(i);
	elementInserted--;
      }
      break;
    default:
      std::cerr << __PRETTY_FUNCTION__ << "unexpected state value (serious bug)" << std::endl;
      break; 
    }
  }
  else {
    switch (state) {
    case VECT : 
      if (i>maxIndex) {
	(*vData).resize(i);
	for (unsigned int j=maxIndex+1;j<i;++j)
	  (*vData)[j]=defaultValue;
	(*vData).push_back(defaultValue);
      }
      if (maxIndex==0 && (*vData).empty()) (*vData).push_back(defaultValue); 
      if ((*vData)[i]==defaultValue) elementInserted++;
      (*vData)[i]= value;
      break;
    case HASH :
      if (hData->find(i)==hData->end()) elementInserted++;
      (*hData)[i]= value;
      break;
    default:
      std::cerr << __PRETTY_FUNCTION__ << "unexpected state value (serious bug)" << std::endl;
      break; 
    }
    maxIndex= maxIndex >? i;
  }
  
  toCompress++;
  if (toCompress>MAXSET) {
    compress();
    toCompress=0;
  }
}

//const TYPE &  MutableContainer<TYPE>::get(unsigned int i) const {
template <typename TYPE>   
typename ReturnType<TYPE>::Value MutableContainer<TYPE>::get(const unsigned int i) const {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  typename stdext::hash_map<unsigned int,TYPE>::iterator it;
  switch (state) {
  case VECT:
    if (i>maxIndex || (*vData).empty()) 
      return defaultValue;
    else 
      return ((*vData)[i]);
    break;
  case HASH:
    if ((it=hData->find(i))!=hData->end())
      return (*it).second;
    else
      return defaultValue;
    break;
  default:
    std::cerr << __PRETTY_FUNCTION__ << "unexpected state value (serious bug)" << std::endl;
    return defaultValue;
    break;
  }
}

template <typename TYPE> 
void MutableContainer<TYPE>::vecttohash() {
  //  std::cerr << __PRETTY_FUNCTION__ << std::endl << std::flush;
  hData=new stdext::hash_map<unsigned int,TYPE>(elementInserted);
  unsigned int newMaxIndex=0;
  for (unsigned int i=0;i<=maxIndex;++i) {
    if ((*vData)[i]!=defaultValue) {
      (*hData)[i]=(*vData)[i];
      newMaxIndex=newMaxIndex >? i;
    }
  }
  maxIndex=newMaxIndex;
  delete vData;
  state=HASH;
}

template <typename TYPE> 
void MutableContainer<TYPE>::hashtovect() {
  //  std::cerr << __PRETTY_FUNCTION__ << std::endl << std::flush;
  vData=new std::vector<TYPE>(maxIndex+1);
  typename stdext::hash_map<unsigned int,TYPE>::iterator it;
  unsigned int current=0;
  for (it=hData->begin();it!=hData->end();++it) {
    for (;current<it->first;++current)
      (*vData)[current]=defaultValue;
    (*vData)[it->first]=it->second;
    ++current;
  }
  delete hData;
  state=VECT;
}

template <typename TYPE> 
void MutableContainer<TYPE>::compress() {
  // cerr << __PRETTY_FUNCTION__ << endl;
  if (maxIndex==0) return;
  double limitValue= (ratio*(double(maxIndex)+1.0f));
  switch (state) {
  case VECT: 
    if ( double(elementInserted) < limitValue) {
      vecttohash();
    }
    break;
  case HASH:
    if ( double(elementInserted) > limitValue)
      hashtovect();
    break;
  default:
    std::cerr << __PRETTY_FUNCTION__ << "unexpected state value (serious bug)" << std::endl;
    break; 
  }
}

#endif
