//-*-c++-*-
/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 26/09/2001
 This program is free software; you can redistribute it and/or modify  *
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/

template <class Tnode, class Tedge>
TemplateFactory<PropertyFactory<Property<Tnode,Tedge> >, Property<Tnode,Tedge>,PropertyContext > PropertyProxy<Tnode,Tedge>::factory;

template <class Tnode, class Tedge>
PropertyProxy<Tnode,Tedge>::PropertyProxy(SuperGraph *graph):
  nodeDefaultValue(Tnode::defaultValue()),
  edgeDefaultValue(Tedge::defaultValue()),
  currentProperty(0),
  superGraph(graph) {
  nodeComputed.setAll(false);
  edgeComputed.setAll(false);
  nodeProperties.setAll(nodeDefaultValue);
  edgeProperties.setAll(edgeDefaultValue);
}


//=============================================================
template <class Tnode, class Tedge>
bool PropertyProxy<Tnode,Tedge>::compute(const std::string &algorithm, std::string &msg, const PropertyContext& context) {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << endl;
#endif
  Observable::holdObservers();

  PropertyContext tmpContext(context);
  tmpContext.propertyProxy = this;
  Property<Tnode,Tedge> *tmpProperty = factory.getObject(algorithm, tmpContext);

  bool result;
  if (tmpProperty!=NULL) {
    result=tmpProperty->check(msg);
    if(currentProperty!=NULL) delete currentProperty;
    currentProperty=tmpProperty;
    //    changeCurrentProperty(currentProperty,algorithm);
    if (result) {
      reset();
      currentProperty->run();
    }
  }
  else {
    msg = "No algorithm available with this name";
    result=false;
  }
  notifyObservers();
  Observable::unholdObservers();
  return result;
}
//=============================================================
template <class Tnode, class Tedge>
PropertyProxy<Tnode,Tedge>::~PropertyProxy() {
  if (currentProperty!=0) delete currentProperty;
}
//=============================================================
template <class Tnode, class Tedge>
void PropertyProxy<Tnode,Tedge>::reset() {
  Observable::holdObservers();
  reset_handler();
  nodeComputed.setAll(false);
  edgeComputed.setAll(false);
  nodeProperties.setAll(nodeDefaultValue);
  edgeProperties.setAll(edgeDefaultValue);
  notifyObservers();
  Observable::unholdObservers();
}
//=============================================================
/*template <class Tnode, class Tedge>
void PropertyProxy<Tnode,Tedge>::changeCurrentProperty(Property< Tnode ,Tedge > *c, const std::string &s) {
  name = s;
currentProperty=c;
}*/
//================================================================
/*template <class Tnode, class Tedge>
std::string PropertyProxy<Tnode,Tedge>::getName() {
  if (currentProperty!=NULL) return name;
  else return std::string("none");
  }*/
//=============================================================
template <class Tnode, class Tedge>
typename Tnode::RealType PropertyProxy<Tnode,Tedge>::getNodeDefaultValue() { 
  return nodeDefaultValue; 
}
//=============================================================
template <class Tnode, class Tedge>
typename Tedge::RealType PropertyProxy<Tnode,Tedge>::getEdgeDefaultValue() { 
  return edgeDefaultValue; 
}
//=============================================================
template <class Tnode, class Tedge>
typename ReturnType<typename Tnode::RealType>::Value PropertyProxy<Tnode,Tedge>::getNodeValue(const node n ) {
  if ((currentProperty!=0) && !nodeComputed.get(n.id) ) {
    nodeProperties.set(n.id,currentProperty->getNodeValue(n));
    nodeComputed.set(n.id,true);
  }
  return nodeProperties.get(n.id);
}
//=============================================================
template <class Tnode, class Tedge>
typename ReturnType<typename Tedge::RealType>::Value PropertyProxy<Tnode,Tedge>::getEdgeValue(const edge e) {
  if ((currentProperty!=0) && !edgeComputed.get(e.id) ) {
    edgeProperties.set(e.id,currentProperty->getEdgeValue(e));
    edgeComputed.set(e.id,true);
  }
  return edgeProperties.get(e.id);
} 
//=============================================================
template <class Tnode, class Tedge>
void PropertyProxy<Tnode,Tedge>::setNodeValue(const node n,const typename Tnode::RealType v) {
  nodeProperties.set(n.id,v);
  nodeComputed.set(n.id,true);
  setNodeValue_handler(n);
  notifyObservers();
}
//=============================================================
template <class Tnode, class Tedge>
void PropertyProxy<Tnode,Tedge>::setEdgeValue(const edge e,const typename Tedge::RealType v) {
  edgeProperties.set(e.id,v);
  edgeComputed.set(e.id,true);
  setEdgeValue_handler(e);
  notifyObservers();
}
//=============================================================
template <class Tnode, class Tedge>
void PropertyProxy<Tnode,Tedge>::setAllNodeValue(const typename Tnode::RealType v) {
  nodeDefaultValue=v;
  nodeProperties.setAll(v);
  nodeComputed.setAll(true);
  setAllNodeValue_handler();
  notifyObservers();
}
//============================================================
template <class Tnode, class Tedge>
void PropertyProxy<Tnode,Tedge>::setAllEdgeValue(const typename Tedge::RealType v) {
  edgeDefaultValue=v;
  edgeProperties.setAll(v);
  edgeComputed.setAll(true);
  setAllEdgeValue_handler();
  notifyObservers();
}

//==============================================================
/*
template <class Tnode, class Tedge>
bool PropertyProxy<Tnode,Tedge>::recompute(std::string &erreurMsg) {
#ifndef NDEBUG
  cerr << "PropertyProxy::reCompute" << endl;
#endif

  recompute_handler();
  reset();
  if (this->currentProperty!=NULL) {
    Observable::holdObservers();
    currentProperty->reset();
    if (!currentProperty->check(erreurMsg)) {
      Observable::unholdObservers();
      return false;
    }
    currentProperty->run();
    Observable::unholdObservers();
    return true;
  }
  else  {
    erreurMsg="PropertyProxy::reCompute error Current property is NULL cannot reCompute";
    return false;
  }
}
*/
//==============================================================
template <class Tnode, class Tedge>
void PropertyProxy<Tnode,Tedge>::erase(const node n) {
  nodeProperties.set(n.id,nodeDefaultValue);
}
//=================================================================================
template <class Tnode, class Tedge>
void PropertyProxy<Tnode,Tedge>::erase(const edge e) {
  edgeProperties.set(e.id,edgeDefaultValue);
}

//=================================================================================
template <class Tnode, class Tedge>
PropertyProxy<Tnode,Tedge>& PropertyProxy<Tnode,Tedge>::operator =(PropertyProxy<Tnode,Tedge> &proxy) {
  if (this!= &proxy) {
    //=============================================================
    //The backup is necessary, if a property is a function which use the value of "*this"
    //Future implementation should take into account : Buffered or not recursive or not
    //It will enable to presserve the backup cost in a lot of case.
    //=============================================================
    if (superGraph==0) superGraph=proxy.superGraph;
    MutableContainer<typename Tnode::RealType> backupNode;
    MutableContainer<typename Tedge::RealType> backupEdge;
    backupNode.setAll(proxy.nodeDefaultValue);
    backupEdge.setAll(proxy.edgeDefaultValue);
    Iterator<node> *itN=superGraph->getNodes();
    for (;itN->hasNext();) {
      node itn=itN->next();
      backupNode.set(itn.id,proxy.getNodeValue(itn));
    } delete itN;
    Iterator<edge> *itE=superGraph->getEdges();
    for (;itE->hasNext();) {
      edge ite=itE->next();
      backupEdge.set(ite.id,proxy.getEdgeValue(ite));
    } delete itE;
    //==============================================================*
    reset();
    if (currentProperty!=0) {
      delete currentProperty;
      currentProperty=0;
    }
    if (superGraph==proxy.superGraph) {
      setAllNodeValue(proxy.getNodeDefaultValue());
      setAllEdgeValue(proxy.getEdgeDefaultValue());
    }
    itN=superGraph->getNodes();
    for (;itN->hasNext();) {
      node itn=itN->next();
      if (proxy.superGraph->isElement(itn))
	  nodeProperties.set(itn.id,backupNode.get(itn.id));
    } delete itN;
    itE=superGraph->getEdges();
    for (;itE->hasNext();) {
      edge ite=itE->next();
      if (proxy.superGraph->isElement(ite))
	edgeProperties.set(ite.id,backupEdge.get(ite.id));
    } delete itE;
    clone_handler(proxy);
  }
  return *this;
}

//==============================================================

template <class Tnode, class Tedge>
std::string
PropertyProxy<Tnode,Tedge>::getTypename()
{
	return PProxy::getTypename( this );
}


//==============================================================

// Untyped accessors
template <class Tnode, class Tedge>
std::string
PropertyProxy<Tnode,Tedge>::getNodeDefaultStringValue()
{
	typename Tnode::RealType v = getNodeDefaultValue();
	return Tnode::toString( v );
}

template <class Tnode, class Tedge>
std::string
PropertyProxy<Tnode,Tedge>::getEdgeDefaultStringValue()
{
	typename Tedge::RealType v = getEdgeDefaultValue();
	return Tedge::toString( v );
}

template <class Tnode, class Tedge>
std::string
PropertyProxy<Tnode,Tedge>::getNodeStringValue( const node inN )
{
	typename Tnode::RealType v = getNodeValue( inN );
	return Tnode::toString( v );
}

template <class Tnode, class Tedge>
std::string
PropertyProxy<Tnode,Tedge>::getEdgeStringValue( const edge inE )
{
	typename Tedge::RealType v = getEdgeValue( inE );
	return Tedge::toString( v );
}

template <class Tnode, class Tedge>
bool
PropertyProxy<Tnode,Tedge>::setNodeStringValue( const node inN, const std::string & inV )
{
	typename Tnode::RealType v;
	if( !Tnode::fromString(v,inV) )
		return false;
	setNodeValue( inN, v );
	return true;
}

template <class Tnode, class Tedge>
bool
PropertyProxy<Tnode,Tedge>::setEdgeStringValue( const edge inE, const std::string & inV )
{
	typename Tedge::RealType v;
	if( !Tedge::fromString(v,inV) )
		return false;
	setEdgeValue( inE, v );
	return true;
}

template <class Tnode, class Tedge>
bool
PropertyProxy<Tnode,Tedge>::setAllNodeStringValue( const std::string & inV )
{
	typename Tnode::RealType v;
	if( !Tnode::fromString(v,inV) )
		return false;
	setAllNodeValue( v );
	return true;
}

template <class Tnode, class Tedge>
bool
PropertyProxy<Tnode,Tedge>::setAllEdgeStringValue( const std::string & inV )
{
	typename Tedge::RealType v;
	if( !Tedge::fromString(v,inV) )
		return false;
	setAllEdgeValue( v );
	return true;
}



