#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <GL/glu.h>

#include <tulip/SelectionProxy.h>
#include <tulip/StringProxy.h>
#include <tulip/MetricProxy.h>
#include <tulip/MetaGraphProxy.h>
#include <tulip/IntProxy.h>
#include <tulip/LayoutProxy.h>
#include <tulip/SizesProxy.h>
#include <tulip/ColorsProxy.h>
#include <tulip/TlpTools.h>
#include "tulip/GlRenderer.h"
#include "tulip/GlLines.h"
#include "tulip/GlGraph.h"
#include "tulip/Glyph.h"
#include "tulip/TLPPixmapGlyph.h"
#include "tulip/TLPPixmapFont.h"
#include "tulip/TextRenderer.h"
#include "tulip/DrawingTools.h"
using namespace std;
//====================================================
static const unsigned int TLPMAXDEPTH=10;
const Color colorSelect= Color(255, 102, 255, 255);
static float colorMat[4] = { 0.0, 0.0, 0.0, 1.0 };
//====================================================
bool glTest(){
  if (glGetError()!=GL_NO_ERROR) 
    return false;
  return true;
}
//====================================================
inline void SetColor(const Color &c) {
  glColor4ub(c[0],c[1],c[2], 255);
}
//====================================================
inline void setMaterial(const Color &C) {
  colorMat[0]=((float)C[0])/255.0;
  colorMat[1]=((float)C[1])/255.0;
  colorMat[2]=((float)C[2])/255.0;
  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, colorMat);
}
//====================================================
float GlGraph::projectSize(const Coord& position, const Size& size) {
  bool project;  
  double x1Scr,y1Scr,zScr;
  double x2Scr,y2Scr;
  bool visible = false;
  double max = 0;
  //test x-axis
  if (project=(gluProject(position[0] - size[0]/2.0, position[1], position[2],
			  modelviewMatrix,projectionMatrix,viewportArray,&x1Scr,&y1Scr,&zScr) &&
	       gluProject(position[0] + size[0]/2.0,position[1], position[2],
			  modelviewMatrix,projectionMatrix,viewportArray,&x2Scr,&y2Scr,&zScr)))  {
    if (! ((x1Scr<0.0 && x2Scr<0.0) || (x1Scr>winW && x2Scr>winW) ||
	   (y1Scr<0.0 && y2Scr<0.0) || (y1Scr>winH && y2Scr>winH)) ) {
      max >?= (x1Scr-x2Scr)*(x1Scr-x2Scr)+(y1Scr-y2Scr)*(y1Scr-y2Scr);
      visible = true;
    }
  }
  //test Y-axis
  if (project=(gluProject(position[0], position[1] - size[1]/2.0, position[2],
			  modelviewMatrix,projectionMatrix,viewportArray,&x1Scr,&y1Scr,&zScr) &&
	       gluProject(position[0], position[1] + size[1]/2.0, position[2],
			  modelviewMatrix,projectionMatrix,viewportArray,&x2Scr,&y2Scr,&zScr)))  {
    if (!( (x1Scr<0.0 && x2Scr<0.0) || (x1Scr>winW && x2Scr>winW) ||
	   (y1Scr<0.0 && y2Scr<0.0) || (y1Scr>winH && y2Scr>winH))) {
      max >?= (x1Scr-x2Scr)*(x1Scr-x2Scr)+(y1Scr-y2Scr)*(y1Scr-y2Scr);
      visible = true;
    }
  }
  //test z-axis
  if (project=(gluProject(position[0], position[1], position[2] - size[2]/2.0,
			  modelviewMatrix,projectionMatrix,viewportArray,&x1Scr,&y1Scr,&zScr) &&
	       gluProject(position[0], position[1], position[2] + size[2]/2.0,
			  modelviewMatrix,projectionMatrix,viewportArray,&x2Scr,&y2Scr,&zScr)))  {
    if (! ((x1Scr<0.0 && x2Scr<0.0) || (x1Scr>winW && x2Scr>winW) ||
	   (y1Scr<0.0 && y2Scr<0.0) || (y1Scr>winH && y2Scr>winH))) {
      max >?= (x1Scr-x2Scr)*(x1Scr-x2Scr)+(y1Scr-y2Scr)*(y1Scr-y2Scr);
      visible = true;
    }
  }
  return max;
}
//====================================================
void GlGraph::drawMetaNode(node n,unsigned int depth, bool labels, bool selectedLabels) {
  SuperGraph *sp = _superGraph;
  _superGraph = elementMetaGraph->getNodeValue(n);
  string layoutNameBackup(layoutName);
  layoutName = "viewLayout";
  initProxies();
  pair<Coord, Coord> bboxes = tlp::computeBoundingBox(_superGraph, elementLayout, elementSize);
  Coord maxC = bboxes.first;
  Coord minC = bboxes.second;
  double saveMatrix[16];
  for (unsigned int i=0;i<16;++i) saveMatrix[i] = modelviewMatrix[i];
  Coord translate= (maxC + minC)/ (-2.0);
  double dept  = maxC[2] - minC[2];
  double width  = maxC[0] - minC[0];
  double height = maxC[1] - minC[1];
  if (width<0.0001) width=1;
  if (height<0.0001) height=1;
  if (dept<0.0001) dept=1;
  glPushMatrix();
  glScalef(1.0/width, 1.0/height, 1.0/dept);
  glTranslatef(translate[0],translate[1],translate[2]);
  glGetDoublev (GL_MODELVIEW_MATRIX, modelviewMatrix);
  if (_displayEdges) {
    Iterator<edge> *itE=_superGraph->getEdges();
    if (!labels)
      drawEdges(_superGraph->numberOfEdges(), itE, depth+1);
    else
      drawEdgeLabels(_superGraph->numberOfEdges(), itE , selectedLabels, depth+1);
  }
  Iterator<node> *itN=_superGraph->getNodes();
  if (!labels)
    drawNodes(_superGraph->numberOfNodes(), itN , depth+1);
  else
    drawNodeLabels(_superGraph->numberOfNodes(), itN , selectedLabels, depth+1);
  delete itN;
  glPopMatrix();
  _superGraph=sp;
  layoutName = layoutNameBackup;
  initProxies();
  for (unsigned int i=0;i<16;++i) modelviewMatrix[i]=saveMatrix[i];
}
//====================================================
void GlGraph::drawNode(node n, unsigned int depth) {
  if (depth>TLPMAXDEPTH) return;
  this->desactivateTexture();
  const Coord &nodeCoord = elementLayout->getNodeValue(n);
  const Size &nodeSize = elementSize->getNodeValue(n);
  if( nodeSize == Size(0,0,0) )	return;
  float lod= projectSize(nodeCoord,nodeSize);
  if (lod==0) return;
  if (lod <10.0) { //less than four pixel on screen
    const Color &nodeColor = elementColor->getNodeValue(n);

    if (!elementSelected->getNodeValue(n)) {
      glPointSize(3);
      setMaterial(nodeColor);
    } else {
      glPointSize(4);
      setMaterial(Color(255, 102, 255, 255));
    }
    glPassThrough(n.id);

    glBegin(GL_POINTS);
    glVertex3f(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
    glEnd();
  }
  else {
    glPushMatrix();
    glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
    glScalef(nodeSize[0], nodeSize[1], nodeSize[2]);

    glPassThrough(n.id);
    if (elementMetaGraph->getNodeValue(n) == 0) {
      glyphs.get(elementShape->getNodeValue(n))->draw(n);
    }
    else {
      assert(glIsList(metaGraphDL));
      glCallList(metaGraphDL); //cube
      drawMetaNode(n,depth);
    }
    if (elementSelected->getNodeValue(n)) {
      assert(glIsList(selectionDL));
      glCallList(selectionDL);
    }
    glPopMatrix();
  }
}
//====================================================
unsigned int GlGraph::drawNodes(unsigned int number, Iterator<node> *itN, unsigned int depth) {
  if (!itN->hasNext() || number==0) return 0;
  if (depth>TLPMAXDEPTH) return 0;
  unsigned int tmp=number;
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glMatrixMode(GL_MODELVIEW); 
  glEnable(GL_LIGHTING);
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_COLOR_MATERIAL);
  initProxies();
  glPassThrough(-1.0);	  //mark beginning of nodes drawing for feedback mode
  while ((itN->hasNext()) && (number >0)) {
    number--;
    drawNode(itN->next(),depth);
  }
  glPassThrough(-1.0);  //mark end of nodes drawing
  glPopAttrib();
  return tmp-number;
}
//====================================================
unsigned int GlGraph::drawEdges(unsigned int number,Iterator<edge> *itE, unsigned int depth) {
  if (!itE->hasNext() || number==0) return 0;
  if (depth>TLPMAXDEPTH) return 0;
  unsigned int tmp=number;
  glPushAttrib(GL_ALL_ATTRIB_BITS); 
  glDisable(GL_LIGHTING);
  initProxies();
  while ((itE->hasNext()) && (number>0)) {
    number--;
    drawEdge(itE->next());
  }
  glPopAttrib();
  return tmp-number;
}
//====================================================
unsigned int GlGraph::drawEdgeLabels(unsigned int number, Iterator<edge> *itE, bool mode, unsigned int depth) {
  if (!itE->hasNext() || number==0) return 0;
  if (depth>TLPMAXDEPTH) return 0;
  unsigned int tmp=number;
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDisable(GL_LIGHTING);
  glDepthFunc(GL_ALWAYS );
  glDisable(GL_COLOR_MATERIAL);
  initProxies();
  if(mode)
    fontRenderer->setContext(fontsPath + "fontb.ttf", 14, 0, 0, 255);
  else	
    fontRenderer->setContext(fontsPath + "font.ttf", 12, 255, 255, 255);
  while ((itE->hasNext()) && (number >0)) {
    edge ite=itE->next();
    if (elementSelected->getEdgeValue(ite)==mode){
      number--;
      drawEdgeLabel(ite, mode);
    }
  }
  glPopAttrib();
  return tmp-number;
}
//====================================================
unsigned int GlGraph::drawNodeLabels(unsigned int number, Iterator<node> *itN, bool mode, unsigned int depth) {
  if (!itN->hasNext() || number==0) return 0;
  if (depth>TLPMAXDEPTH) return 0;
  unsigned int tmp=number;
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDisable(GL_LIGHTING);
  glDepthFunc(GL_ALWAYS );
  glDisable(GL_COLOR_MATERIAL);
  initProxies();
  if(mode)
    fontRenderer->setContext(fontsPath + "fontb.ttf", 14, 0, 0, 255);
   else	
     fontRenderer->setContext(fontsPath + "font.ttf", 12, 255, 255, 255);
  while ((itN->hasNext()) && (number >0)) {
    node itv=itN->next();
    if (elementSelected->getNodeValue(itv)==mode){
      number--;
      drawNodeLabel(itv, mode, depth);
    }
  }
  glPopAttrib();
  return tmp-number;
}
//====================================================
void GlGraph::drawPixmapFont(const string &str, const Coord &position, bool selected, float width) {
  int rastPos[4];
  float w_max = 300;
  float w,h;
  glRasterPos3f(position[0], position[1], position[2]);
  glGetIntegerv(GL_CURRENT_RASTER_POSITION,rastPos);
  if(occlusionTest.testRectangle(RectangleInt2D(rastPos[0] - labelsBorder - 5,
						rastPos[1] - labelsBorder - 5,
						rastPos[0] + labelsBorder + 5,
						rastPos[1] + labelsBorder + 5)))
    return;
  if (selected)
    glColor4ub(255,0,0,255);
  else
    glColor4ub(0,0,0,255);
  fontRenderer->setMode(TLP_TLPPIXMAP);
  fontRenderer->setString(str, VERBATIM);
  //  w_max = width;
  fontRenderer->getBoundingBox(w_max, h, w);
  if(!occlusionTest.addRectangle(RectangleInt2D(rastPos[0]-(int)(w/2.0) - labelsBorder,
						rastPos[1]-(int)(h/2.0) - labelsBorder,
						rastPos[0]+(int)(w/2.0) + labelsBorder,
						rastPos[1]+(int)(h/2.0) + labelsBorder))) {
    fontRenderer->draw(w, w);
  }
}
//====================================================
void GlGraph::drawEdgeLabel(edge e, bool mode) {
  string tmp=elementLabel->getEdgeValue(e);
  if (tmp.length()==0) return;
  const Coord & srcCoord = elementLayout->getNodeValue(_superGraph->source(e));
  const Coord & tgtCoord = elementLayout->getNodeValue(_superGraph->target(e));
  const LineType::RealType &bends = elementLayout->getEdgeValue(e);
  Coord position;
  if (bends.empty()) {
    position = (srcCoord+tgtCoord) / 2.0;
  }
  else {
    if (bends.size()%2 == 0)
      position = (bends[bends.size()/2-1]+bends[bends.size()/2]) / 2.0;
    else
      position = bends[bends.size()/2];
  }
  drawPixmapFont(tmp, position, elementSelected->getEdgeValue(e), 50);
}
//====================================================
void GlGraph::drawNodeLabel(node n, bool mode, unsigned int depth) {
  const Coord &nodeCoord = elementLayout->getNodeValue(n);
  const Size &nodeSize = elementSize->getNodeValue(n);
  if (_viewMetaLabel && elementMetaGraph->getNodeValue(n)!=0) {
    glPushMatrix();
    glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
    glScalef(nodeSize[0], nodeSize[1], nodeSize[2]);
    drawMetaNode(n, depth, true, mode);
    glPopMatrix();
  }
  const string &tmp=elementLabel->getNodeValue(n);
  if (tmp.length()==0) {
    return;
  }
  Color fontColor;
  if (elementSelected->getNodeValue(n))
    fontColor.set(255,0,0,255);
  else
    fontColor.set(0,0,0,255);
  if (_FontsType==0) fontColor.set(0,0,0,255);
  if (_FontsType==2) fontColor.set(255,255,255,255);

  glColor4ub(fontColor.getR(),fontColor.getG(),fontColor.getB(),fontColor.getA());
  setMaterial(fontColor);

  float w_max = 300;
  float w,h;
  float div_w, div_h;
  if (tmp.length()>0) {
    switch(_FontsType){
    case 0:
      if (projectSize(nodeCoord,nodeSize) < 8.0) return;
      fontRenderer->setMode(TLP_POLYGON);
      fontRenderer->setString(tmp, VERBATIM);
      //      w_max = nodeSize.getW()*50.0;
      fontRenderer->getBoundingBox(w_max, h, w);
      glPushMatrix();
      glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
      div_w = nodeSize.getW()/w;
      div_h = nodeSize.getH()/h;
      if(div_h > div_w) 
	glScalef(-div_w, -div_w, 1);
      else 
	glScalef(-div_h, -div_h, 1);
      fontRenderer->draw(w,w);
      glPopMatrix();
      break;
    case 1:
      drawPixmapFont(tmp, nodeCoord, elementSelected->getNodeValue(n), nodeSize.getW());
      break;
    case 2:
      if (projectSize(nodeCoord,nodeSize) < 8.0) return;
      fontRenderer->setMode(TLP_TEXTURE);
      fontRenderer->setString(tmp, VERBATIM);
      //      w_max = nodeSize.getW();
      fontRenderer->getBoundingBox(w_max, h, w);
      glPushMatrix();
      glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
      div_w = nodeSize.getW()/w;
      div_h = nodeSize.getH()/h;
      if(div_h > div_w) 
	glScalef(-div_w, -div_w, 1);
      else 
	glScalef(-div_h, -div_h, 1);
      glEnable( GL_TEXTURE_2D);
      glBlendFunc(GL_ONE_MINUS_DST_COLOR,GL_ONE_MINUS_SRC_COLOR);
      fontRenderer->draw(w, w);
      glDisable( GL_TEXTURE_2D);
      glPopMatrix();
      break;
    default:
      cerr << "GlGraph::DrawNodes unknown fonts" << endl;
      break;
    }
  }

}
//=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*
#define L3D_BIT (1<<9)
void GlGraph::drawEdge(const Coord &startNode, const Coord &finalNode,
                       const Coord &startPoint,const LineType::RealType &bends, const Coord &endPoint,
                       const Color &startColor, const Color &endColor, const Size &size, int shape, const bool selected) {
  const double width = selected ? 3.0 : 1.0;
  
  if (_edge3D) shape |= L3D_BIT;
  if (shape & L3D_BIT) {
    if (shape < (L3D_BIT+16)) {
      glEnable(GL_LIGHTING);
      glEnable(GL_COLOR_MATERIAL);
      if (selected) glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    }
    else shape  = shape & ~L3D_BIT; //no 3D model defined, falling back to Line model
  }
  
  switch (shape) {
  case 0:GlLines::glDrawCurve(startPoint,bends,endPoint,width,0,startColor,endColor);break;
  case 1:GlLines::glDrawCurve(startPoint,bends,endPoint,width,1,startColor,endColor);break;
  case 2:GlLines::glDrawCurve(startPoint,bends,endPoint,width,2,startColor,endColor);break;
  case 3:GlLines::glDrawCurve(startPoint,bends,endPoint,width,3,startColor,endColor);break;
  case 4:GlLines::glDrawBezierCurve(startPoint,bends,endPoint,10,width,0,startColor,endColor);break;
  case 5:GlLines::glDrawBezierCurve(startPoint,bends,endPoint,10,width,1,startColor,endColor);break;
  case 6:GlLines::glDrawBezierCurve(startPoint,bends,endPoint,10,width,2,startColor,endColor);break;
  case 7:GlLines::glDrawBezierCurve(startPoint,bends,endPoint,10,width,3,startColor,endColor);break;
  case 8:GlLines::glDrawSplineCurve(startPoint,bends,endPoint,10,width,0,startColor,endColor);break;
  case 9:GlLines::glDrawSplineCurve(startPoint,bends,endPoint,10,width,1,startColor,endColor);break;
  case 10:GlLines::glDrawSplineCurve(startPoint,bends,endPoint,10,width,2,startColor,endColor);break;
  case 11:GlLines::glDrawSplineCurve(startPoint,bends,endPoint,10,width,3,startColor,endColor);break;
  case 12:GlLines::glDrawSpline2Curve(startPoint,bends,endPoint,10,width,0,startColor,endColor);break;
  case 13:GlLines::glDrawSpline2Curve(startPoint,bends,endPoint,10,width,1,startColor,endColor);break;
  case 14:GlLines::glDrawSpline2Curve(startPoint,bends,endPoint,10,width,2,startColor,endColor);break;
  case 15:GlLines::glDrawSpline2Curve(startPoint,bends,endPoint,10,width,3,startColor,endColor);break;
    //3D lines
  case L3D_BIT: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_PLAIN,
					 GlLines::LINEAR, startColor, endColor); break;
  case L3D_BIT+1: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_DOT,
                                           GlLines::LINEAR, startColor, endColor); break;
  case L3D_BIT+2: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_DASHED,
                                           GlLines::LINEAR, startColor, endColor); break;
  case L3D_BIT+3: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_ALTERNATE,
                                           GlLines::LINEAR, startColor, endColor); break;

  case L3D_BIT+4: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_PLAIN,
                                           GlLines::BEZIER, startColor, endColor); break;
  case L3D_BIT+5: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_DOT,
                                           GlLines::BEZIER, startColor, endColor); break;
  case L3D_BIT+6: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_DASHED,
                                           GlLines::BEZIER, startColor, endColor); break;
  case L3D_BIT+7: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_ALTERNATE,
                                           GlLines::BEZIER, startColor, endColor); break;

  case L3D_BIT+8: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_PLAIN,
                                           GlLines::SPLINE3, startColor, endColor); break;
  case L3D_BIT+9: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_DOT,
                                           GlLines::SPLINE3, startColor, endColor); break;
  case L3D_BIT+10: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_DASHED,
					    GlLines::SPLINE3, startColor, endColor); break;
  case L3D_BIT+11: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_ALTERNATE,
					    GlLines::SPLINE3, startColor, endColor); break;
    
  case L3D_BIT+12: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_PLAIN,
					    GlLines::SPLINE4, startColor, endColor); break;
  case L3D_BIT+13: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_DOT,
					    GlLines::SPLINE4, startColor, endColor); break;
  case L3D_BIT+14: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_DASHED,
					    GlLines::SPLINE4, startColor, endColor); break;
  case L3D_BIT+15: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::TLP_ALTERNATE,
					    GlLines::SPLINE4, startColor, endColor); break;

  default:
    glDisable(GL_LIGHTING);
    GlLines::glDrawCurve(startPoint,bends,endPoint,width,0,startColor,endColor);break;
  }
  
  if ((shape & L3D_BIT) && (shape < (L3D_BIT+16))) {
    if (selected) glPolygonMode(GL_FRONT, GL_FILL);
    glDisable(GL_LIGHTING);
    glDisable(GL_COLOR_MATERIAL);
  }
}
//====================================================
void GlGraph::drawEdge(edge ite) {
  const Coord & srcCoord = elementLayout->getNodeValue(_superGraph->source(ite));
  Coord tgtCoord = elementLayout->getNodeValue(_superGraph->target(ite));
  const LineType::RealType &lCoord = elementLayout->getEdgeValue(ite);
  if (srcCoord == tgtCoord && lCoord.size()==0) return;
  Coord srcAnchor, tgtAnchor, endLineAnchor, tmpAnchor;
  const Size &srcSize = elementSize->getNodeValue(_superGraph->source(ite));
  const Size &tgtSize = elementSize->getNodeValue(_superGraph->target(ite));
  const Size &edgeSize = elementSize->getEdgeValue(ite);
  Color C1,C2;
  // set color
  if (isEdgeColorInterpolate()) {
    C1 = elementColor->getNodeValue(_superGraph->source(ite));
    C2 = elementColor->getNodeValue(_superGraph->target(ite));
  }
  else {
    C1 = C2 = elementColor->getEdgeValue(ite);
  }
  
  // set srcAnchor, tgtAnchor. tmpAnchor will be on the point just before tgtAnchor
  node source = _superGraph->source(ite);
  int srcGlyphId = 1; //cube outlined
  if (elementMetaGraph->getNodeValue(source)==0)
    srcGlyphId = elementShape->getNodeValue(source);
  Glyph *sourceGlyph=glyphs.get(srcGlyphId);
  tmpAnchor = (lCoord.size() > 0) ? lCoord.front() : tgtCoord;
  srcAnchor = sourceGlyph->getAnchor(srcCoord, tmpAnchor, srcSize);

  node target;
  int tgtGlyphId = 1; //cube outlined
  if (elementMetaGraph->getNodeValue(target)==0)
    tgtGlyphId = elementShape->getNodeValue(target);
  Glyph *targetGlyph = glyphs.get(tgtGlyphId);
  //this time we don't take srcCoord but srcAnchor to be oriented to where the line comes from
  tmpAnchor = (lCoord.size() > 0) ? lCoord.back() : srcAnchor;
  tgtAnchor = targetGlyph->getAnchor(tgtCoord, tmpAnchor, tgtSize);

  //draw Arrow
  if (_viewArrow) {
    GLfloat matrix[16];
    float sizeT = edgeSize.getD();
    makeArrowMatrix((GLfloat *)&matrix, tmpAnchor, tgtAnchor, 1.0, 1.0, 1.0);
    glMatrixMode(GL_MODELVIEW);
    glEnable(GL_LIGHTING);
    glPushMatrix();
    glMultMatrixf((GLfloat *)&matrix);
    glScalef(sizeT, sizeT, sizeT);
    setMaterial(C2);
    assert(glIsList(arrowDL));
    glCallList(arrowDL);
    glPopMatrix();
    if (elementSelected->getEdgeValue(ite)==true) {      
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      glPushMatrix();
      glMultMatrixf(matrix);
      glScalef(sizeT+0.1, sizeT+0.1, sizeT+0.1);
      setMaterial(colorSelect);
      assert(glIsList(arrowDL));
      glCallList(arrowDL);
      glPopMatrix();
      glPolygonMode(GL_FRONT, GL_FILL);
    }
    endLineAnchor = tmpAnchor - tgtAnchor;
    float nrm = endLineAnchor.norm();
    if( nrm > 0.00000001f ) {
	    endLineAnchor *= sizeT / nrm;
	    endLineAnchor += tgtAnchor;
    } else {
	    endLineAnchor = tgtAnchor;
    }
    tgtCoord = tgtAnchor; //this defines in drawEdge the arrow head as being the final node
    glDisable(GL_LIGHTING);
  }
  else {
    endLineAnchor = tgtAnchor;
  }
  
  //draw Edges
  if (elementSelected->getEdgeValue(ite)==true) {
    float w,h,d;
    edgeSize.get(w, h, d);
    Size selSize(w+0.05, h+0.05, d);
    drawEdge(srcCoord, tgtCoord, srcAnchor, lCoord, endLineAnchor, colorSelect, colorSelect,
	     selSize, elementShape->getEdgeValue(ite), true);
  }
  
  drawEdge(srcCoord, tgtCoord, srcAnchor, lCoord, endLineAnchor, C1, C2, edgeSize, elementShape->getEdgeValue(ite));
}
