#include "tulip/FindSelection.h"

#include <tulip/SuperGraph.h>
#include <tulip/PropertyProxy.h>
#include <tulip/MetricProxy.h>
#include <tulip/StringProxy.h>
#include <tulip/SelectionProxy.h>
#if (QT_REL == 3)
#include <qregexp.h>
// the 4 include below are only needed for compilation purpose
// (because of FindSelectionData.h generated file)
#include <qvalidator.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qradiobutton.h>
#else
#include <QtCore/qregexp.h>
#include "tulip/Qt3ForTulip.h"
#endif

SelectionWidget::SelectionWidget(SuperGraph *superGraph, QWidget *parent, const char *name, Qt::WFlags fl) : 
  FindSelection(parent, name, fl), 
  superGraph(superGraph){
  insertProperties();
  propertyChanged(-1);
}

bool SelectionWidget::IsEvaluableProxy( PProxy *p) {
  return dynamic_cast<MetricProxy*>(p) || dynamic_cast<StringProxy*>(p) || dynamic_cast<SelectionProxy*>(p);
}

bool SelectionWidget::EvalProxy( PProxy *p, const GItem &gi, std::string value, int mode) {
  assert( IsEvaluableProxy(p) );
  MetricProxy *    metp = dynamic_cast<MetricProxy*>(p);
  StringProxy *    strp = dynamic_cast<StringProxy*>(p);
  SelectionProxy * selp = dynamic_cast<SelectionProxy*>(p);
  
  if( metp ) {
    // Metric Proxy
    DoubleType::RealType v = gi.isnode ? metp->getNodeValue( gi.n ) : metp->getEdgeValue( gi.e );
    DoubleType::RealType v0 = atof( value.c_str() );
    if( mode == 0 )		return v < v0;
    if( mode == 1 )		return v <= v0;
    if( mode == 2 )		return v == v0;
    if( mode == 3 )		return v >= v0;
    if( mode == 4 )		return v > v0;
    else return v != v0;
  }
  
  if( strp ) {
    // String Proxy
    std::string v = gi.isnode ? strp->getNodeValue( gi.n ) : strp->getEdgeValue( gi.e );
    std::string v0 = value.c_str();
    QRegExp rexp( v0.c_str() );
    if( mode == 2 ) return rexp.exactMatch( v.c_str() );
    else return !rexp.exactMatch( v.c_str() );
  }
  
  if( selp ) {
    // Selection Proxy
    bool v  = gi.isnode ? selp->getNodeValue( gi.n )
      : selp->getEdgeValue( gi.e );
    bool v0 = !( value.size() == 0 || value == "False" || value == "false" || value == "0" );
    if( mode == 2 ) return v == v0;
    else return v != v0;
  }
  return false;
}


void SelectionWidget::propertyChanged( int no ) {
  PProxy * p = getProperty();
  if( dynamic_cast<MetricProxy*>(p) ) {
    filterOp->clear();
    filterOp->insertItem( "<" );
    filterOp->insertItem( "<=" );
    filterOp->insertItem( "=" );
    filterOp->insertItem( ">=" );
    filterOp->insertItem( ">" );
    filterOp->insertItem( "!=" );
    filterValue->setText( QString() );
    filterValue->setValidator( new QDoubleValidator(this) );
    filterValue->show();
  }
  else if( dynamic_cast<StringProxy*>(p) ) {
    filterOp->clear();
    filterOp->insertItem( "=" );
    filterOp->insertItem( "!=" );
    filterValue->setValidator( 0 );
    filterValue->show();
  }
  else if( dynamic_cast<SelectionProxy*>(p) ) {
    filterOp->clear();
    filterOp->insertItem( "False" );
    filterOp->insertItem( "True" );
    filterValue->hide();
    filterValue->setValidator( 0 );
      filterValue->setText( QString() );
  }
}

int SelectionWidget::getMode() {
  if( filterOp->count() == 2 )
    // == -> 2 & != -> 5
    return filterOp->currentItem() == 0 ? 2 : 5;
  else
    return filterOp->currentItem();
}

PProxy * SelectionWidget::getProperty() {
  return superGraph->getProperty(inputProp->currentText().latin1());
}

int SelectionWidget::getOperation() {
  if(setToSelectionOpt->isOn())		return 0;
  if(addToSelectionOpt->isOn())		return 1;
  if(rmvFromSelectionOpt->isOn())	return 2;
  return 3;
}

int SelectionWidget::getSource() {
  return srcOpt->currentItem();
}

void SelectionWidget::insertProperties() {
  Iterator<std::string> * propIt = superGraph->getProperties();
  while( propIt->hasNext() ) {
    std::string n = propIt->next();
    PProxy * p = superGraph->getProperty( n );
    if(	SelectionWidget::IsEvaluableProxy(p) )
      inputProp->insertItem( QString(n.c_str()) );
  } delete propIt;
}

int SelectionWidget::exec() {
  return ((FindSelection *)this)->exec();
}

void SelectionWidget::evalNodes(PProxy *p, int mode, std::string fv, int op,
				SelectionProxy *selP) {
  Iterator<node> * nodeIt = superGraph->getNodes();
  while( nodeIt->hasNext() ) {
    node n = nodeIt->next();
    bool v = EvalProxy( p, GItem(n), fv, mode );
    if( op == 0 ) {
      selP->setNodeValue( n, v );
    } else if( op == 1 ) {
      if( v )
	selP->setNodeValue( n, true );
    } else if( op == 2 )  {
      if( v )
	selP->setNodeValue( n, false );
    } else {
      if( !v )
	selP->setNodeValue( n, false );
    }
  }
  delete nodeIt;
}

void SelectionWidget::evalEdges(PProxy *p, int mode, std::string fv, int op, 
				SelectionProxy *selP) {
  Iterator<edge> * edgeIt = superGraph->getEdges();
  while( edgeIt->hasNext() ) {
    edge e = edgeIt->next();
    bool v = EvalProxy( p, GItem(e), fv, mode );
      if( op == 0 ) {
	selP->setEdgeValue( e, v );
      } else if( op == 1 ) {
	if( v )
	  selP->setEdgeValue( e, true );
      } else if( op == 2 ) {
	if( v )
	  selP->setEdgeValue( e, false );
      } else {
	if( !v )
	  selP->setEdgeValue( e, false );
      }
  }
  delete edgeIt;
}

void SelectionWidget::find(SelectionProxy *selP) {
  PProxy * p = getProperty();
  int mode  = getMode();
  std::string fv = filterValue->text().latin1();
  int op = getOperation();
  if( (getSource()+1) & 1 ) 
      evalNodes(p, mode, fv, op, selP);
  if( (getSource()+1) & 2 ) 
    evalEdges(p, mode, fv, op, selP);
}

void SelectionWidget::accept() {
  SelectionProxy * selP = superGraph->getProperty<SelectionProxy>("viewSelection");
  if (!selP) return;
  find(selP);
  close();
}
