//-*-c++-*-
/*
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 22/05/2003
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifndef Tulip_SUPERGRAPH_H
#define Tulip_SUPERGRAPH_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <string>

#if (__GNUC__ < 3)
#include <hash_map>
#else
#include <ext/hash_map>
#endif
#include <climits>
#include "tulipconf.h"
#include "Reflect.h"
#include "PluginContext.h"
#include "PluginProgress.h"
#include "Node.h"
#include "Edge.h"
#include "ObservableGraph.h"
#include "IdManager.h"

class PropertyManager;
template<class C>class Iterator;
class SelectionProxy;

namespace tlp {
  enum ElementType {NODE=0, EDGE};
}

/**
 * \defgroup graphs
 */ 
/*@{*/
/// Interface for a graph
/**
 * The class SuperGraph is the interface of a Graph in the Tulip Library.
 */
class TLP_SIMPLE_SCOPE SuperGraph : public ObservableGraph {

public:  
  SuperGraph() {
    id=SuperGraph::idManager.get();
  }
  virtual ~SuperGraph(){
    SuperGraph::idManager.free(id);
  };
  //=========================================================================  
  // Graph hierarrchy acces and building
  //=========================================================================
  /**
   *  Remove all nodes, edges and subgraphs of the supergraph 
   */
  virtual  void clear()=0;
  /**
   * Create and return a new SubGraph of the graph
   * The elements of the new SubGraph is those selected in the selectionProxy
   * if there is no selectionProxy an empty SubGraph is return.
   */
  virtual SuperGraph *addSubGraph(SelectionProxy *selectionProxy=0)=0;
  /**
   * Del a SubGraph of this graph.
   * The SubGraph's SubGraphs become SubGraphs of the graph.
   */
  virtual void delSubGraph(SuperGraph *)=0;
  /**
   * Del the SubGraph of the graph and all its SubGraphs.
   */
  virtual void delAllSubGraphs(SuperGraph *)=0;
  /**
   * Return the SuperGraph(UpperGraph of the graph), if it has no SuperGraph it 
   * returns itself
   */
  virtual SuperGraph* getFather()const =0;
  /**
   * Return the root graph of the graph hierarchy
   */
  virtual SuperGraph* getRoot() const =0;
  /**
   * Set the SuperGraph of a graph (use very carefully)
   * Normal user should never use this function.
   */
  virtual void setFather(SuperGraph *)=0;
  /**
   * Return an iterator on all the SubGraphs of the graph
   */
  virtual Iterator<SuperGraph *> * getSubGraphs() const=0;
  //==============================================================================
  // Modification of the graph structure
  //==============================================================================
  /** 
   * Add a new node in the graph and return it. This node is also added in all 
   * the SuperGraph of the graph to maintain the sub_graph relation between graphs.
   */
  virtual node addNode()=0;
  /** 
   * Add an existing node in the graph. this node is also added in all 
   * the SuperGraph of the graph to maintain the sub_graph relation between graphs.
   * Warning, the node must be element of the graph hierarchy, thus it must be 
   * element of the root graph.
   */
  virtual void addNode(const node)=0;
  /**
   * Delete a node in the graph. this node is also removed in all 
   * the sub-graph of the graph to maintain the sub-graph relation between graphs.
   * Warning : One can't add an existing node to the root graph
   */
  virtual void delNode(const node)=0;
  /**
   * Delete a node in all the hierarchy of graphs.
   */
  virtual void delAllNode(const node)=0;
  /** 
   * Add a new edge in the graph and return it. This edge is also added in all 
   * the super-graph of the graph to maintain the sub-graph relation between graphs. 
   * If the second parameter is true the ordering of edges will be preserved.
   */
  virtual edge addEdge(const node, const node )=0;
  /** 
   * Add an existing edge in the graph. this edge is also added in all 
   * the super-graph of the graph to maintain the sub-graph relation between graphs.
   * Warning, the edge must be element of the graph hierarchy, thus it must be 
   * element of the root graph.
   * Warning : One can't add an existing edge to the root graph
   */
  virtual void addEdge(const edge )=0;
  /**
   * Delete an edge in the graph. this edge is also removed in all 
   * the sub-graph of the graph to maintain the sub-graph relation between graphs.
   * The ordering of edges is preserved.
   */
  virtual  void delEdge(const edge )=0;
  /**
   * Delete an edge in all the hierarchy of graphs. 
   * The ordering of edges around the node
   * is preserved.
   */
  virtual  void delAllEdge(const edge)=0;
  /**
   * Set the order of the edges around a node. This operation
   * ensure that adjacent edges of a node will 
   * be odered and consecutive as they are in the vector given in parameter.
   */
  virtual void setEdgeOrder(const node,const std::vector<edge> & )=0;
  /**
   * Swap two edges in the adjacent list of a node.
   */
  virtual void swapEdgeOrder(const node,const edge , const edge )=0;
  /**
   *  Reverse the direction of an edge, the source becomes the target and the target.
   *  becomes the source.
   *  Warning : The ordering is global to the entire graph hierarchy. Thus, by changing
   *  The ordering of a graph you change the ordering of the hierarchy.
   */
  virtual void reverse(const edge)=0;
  //================================================================================
  //Iterators on the graph structure.
  //================================================================================
  /// Return an existing node of the graph
  virtual node getOneNode() const =0;
  ///Return an iterator on the nodes.
  virtual Iterator<node>* getNodes() const =0;
  /// Return the ith successor of a node
  virtual node getInNode(const node,unsigned int )const =0;
  /// Return an iterator on the predecessors of a node. 
  virtual Iterator<node>* getInNodes(const node) const =0;
  /// Return the ith predecessor of a node
  virtual node getOutNode(const node,unsigned int) const =0;
  /// Return an iterator on the successors of a node.
  virtual Iterator<node>* getOutNodes(const node) const =0;
  /// Return an iterator on the neighbours of a node.
  virtual Iterator<node>* getInOutNodes(const node) const =0;
  /// Return an iterator on the edges.
  virtual Iterator<edge>* getEdges() const =0;
  /// Return an existing edge of the graph
  virtual edge getOneEdge() const =0;
  /// Return an iterator on the out-edges of a node.
  virtual Iterator<edge>* getOutEdges(const node) const =0;
  /// Return an iterator on the in-out-edges of a node.
  virtual Iterator<edge>* getInOutEdges(const node) const =0;
  /// Return an iterator on the in--edges of a node.
  virtual Iterator<edge>* getInEdges(const node) const =0;
  //================================================================================
  // Graph, nodes and adges informations about the graph stucture
  //================================================================================
  /// Return the graph's id, this id is unique.
  int getId() const {return id;}
  /// Return the number of nodes in the graph
  virtual unsigned int numberOfNodes()const =0;
  /// Return the number of edges in the graph
  virtual unsigned int numberOfEdges()const =0;
  /// Return degree of a node.
  virtual unsigned int deg(const node)const =0;
  /// Return indegree of a node.
  virtual unsigned int indeg(const node)const =0;
  /// Return outdegree of a node.
  virtual unsigned int outdeg(const node)const =0;
  /// Return the source of the edge.
  virtual node source(const edge)const =0;
  /// Return the target of the edge.
  virtual node target(const edge)const =0;
  /// Return the opposite node for s in the edge e
  virtual node opposite(const edge, const node)const =0;
  /// Return true if the node is element of the graph.
  virtual bool isElement(const node ) const =0;
  /// Return true if the edge is element of the graph.
  virtual bool isElement(const edge ) const =0;
  /** Return the edge if it exists an edge between two node 
   *  sens of the edge is not taken into account)
   *  If no edge is found return an invalid edge.
   */
  virtual edge existEdge(const node , const node) const =0;
  //================================================================================
  // Access to the graph attributes and to the node/edge property.
  //================================================================================
  ///Return graph attributes
  virtual DataSet & getAttributes() =0;
  ///Get an attribute of the graph
  template<typename ATTRIBUTETYPE> 
  ATTRIBUTETYPE getAttribute(const std::string &name);
  ///Set an attribute of the graph
  template<typename ATTRIBUTETYPE> 
  void setAttribute(const std::string &name,const ATTRIBUTETYPE &value);
  /**
   * Return a pointer to a propertyProxy which is in the pool.
   * The real type of the propertyproxy is tested with the template parameter.
   * If the propertyProxy is not in the pool, a new one is created and return.
   */
  template<typename Proxytype> 
  Proxytype* getLocalProperty(const std::string &name);
  /**
   * Compute a property on this graph using an external algorithm (plug-in) 
   * The result is stored in result, Warning all information in result will be deleted
   * If the function return false error message are stored in msg.
   * One can give a PluginProgress to the algortihm in order to have feed back or to stop
   * the algorithm during its computation. One can give parameter to the algorithm 
   * using the DataSet. In some cases algorithms can use this DataSet in order 
   * to give as result external information (not stored in result).
   */
  template<typename Proxytype>
  bool computeProperty(const std::string &algorithm, Proxytype result, std::string &msg, 
		       PluginProgress *progress=0, DataSet *data=0);
  /**
   * Return a pointer to a propertyProxy which is in the pool or in the pool of an ascendant
   * The real type of the propertyproxy is tested with the template parameter.
   * If the propertyProxy is not the pool it creates a new one and return it.
   */
  template<typename Proxytype>
  Proxytype* getProperty(const std::string &name);
  /**
   * Return a pointer to a propertyProxy which is in the pool or in the pool of an ascendant.
   * The real type of the propertyproxy is tested with the template parameter.
   * If the propertyProxy is in the pool the cached value return true else false.
   * The resultBool value indicate if the check algorithm of the associated property was ok;
   * The resultStr is the error message which has been return by the Property.
   */
  /**
   * Return a pointer on an existing property. If the property does not 
   * exist return 0.
   * In DEBUG the existence of a property is checked using an assertion.
   */
  virtual PProxy* getProperty(const std::string &name)=0;  
  /**
   *  Return true if the propertyProxy exists
   *  in the Graph or in a SuperGraph
   */
  virtual  bool existProperty(const std::string&name)=0;
  /**
   * Return true if the propertyProxy is in the graph
   */
  virtual  bool existLocalProperty(const std::string&name)=0;
  /**
   * Remove a PropertyProxy from the graph
   */
  virtual  void delLocalProperty(const std::string&name)=0;
  /**
   * Return an iterator on the local properties
   */
  virtual Iterator<std::string>* getLocalProperties()=0;
  /**
   * Return an iterator on the inherited properties
   */
  virtual Iterator<std::string>* getInheritedProperties()=0;
  /**
   * Return an iterator on the properties
   */
  virtual Iterator<std::string>* getProperties()=0;

private:
  /**
   *  Return a pointer on the porperty proxy container associated to the 
   *  supergraph.
   */
  virtual  PropertyManager *getPropertyManager()=0;

  static IdManager idManager;
  unsigned int id;
};

///Print the graph (only nodes and edges) in ostream, in the tulip format
TLP_SCOPE std::ostream& operator<< (std::ostream &,const SuperGraph *);
/*@}*/
//================================================================================
// Specilization of some template class
//================================================================================
#ifndef DOXYGEN_NOTFOR_DEVEL

namespace stdext {
  template <>
  struct TLP_SCOPE hash<const SuperGraph *> {
    size_t operator()(const SuperGraph *s) const {return size_t(s->getId());}
  };
  template <>
  struct TLP_SCOPE hash<SuperGraph *> {
    size_t operator()(SuperGraph *s) const {return size_t(s->getId());}
  };
}

#endif // DOXYGEN_NOTFOR_DEVEL
//include the template code
#include "cxx/SuperGraph.cxx"
#endif
