//-*-c++-*-
/**
 Authors: David Auber, Romain Bourqui, Patrick Mary
 from the LaBRI Visualization Team
 Email : auber@tulip-software.org
 Last modification : 13/07/2007 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#include <typeinfo>
#include "tulip/GlADComposite.h"
#include "tulip/GlHud.h"

using namespace tlp;
using namespace std;

typedef stdext::hash_map<string, GlAugmentedDisplay *>::const_iterator ITM;
//============================================================
GlADComposite::GlADComposite() {
}
//============================================================
GlADComposite::~GlADComposite() {
  reset(false);
}
//============================================================
void GlADComposite::reset(bool deleteElems) {
  if (deleteElems)
    for(ITM i = elements.begin(); i != elements.end(); ++i)
      delete (*i).second;
  elements.clear();
  _sortedElements.clear();
}
//============================================================
void GlADComposite::addGlAugmentedDisplay(GlAugmentedDisplay *augmentedDisplay, const string &key) {
  elements[key] = augmentedDisplay;
  _sortedElements.push_back(augmentedDisplay);
}
//============================================================
void GlADComposite::deleteGlAugmentedDisplay(const string &key) {
  _sortedElements.remove(elements[key]);
  elements.erase(key);
}
//============================================================
void GlADComposite::deleteGlAugmentedDisplay(GlAugmentedDisplay *augmentedDisplay) {
  for(ITM i = elements.begin(); i != elements.end(); ++i) {
    if(augmentedDisplay == (*i).second) {
      _sortedElements.remove((*i).second);
      elements.erase(i->first);
      return;
    }
  }
}
//============================================================
string GlADComposite::findKey(GlAugmentedDisplay *augmentedDisplay) {
  for(ITM it = elements.begin(); it != elements.end(); ++it) {
    if(augmentedDisplay == (*it).second) {
      return it->first;
    }
  }
  return string("");
}
//============================================================
GlAugmentedDisplay* GlADComposite::findGlAugmentedDisplay(const string &key) {
  ITM ite = elements.find(key);
  if (ite == elements.end())
    return NULL;
  return (*ite).second;
}
//============================================================
void GlADComposite::draw(GlGraph *glGraph) {
  list<GlAugmentedDisplay *>::iterator it;
  for(it = _sortedElements.begin(); it!=_sortedElements.end(); ++it) {
    (*it)->draw(glGraph);
  }
}
//======================================================================
//see header for complete documentation
void GlADComposite::makeAugmentedDisplaySelect (GlGraph *glGraph, 
					        int x, int y, 
						const int w, const int h,
						const Vector<int, 4>& viewport) {
  glMatrixMode(GL_MODELVIEW);
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  list<GlAugmentedDisplay *>::iterator it;
  for(it = _sortedElements.begin(); it!=_sortedElements.end(); ++it) {
    glLoadName((unsigned long)(*it));
    glPushMatrix();
    /*
     * we need to reapply the picking matrix in case of GlHUD
     * because GlHud coordinates use screen coordinates (computed with 
     * initial matrix, set upt initDoselect)
     */
    if (dynamic_cast<GlHud*>(*it) != 0) {
      int xx  = x + w/2;
      int yy  = viewport[3] - (y + h/2);
      Matrix<float, 4> tmp;
      glGetFloatv (GL_PROJECTION_MATRIX, (GLfloat *)&tmp);
      glMatrixMode(GL_PROJECTION);
      glPushMatrix();
      glLoadIdentity();
      gluPickMatrix(xx, yy, w, h, (GLint *)&viewport);
      glMultMatrixf((GLfloat *)&tmp);
      glMatrixMode(GL_MODELVIEW);
    }
    (*it)->draw(glGraph);
    if (dynamic_cast<GlHud*>(*it) != 0) {
      glMatrixMode(GL_PROJECTION);
      glPopMatrix();
      glMatrixMode(GL_MODELVIEW);
    }
    glPopMatrix();
  }
  glPopAttrib();
}
//============================================================
//see headder for complete documentation
bool GlADComposite::doSelect(const int x, const int y, 
			     const int width, const int height,
			     vector <pair <string, GlAugmentedDisplay *> >
			     &pickedAugmentedDisplays,
			     GlGraph *glGraph) {
  if (glGraph == 0) return false;
  //  if (glGraph->getGraph() == 0) return false;
  if (_sortedElements.size() <= 0) return false;
    glGraph->initDoSelect(x , y, width, height, _sortedElements.size());
  makeAugmentedDisplaySelect(glGraph, x, y, width, height, 
			     glGraph->getRenderingParameters().getViewport());
  glFlush();
  GLint hits = glRenderMode(GL_RENDER);
  if (hits <= 0) {
    glGraph->endSelect(); 
    return false;
  }
  while(hits > 0) {
    pair<string, GlAugmentedDisplay *> tmp;
    tmp.second = (GlAugmentedDisplay *)(glGraph->selectBuf[hits-1][3]);
    tmp.first = findKey(tmp.second);
    pickedAugmentedDisplays.push_back (tmp);
    --hits;
  }
  glGraph->endSelect();
  return true;
}
